#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from datetime import timedelta

from CliModel import Dict, Int, Model
from TableOutput import createTable, Format
from Tac import utcNow

class AgentUptimeEntry( Model ):
   restartCount = Int( help="Number of agent restarts" )
   agentStartTime = Int( help="Agent start time in UTC seconds since epoch" )

class AgentUptime( Model ):
   agents = Dict( keyType=str, valueType=AgentUptimeEntry,
                  help="Mapping from Agent name to Agent restart statistics" )

   def render( self ):
      headings = ( "Agent Name", "Restarts", "Uptime" )
      fl = Format( justify='left' )
      fr = Format( justify='right' )
      table = createTable( headings )
      table.formatColumns( fl, fr, fr )
      # Secondary key: sort by the agent name
      agentList = sorted( self.agents.items(), key=lambda k: k[ 0 ] )
      # Primary key: reverse sort by the number of restarts
      agentList = sorted( agentList, key=lambda k: k[ 1 ].restartCount,
                          reverse=True )
      for agent, stats in agentList:
         table.newRow( agent, stats.restartCount,
                       timedelta( seconds=int( utcNow() ) - stats.agentStartTime ) )

      print table.output()

class AgentOomScoresEntry( Model ):
   rssAnon = Int( help="Agent resident anonymous memory size" )
   vmSize = Int( help="Agent virtual memory size" )
   maxVmSize = Int( help="Agent peak virtual memory size" )
   oomScore = Int( help="Out-of-memory score of agent: -1000 (least likely to be a "
                        "victim when system needs to find memory) to 1000" )
   oomScoreAdj = Int( help="Adjustment up or down (1000 to -1000) to apply to the "
                           "computed OOM-score of the agent in order to make it more"
                           " or less likely to become a target/victim of "
                           "out-of-memory killer)" )

# the optimal unit e.g. _humanReadableFormat( 168963795964 ) ---> '157.4GB'
# Input num should be in KB
def _humanReadableFormat( num, suffix='B' ):
   # Convert to bytes first
   num = num * 1024
   for unit in ['','K','M','G','T','P','E','Z']:
      if abs(num) < 1024.0:
         return "%3.1f%s%s" % (num, unit, suffix)
      num /= 1024.0
   return "%.1f%s%s" % (num, 'Y', suffix)

class AgentOomScores( Model ):
   agents = Dict( keyType=str, valueType=AgentOomScoresEntry,
                  help="Mapping from Agent name to Agent OOM scores information" )

   def render( self ):
      headings = ( "Agent Name", "RssAnon", "VmSize", "Max VmSize", "oom_score",
                   "oom_score_adj" )
      fl = Format( justify='left' )
      fr = Format( justify='right' )
      table = createTable( headings )
      table.formatColumns( fl, fr, fr )
      # Secondary key: sort by the agent name
      agentList = sorted( self.agents.items(), key=lambda k: k[ 0 ] )
      # Primary key: reverse sort by oom_score
      agentList = sorted( agentList, key=lambda k: k[ 1 ].oomScore,
                          reverse=True )
      for agent, stats in agentList:
         rssAnonDisp = _humanReadableFormat( stats.rssAnon )
         vmSizeDisp = _humanReadableFormat( stats.vmSize )
         maxVmSizeDisp = _humanReadableFormat( stats.maxVmSize )
         table.newRow( agent, rssAnonDisp, vmSizeDisp, maxVmSizeDisp, stats.oomScore,
                       stats.oomScoreAdj )

      print table.output()
