# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import List
from CliModel import Model
from CliModel import Dict
from CliModel import Int
from CliModel import Str
from CliModel import Bool
from CliModel import Float
from TableOutput import createTable, Format
import operator

class ShowAgentNames( Model ):
   agentNames = List ( valueType=str, 
                       help="List of Agent names in Sysdb's launcher config" )

   def render( self ):
      for agentName in self.agentNames:
         print agentName

def boolToStr( b ):
   return 'enabled' if b else 'disabled'

class StateAndLevel( Model ):
   levels = List( valueType=str,
                   help="List of levels for which tracing has been enabled" )

class ShowTraceAgent( Model ):
   facilities = Dict( keyType=str, valueType=StateAndLevel,
                      help="A mapping of facility name "
                           "to its trace settings" )
   showDate = Bool( help="Date in trace messages", default=False )
   showTraceTimestamp = Bool( help="Timestamp in trace messages", default=False )
   showPid = Bool( help="Pid in trace messages", default=False )
   showTid = Bool( help="Tid in trace messages", default=False )
   showFacilityName = Bool( help="Facility name in trace messages", default=False )
   showTracelevel = Bool( help="Trace level in trace messages", default=False  )
   showFileName = Bool( help="File name in trace messages", default=False )
   showLineNumber = Bool( help="Line number in trace messages", default=False )
   showFunctionName = Bool( help="Method name in trace messages", default=False)
   filenameWidth = Int( help="Set the width of file name in the trace output",
                        optional=True )
   facilityNameWidth = Int( help="Set the width of facility name "
                                 "in the trace output", optional=True )
   functionNameWidth = Int( help="Set the width of function name "
                                 "in the trace output", optional=True )
   _agent = Str( help="Name of the agent" )
   destination = Str( help="File name where trace output goes if not using "
                           "the standard log file", optional=True )
   configTrace = List( valueType=str, help="Trace setting for agent "
                                           "when agent is not running" )
   _agentConfig = Bool( default=True, help="Detailed trace information if True" )

   def render( self ):
      print "Global trace settings for agent %s" % self._agent
      print "-----------------------------------------------"
      if not self._agentConfig:
         if self.destination:
            print "Tracing sent to %s" % self.destination
         if self.configTrace:
            print
            print "Trace facility settings for agent %s is %s " % ( self._agent,
                                ','.join( self.configTrace ) )
         return
      print "Tracing sent to %s" % self.destination
      print
      print "date:     %s" % boolToStr( self.showDate )
      print "time:     %s" % boolToStr( self.showTraceTimestamp )
      print "PID:           %s" % boolToStr( self.showPid )
      print "TID:           %s" % boolToStr( self.showTid )
      print "facility name: %-8s (width %d)" % ( boolToStr( self.showFacilityName ),
                                  self.facilityNameWidth )
      print "trace level:   %s" % boolToStr( self.showTracelevel )
      print "filename:      %-8s (width %d)" % ( boolToStr( self.showFileName ),
                                  self.filenameWidth )
      print "line number:   %s" % boolToStr( self.showLineNumber )
      print "function name: %-8s (width %d)" % ( boolToStr( self.showFunctionName ),
                                  self.functionNameWidth )
      print
      print "-----------------------------------------------"
      for facility, stateLevel in sorted ( self.facilities.iteritems() ):
         print "%-20s enabled  %s" % ( facility, ''.join( stateLevel.levels ) )

class ShowAgentWarmRestartEntry( Model ):
   windowLength = Float( help="Window length in seconds for warm restarts" )
   allowedCount = Int( help="Number of warm restarts allowed" )

class ShowAgentWarmRestartTable( Model ):
   agents = Dict( keyType=str, valueType=ShowAgentWarmRestartEntry,
                  help="Mapping from Agent name to Agent warm restart "
                        "configurations" )

   def render( self ):
      headings = ( "Agent Name", "Allowed", "Window" )
      fl = Format( justify='left' )
      fr = Format( justify='right' )
      table = createTable( headings, tableWidth=60 )
      table.formatColumns( fl, fr, fr )
      agentList = sorted( self.agents.items(), key=operator.itemgetter( 0 ) )
      for agent, config in agentList:
         table.newRow( agent, config.allowedCount, config.windowLength )

      print table.output()
