#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import AgentCommandRequest
import ArPyUtils
import BasicCli
import Cell
import CliCommand
import CliMatcher
import LazyMount
import ShowCommand
from CliPlugin.AegisCli import aegisTrafficPolicyGuard
from CliToken.Platform import platformMatcherForShow

policiesStatus = None

def chipNamesFunc( _ ):
   chipNames = {}
   for tpStatus in policiesStatus.itervalues():
      for chipName in tpStatus.chipNameToId.keys():
         assert chipName not in chipNames
         chipNames[ chipName ] = ''
   return chipNames

def chipNameToId( chipName ):
   for tpStatus in policiesStatus.itervalues():
      if chipName in tpStatus.chipNameToId:
         return tpStatus.chipNameToId[ chipName ]
   return None

def doShowPlatformAegisAlgo( mode, args ):
   if 'CHIP' in args:
      chips = [ args[ 'CHIP' ] ]
   else:
      chips = ArPyUtils.naturalsorted( chipNamesFunc( None ) )

   cmds = [] # [ ( cmdTemplate, asyncCommand ) ]
   if 'transforms' in args:
      if 'prefix' in args:
         transforms = [ 'prefix' ]
      elif 'port' in args:
         transforms = [ 'port' ]
      else:
         transforms = [ 'prefix', 'port' ]

      if 'prefix' in transforms:
         cmds += [ ( 'dumpPdConfigPrefixXform chipId=%d', True ) ]
      if 'port' in transforms:
         cmds += [ ( 'dumpPdConfigPortXform chipId=%d', False ) ]
   elif 'rules' in args:
      cmds += [ ( 'dumpPdConfigRules chipId=%d', True ) ]
   elif 'summary' in args:
      cmds += [ ( 'summary chipId=%d', False ) ]
   else:
      assert False, str( args ) + ' is not a valid combination of args'

   for chipName in chips:
      print( chipName )
      chipId = chipNameToId( chipName )
      if chipId is None:
         print( 'warning: %s could not be mapped to a chipId' % chipName )
         continue

      for cmdTemplate, asyncCommand in cmds:
         AgentCommandRequest.runSocketCommand( mode.entityManager,
                                               "aegis",
                                               "Aegis",
                                               cmdTemplate % chipId,
                                               asyncCommand=asyncCommand )


#------------------------------------------------------------------------
# show platform aegisalgo [ chip CHIP ]
#                         ( ( transforms [ prefix | port ] ) | rules | summary )
#------------------------------------------------------------------------
aegisalgo = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher( 'aegisalgo', helpdesc='Aegis ACL' ),
   guard=aegisTrafficPolicyGuard )

class ShowPlatformAegisAlgo( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform aegisalgo [ chip CHIP ]' + \
            '( ( transforms [ prefix | port ] ) | rules | summary )'
   data = {
      'platform' : platformMatcherForShow,
      'aegisalgo' : aegisalgo,
      'chip' : 'select a chip',
      'CHIP' : CliMatcher.DynamicKeywordMatcher( chipNamesFunc ),
      'transforms' : 'show transforms',
      'prefix' : 'show prefix transforms',
      'port' : 'show port transforms',
      'rules' : 'show rules',
      'summary' : 'show summary',
   }

   handler = doShowPlatformAegisAlgo
   privileged = True

BasicCli.addShowCommandClass( ShowPlatformAegisAlgo )

#-------------------------------------------------------------------------------
# Mount the needed sysdb state
#-------------------------------------------------------------------------------
def Plugin( em ):
   global policiesStatus
   policiesCellStatusPath = 'cell/%d/trafficPolicies/status' % Cell.cellId()
   policiesStatusType = 'Tac::Dir'
   policiesStatus = LazyMount.mount( em, policiesCellStatusPath, policiesStatusType,
                                     'ri' )
