# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentDirectory
import IntfCli
import Cell
import CliCommand
import CliMatcher
import CliParser
import ConfigMount
import LazyMount
import PolicyMapCliLib
from Toggles.AegisToggleLib import toggleEgressAegisEnabled

policiesCliConfig = None
intfConfigIngress = None
intfConfigEgress = None
policiesStatusRequestDir = None
policiesStatus = None
entityManager = None

def aegisTrafficPolicyGuard( mode, token ):
   if AgentDirectory.agent( mode.sysname, 'SandAegis' ):
      return None
   return CliParser.guardNotThisPlatform

# Put traffic-policy token behind guard
trafficPolicyToken = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher( 'traffic-policy',
                                      helpdesc='Apply a traffic-policy' ),
   guard=aegisTrafficPolicyGuard )

def getTrafficPolicyNames():
   return policiesCliConfig.pmapType.pmap

#----------------------------------------------------------------------
# Modelet under Intf mode for traffic-policy command
#----------------------------------------------------------------------
class IntfAegisModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( mode.intf.name.startswith( 'Ethernet' ) or
               mode.intf.name.startswith( 'Port-Channel' ) )

IntfCli.IntfConfigMode.addModelet( IntfAegisModelet )

#----------------------------------------------------------------------
# [ no|default ] traffic-policy (input | output) [ TRAFFIC_POLICY ]
#----------------------------------------------------------------------
class TrafficPolicyAegis( CliCommand.CliCommandClass ):
   syntax = "traffic-policy input TRAFFIC_POLICY"
   noOrDefaultSyntax = "traffic-policy input [ TRAFFIC_POLICY ]"
   data = {
      'traffic-policy' : trafficPolicyToken,
      'input' : 'Assign traffic-policy to the input of an interface',
      'TRAFFIC_POLICY' : CliMatcher.DynamicNameMatcher(
         lambda mode : getTrafficPolicyNames(),
         "Traffic Policy Name" )
   }

   @staticmethod
   def _handleServicePolicy( no, mode, args ):
      trafficPolicyName = args.get( 'TRAFFIC_POLICY' )
      PolicyMapCliLib.handleServicePolicy( mode, no, trafficPolicyName,
                                           policiesCliConfig,
                                           policiesStatusRequestDir,
                                           policiesStatus, None,
                                           intfConfig=intfConfigIngress,
                                           configSessionRollbackSupported=True )

   @staticmethod
   def handler( mode, args ):
      # handle policy application
      TrafficPolicyAegis._handleServicePolicy( False, mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # handle policy removal
      TrafficPolicyAegis._handleServicePolicy( True, mode, args )

#----------------------------------------------------------------------
# [ no|default ] traffic-policy output [ TRAFFIC_POLICY ]
#----------------------------------------------------------------------
class TrafficPolicyAegisEgress( CliCommand.CliCommandClass ):
   syntax = "traffic-policy output TRAFFIC_POLICY"
   noOrDefaultSyntax = "traffic-policy output [ TRAFFIC_POLICY ]"
   data = {
      'traffic-policy' : trafficPolicyToken,
      'output' : 'Assign traffic-policy to the output of an interface',
      'TRAFFIC_POLICY' : CliMatcher.DynamicNameMatcher(
         lambda mode : getTrafficPolicyNames(),
         "Traffic Policy Name" )
   }

   @staticmethod
   def _handleServicePolicy( no, mode, args ):
      trafficPolicyName = args.get( 'TRAFFIC_POLICY' )
      PolicyMapCliLib.handleServicePolicy( mode, no, trafficPolicyName,
                                           policiesCliConfig,
                                           policiesStatusRequestDir,
                                           policiesStatus, None,
                                           intfConfig=intfConfigEgress,
                                           configSessionRollbackSupported=True )

   @staticmethod
   def handler( mode, args ):
      # handle policy application
      TrafficPolicyAegisEgress._handleServicePolicy( False, mode, args )
      assert intfConfigEgress.intf

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # handle policy removal
      TrafficPolicyAegisEgress._handleServicePolicy( True, mode, args )

IntfAegisModelet.addCommandClass( TrafficPolicyAegis )

if toggleEgressAegisEnabled():
   IntfAegisModelet.addCommandClass( TrafficPolicyAegisEgress )

#----------------------------------------------------------------------
# Support for default interface command
#----------------------------------------------------------------------
class AegisIntfJanitor( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      if self.intf_.name in intfConfigIngress.intf:
         del intfConfigIngress.intf[ self.intf_.name ]
      if self.intf_.name in intfConfigIngress.intfFallback:
         del intfConfigIngress.intfFallback[ self.intf_.name ]

      if self.intf_.name in intfConfigEgress.intf:
         del intfConfigEgress.intf[ self.intf_.name ]
      if self.intf_.name in intfConfigEgress.intfFallback:
         del intfConfigEgress.intfFallback[ self.intf_.name ]

IntfCli.Intf.registerDependentClass( AegisIntfJanitor )

def Plugin( em ):
   global policiesCliConfig, intfConfigIngress, intfConfigEgress
   global policiesStatusRequestDir, policiesStatus
   global entityManager
   policiesCellStatusPath = 'cell/%d/trafficPolicies/status' % Cell.cellId()
   policiesStatusType = 'Tac::Dir'
   policiesCliConfigPath = 'trafficPolicies/input/cli'
   policiesCliConfigType = 'TrafficPolicy::TrafficPolicyConfig'
   policiesStatusRequestPath = 'trafficPolicies/statusRequest'
   policiesStatusRequestDirType = 'PolicyMap::PolicyMapStatusRequestDir'
   intfConfigIngressAegisPath = 'trafficPolicies/intf/input/aegis'
   intfConfigEgressAegisPath = 'trafficPolicies/intf/output/aegis'
   intfConfigType = "PolicyMap::IntfConfig"
   entityManager = em

   mountGroup = entityManager.mountGroup()
   policiesStatus = mountGroup.mount( policiesCellStatusPath, policiesStatusType,
                                      'ri' )
   mountGroup.close( callback=None, blocking=False )

   policiesCliConfig = ConfigMount.mount( entityManager, policiesCliConfigPath,
                                          policiesCliConfigType, 'wi' )
   intfConfigIngress = ConfigMount.mount( entityManager, intfConfigIngressAegisPath,
                                          intfConfigType, 'wi' )
   intfConfigEgress = ConfigMount.mount( entityManager, intfConfigEgressAegisPath,
                                         intfConfigType, 'wi' )
   policiesStatusRequestDir = LazyMount.mount( entityManager,
                                               policiesStatusRequestPath,
                                               policiesStatusRequestDirType,
                                               'w' )
