#!/usr/bin/env python
# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import AclLib
import AclCli as AclCli
import AclCliLib
import CliCommand
import CliCommon
import CliMatcher
import CliParser
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
from CliPlugin.IraIpCli import getNexthopGroupNames, nexthopGroupSupportedGuard
import Ethernet
from eunuchs.in_h import IPPROTO_TCP, IPPROTO_UDP
import MultiRangeRule
import Tac
import Toggles.AclToggleLib

#---------------------------------------------------
# Rules used by the Cli Plugin
#---------------------------------------------------

#---------------------------------------------------
# IP Rules
#---------------------------------------------------
BadPortSpecError = object( )
BadIpLenSpecError = object( )

# protocol takes a keyword and a hidden numeric value
# for example, you can do "permit tcp" or "permit 6"
def ipProtocolMatcher( protoName, protoValue, helpdesc, status=None, validate=None ):
   def supported( mode, token ):
      if status is None or validate is None:
         return None
      if getattr( status, validate ):
         return None
      return CliParser.guardNotThisPlatform
   protoNameMatcher = CliMatcher.KeywordMatcher( protoName, helpdesc=helpdesc )
   protoValueMatcher = CliMatcher.KeywordMatcher( str( protoValue ),
                                                  helpdesc=helpdesc )
   class IpProcotolExpression( CliCommand.CliExpression ):
      expression = '<protoName-%s> | <protoValue-%s>' % ( protoName, protoName )
      data = { '<protoName-%s>' % protoName: CliCommand.Node( protoNameMatcher,
                                                              guard=supported ),
               '<protoValue-%s>' % protoName: CliCommand.Node( protoValueMatcher,
                                                               hidden=True ) }
      @staticmethod
      def adapter( mode, args, argsList ):
         if '<protoName-%s>' % protoName in args:
            args.pop( '<protoName-%s>' % protoName )
            assert 'protocol' not in args
            args[ 'protocol' ] = protoValue
         elif '<protoValue-%s>' % protoName in args:
            args.pop( '<protoValue-%s>' % protoName )
            assert 'protocol' not in args
            args[ 'protocol' ] = protoValue

   return IpProcotolExpression

def innerIpProtocolMatcher( innerProtoName, innerProtoValue, helpdesc ):
   innerProtoNameMatcher = CliMatcher.KeywordMatcher( innerProtoName,
                                                      helpdesc=helpdesc )
   innerProtoValueMatcher = CliMatcher.KeywordMatcher( str( innerProtoValue ),
                                                       helpdesc=helpdesc )
   class InnerIpProcotolExpression( CliCommand.CliExpression ):
      expression = '<innerProtoName-%s> | <innerProtoValue-%s>' % (
         innerProtoName, innerProtoName )
      data = { '<innerProtoName-%s>' % innerProtoName:
               CliCommand.Node( innerProtoNameMatcher ),
               '<innerProtoValue-%s>' % innerProtoName:
               CliCommand.Node( innerProtoValueMatcher, hidden=True ) }
      @staticmethod
      def adapter( mode, args, argsList ):
         if '<innerProtoName-%s>' % innerProtoName in args:
            args.pop( '<innerProtoName-%s>' % innerProtoName )
            assert 'innerProtocol' not in args
            args[ 'innerProtocol' ] = innerProtoValue
         elif '<innerProtoValue-%s>' % innerProtoName in args:
            args.pop( '<innerProtoValue-%s>' % innerProtoName )
            assert 'innerProtocol' not in args
            args[ 'innerProtocol' ] = innerProtoValue

   return InnerIpProcotolExpression

def diProtocolMatcher( protoName, protoValue, helpdesc, status=None, validate=None,
                       alias='protocol' ):
   def validateSupported( mode, token ):
      if not validate:
         return None
      if getattr( status, validate ) or getattr( status, 'dpAclDIParsingSupported' ):
         return None
      return CliParser.guardNotThisPlatform

   class DIProtocolExpression( CliCommand.CliExpression ):
      expression = '<protoName-%s>' % protoName
      data = {
         '<protoName-%s>' % protoName: CliCommand.guardedKeyword( protoName,
            helpdesc=helpdesc, guard=validateSupported )
      }

      @staticmethod
      def adapter( mode, args, argsList ):
         if '<protoName-%s>' % protoName in args:
            args.pop( '<protoName-%s>' % protoName )
            assert alias not in args
            args[ alias ] = protoValue

   return DIProtocolExpression

class DscpMatcher( CliCommand.CliExpressionFactory ):
   def __init__( self, status, aclType ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.status = status
      self.aclType = aclType

   def generate( self, name ): 
      def dscpSupported( mode, token ):
         if self.aclType == 'ip' and self.status.dscpFilterSupported:
            return None
         elif self.aclType == 'ipv6' and self.status.dpAclTcSupported:
            return None
         return CliParser.guardNotThisPlatform

      def dscpMaskSupported( mode, token ):
         if self.aclType == 'ip' and self.status.dpAclDscpMaskSupported:
            return None
         elif self.aclType == 'ipv6' and self.status.dpAclDscpMaskIpv6Supported:
            return None
         return CliParser.guardNotThisPlatform

      dscpKwMatcher = CliCommand.Node( 
                        matcher=CliMatcher.KeywordMatcher(
                           'dscp', helpdesc='Match packets by DSCP value or name' ),
                        guard=dscpSupported,
                        maxMatches=1 )

      dscpValueMatcher = CliCommand.Node(
                           matcher=CliMatcher.IntegerMatcher(
                              0, AclLib.MAX_DSCP, helpdesc='DSCP Value' ),
                           maxMatches=1 )

      dscpNameMatcher = CliCommand.Node(
            matcher=CliMatcher.EnumMatcher(
               { k: v[ 1 ] for k, v in AclCliLib.dscpAclNames.iteritems() } ),
            maxMatches=1 )

      maskHelpdesc = \
            'DSCP mask (1 bits indicate to ignore the corresponding value bit)'
      dscpMaskMatcher = CliCommand.Node(
                           matcher=CliMatcher.IntegerMatcher( 
                              0, 0x3F, helpname='0x00-0x3F', helpdesc=maskHelpdesc ),
                           guard=dscpMaskSupported, 
                           maxMatches=1 )

      class DscpExpression( CliCommand.CliExpression ):
         expression = '''dscp ( DSCP | DSCP_NAME ) [ DSCP_MASK ]'''
         data = { 'dscp': dscpKwMatcher,
                  'DSCP': dscpValueMatcher,
                  'DSCP_NAME': dscpNameMatcher,
                  'DSCP_MASK': dscpMaskMatcher }

         @staticmethod
         def adapter( mode, args, argList ):
            if 'dscp' not in args:
               return
            args.pop( 'dscp' )
            if 'DSCP_NAME' in args:
               args[ 'DSCP' ] = args.pop( 'DSCP_NAME' )
            args[ 'DSCP_MASK' ] = args.pop( 'DSCP_MASK', 0 )

      return DscpExpression 

class IpEcnMatcher( CliCommand.CliExpressionFactory ):
   def __init__( self, status ):
      super( IpEcnMatcher, self ).__init__()

      def validateSupported( mode, token ):
         if getattr( status, 'ecnFilterSupported' ):
            return None
         return CliParser.guardNotThisPlatform

      ecnKeywordMatcher = CliMatcher.KeywordMatcher( 'ecn',
             helpdesc='Match packets by ECN Codepoints' )
      self.ecnNode_ = CliCommand.Node( matcher=ecnKeywordMatcher,
            guard=validateSupported, maxMatches=1, noResult=True )

      ecnOpts = {
         'ect-ce': 'ECN Codepoint ECT(0), ECT(1) or CE',
         'non-ect': 'ECN Codepoint Not-ECT',
         'ce': 'ECN Codepoint CE',
         'ect': 'ECN Codepoint ECT(0) or ECT(1)'
      }
      self.ecnValNode_ = CliCommand.singleNode( CliMatcher.EnumMatcher( ecnOpts ) )

   def generate( self, name ):
      class IpEcnExpression( CliCommand.CliExpression ):
         expression = 'ecn %s' % name
         data = {
            'ecn': self.ecnNode_,
            name : self.ecnValNode_
         }

      return IpEcnExpression

# A rule for specifying source or destination in one of the four forms
# any
# host ipv6-address
# ipv6-address/prefixLength
# ipv6-address ipv6-mask
class Ip6HostMatcher( CliCommand.CliExpressionFactory ):
   def __init__( self, tag, fullMaskValid=False, fullMaskGuard=None, name=None ):
      super( Ip6HostMatcher, self ).__init__()
      if fullMaskGuard:
         assert fullMaskValid
      self.tag_ = tag
      self.name_ = name
      self.fullMaskValid_ = fullMaskValid
      self.fullMaskGuard_ = fullMaskGuard
      self.matcherAnyHost_ = CliMatcher.KeywordMatcher( 'any',
            helpdesc='Any %s IPv6 host' % tag )
      self.matcherHost_ = CliMatcher.KeywordMatcher( 'host',
            helpdesc='A single %s IPv6 host' % tag )
      self.matcherIp6Addr_ = Ip6AddrMatcher.Ip6AddrMatcher(
            '%s IPv6 address' % tag.capitalize() )

   def generate( self, name ):
      if self.name_:
         name = self.name_
      anyName = 'ANY_KW_%s' % name
      hostName = 'HOST_KW_%s' % name
      ipv6AddrName = 'IPV6_ADDR_%s' % name
      matcherIp6FullAddr = Ip6AddrMatcher.ip6AddrWithPrefixOrFullMaskExpr( name,
         self.tag_.capitalize(), self.fullMaskValid_, self.fullMaskGuard_,
         inverseMask=True )
      class Ip6HostExpression( CliCommand.CliExpression ):
         expression = '%s | ANY_KW_%s | ( HOST_KW_%s IPV6_ADDR_%s )' % (
            name, name, name, name )
         data = {
            name : matcherIp6FullAddr,
            anyName : self.matcherAnyHost_,
            hostName : self.matcherHost_,
            ipv6AddrName : self.matcherIp6Addr_,
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            # we are going to put our result into args, so make sure tat
            # we aren't going to override anything important there
            if name in args:
               return
            if anyName in args:
               args.pop( anyName )
               args[ name ] = AclLib.anyIp6AddrWithFullMask
            elif hostName in args:
               args.pop( hostName )
               ip6Addr = args.pop( ipv6AddrName )
               args[ name ] = Arnet.Ip6AddrWithFullMask( ip6Addr.stringValue )

      return Ip6HostExpression

# A rule for specifying source or destination in one of the four forms
# any
# host ip-address
# ip-address inverse-netmask
# ip-address/prefixLength
def ipHostRuleMatcher( tag, name ):
   tagCap = tag.capitalize()

   ipPrefixMatcher = IpAddrMatcher.ipAddrWithFullMaskExpr(
      '%s address' % tagCap,
      '%s wildcard bits' % tagCap,
      '%s IP address with mask length' % tagCap,
      inverseMask=True )

   anyName = "%s_any" % name
   hostName = "%s_host" % name
   hostAddrName = "%s_IPADDR" % name

   class IpHostRuleExpression( CliCommand.CliExpression ):
      # <ipPrefix> | any | host <ipAddr>
      expression = ( '%s | %s | ( %s %s )' %
                     ( name, anyName, hostName, hostAddrName ) )
      data = { name: ipPrefixMatcher,
               anyName: CliMatcher.KeywordMatcher( 'any',
                                                   helpdesc='Any %s host' % tag ),
               hostName: CliMatcher.KeywordMatcher( 'host',
                                     helpdesc='A single %s host' % tag ),
               hostAddrName: IpAddrMatcher.ipAddrMatcher }

      @staticmethod
      def adapter( mode, args, argsList ):
         # we are going to put our result into args, so make sure tat
         # we aren't going to override anything important there
         if anyName in args:
            assert name not in args
            args.pop( anyName )
            args[ name ] = AclLib.anyIpAddr
         elif hostName in args:
            assert name not in args
            ipAddrArg = args.pop( hostAddrName )
            args.pop( hostName )
            args[ name ] = Arnet.AddrWithFullMask( str( ipAddrArg ), 0xffffffff )

   return IpHostRuleExpression

# A special ipHostRule for MplsOverGre which now only supports any any for outer
# ip address
def ipMplsOverGreHostMatcher( tag, name, anyIpAddr ):
   class IpMplsOverGreHostMatcher( CliCommand.CliExpression ):
      expression = '''<mplsAnyKw-%s>''' % tag
      data = { '<mplsAnyKw-%s>' % tag: CliMatcher.KeywordMatcher( 'any',
                                                      helpdesc='Any %s host' % tag )
             }

      @staticmethod
      def adapter( mode, args, argsList ):
         if '<mplsAnyKw-%s>' % tag in args:
            args.pop( '<mplsAnyKw-%s>' % tag )
            args[ name ] = anyIpAddr

   return IpMplsOverGreHostMatcher

# A rule for specifying inner IPv4/IPv6 source or destination
def innerIpHostMatcher( status, name ):
   def validateSupported( mode, token ):
      if getattr( status, 'dpAclInnerIpSupported' ) or \
         getattr( status, 'dpAclMplsOverGreSupported' ):
         return None
      return CliParser.guardNotThisPlatform

   innerKwMatcher = CliMatcher.KeywordMatcher( 'inner', helpdesc='inner qualifier' )
   class InnerIpHostExpression( CliCommand.CliExpression ):
      # inner ( ( ip <innerIpSrcOpt> <innerIpDstOpt> ) |
      #         ( ipv6 innerIp6SrcOpt innerIp6DstOpt) )
      expression = ( '<innerIpInnerKw> ('
                        '( <innerIpIpKw> <innerIpSrcOpt> <innerIpDstOpt> ) |'
                        '( <innerIpIpv6Kw> innerIp6SrcOpt innerIp6DstOpt ) ) ' )
      data = { '<innerIpInnerKw>': CliCommand.Node( innerKwMatcher,
                                                    maxMatches=1,
                                                    guard=validateSupported ),
               '<innerIpIpKw>': CliMatcher.KeywordMatcher( 'ip',
                                                       helpdesc='inner IP address' ),
               '<innerIpSrcOpt>': ipHostRuleMatcher( 'source', 'innerIpSrcOpt' ),
               '<innerIpDstOpt>': ipHostRuleMatcher( 'destination',
                                                     'innerIpDstOpt' ),
               '<innerIpIpv6Kw>': CliMatcher.KeywordMatcher( 'ipv6',
                                                     helpdesc='inner IPv6 address' ),
               'innerIp6SrcOpt': Ip6HostMatcher( 'source', fullMaskValid=False ),
               'innerIp6DstOpt': Ip6HostMatcher( 'destination', fullMaskValid=False )
               }

      @staticmethod
      def adapter( mode, args, argsList ):
         if '<innerIpInnerKw>' not in args:
            return
         args.pop( '<innerIpInnerKw>' )
         if '<innerIpIpKw>' in args:
            args.pop( '<innerIpIpKw>' )
            args[ name ] = ( 'ip', args[ 'innerIpSrcOpt' ],
                                     args[ 'innerIpDstOpt' ] )
            args.pop( 'innerIpSrcOpt' )
            args.pop( 'innerIpDstOpt' )
         elif '<innerIpIpv6Kw>' in args:
            args.pop( '<innerIpIpv6Kw>' )
            innerIp6SrcOpt = args[ 'innerIp6SrcOpt' ]
            innerIp6DstOpt = args[ 'innerIp6DstOpt' ]
            args[ name ] = ( 'ipv6',
                  Arnet.Ip6AddrWithMask( innerIp6SrcOpt.address,
                     innerIp6SrcOpt.prefixLen() ),
                  Arnet.Ip6AddrWithMask( innerIp6DstOpt.address,
                     innerIp6DstOpt.prefixLen() ) )
            args.pop( 'innerIp6SrcOpt' )
            args.pop( 'innerIp6DstOpt' )

   return InnerIpHostExpression

def innerIpHostProtoMatcher( status, name ):
   def validateSupported( mode, token ):
      if status.dpAclInnerProtocolSupported:
         return None
      return CliParser.guardNotThisPlatform

   innerKwMatcher = CliMatcher.KeywordMatcher( 'inner', helpdesc='inner qualifier' )
   class InnerIpHostProtoExpression( CliCommand.CliExpression ):
      expression = ( '<innerKw> ( '
                        '( <innerIp4Kw> <innerIp4Src> <innerIp4Dst> ) '
                      '| ( <innerIp4Kw> <innerTcp> <innerIp4Src> [ <sportTcp> ] '
                                                  '<innerIp4Dst> [ <dportTcp> ] ) '
                      '| ( <innerIp4Kw> <innerUdp> <innerIp4Src> [ <sportUdp> ] '
                                                  '<innerIp4Dst> [ <dportUdp> ] ) '

                      '| ( <innerIp6Kw> innerIp6Src innerIp6Dst ) '
                      '| ( <innerIp6Kw> <innerTcp> innerIp6Src [ <sportTcp> ] '
                                                  'innerIp6Dst [ <dportTcp> ] )'
                      '| ( <innerIp6Kw> <innerUdp> innerIp6Src [ <sportUdp> ] '
                                                  'innerIp6Dst [ <dportUdp> ] )'
                     ')'
                   )
      data = { '<innerKw>': CliCommand.Node( innerKwMatcher,
                                             maxMatches=1,
                                             guard=validateSupported ),
               '<innerIp4Kw>': CliMatcher.KeywordMatcher( 'ip',
                                 helpdesc='inner IP address' ),
               '<innerIp4Src>': ipHostRuleMatcher( 'source', 'innerIp4Src' ),
               '<innerIp4Dst>': ipHostRuleMatcher( 'destination', 'innerIp4Dst' ),
               '<innerIp6Kw>': CliMatcher.KeywordMatcher( 'ipv6',
                                 helpdesc='inner IPv6 address' ),
               'innerIp6Src': Ip6HostMatcher( 'source', fullMaskValid=False ),
               'innerIp6Dst': Ip6HostMatcher( 'destination', fullMaskValid=False ),

               '<innerTcp>' : innerIpProtocolMatcher( 'tcp', IPPROTO_TCP,
                              helpdesc='TCP' ),
               '<sportTcp>' : portSpecMatcher( IPPROTO_TCP, 'src', 'innerTcp',
                                               alias='sportTcp' ),
               '<dportTcp>' : portSpecMatcher( IPPROTO_TCP, 'dst', 'innerTcp',
                                               alias='dportTcp' ),
               '<innerUdp>' : innerIpProtocolMatcher( 'udp', IPPROTO_UDP,
                              helpdesc='UDP' ),
               '<sportUdp>' : portSpecMatcher( IPPROTO_UDP, 'src', 'innerUdp',
                                               alias='sportUdp' ),
               '<dportUdp>' : portSpecMatcher( IPPROTO_UDP, 'dst', 'innerUdp',
                                               alias='dportUdp' ),
              }

      @staticmethod
      def adapter( mode, args, argsList ):
         def portSpecProtocolAssignment( args ):
            if args[ 'innerProtocol' ] == IPPROTO_TCP:
               sportExp = 'sportTcp'
               dportExp = 'dportTcp'
            elif args[ 'innerProtocol' ] == IPPROTO_UDP:
               sportExp = 'sportUdp'
               dportExp = 'dportUdp'
            else:
               assert False

            srcPort = args[ sportExp ] if sportExp in args else None
            dstPort = args[ dportExp ] if dportExp in args else None
            if sportExp in args:
               args.pop( sportExp )
            if dportExp in args:
               args.pop( dportExp )
            return ( srcPort, dstPort )

         if '<innerKw>' not in args:
            return
         args.pop( '<innerKw>' )
         srcIp, dstIp = None, None
         ipVersion = 'ip'
         if '<innerIp4Kw>' in args:
            args.pop( '<innerIp4Kw>' )

            srcIp = args[ 'innerIp4Src' ]
            dstIp = args[ 'innerIp4Dst' ]
            args.pop( 'innerIp4Src' )
            args.pop( 'innerIp4Dst' )
         elif '<innerIp6Kw>' in args:
            args.pop( '<innerIp6Kw>' )
            srcIp = args[ 'innerIp6Src' ]
            dstIp = args[ 'innerIp6Dst' ]
            srcIp = Arnet.Ip6AddrWithMask( srcIp.address, srcIp.prefixLen() )
            dstIp = Arnet.Ip6AddrWithMask( dstIp.address, dstIp.prefixLen() )
            args.pop( 'innerIp6Src' )
            args.pop( 'innerIp6Dst' )
            ipVersion = 'ipv6'
         else:
            mode.addError( "Incorrect inner IP type" )

         if 'innerProtocol' in args:
            srcPort, dstPort = portSpecProtocolAssignment( args )
            args[ name ] = ( ipVersion, args[ 'innerProtocol' ],
                                          srcIp, srcPort, dstIp, dstPort )
            args.pop( 'innerProtocol' )
         else:
            args[ name ] = ( ipVersion, srcIp, dstIp )

   return InnerIpHostProtoExpression

# A rule for specifying vlan(and/or inner vlan) in the form of
# vlan id inverse-mask inner id inverse-mask
def vlanMatcher( status=None, validate='vlanSupported' ):
   def validateSupported( mode, token ):
      if getattr( status, validate ):
         return None
      return CliParser.guardNotThisPlatform

   vlanKeywordMatcher = CliMatcher.KeywordMatcher( 'vlan', helpdesc='Vlan' )
   vlanIntegerMatcher = CliMatcher.IntegerMatcher( 1, 4094,
                                             helpdesc='Match packets by vlan value' )
   vlanMaskIntegerMatcher = CliMatcher.IntegerMatcher( 0, 0xFFF,
                                                       helpname='0x000-0xFFF',
                                                       helpdesc='Vlan mask' )
   innerVlanIntegerMatcher = CliMatcher.IntegerMatcher( 1, 4094,
                                       helpdesc='Match packets by inner vlan value' )
   innerVlanMaskIntegerMatcher = CliMatcher.IntegerMatcher( 0, 0xFFF,
                                                        helpname='0x000-0xFFF',
                                                        helpdesc='Inner vlan mask' )

   class VlanVlanMaskExpression( CliCommand.CliExpression ):
      expression = '''<vlan> <vlanMask>'''
      data = { '<vlan>': vlanIntegerMatcher,
               '<vlanMask>': vlanMaskIntegerMatcher }

   class InnerVlanExpression( CliCommand.CliExpression ):
      expression = '''inner <innerVlan> <innerVlanMask>'''
      data = { 'inner': 'Inner vlan',
               '<innerVlan>': innerVlanIntegerMatcher,
               '<innerVlanMask>': innerVlanMaskIntegerMatcher }

   class VlanExpression( CliCommand.CliExpression ):
      expression = ( 'vlan ( ( <vlanVlanMaskSyntax> [ <innerVlanSyntax> ] ) | '
                     '<innerVlanSyntax> )' )
      data = { 'vlan': CliCommand.Node( matcher=vlanKeywordMatcher,
                                        guard=validateSupported ),
               '<vlanVlanMaskSyntax>': VlanVlanMaskExpression,
               '<innerVlanSyntax>': InnerVlanExpression }

      @staticmethod
      def adapter( mode, args, argsList ):
         if 'vlan' not in args:
            if 'vlanSpec' not in args:
               args[ 'vlanSpec' ] = None
            return
         vlan = args.get( '<vlan>', 0 )
         vlanMask = args.get( '<vlanMask>', 0xFFF ) ^ 0xFFF
         innerVlan = args.get( '<innerVlan>', 0 )
         innerVlanMask = args.get( '<innerVlanMask>', 0xFFF ) ^ 0xFFF
         args.pop( 'vlan' )
         if '<vlan>' in args:
            args.pop( '<vlan>' )
            args.pop( '<vlanMask>' )
         if 'inner' in args:
            args.pop( 'inner' )
            args.pop( '<innerVlan>' )
            args.pop( '<innerVlanMask>' )
         args[ 'vlanSpec' ] = ( vlan, vlanMask, innerVlan, innerVlanMask )

   return VlanExpression

# A rule for specifying mpls label (and/or inner label). This is an old command used
# in older platforms (e.g. Alta)
def mplsTwoLabelMatcher( status=None, validate='dpAclMplsTwoLabelSupported' ):
   def validateSupported( mode, token ):
      if getattr( status, validate ) or \
            getattr( status, 'dpAclDIParsingSupported' ):
         return None
      return CliParser.guardNotThisPlatform

   class MplsTwoLabelExpression( CliCommand.CliExpression ):
      expression = ( 'LABEL_KW ( LABEL MASK [ INNER_KW INNER_LABEL INNER_MASK ] ) | '
                            '( INNER_KW INNER_LABEL INNER_MASK )' )
      data = { 
               'LABEL_KW': CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( 'label',
                     helpdesc='MPLS outer label' ),
                  guard=validateSupported ),
               'INNER_KW': CliMatcher.KeywordMatcher( 'inner',
                     helpdesc='Inner label' ),
               'LABEL': CliMatcher.IntegerMatcher( 0, ( 2 ** 20 ) - 1,
                     helpdesc='Match packets by label value' ),
               'INNER_LABEL': CliMatcher.IntegerMatcher( 0, ( 2 ** 20 ) - 1,
                     helpdesc='Match packets by inner label value' ),
               'MASK': CliMatcher.IntegerMatcher( 0, 0xFFFFF,
                     helpdesc='Label mask' ),
               'INNER_MASK': CliMatcher.IntegerMatcher( 0, 0xFFFFF,
                     helpdesc='Inner label mask' )
             }

      @staticmethod
      def adapter( mode, args, argsList ):
         if 'LABEL_KW' not in args:
            return
         assert 'mplsTwoLabelSpec' not in args
         args.pop( 'LABEL_KW' )
         args.pop( 'INNER_KW', None )
         label = args.pop( 'LABEL', 0 )
         labelMask = args.pop( 'MASK', AclLib.MPLSLABEL_MAX )
         innerLabel = args.pop( 'INNER_LABEL', 0 )
         innerLabelMask = args.pop( 'INNER_MASK', AclLib.MPLSLABEL_MAX )
         mplsTwoLabelSpec = ( label, labelMask ^ AclLib.MPLSLABEL_MAX,
                              innerLabel, innerLabelMask ^ AclLib.MPLSLABEL_MAX )
         args[ 'mplsTwoLabelSpec' ] = mplsTwoLabelSpec

   return MplsTwoLabelExpression

# A rule for specifying mpls filter
def mplsFilterMatcher( status=None ):
   def validateSupported( mode, token ):
      if not getattr( status, 'dpAclMplsSupported' ):
         return CliParser.guardNotThisPlatform
      return None

   class MplsFilterExpression( CliCommand.CliExpression ):
      expression = '{ label LABEL_INDEX LABEL_VALUE LABEL_MASK } [ bos BOS ]'
      data = {
               'label': CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( 'label',
                     helpdesc='MPLS label' ),
                  guard=validateSupported,
                  maxMatches=3 ),
               'LABEL_INDEX': CliMatcher.IntegerMatcher( 1, 3,
                  helpdesc='Label index (1 is top of stack)' ),
               'LABEL_VALUE': CliMatcher.IntegerMatcher( 0, ( 2 ** 20 ) - 1,
                  helpdesc='Value of the MPLS label' ),
               'LABEL_MASK': CliMatcher.IntegerMatcher( 0, 0xFFFFF,
                  helpname='0x00000-0xFFFFF', helpdesc='Label mask' ),
               'bos': 'MPLS BOS (Bottom Of Stack) setting',
               'BOS': CliMatcher.IntegerMatcher( 0, 1,
                  helpdesc='BOS of the largest specified label' )
             }

      @staticmethod
      def adapter( mode, args, argsList ):
         if 'label' not in args:
            return
         assert 'mplsFilter' not in args
         args.pop( 'label' )
         args.pop( 'bos', None )
         bos = args.pop( 'BOS', Tac.Type( 'Acl::MplsBosEnum' ).bosDontCare )
         mplsFilter = AclLib.MplsFilterValue( bos=bos )
         labelIndexes = args.pop( 'LABEL_INDEX' )
         labelValues = args.pop( 'LABEL_VALUE' )
         labelMasks = args.pop( 'LABEL_MASK' )
         for counter, index in enumerate( labelIndexes ):
            if mplsFilter.label.get( index ):
               # Making sure the label is defined only once
               raise CliParser.InvalidInputError( 'Multiple entries for label with'
                                                  ' index %d' % index )
            labelValue = labelValues[ counter ]
            labelMask = labelMasks[ counter ] ^ AclLib.MPLSLABEL_MAX
            value = AclLib.MplsLabelFilterValue( value=labelValue, mask=labelMask )
            mplsFilter.label[ index ] = value
         args[ 'mplsFilter' ] = mplsFilter

   return MplsFilterExpression

patternMatcher = CliMatcher.KeywordMatcher( 'pattern',
                                            helpdesc='Payload pattern mask' )
payloadPatternMatcher = CliMatcher.IntegerMatcher( 0, 0xFFFFFFFF,
                                                   helpname='0x00000000-0xFFFFFFFF',
                                                   helpdesc='Word pattern to match' )
maskMatcher = CliMatcher.KeywordMatcher( 'mask',
                                         helpdesc='Payload pattern mask' )
payloadMaskMatcher = CliMatcher.IntegerMatcher( 0, 0xFFFFFFFF,
                                                helpname='0x00000000-0xFFFFFFFF',
                                                helpdesc='Word pattern mask' )
def offsetMatcher( macFilter, status=None ):
   if status is None:
      status = AclCli.status
   def offsetRangeFn( mode ):
      maxRange = 0
      if macFilter:
         maxRange = status.dpAclPayloadMaxL2Index
      else:
         maxRange = status.dpAclPayloadMaxL4Index
      return ( 0 , maxRange )

   offsetDynamicMatcher = CliMatcher.DynamicIntegerMatcher( rangeFn=offsetRangeFn,
                      helpdesc='Offsets of 32 bits from the start of DPI window' )
   class OffsetExpression( CliCommand.CliExpression ):
      # offset <offset> pattern <pattern> mask <mask>
      expression = ( '<offsetKw> <payload-offset> '
                     '<patternKw-offset> <payload-pattern> '
                     '<maskKw-offset> <payload-mask>' )
      data = { '<offsetKw>': CliMatcher.KeywordMatcher( 'offset',
                             helpdesc='Payload offset' ),
               '<payload-offset>': offsetDynamicMatcher,
               '<patternKw-offset>': patternMatcher,
               '<payload-pattern>': payloadPatternMatcher,
               '<maskKw-offset>': maskMatcher,
               '<payload-mask>': payloadMaskMatcher }

      @staticmethod
      def adapter( mode, args, argsList ):
         if '<offsetKw>' not in args:
            return
         results = []

         offsets = args[ '<payload-offset>' ]
         patterns = args[ '<payload-pattern>' ]
         patternMasks = args[ '<payload-mask>' ]
         
         if isinstance( offsets, list ):
            for i, _ in enumerate( offsets ):
               patterMask = patternMasks[ i ] ^ 0xFFFFFFFF # inverse mask
               results.append( AclLib.PayloadValue( offsets[ i ],
                                                    patterns[ i ],
                                                    patterMask ) )
         else:
            patterMask = patternMasks ^ 0xFFFFFFFF # inverse mask
            results.append( AclLib.PayloadValue( offsets,
                                                 patterns,
                                                 patterMask ) )

         args.pop( '<offsetKw>' )
         args.pop( '<payload-offset>' )
         args.pop( '<patternKw-offset>' )
         args.pop( '<payload-pattern>' )
         args.pop( '<maskKw-offset>' )
         args.pop( '<payload-mask>' )
         args[ '<payloadOffsets>' ] = results

   return OffsetExpression

class AliasNameMatcher( CliCommand.CliExpressionFactory ):
   def __init__( self, macFilter=False ):
      super( AliasNameMatcher, self ).__init__()
      self.macFilter_ = macFilter

   def generate( self, name ):
      if AclCli.udfAlias:
         aliasColl = ( AclCli.udfAlias.macAlias
            if self.macFilter_ else AclCli.udfAlias.ipAlias )
      else:
         aliasColl = []

      class AliasNameExpression( CliCommand.CliExpression ):
         expression = '''alias ALIAS'''
         data = {
            'alias': 'Alias name for payload',
            'ALIAS' : CliMatcher.DynamicNameMatcher( lambda mode: aliasColl,
               helpdesc='Alias Name' )
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            if 'alias' not in args:
               return
            args.pop( 'alias' )
            args[ name ] = args.pop( 'ALIAS' )

      return AliasNameExpression

def payloadAliasMatch( macFilter ):
   nodeNameTypeMaping = { 'ALIAS': 'aliasName',
                          '<alias-pattern>': 'pattern',
                          '<alias-mask>': 'mask' }
   class PayloadAliasExpression( CliCommand.CliExpression ):
      expression = ( 'ALIAS '
      '[ ( <patternKw-alias> <alias-pattern> [ <maskKw-alias> <alias-mask> ] ) |'
      '  ( <maskKw-alias> <alias-mask> [ <patternKw-alias> <alias-pattern> ] ) ]' )
      data = { 'ALIAS': AliasNameMatcher( macFilter ),
               '<patternKw-alias>': patternMatcher,
               '<alias-pattern>': payloadPatternMatcher,
               '<maskKw-alias>': maskMatcher,
               '<alias-mask>': payloadMaskMatcher }

      @staticmethod
      def adapter( mode, args, argsList ):
         if 'ALIAS' not in args:
            return
         currAlias = None
         aliases = {}
         for nodeName, nodeResult in argsList:
            if nodeName not in nodeNameTypeMaping:
               continue
            typ = nodeNameTypeMaping[ nodeName ]
            if typ == 'aliasName':
               currAlias = nodeResult
               aliases[ currAlias ] = {}
            else:
               assert typ not in aliases[ currAlias ]
               aliases[ currAlias ][ typ ] = nodeResult
         for i in ( 'ALIAS', '<patternKw-alias>',
                    '<alias-pattern>', '<maskKw-alias>', '<alias-mask>' ):
            if i in args:
               args.pop( i )
         result = []
         for alias, typs in aliases.iteritems():
            subresult = [ ( typ, value ) for typ, value in typs.iteritems() ]
            subresult.append( ( 'aliasName', alias ) )
            result.append( subresult )
         args[ '<offsetAliases>' ] = result

   return PayloadAliasExpression

def nodePayloadFactory( status=None, ipv6Filter=False, macFilter=False,
      validate='dpAclPayloadSupported' ):

   def validateSupported( mode, token ):

      _status = AclCli.status if status is None else status

      if getattr( _status, validate ) or \
         ( getattr( _status, 'dpAclDIParsingSupported' ) and \
            not ipv6Filter ) or \
         ( getattr( _status, 'dpAclIp6DIParsingSupported' ) and \
            ipv6Filter ):
         return None
      return CliParser.guardNotThisPlatform

   return CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'payload',
            helpdesc='Match packets based on payload' ),
         guard=validateSupported )

matcherStartOrEnd = CliMatcher.EnumMatcher( {
   'start': 'offset from start of header',
   'end': 'offset from end of header',
} )




def payloadMatcher( status=None, ipv6Filter=False, macFilter=False,
                    validate='dpAclPayloadSupported' ):
   nodePayload = nodePayloadFactory( status=status, ipv6Filter=ipv6Filter,
         macFilter=macFilter, validate=validate )

   def handleIpAlias( mode, udfIterRule=None, startOrEnd=None ):
      return handleAlias( mode, udfIterRule=udfIterRule, startOrEnd=startOrEnd,
                           aliasType='ip' )

   def handleMacAlias( mode, udfIterRule=None ):
      return handleAlias( mode, udfIterRule=udfIterRule, aliasType='mac' )

   def handleAlias( mode, udfIterRule=None, startOrEnd=None, aliasType='ip' ):
      if aliasType == 'mac':
         udfAliasColl = AclCli.udfAlias.macAlias
      else:
         udfAliasColl = AclCli.udfAlias.ipAlias
      payloadValueList = []
      uniqueOffsets = set()
      for entry in udfIterRule:
         if isinstance( entry, list ):
            aliasName = None
            pattern = None
            mask = None
            patternOverride = False
            maskOverride = False
            for ( key, value ) in entry:
               if key == 'aliasName':
                  aliasName = value
               elif key == 'pattern':
                  pattern = value
                  patternOverride = True
               elif key == 'mask':
                  mask = value ^ 0xFFFFFFFF
                  maskOverride = True
            if aliasName not in udfAliasColl:
               raise CliCommon.InvalidInputError( 'Alias does not exit' )
            aliasPayload = udfAliasColl[ aliasName ].payload
            if not pattern:
               pattern = aliasPayload.pattern
            if not mask:
               mask = aliasPayload.mask
            payload = AclLib.PayloadValue( aliasPayload.offset, pattern, mask )
            payload.alias = aliasName
            payload.patternOverride = patternOverride
            payload.maskOverride = maskOverride
            offset = payload.offset
            if startOrEnd is None:
               startOrEnd = udfAliasColl[ aliasName ].headerStart
            if startOrEnd != udfAliasColl[ aliasName ].headerStart:
               raise CliCommon.InvalidInputError( "Header start/end mismatch" )
            payloadValueList.append( payload )
         else:
            offset = entry.offset
            payloadValueList.append( entry )
         if offset in uniqueOffsets:
            mode.addWarning( 'Only one entry per offset will be saved' )
         else:
            uniqueOffsets.add( offset )
      return handle( mode, payloadValueList, startOrEnd=startOrEnd )

   def handle( mode, offsetList, startOrEnd=None ):
      # Currently, 'header' and 'start' or 'end' keywords are only
      # supported for IP. If they are to be supported for mac, this code will
      # need to move and become aware of the type of ACL.
      if startOrEnd != 'start':
         if getattr( status, 'dpAclPayloadL4HeaderEndReducesOffsets' ):
            #ip header is 20 bytes, so reduce 4 offsets
            maxOffsets = status.dpAclPayloadMaxL4Index - 5
            for offset in offsetList:
               if status.dpAclPayloadL4HeaderEndReducesOffsets\
                     and ( offset.offset >  maxOffsets ):
                  mode.addError( "Only %d offsets allowed from end of header" %
                                 maxOffsets )
                  raise CliCommon.AlreadyHandledError( )
      payloadSpec = AclLib.payloadOptToPayloadSpec( offsetList, startOrEnd )
      return payloadSpec

   def adapterFunc( mode, args, argsList ):
      def maxIteration():
         index = ( status.dpAclPayloadMaxL2Index if macFilter
                   else status.dpAclPayloadMaxL4Index )
         return index + 1

      if 'payload' not in args:
         return
      args.pop( 'payload' )
      if 'HEADER' in args:
         startOrEnd = args.pop( 'HEADER' )
      else:
         startOrEnd = None

      udfIterRule = []
      if '<payloadOffsets>' in args:
         udfIterRule.extend( args[ '<payloadOffsets>' ] )
         args.pop( '<payloadOffsets>' )
      if '<offsetAliases>' in args:
         udfIterRule.extend( args[ '<offsetAliases>' ] )
         args.pop( '<offsetAliases>' )

      if not macFilter:
         payload = handleIpAlias( mode, udfIterRule=udfIterRule,
               startOrEnd=startOrEnd )
      else:
         payload = handleMacAlias( mode, udfIterRule=udfIterRule )
      if len( udfIterRule ) > maxIteration():
         raise CliCommon.InvalidInputError( 'Too many payload entries' )
      args[ 'payloadOpt' ] = payload

   class PayloadExpression( CliCommand.CliExpression ):
      expression = ( 'payload [ header HEADER ] '
                     '{ OFFSET_EXPR | <aliasExpression> }' )
      data = { 'payload': nodePayload,
               'header': 'start of offset relative to IP header',
               'HEADER': matcherStartOrEnd,
               'OFFSET_EXPR': offsetMatcher( macFilter, status ),
               '<aliasExpression>': payloadAliasMatch( macFilter ) }
      adapter = adapterFunc

   class MacPayloadExpression( CliCommand.CliExpression ):
      expression = ( 'payload { OFFSET_EXPR | <aliasExpression> }' )
      data = { 'payload': nodePayload,
               'OFFSET_EXPR': offsetMatcher( macFilter, status ),
               '<aliasExpression>': payloadAliasMatch( macFilter ) }
      adapter = adapterFunc

   return PayloadExpression if not macFilter else MacPayloadExpression

class MetadataOffsetExprFactory( CliCommand.CliExpressionFactory ):
   def __init__( self, status=None, validate='dpAclMetadataSupported' ):
      if status is None:
         status = AclCli.status
      self.status = status
      self.validate = validate
      CliCommand.CliExpressionFactory.__init__( self )
   
   def generate( self, name ):
      def metadataSupported( mode, token ):
         if getattr( self.status, self.validate ):
            return None
         return CliParser.guardNotThisPlatform 

      metadataKwMatcher = CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'metadata',
            helpdesc='Match packets based on packet internal attributes' ),
         guard=metadataSupported )
      metadataSourceMatcher = CliMatcher.EnumMatcher( {
         '1' : 'Offset from start of metadata1 buffer',
         '2' : 'Offset from start of metadata2 buffer' } )
      metadataOffsetMatcher = CliMatcher.IntegerMatcher( 0, 255,
            helpdesc='Offsets of 32 bits from the start of DPI window' )
      metadataPatternMatcher = CliMatcher.IntegerMatcher( 0, 0xFFFFFFFF,
            helpname='0x00000000-0xFFFFFFFF',
            helpdesc='Word pattern to match' )
      metadataMaskMatcher = CliMatcher.IntegerMatcher( 0, 0xFFFFFFFF,
            helpname='0x00000000-0xFFFFFFFF',
            helpdesc='Word pattern mask' )
      offsetKw = CliMatcher.KeywordMatcher( 'offset', helpdesc='Metadata offset' )
      patternKw = CliMatcher.KeywordMatcher( 'pattern', helpdesc='Metadata pattern' )
      maskKw = CliMatcher.KeywordMatcher( 'mask', helpdesc='Metadata mask' )
      
      class MetadataOffsetExpression( CliCommand.CliExpression ):
         expression = '''{ METADATA_KW METADATA_SOURCE OFFSET_KW METADATA_OFFSET
                         PATTERN_KW METADATA_PATTERN MASK_KW METADATA_MASK }'''
         data = { 'METADATA_KW' : metadataKwMatcher,
                  'METADATA_SOURCE' : metadataSourceMatcher,
                  'OFFSET_KW' : offsetKw,
                  'METADATA_OFFSET' : metadataOffsetMatcher,
                  'PATTERN_KW' : patternKw,
                  'METADATA_PATTERN' : metadataPatternMatcher,
                  'MASK_KW' : maskKw,
                  'METADATA_MASK' : metadataMaskMatcher }
         
         @staticmethod
         def adapter( mode, args, argList ):
            if 'METADATA_KW' not in args:
               return
            
            args.pop( 'METADATA_KW' )
            args.pop( 'OFFSET_KW' )
            args.pop( 'PATTERN_KW' )
            args.pop( 'MASK_KW' )
            sources = args.pop( 'METADATA_SOURCE' )
            offsets = args.pop( 'METADATA_OFFSET' )
            patterns = args.pop( 'METADATA_PATTERN' )
            masks = args.pop( 'METADATA_MASK' )
           
            metadataList = []
            for i, _ in enumerate( sources ):
               metadataSource = \
                     'metadataLsb' if sources[ i ] == '1' else 'metadataMsb' 
               metadataList.append( AclLib.MetadataValue( offsets[ i ],
                                                          patterns[ i ],
                                                          masks[ i ] ^ 0xFFFFFFFF,
                                                          metadataSource ) )
            args[ 'metadataSpec' ] = \
                  AclLib.metadataListToMetadataSpec( metadataList )
      
      return MetadataOffsetExpression

class UdfAliasMatcher( CliCommand.CliExpressionFactory ):
   def __init__( self, status, macFilter=False ):
      super( UdfAliasMatcher, self ).__init__()
      self.status_ = status
      self.macFilter_ = macFilter

   def generate( self, name ):
      nodePayload = nodePayloadFactory( status=self.status_,
            macFilter=self.macFilter_ )

      def adapterFunc( mode, args, argsList ):
         if 'payload' not in args:
            return

         result = {}
         args.pop( 'payload' )
         if 'HEADER' in args:
            args.pop( 'header' )
            result[ 'headerStart' ] = args.pop( 'HEADER' )
         else:
            result[ 'headerStart' ] = 'end'
         result[ 'payload' ] = args.pop( '<payloadOffsets>' )[ 0 ]
         result[ 'payload' ].alias = args.pop( 'ALIAS' )

         args[ name ] = result

      if not self.macFilter_:
         class _UdfAliasExpression( CliCommand.CliExpression ):
            expression = 'payload ALIAS [ header HEADER ] OFFSET_EXPR'
            data = {
               'payload': nodePayload,
               'ALIAS': AliasNameMatcher( self.macFilter_ ),
               'header': 'start of offset relative to IP header',
               'HEADER': matcherStartOrEnd,
               'OFFSET_EXPR': offsetMatcher( self.macFilter_, self.status_ )
            }
            adapter = adapterFunc
         return _UdfAliasExpression
      else:
         class _UdfMacAliasExpression( CliCommand.CliExpression ):
            expression = 'payload ALIAS OFFSET_EXPR'
            data = {
               'payload': nodePayload,
               'ALIAS': AliasNameMatcher( self.macFilter_ ),
               'OFFSET_EXPR': offsetMatcher( self.macFilter_, self.status_ )
            }
            adapter = adapterFunc
         return _UdfMacAliasExpression

class tcpFlagMaskMatcher( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      
      def tcpNotFlagsEnabledGuard( mode, token ):
         TcpNotFlagsEnabled = Toggles.AclToggleLib.toggleTcpNotFlagsEnabled()
         if TcpNotFlagsEnabled:
            return None
         else:
            return CliParser.guardNotThisEosVersion

      tcpTokens = { t: 'Match on the %s bit' % t for t in AclLib.tcpFlagTokens }
      
      notFlagsKeywordMatcher = CliMatcher.KeywordMatcher( 'not-flags',
                               helpdesc='Match on packets with the' \
                                        + 'specified tcp flags set to 0' )

      class _TcpFlagsExpression( CliCommand.CliExpression ):
         expression = 'not-flags { NOT_MATCH_FLAGS }'
         data = {
                  'not-flags': CliCommand.Node( matcher=notFlagsKeywordMatcher,
                                                guard=tcpNotFlagsEnabledGuard,
                                                hidden=True,
                                                maxMatches=1 ),
                  'NOT_MATCH_FLAGS': CliMatcher.EnumMatcher( tcpTokens )
                }

         @staticmethod
         def adapter( mode, args, argList ):
            if 'not-flags' not in args:
               return

            tcpFlagMask = 0
            for tcpFlag, tcpFlagValue in AclLib.tcpFlagTokens.iteritems():
               if tcpFlag not in args[ 'NOT_MATCH_FLAGS' ]:
                  continue
               tcpFlagMask |= tcpFlagValue
            
            del args[ 'not-flags' ]
            del args[ 'NOT_MATCH_FLAGS' ]
            args[ 'TCP_FLAGS_MASK' ] = AclLib.TcpFlagValue( tcpFlagMask )

      return _TcpFlagsExpression

def userL4Matcher( status=None, validate='dpAclUserL4Supported' ):
   def validateSupported( mode, token ):
      if getattr( status, validate ) or getattr( status, 'dpAclDIParsingSupported' ):
         return None
      return CliParser.guardNotThisPlatform

   userL4KeywordMatcher = CliMatcher.KeywordMatcher( 'userl4',
                       helpdesc='Match packets based on first word after L3 header' )
   rangeMatcher = CliMatcher.IntegerMatcher( 0, 0xFFFFFFFF,
                                             helpname='0x00000000-0xFFFFFFFF',
                                             helpdesc='Word pattern to match' )
   class UserL4Expression( CliCommand.CliExpression ):
      expression = 'userl4 <userL4> mask <userL4Mask>'
      data = { 'userl4': CliCommand.Node( matcher=userL4KeywordMatcher,
                                          guard=validateSupported,
                                          maxMatches=1 ),
               '<userL4>': rangeMatcher,
               'mask': 'Pattern mask',
               '<userL4Mask>': rangeMatcher }
      @staticmethod
      def adapter( mode, args, argsList ):
         if 'userl4' not in args:
            return
         userL4 = args[ '<userL4>' ]
         userL4Mask = args[ '<userL4Mask>' ]
         args[ 'userL4Value' ] = ( userL4, ( userL4Mask ^ 0xFFFFFFFF ) )
         args.pop( 'userl4' )
         args.pop( '<userL4>' )
         args.pop( 'mask' )
         args.pop( '<userL4Mask>' )

   return UserL4Expression

def gtpVersionProtoMatcher( status, validate='dpAclGtpSupported' ):
   def validateSupported( mode, token ):
      if not validate:
         return None
      if getattr( status, validate ) or getattr( status, 'dpAclDIParsingSupported' ):
         return None
      return CliParser.guardNotThisPlatform


   versionMatcher = CliMatcher.KeywordMatcher( 'version',
                                         helpdesc='GPRS Tunneling Protocol version' )
   versionNode = CliCommand.Node( matcher=versionMatcher,
                                  guard=validateSupported )
   servmap = AclLib.gtpProtoByName
   protoOpts = { s: port[ 1 ] for s, port in servmap.iteritems( ) }

   class GtpVersionProtoExpression( CliCommand.CliExpression ):
      expression = '''<gtpVersionKw> <gtpVersion> <gtpProtocolKw> <gtpProto>'''
      data = {  '<gtpVersionKw>': versionNode,
                '<gtpVersion>': CliMatcher.IntegerMatcher( 1, 2,
                                        helpdesc='GPRS Tunneling Protocol version' ),
                '<gtpProtocolKw>': CliMatcher.KeywordMatcher( 'protocol',
                                                      helpdesc='GTP protocol type' ),
                '<gtpProto>': CliMatcher.EnumMatcher( protoOpts ),
            }

      @staticmethod
      def adapter( mode, args, argsList ):
         if '<gtpVersionKw>' not in args:
            return
         gtpProto = AclLib.gtpProtoByName[ args[ '<gtpProto>' ] ][ 0 ]
         args[ 'gtpVersionProto' ] = ( args[ '<gtpVersion>' ], gtpProto )
         args.pop( '<gtpVersionKw>' )
         args.pop( '<gtpProtocolKw>' )
         args.pop( '<gtpVersion>' )
         args.pop( '<gtpProto>' )

   return GtpVersionProtoExpression

class IpLenSpecMatcher( CliCommand.CliExpressionFactory ):
   def __init__( self, status ):
      super( IpLenSpecMatcher, self ).__init__()
      self.status_ = status

   def _validateSupported( self, mode, token ):
      if getattr( self.status_, 'allowIpLenFilter' ):
         return None
      return CliParser.guardNotThisPlatform

   def generate( self, name ):
      ipLenKeywordMatcher = CliMatcher.KeywordMatcher( 'ip-length',
            helpdesc='Match IP packet length' )
      ipLenMatcher = CliMatcher.IntegerMatcher( 0, AclLib.MAX_IP_PACKET_LEN,
            helpdesc='IP packet length' )

      class IpLenSpecExpression( CliCommand.CliExpression ):
         expression = ( 'ip-length ( ( <ipLenEqKw> { <ipLenEq> } ) | '
                        '( <ipLenGtKw> <ipLenGt> ) | '
                        '( <ipLenLtKw> <ipLenLt> ) | '
                        '( <ipLenRangeKw> <ipLenRangeMin> <ipLenRangeMax> ) )' )
         data = {
            'ip-length':  CliCommand.singleNode( matcher=ipLenKeywordMatcher,
               guard=self._validateSupported ),
            '<ipLenEqKw>': CliMatcher.KeywordMatcher( 'eq',
               helpdesc='Match IP packet length' ),
            '<ipLenEq>': CliCommand.Node( matcher=ipLenMatcher,
               maxMatches=10 ),
            '<ipLenGtKw>': CliMatcher.KeywordMatcher( 'gt',
                    helpdesc='Match IP packet lengths greater than this number' ),
            '<ipLenGt>': ipLenMatcher,
            '<ipLenLtKw>': CliMatcher.KeywordMatcher( 'lt',
                       helpdesc='Match IP packet lengths less than this number' ),
            '<ipLenLt>': CliMatcher.IntegerMatcher( 1, AclLib.MAX_IP_PACKET_LEN,
                                                    helpdesc='IP packet length' ),
            '<ipLenRangeKw>': CliMatcher.KeywordMatcher( 'range',
                              helpdesc='Match IP packet lengths within a range' ),
            '<ipLenRangeMin>': ipLenMatcher,
            '<ipLenRangeMax>': ipLenMatcher
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            if 'ip-length' not in args:
               return
            args.pop( 'ip-length' )

            ipLen = None
            if '<ipLenEqKw>' in args:
               ipLens = ' '.join( [ str( i ) for i in args[ '<ipLenEq>' ] ] )
               ipLen = AclLib.IpLenValue( 'eq', ipLens )
               args.pop( '<ipLenEqKw>' )
               args.pop( '<ipLenEq>' )
            elif '<ipLenGtKw>' in args:
               ipLen = AclLib.IpLenValue( 'gt', str( args[ '<ipLenGt>' ][ 0 ] ) )
               args.pop( '<ipLenGtKw>' )
               args.pop( '<ipLenGt>' )
            elif '<ipLenLtKw>' in args:
               ipLen = AclLib.IpLenValue( 'lt', str( args[ '<ipLenLt>' ][ 0 ] ) )
               args.pop( '<ipLenLtKw>' )
               args.pop( '<ipLenLt>' )
            elif '<ipLenRangeKw>' in args:
               iplenMin = args[ '<ipLenRangeMin>' ][ 0 ]
               ipLenHigh = args[ '<ipLenRangeMax>' ][ 0 ]
               args.pop( '<ipLenRangeKw>' )
               args.pop( '<ipLenRangeMin>' )
               args.pop( '<ipLenRangeMax>' )
               if iplenMin > ipLenHigh:
                  mode.addError( "Invalid IP packet length range %d, %d" %
                                 ( iplenMin, ipLenHigh ) )
                  ipLen = BadIpLenSpecError
               else:
                  ipLen = AclLib.IpLenValue( 'range', '%d %d' %
                                             ( iplenMin, ipLenHigh ) )
            assert ipLen is not None
            assert name not in args
            args[ name ] = ipLen

      return IpLenSpecExpression

def portNumberMatcher( proto, tag, protoName, typ, low=0, high=65535, maxMatches=1 ):
   servmap = AclLib.getServiceMap( proto )
   portNameOpts = { s: "%s (%d)" % ( port[ 1 ], port[ 0 ] )
                    for s, port in servmap.iteritems( ) }
   portNameMatcher = CliMatcher.EnumMatcher( portNameOpts )
   prefix = '%s_%s_%s' % ( protoName, tag, typ )
   portNum = '%s_PORTNUM' % prefix
   portName = '%s_PORTNAME' % prefix
   sharedMatchObj = object()
   class PortNumberExpression( CliCommand.CliExpression ):
      expression = '''{ %s | %s }''' % ( portNum, portName )
      data = { portNum: CliCommand.Node(
                           matcher=CliMatcher.IntegerMatcher( low, high,
                              helpdesc='Number of the port to use' ),
                           sharedMatchObj=sharedMatchObj, maxMatches=maxMatches ),
               portName: CliCommand.Node( matcher=portNameMatcher,
                           sharedMatchObj=sharedMatchObj, maxMatches=maxMatches ) }
      @staticmethod
      def adapter( mode, args, argsList ):
         if portNum in args or portName in args:
            argName = '%s-%s-portSpec-port' % ( protoName, tag )
            if argName not in args:
               args[ argName ] = []
            if portNum in args:
               if maxMatches == 1:
                  args[ argName ].append( int( args.pop( portNum ) ) )
               else:
                  for port in args.pop( portNum ):
                     args[ argName ].append( int( port ) )

            if portName in args:
               if maxMatches == 1:
                  args[ argName ].append( servmap[ args.pop( portName ) ][ 0 ] )
               else:
                  for port in args.pop( portName ):
                     args[ argName ].append( servmap[ port ][ 0 ] )

   return PortNumberExpression

def portSpecMatcher( proto, tag, protoName, guard=None, alias=None ):
   prefix = '%s-%s-portSpec' % ( protoName, tag )
   assert alias is not None
   eqKwMatcher = CliMatcher.KeywordMatcher( 'eq',
                            helpdesc='Match one or more port numbers (up to 10)' )
   neqKwMatcher = CliMatcher.KeywordMatcher( 'neq',
                    helpdesc='Match ports not equal to these numbers (up to 10)' )
   gtKwMatcher = CliMatcher.KeywordMatcher( 'gt',
                                 helpdesc='Match ports greater than this number' )
   ltKwMatcher = CliMatcher.KeywordMatcher( 'lt',
                                   helpdesc='Match ports lower than this number' )
   rangeKwMatcher = CliMatcher.KeywordMatcher( 'range',
                                  helpdesc='Match ports within two port numbers' )
   operAlias = '%s-oper' % prefix
   eqKwNode = CliCommand.Node( matcher=eqKwMatcher, guard=guard, alias=operAlias )
   neqKwNode = CliCommand.Node( matcher=neqKwMatcher, guard=guard, alias=operAlias )
   gtKwNode = CliCommand.Node( matcher=gtKwMatcher, guard=guard, alias=operAlias )
   ltKwNode = CliCommand.Node( matcher=ltKwMatcher, guard=guard, alias=operAlias )
   rangeKwNode = CliCommand.Node( matcher=rangeKwMatcher, guard=guard,
                                  alias=operAlias )
   class PortSpecExpression( CliCommand.CliExpression ):
      expression = ( '( <%s-eqKw> <eqPortMatcher> ) |'
                     '( <%s-neqKw> <neqPortMatcher> ) |'
                     '( <%s-gtKw> <gtPortMatcher> ) |'
                     '( <%s-ltKw> <ltPortMatcher> ) |'
                     '( <%s-rangeKw> <lPortMatcher> <uPortMatcher> )' ) % (
                     prefix, prefix, prefix, prefix, prefix )
      data = { '<%s-eqKw>' % prefix: eqKwNode,
               '<eqPortMatcher>':
                            portNumberMatcher( proto, tag, protoName, 'eq',
                               maxMatches=10 ),
               '<%s-neqKw>' % prefix: neqKwNode,
               '<neqPortMatcher>':
                           portNumberMatcher( proto, tag, protoName, 'neq',
                              maxMatches=10 ),
               '<%s-gtKw>' % prefix: gtKwNode,
               '<gtPortMatcher>': portNumberMatcher( proto, tag, protoName, 'gt',
                                                     high=65534 ),
               '<%s-ltKw>' % prefix: ltKwNode,
               '<ltPortMatcher>': portNumberMatcher( proto, tag, protoName, 'lt',
                                                     low=1 ),
               '<%s-rangeKw>' % prefix: rangeKwNode,
               '<lPortMatcher>': portNumberMatcher( proto, tag, protoName, 'l' ),
               '<uPortMatcher>': portNumberMatcher( proto, tag, protoName, 'u' ) }
      @staticmethod
      def adapter( mode, args, argsList ):
         if operAlias not in args:
            return
         oper = args[ operAlias ]
         ports = args[ '%s-%s-portSpec-port' % ( protoName, tag ) ]
         if oper == 'range':
            if ports[ 1 ] > ports[ 0 ]:
               mode.addError( "Invalid port range %d, %d" % ( ports[ 1 ],
                                                              ports[ 0 ] ) )
               args[ alias ] = BadPortSpecError
            else:
               ports = '%s %s' % ( ports[ 1 ], ports[ 0 ] )
               args[ alias ] = AclLib.PortValue( oper, ports )
         else:
            ports = ' '.join( [ str( i ) for i in sorted( ports ) ] )
            args[ alias ] = AclLib.PortValue( oper, ports )
         args.pop( operAlias )
         args.pop( '%s-%s-portSpec-port' % ( protoName, tag ) )

   return PortSpecExpression

def tcpOptionsMatcher():
   tcpNodes = {}
   for token in AclLib.tcpFlagTokens:
      matcher = CliMatcher.KeywordMatcher( token,
                                     helpdesc='Match on the %s bit' % token.upper() )
      tcpNodes[ token ] = CliCommand.Node( matcher=matcher, maxMatches=1 )
   establishedMatcher = CliMatcher.KeywordMatcher( 'established',
                               helpdesc='Match packets in existing TCP connections' )

   class TcpOptionsExpression( CliCommand.CliExpression ):
      expression = ( '{ established | fin | syn | psh | urg } |'
                     '{ fin | syn | rst | psh | ack | urg }' )
      data = { 'established': CliCommand.Node( matcher=establishedMatcher,
                                               maxMatches=1 ),
               'fin': tcpNodes[ 'fin' ],
               'syn': tcpNodes[ 'syn' ],
               'rst': tcpNodes[ 'rst' ],
               'psh': tcpNodes[ 'psh' ],
               'ack': tcpNodes[ 'ack' ],
               'urg': tcpNodes[ 'urg' ] }
      @staticmethod
      def adapter( mode, args, argsList ):
         if 'tcpOptions' in args:
            return
         flags = 0
         for tcpFlag, tcpFlagValue in AclLib.tcpFlagTokens.iteritems():
            if tcpFlag not in args:
               continue
            flags |= tcpFlagValue
            args.pop( tcpFlag )
         if 'established' in args:
            args.pop( 'established' )
            established = True
         else:
            established = False
         args[ 'tcpOptions' ] = ( AclLib.TcpFlagValue( flags ), established )

   return TcpOptionsExpression

ttlMatcher = CliMatcher.KeywordMatcher( 'ttl',
                                        helpdesc='Match TTL (Time-to-Live) value' )
class TtlSpecExpression( CliCommand.CliExpression ):
   expression = ( '<ttlKw> ( ( eq <eqTtl> ) |'
                            '( neq <neqTtl> ) |'
                            '( gt <gtTtl> ) |'
                            '( lt <ltTtl> ) )' )
   data = { '<ttlKw>': CliCommand.Node( matcher=ttlMatcher, maxMatches=1 ),
            'eq': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'eq',
                                               helpdesc='Match a single TTL value' ),
                                   alias='<ttlOper>' ),
            '<eqTtl>': CliCommand.Node( CliMatcher.IntegerMatcher( 0, 255,
                                          helpdesc='TTL value' ), alias='<ttlVal>' ),
            'neq': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'neq',
                                      helpdesc='Match TTL not equal to this value' ),
                                    alias='<ttlOper>' ),
            '<neqTtl>': CliCommand.Node( CliMatcher.IntegerMatcher( 0, 255,
                                          helpdesc='TTL value' ), alias='<ttlVal>' ),
            'gt': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'gt',
                                     helpdesc='Match TTL greater than this number' ),
                                    alias='<ttlOper>' ),
            '<gtTtl>': CliCommand.Node( CliMatcher.IntegerMatcher( 0, 254,
                                          helpdesc='TTL value' ), alias='<ttlVal>' ),
            'lt': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'lt',
                                       helpdesc='Match TTL lower than this number' ),
                                    alias='<ttlOper>' ),
            '<ltTtl>': CliCommand.Node( CliMatcher.IntegerMatcher( 1, 255,
                                         helpdesc='TTL value' ), alias='<ttlVal>' ) }

   @staticmethod
   def adapter( mode, args, argsList ):
      if '<ttlKw>' not in args:
         return
      args.pop( '<ttlKw>' )
      oper = args[ '<ttlOper>' ][ 0 ]
      ttl = args[ '<ttlVal>' ][ 0 ]
      args.pop( '<ttlOper>' )
      args.pop( '<ttlVal>' )
      args[ 'ttl' ] = AclLib.TtlValue( oper, ttl )

#---------------------------------------------------
# IPv6 Rules
#---------------------------------------------------
class HopLimitSpecMatcher( CliCommand.CliExpressionFactory ):
   def __init__( self ):
      super( HopLimitSpecMatcher, self ).__init__()
      self.matcherHopLimitOperEq_ = CliMatcher.KeywordMatcher( 'eq',
            helpdesc='Match a single Hop Limit value' )
      self.matcherHopLimitOperNeq_ = CliMatcher.KeywordMatcher( 'neq',
            helpdesc='Match Hop Limit not equal to this value' )
      self.matcherHopLimitOperGt_ = CliMatcher.KeywordMatcher( 'gt',
            helpdesc='Match Hop Limit greater than this number' )
      self.matcherHopLimitOperLt_ = CliMatcher.KeywordMatcher( 'lt',
            helpdesc='Match Hop Limit lower than this number' )
      self.hopLimitMatcher_ = CliMatcher.IntegerMatcher( 0, 255,
            helpdesc='Hop limit value' )
      self.hopLimitGtMatcher_ = CliMatcher.IntegerMatcher( 0, 254,
            helpdesc='Hop limit value' )
      self.hopLimitLtNumberMatcher_ = CliMatcher.IntegerMatcher( 1, 255,
            helpdesc='Hop limit value' )

   def generate( self, name ):
      operName = '%s_OPER' % name
      hopLimitValName = '%s_VAL' % name

      class HopLimitSpecExpression( CliCommand.CliExpression ):
         expression = ( 'hop-limit ( ( %s_EQ %s_EQ_VAL ) |'
                                    '( %s_NEQ %s_NEQ_VAL ) |'
                                    '( %s_GT %s_GT_VAL ) |'
                                    '( %s_LT %s_LT_VAL ) )' % (
                          name, name, name, name,
                          name, name, name, name ) )
         data = {
            'hop-limit' : CliCommand.singleKeyword( 'hop-limit',
               helpdesc='Match Hop Limit value' ),
            '%s_EQ' % name: CliCommand.Node( matcher=self.matcherHopLimitOperEq_,
               alias=operName ),
            '%s_EQ_VAL' % name: CliCommand.Node( matcher=self.hopLimitMatcher_,
               alias=hopLimitValName ),
            '%s_NEQ' % name: CliCommand.Node( matcher=self.matcherHopLimitOperNeq_,
               alias=operName ),
            '%s_NEQ_VAL' % name: CliCommand.Node( matcher=self.hopLimitMatcher_,
               alias=hopLimitValName ),
            '%s_GT' % name: CliCommand.Node( matcher=self.matcherHopLimitOperGt_,
               alias=operName ),
            '%s_GT_VAL' % name: CliCommand.Node( matcher=self.hopLimitGtMatcher_,
               alias=hopLimitValName ),
            '%s_LT' % name: CliCommand.Node( matcher=self.matcherHopLimitOperLt_,
               alias=operName ),
            '%s_LT_VAL' % name: CliCommand.Node(
               matcher=self.hopLimitLtNumberMatcher_,
               alias=hopLimitValName )
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            if 'hop-limit' not in args:
               return
            args.pop( 'hop-limit' )
            oper = args.pop( operName )[ 0 ]
            hoplimit = args.pop( hopLimitValName )[ 0 ]
            args[ name ] = AclLib.TtlValue( oper, hoplimit )

      return HopLimitSpecExpression

#---------------------------------------------------
# MAC Rules
#---------------------------------------------------
# An expression for specifying source or destination in one of the four forms
# any
# mac-address inverse-mac-mask
#
# Returns: a ( addr, mask ) tuple
def macHostRuleMatcher( tag ):
   tagCap = tag.capitalize()
   tagUpper = tag.upper()

   macAddrName = 'MAC_ADDRESS_%s' % tagUpper
   macMaskName = 'MAC_MASK_%s' % tagUpper
   anyName = 'ANY_KW_%s' % tagUpper
   macAddrMatcher = CliMatcher.PatternMatcher(
      Ethernet.macAddrPattern, helpname='H.H.H',
      helpdesc='MAC address of the %s' %  tag )
   macMaskMatcher = CliMatcher.PatternMatcher(
      Ethernet.macAddrPattern, helpname='H.H.H',
      helpdesc='%s MAC wildcard bits' % tagCap )

   class MacHostRuleExpression( CliCommand.CliExpression ):
      expression = '( %s %s ) | %s' % ( macAddrName, macMaskName, anyName )
      data = {
               macAddrName: macAddrMatcher,
               macMaskName: macMaskMatcher,
               anyName: CliMatcher.KeywordMatcher( 'any',
                  helpdesc='Any %s address' % tag ),
             }

      @staticmethod
      def adapter( mode, args, argsList ):
         # we are going to put our result into args, so make sure tat
         # we aren't going to override anything important there
         if macAddrName in args:
            assert tag not in args
            macMask = args.pop( macMaskName )
            macAddr = args.pop( macAddrName )
            macMask = AclLib.inverseMacMask(
               Ethernet.convertMacAddrToCanonical( macMask ) )
            if macMask == AclLib.zeroMacAddrString:
               macAddr = AclLib.zeroMacAddrString
            else:
               macAddr = Ethernet.convertMacAddrToCanonical( macAddr )
            args[ tag ] = ( macAddr, macMask )
         elif anyName in args:
            assert tag not in args
            args.pop( anyName )
            args[ tag ] = ( AclLib.zeroMacAddrString, AclLib.zeroMacAddrString )

   return MacHostRuleExpression

#---------------------------------------------------
# Action Rules
#---------------------------------------------------
def aclMirrorSessionNameFn( mode ):
   if AclCli.gv.mirrorSessionNameFn:
      return AclCli.gv.mirrorSessionNameFn( mode )
   return {}

class MirrorSessionMatcher( CliCommand.CliExpressionFactory ):
   def __init__( self, status, aclType ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.status = status
      self.aclType = aclType

      self.mirrorKeywordMatcher = CliMatcher.KeywordMatcher(
            'mirror',
            helpdesc='Mirror matches against this rule' )

      excludeNames = r'^(?!detail$|default$|summary$)([A-Za-z0-9_:{}\[\]-]+)'

      self.sessionNameMatcher = CliMatcher.DynamicNameMatcher(
         aclMirrorSessionNameFn,
         helpdesc='Session name',
         pattern=excludeNames )

   def mirrorActionSupported( self, mode, token ):
      if self.aclType == 'ip' and self.status.dpIpAclMirrorActionSupported:
         return None
      if self.aclType == 'mac' and self.status.dpMacAclMirrorActionSupported:
         return None
      if self.aclType == 'ipv6' and self.status.dpIpv6AclMirrorActionSupported:
         return None
      return CliParser.guardNotThisPlatform

   def generate( self, name ):
      class MirrorSessionExpression( CliCommand.CliExpression ):
         expression = 'mirror session SESSION_NAME'
         data = { 'mirror' : CliCommand.Node( matcher=self.mirrorKeywordMatcher,
                                              guard=self.mirrorActionSupported,
                                              maxMatches=1 ),
                  'session' : 'Mirror session',
                  'SESSION_NAME' : CliCommand.Node( matcher=self.sessionNameMatcher,
                                                    maxMatches=1 ) }

         @staticmethod
         def adapter( mode, args, argsList ):
            args[ 'mirrorSession' ] = args.pop( 'SESSION_NAME', None )
            args.pop( 'mirror', None )
            args.pop( 'session', None )

      return MirrorSessionExpression

def _getCopyToCpuAction( mode, context ):
   copyActions = {}
   if context.sharedResult[ 'ACTION' ] == 'deny':
      copyActions[ 'copy' ] = 'Copy packet to CPU matches against this rule'
   return copyActions

class CopyToCpuDstMatcher( CliCommand.CliExpressionFactory ):
   def __init__( self, status, aclType, sharedMatchObj ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.status = status
      self.aclType = aclType
      self.sharedMatchObj = sharedMatchObj

      self.copyKeywordMatcher = CliMatcher.DynamicKeywordMatcher( 
            _getCopyToCpuAction, passContext=True )
      self.copyToCpuDstMatcher = CliMatcher.EnumMatcher( {
            'captive-portal' : 'Copy packet to CPU queue for dot1x captive-portal'
            } )

   def copyToCpuActionSupported( self, mode, token ):
      if ( Toggles.AclToggleLib.toggleDpAclCopyToCpuEnabled() and
           self.status.dpIngressCopyToCpuSupported and
           self.aclType in [ 'ip', 'ipv6' ] ):
         return None
      return CliParser.guardNotThisPlatform

   def generate( self, name ):
      class CopyToCpuDstExpression( CliCommand.CliExpression ):
         expression = 'copy COPY_CPU_DST'
         data = { 'copy' : CliCommand.Node( matcher=self.copyKeywordMatcher,
                                            guard=self.copyToCpuActionSupported,
                                            maxMatches=1,
                                            sharedMatchObj=self.sharedMatchObj ),
                  'COPY_CPU_DST': CliCommand.Node( matcher=self.copyToCpuDstMatcher,
                                                   maxMatches=1 ) }

         @staticmethod
         def adapter( mode, args, argsList ):
            args.pop( 'copy', None )
            args[ 'copyToCpuDst' ] = args.pop( 'COPY_CPU_DST', None ) 

      return CopyToCpuDstExpression 

logMatcher = CliMatcher.KeywordMatcher( 'log',
      helpdesc='Log matches against this rule' )

class RuleActionExprFactory( CliCommand.CliExpressionFactory ):
   def __init__( self, status, aclType ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.status = status
      self.aclType = aclType
      self.mirrorSessionMatcher = MirrorSessionMatcher( self.status, self.aclType )
      # make log option exclusive with copy option
      sharedMatchObj = object()
      self.logNode = CliCommand.Node( matcher=logMatcher, maxMatches=1,
                                      sharedMatchObj=sharedMatchObj )
      self.copyToCpuDstNode = CopyToCpuDstMatcher( self.status, self.aclType,
                                                   sharedMatchObj ) 

   def generate( self, name ):
      class RuleActionExpression( CliCommand.CliExpression ):
         expression = '{ MIRROR | log | COPY_TO_CPU }'
         data = { 'MIRROR' : self.mirrorSessionMatcher,
                  'log' : self.logNode,
                  'COPY_TO_CPU' : self.copyToCpuDstNode }
      return RuleActionExpression

#---------------------------------------------------
# Generic Protocols
#---------------------------------------------------
class GenericProtocolsMatcher( CliCommand.CliExpressionFactory ):
   def __init__( self, protocols ):
      super( GenericProtocolsMatcher, self ).__init__()
      self.protocols_ = protocols

   def generate( self, name ):
      class GenericProtocolsExpression( CliCommand.CliExpression ):
         expression = 'PROTO_NAME | PROTO_VALUE'
         data = {
               'PROTO_NAME': CliMatcher.EnumMatcher( { p: v[ 1 ]
                  for p, v in self.protocols_.iteritems() } ),
               'PROTO_VALUE': CliCommand.Node(
                  matcher=CliMatcher.EnumMatcher( { str( v[ 0 ] ): v[ 1 ]
                     for v in self.protocols_.itervalues() } ),
                  hidden=True )
               }

         @staticmethod
         def adapter( mode, args, argsList ):
            if 'PROTO_NAME' in args:
               assert 'protocol' not in args
               protoName = args.pop( 'PROTO_NAME' )
               args[ 'protocol' ] = self.protocols_[ protoName ][ 0 ]
            elif 'PROTO_VALUE' in args:
               assert 'protocol' not in args
               args[ 'protocol' ] = int( args.pop( 'PROTO_VALUE' ) )
      return GenericProtocolsExpression

#---------------------------------------------------
# ICMP
#---------------------------------------------------
class IcmpOptMatcher( CliCommand.CliExpressionFactory ):
   def __init__( self, messages, icmpAllVal, name ):
      super( IcmpOptMatcher, self ).__init__()
      self.name_ = name
      self.messages_ = messages
      self.icmpOptExpressionData_ = {}
      for k, v in messages.iteritems():
         # generate the help description
         helpdesc = v[ 0 ]
         if v[ 2 ] == icmpAllVal:
            helpdesc += ' (%d)' % v[ 1 ]
         else:
            helpdesc += ' (%d/%d)' % ( v[ 1 ], v[ 2 ] )
         self.icmpOptExpressionData_[ k ] = helpdesc

   def generate( self, name ):
      class IcmpOptExpression( CliCommand.CliExpression ):
         expression = '( ICMP_MSG_TYPE [ ICMP_MSG_CODE ] ) | ICMP_TYPE_CODE_MATCHER'
         data = { 'ICMP_MSG_TYPE': CliMatcher.IntegerMatcher( 0, 255,
                                          helpdesc='Message type for ICMP packets' ),
                  'ICMP_MSG_CODE': CliMatcher.IntegerMatcher( 0, 255,
                                          helpdesc='Message code for ICMP packets' ),
                  'ICMP_TYPE_CODE_MATCHER': CliMatcher.EnumMatcher(
                     self.icmpOptExpressionData_ ) }
         @staticmethod
         def adapter( mode, args, argsList ):
            if 'ICMP_MSG_TYPE' in args:
               icmpType = args.pop( 'ICMP_MSG_TYPE' )
               icmpCode = args.get( 'ICMP_MSG_CODE', None )
               assert self.name_ not in args
               args[ self.name_ ] = ( icmpType, icmpCode )
               if 'ICMP_MSG_CODE' in args:
                  args.pop( 'ICMP_MSG_CODE' )
            elif 'ICMP_TYPE_CODE_MATCHER' in args:
               icmpTypeCode = args.pop( 'ICMP_TYPE_CODE_MATCHER' )
               icmpType = self.messages_[ icmpTypeCode ][ 1 ]
               icmpCode = self.messages_[ icmpTypeCode ][ 2 ]
               assert self.name_ not in args
               args[ self.name_ ] = ( icmpType, icmpCode )

      return IcmpOptExpression

#---------------------------------------------------
# Other Rules
#---------------------------------------------------
class TniMaskExpression( CliCommand.CliExpression ):
   expression = '''tni TNI TNI_MASK'''
   data = {
      'tni': 'Tenant Network Identifier',
      'TNI': CliMatcher.IntegerMatcher( 0, ( 2 ** 24 ) - 1,
         helpdesc='Tenant Network Identifier' ),
      'TNI_MASK': CliMatcher.IntegerMatcher( 0, 0xFFFFFF,
         helpname='0x000000-0xFFFFFF', helpdesc='TNI mask' )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if not args.pop( 'tni', None ):
         return
      tni = args.pop( 'TNI' )
      tniMask = args.pop( 'TNI_MASK' ) ^ AclLib.TNI_MAX
      args[ 'nvgre' ] = ( AclLib.NVGRE_PROTO, ( tni, tniMask ) )

class VniMaskExpression( CliCommand.CliExpression ):
   expression = 'vni VNI VNI_MASK'
   data = {
      'vni': 'VXLAN Network Identifier',
      'VNI': CliMatcher.IntegerMatcher( 0, ( 2 ** 24 ) - 1,
         helpdesc='Virtual Network Identifier' ),
      'VNI_MASK': CliMatcher.IntegerMatcher( 0, 0xFFFFFF,
         helpname='0x000000-0xFFFFFF', helpdesc='VNI mask' )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if not args.pop( 'vni', None ):
         return
      vni = args.pop( 'VNI' )
      vniMask = args.pop( 'VNI_MASK' )
      args[ 'vxlan' ] = ( True, ( vni, vniMask ^ AclLib.VNI_MAX ) )

class GreExprFactory( CliCommand.CliExpressionFactory ):
   def __init__( self, status ):
      super( GreExprFactory, self ).__init__()
      self.status_ = status

   def _validateSupported( self, mode, token ):
      if ( getattr( self.status_, 'dpAclGreSupported' ) or
           getattr( self.status_, 'dpAclDIParsingSupported' ) ):
         return None
      return CliParser.guardNotThisPlatform

   def generate( self, name ):
      class GreExpression( CliCommand.CliExpression ):
         expression = 'GRE_PROTO GRE_PROTO_VAL'
         data = {
            'GRE_PROTO': CliCommand.guardedKeyword( 'proto',
               helpdesc='Generic Routing Encapsulation',
               guard=self._validateSupported ),
            'GRE_PROTO_VAL': CliMatcher.IntegerMatcher( 0, ( 2 ** 16 ) - 1,
               helpdesc='GRE Protocol' )
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            if not args.pop( 'GRE_PROTO', None ):
               return

            protoVal = args.pop( 'GRE_PROTO_VAL' )
            args[ 'gre' ] = ( True, ( protoVal, AclLib.GRE_ALL ) )

      return GreExpression

class TeidMaskExpression( CliCommand.CliExpression ):
   expression = 'TEID_KW TEID TEID_MASK'
   data = {
      'TEID_KW': CliMatcher.KeywordMatcher( 'teid',
         helpdesc='Tunnel Endpoint Identifier' ),
      'TEID': CliMatcher.IntegerMatcher( 0, ( 2 ** 32 ) - 1, helpdesc='TEID' ),
      'TEID_MASK': CliMatcher.IntegerMatcher( 0, 0xFFFFFFFF,
         helpname='0x000000-0xFFFFFFF', helpdesc='TEID mask' )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if not args.pop( 'TEID_KW', None ):
         return
      teid = args.pop( 'TEID' )
      teidMask = args.pop( 'TEID_MASK' )
      args[ 'teidVal' ] = ( teid, teidMask ^ AclLib.TEID_MAX )

class NexthopGroupNameExpression( CliCommand.CliExpression ):
   expression = 'nexthop-group NEXTHOP_GROUP'
   data = {
      'nexthop-group' : CliCommand.guardedKeyword( 'nexthop-group',
         helpdesc='Match nexthop-group',
         guard=nexthopGroupSupportedGuard ),
      'NEXTHOP_GROUP': CliMatcher.DynamicNameMatcher( getNexthopGroupNames,
         'Nexthop-group name' )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if not args.pop( 'nexthop-group', None ):
         return
      args[ 'nexthopGroup' ] = args.pop( 'NEXTHOP_GROUP' )

seqRangeMatcher = MultiRangeRule.MultiRangeMatcher(
      lambda: ( 1, AclLib.MAX_SEQ ),
      False,
      helpdesc='Index in the sequence' )

