# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AclCliLib
from AclCliLib import sortedSequenceNumbers
from AclCliLib import getRuleById
import AclLib
import AclCliModel
import ArnetModel
import Tac

import Tracing
t0 = Tracing.trace0
t8 = Tracing.trace8

AclType = Tac.Type( 'Acl::AclType' )
AclPayloadHdrStart = Tac.Type( 'Acl::AclPayloadHdrStart' )

# The functions in this module instantiate ACL CLI model objects.
# They use mounted config and status as well as CLI arguments.
# AclCli.py calls the functions getAclModel() and getAclSummaryModel().
# The getIp4Rule, getIp6Rule, and getMacRule are used by the PolicyMap
# CLI model code.


def getAclModel( config, allVrfConfig, status,
                 name, aclType, aclListModel,
                 countersAreStale,
                 ruleStatsFunc, paramConfig=None, dynamic=False, chipName=None ):
   """ Returns an AclCliModel.Acl instance. """

   # Pull useful bits of config.
   aclTypeConfig = config.config[ aclType ]
   aclConfig = aclTypeConfig.acl.get( name )
   if not aclConfig or ( dynamic and not aclConfig.dynamic ):
      return None #true to orig. impl.
   subConfig = aclConfig.currCfg
   if not subConfig:
      return None #true to orig. impl.

   # Pull useful bits of status.
   aclTypeStatus = status.status[ aclType ]
   aclStatus = aclTypeStatus.acl.get( name )

   aclModel = aclListModel.newAcl()
   # Instantiate an ACL model and populate it with common attrs.
   _populateAclModelBase( aclModel, aclType, name, aclConfig, subConfig, aclStatus )
   aclModel.staleCounters = countersAreStale
   aclModel.sequence = []
   aclModel.chipName = chipName if chipName is not None else ""

   # Model all list elements, ordered by sequence number.
   for seqNo in sortedSequenceNumbers( subConfig ):
      element = _getModelListElement( aclModel, aclType, seqNo,
                                      aclConfig.standard,
                                      subConfig, paramConfig, aclStatus,
                                      ruleStatsFunc, chipName=chipName )
      aclModel.sequence.append( element )

   return aclModel

# PolicyMap calls getAclListModel() to instantiate CLI model objects
# for ACLs configured in the policymap.
def getAclListModel( aclConfig, aclStatus, aclType, name,
                     ruleStatsFunc ):

   subConfig = aclConfig.currCfg
   if not subConfig:
      return None
   if aclType == AclType.ip:
      aclListModel = AclCliModel.IpAclList()
   elif aclType == AclType.ipv6:
      aclListModel = AclCliModel.Ipv6AclList()
   elif aclType == AclType.mac:
      aclListModel = AclCliModel.MacAclList()
   else:
      assert True

   aclModel = aclListModel.newAcl()
   _populateAclModelBase( aclModel, aclType, name, aclConfig, subConfig, None )
   aclModel.staleCounters = False
   for seqNo in sortedSequenceNumbers( subConfig ):
      element = _getModelListElement( aclModel, aclType, seqNo,
                                      aclConfig.standard,
                                      subConfig, None, aclStatus,
                                      ruleStatsFunc )
      aclModel.sequence.append( element )

   aclListModel.aclList.append( aclModel )
   return aclListModel

def getServiceAclModel( config, status, ckptStatus, name, aclType,
                        aclListModel, ruleStatsFunc,
                        paramConfig=None, serviceName=None ):
   """ Returns an AclCliModel.Acl instance. """

   # Pull useful bits of config.
   aclTypeConfig = config.config[ aclType ]
   aclConfig = aclTypeConfig.acl.get( name )
   if not aclConfig:
      return None
   subConfig = aclConfig.currCfg
   if not subConfig:
      return None

   # Pull useful bits of status.
   aclStatus = None
   if status:
      aclTypeStatus = status.status.get( aclType )
      if aclTypeStatus:
         aclStatus = aclTypeStatus.acl.get( name )

   aclModel = aclListModel.newAcl()
   # Instantiate an ACL model and populate it with common attrs.
   _populateAclModelBase( aclModel, aclType, name, aclConfig, subConfig, aclStatus )
   aclModel.staleCounters = False
   aclModel.sequence = []

   # Model all list elements, ordered by sequence number.
   ckpt = None
   if ckptStatus:
      ckpt = ckptStatus.acl.get( name )
   for seqNo in sortedSequenceNumbers( subConfig ):
      element = _getModelListElement( aclModel, aclType, seqNo,
                                      aclConfig.standard,
                                      subConfig, paramConfig, aclStatus,
                                      ruleStatsFunc )
      if seqNo in subConfig.ruleBySequence:
         ruleid = subConfig.ruleBySequence[ seqNo ]
         if ckpt:
            rule = ckpt.ruleStatus.get( ruleid )
            if rule:
               element.counterData.checkpointPacketCount = rule.pkts
            connRule = ckpt.connRuleStatus.get( ruleid )
            if connRule:
               element.counterData.checkpointConnCount = connRule.pkts
      aclModel.sequence.append( element )

   if aclStatus:
      if aclStatus.noRuleMatches.pkts or aclStatus.noConnRuleMatches.pkts:
         noMatchCounter = AclCliModel.AclCounter()
         noMatchCounter.packetCount = aclStatus.noRuleMatches.pkts
         noMatchCounter.connCount = aclStatus.noConnRuleMatches.pkts
         if ckpt:
            noMatchCounter.checkpointPacketCount = ckpt.noRuleMatches.pkts
            noMatchCounter.checkpointConnCount = ckpt.noConnRuleMatches.pkts
         else:
            noMatchCounter.checkpointPacketCount = 0
            noMatchCounter.checkpointConnCount = 0
         noMatchCounter.lastChangedTime = max(
                  aclStatus.noRuleMatches.lastChangedTime,
                  aclStatus.noConnRuleMatches.lastChangedTime )
         aclModel.noMatchCounter = noMatchCounter

   return aclModel

def getServiceAclSummaryModel( config, serviceAclConfig, serviceAclStatus,
                               allVrfConfig, name, aclType, supressVrf,
                               vrfUnaware=False, serviceName=None ):
   """ Returns an AclCliModel.AclSummary instance. """

   # Pull useful bits of config.
   aclTypeConfig = config.config[ aclType ]
   aclConfig = aclTypeConfig.acl.get( name )
   if not aclConfig:
      return None
   subConfig = aclConfig.currCfg
   if not subConfig:
      return None

   # Instantiate a model and populate it with common attrs.
   model = AclCliModel.ServiceAclSummary()
   _populateAclModelBase( model, aclType, name, aclConfig, subConfig, None )

   model.sequenceLength = AclCliLib.numAclRules( subConfig )

   # Build an all-inclusive list of VRF names
   allVrfNames = _getVrfNames( allVrfConfig )
   vrfNames = set()
   assert name
   if not serviceName:
      # per agent based service ACL
      for aclTypeAndVrfName, aclName in serviceAclConfig.aclName.iteritems():
         if aclTypeAndVrfName.aclType == aclType and aclName == name:
            # There is only one entry for default vrf but apply to all vrfs
            if supressVrf:
               vrfNames = set( allVrfNames )
               break
            else:
               vrfNames.add( aclTypeAndVrfName.vrfName )
   else:
      # iptables based service ACL
      cpAclTypeConfig = serviceAclConfig.cpConfig[ aclType ]
      for vrfName, serviceAclVrfConfig in cpAclTypeConfig.serviceAcl.iteritems():
         serviceAcl = serviceAclVrfConfig.service.get( serviceName )
         if serviceAcl and serviceAcl.aclName == name:
            if supressVrf:
               vrfNames = set( allVrfNames )
               break
            else:
               vrfNames.add( vrfName )

   if not vrfUnaware:
      for vrf in vrfNames:
         model.configuredVrfs.append( vrf )

   vrfNames.clear()
   if serviceAclStatus and not serviceName:
      # per agent based service ACL
      for aclTypeAndVrfName, aclVer in serviceAclStatus.aclVersion.iteritems():
         if aclTypeAndVrfName.aclType == aclType and aclVer.aclName == name:
            if supressVrf:
               vrfNames = set( allVrfNames )
               break
            else:
               vrfNames.add( aclTypeAndVrfName.vrfName )
   elif serviceAclStatus and serviceName:
      # iptables based service ACL
      cpAclTypeStatus = serviceAclStatus.cpStatus[ aclType ]
      for vrfName, serviceAclVrfStatus in cpAclTypeStatus.serviceAcl.iteritems():
         serviceAcl = serviceAclVrfStatus.service.get( serviceName )
         if serviceAcl == name:
            if supressVrf:
               vrfNames = set( allVrfNames )
               break
            else:
               vrfNames.add( vrfName )

   if not vrfUnaware:
      for vrf in vrfNames:
         model.activeVrfs.append( vrf )
   return model

def _lagDirStatus( configuredIntfs, activeIntfs, lagToPort ):
   """ Returns list of active ACL intfs where phyical-ports have been converted
       to lag if all members present """
   configuredIntfs = set( configuredIntfs )
   activeIntfs = set( activeIntfs )
   if not lagToPort:
      return activeIntfs
   for cintf in configuredIntfs:
      lagPorts = lagToPort.get( cintf )
      if lagPorts:
         if set( lagPorts ).issubset( activeIntfs ):
            # add LAG to activeIntfs if all members are present
            activeIntfs.add( cintf )
         # remove all physical members of LAG from active intfs
         activeIntfs = activeIntfs.difference( set( lagPorts ) )
   return list( activeIntfs )

def getAclSummaryModel( config, cpConfig, paramConfig, intfConfig, allVrfConfig,
                        status, name, aclType,
                        countersAreStale, ruleStatsFunc, lagToPort=None,
                        dynamic=False ):

   """ Returns an AclCliModel.AclSummary instance. """

   configuredIngressIntfs = []
   configuredEgressIntfs = []

   # Pull useful bits of config.
   aclTypeConfig = config.config[ aclType ]
   aclConfig = aclTypeConfig.acl.get( name )
   if not aclConfig or ( dynamic and not aclConfig.dynamic ):
      return None #true to orig. impl.
   subConfig = aclConfig.currCfg
   if not subConfig:
      return  None #true to orig. impl.
   aclIntfTypeConfig = intfConfig.config[ aclType ]
   cpAclTypeConfig = cpConfig.cpConfig.get( aclType )
   cpAclNameDefault = paramConfig.cpAclNameDefault

   # Pull useful bits of status.
   aclTypeStatus = status.status[ aclType ]
   aclStatus = aclTypeStatus.acl.get( name )
   aclIntfStatus = aclTypeStatus.intf
   cpAclTypeStatus = status.cpStatus.get( aclType )

   # Instantiate a model and populate it with common attrs.
   model = AclCliModel.AclSummary()
   _populateAclModelBase( model, aclType, name, aclConfig, subConfig, aclStatus )

   model.notFullySupportedByHardware = False
   model.sequenceLength = AclCliLib.numAclRules( subConfig )

   # Build an all-inclusive list of VRF names, used to
   # identify control-plane and service interfaces.
   vrfNames = _getVrfNames( allVrfConfig )
   allVrfNames = list( vrfNames )
   if aclType == 'ip' or aclType == 'ipv6':
      allVrfNames.extend( cpAclTypeConfig.globalCpAcl.keys() )
      allVrfNames.extend( cpAclTypeConfig.serviceAcl.keys() )
   allVrfNames = set( allVrfNames )

   # In the case of IP, get models of all configured ingress control-plane and
   # services interfaces, per VRF.
   if aclType == 'ip' or aclType == 'ipv6':
      for vrf in allVrfNames:
         cpAclName = cpAclTypeConfig.globalCpAcl.get( vrf, cpAclNameDefault )
         if cpAclName == name:
            model.configuredIngressIntfs.append( _getIntfSummary( "control-plane",
                                                                  vrf=vrf ) )
         serviceAclVrfConfig = cpAclTypeConfig.serviceAcl.get( vrf )
         if serviceAclVrfConfig:
            for service, serviceAcl in serviceAclVrfConfig.service.iteritems():
               if serviceAcl.aclName == name:
                  model.configuredIngressIntfs.append( _getIntfSummary( service,
                                                                        vrf=vrf ) )
   # Get configured ingress and egress models for the explicitly ACL-ed interfaces.
   intfConfigIn = aclIntfTypeConfig.intf[ 'in' ].intf
   for i, n in intfConfigIn.iteritems( ):
      if n == name:
         model.configuredIngressIntfs.append( _getIntfSummary( i ) )
         configuredIngressIntfs.append( i )
   intfConfigOut = aclIntfTypeConfig.intf[ 'out' ].intf
   for i, n in intfConfigOut.iteritems( ):
      if n == name: 
         model.configuredEgressIntfs.append( _getIntfSummary( i ) )
         configuredEgressIntfs.append( i )

   if aclType == 'ip' or aclType == 'ipv6':
      for vrf in vrfNames:
         activeCpAcl = cpAclTypeStatus.globalCpAcl.get( vrf )
         if activeCpAcl and activeCpAcl == name:
            model.activeIngressIntfs.append( _getIntfSummary( "control-plane",
                                                              vrf=vrf ) )
         activeServiceAcl = cpAclTypeStatus.serviceAcl.get( vrf )
         if activeServiceAcl:
            for service, serviceAcl in activeServiceAcl.service.iteritems():
               if serviceAcl == name:
                  model.activeIngressIntfs.append( _getIntfSummary( service,
                                                                    vrf=vrf ) )
   # Get active ingress and egress models for the explicitly ACL-ed interfaces.
   activeInterfaces = [
                  ( 'in', model.activeIngressIntfs, configuredIngressIntfs ),
                  ( 'out', model.activeEgressIntfs, configuredEgressIntfs ) ]
   if status.dpAclDirStatusEnabled and aclStatus:
      for direction, intfList, configuredIntfs in activeInterfaces:
         dirStatus = aclStatus.directionStatus.get( direction )
         if dirStatus:
            modDirStatus = _lagDirStatus( configuredIntfs,
                                          dirStatus.intf.keys(),
                                          lagToPort )
            intfList.extend( map( _getIntfSummary, modDirStatus ) )
   else:
      for direction, intfList, _ in activeInterfaces:
         dirStatus = aclIntfStatus[ direction ].intf
         for i, n in dirStatus.iteritems():
            if n == name:
               intfList.append( _getIntfSummary( i ) )

   # Make note of hardware support issue.
   if ( name in intfConfigIn.values( ) or name in intfConfigOut.values() ) and \
          ( not AclCliLib.dpAclOk( aclType, aclIntfTypeConfig,
                                   aclConfig, status ) or
            not AclCliLib.dpAclVlanInterfaceOk( aclType,
                                                aclIntfTypeConfig,
                                                aclConfig, status ) ):
      model.notFullySupportedByHardware = True

   return model


def _getVrfNames( allVrfConfig ):
   """ Extracts the VRF names from config. """

   vrfNames = allVrfConfig.vrf.members()
   vrfNames.append( AclLib.defaultVrf )
   return vrfNames


def _getModelListElement( aclModel, aclType, seqNo, standard,
                          subConfig, paramConfig, aclStatus, ruleStatsFunc,
                          chipName=None ):
   """ Determines the type of a ACL element and returns a model for it. """
   convertSymbols = ( not paramConfig or paramConfig.convertSymbols )

   if seqNo in subConfig.ruleBySequence:
      ruleid = subConfig.ruleBySequence[ seqNo ]
      ruleConfig = getRuleById( subConfig, aclType )[ ruleid ]

      if aclType == 'mac':
         rule = getMacRule( ruleid, ruleConfig, aclStatus,
                            seqNo, standard, aclType, ruleStatsFunc,
                            convertSymbols=convertSymbols, chipName=chipName )
      if aclType == 'ip':
         rule = getIp4Rule( ruleid, ruleConfig, aclStatus,
                            seqNo, standard, aclType, ruleStatsFunc,
                            convertSymbols=convertSymbols, chipName=chipName )
      if aclType == 'ipv6':
         rule = getIp6Rule( ruleid, ruleConfig, aclStatus,
                            seqNo, standard, aclType, ruleStatsFunc,
                            convertSymbols=convertSymbols, chipName=chipName )
   else: #remarks, store "remark" in text
      if aclType == 'mac':
         rule = AclCliModel.MacRule()
      if aclType == 'ip':
         rule = AclCliModel.IpRule()
      if aclType == 'ipv6':
         rule = AclCliModel.Ipv6Rule()

      rule.sequenceNumber = seqNo
      rule.convertSymbols = convertSymbols
      rule.text = subConfig.remarkBySequence[ seqNo ].remark
      # The text should start with "remark" to be consistent with
      # other rules. However, we used to not do this, and people might
      # have manually prepended "remark". So let's just be cautious and
      # only add "remark" if it's not already there to avoid an infinite
      # loop each time people fetch and reapply the rule.
      if not rule.text.startswith( "remark " ):
         rule.text = "remark " + rule.text
      rule.ruleFilter = None #indicator that it's a remark
      rule.counterData = None
      rule.action = None
      rule.mirrorSession = None
      rule.log = None
      rule.copyToCpuDst = None 
      rule.payload = None

   return rule

def _populateMacRule( rule, ruleid, ruleConfig, aclStatus, seqNo,
                      standard, aclType, ruleStatsFunc, chipName=None ):
   _populateRuleBase( rule, ruleid, ruleConfig, aclStatus, seqNo,
                      aclType, ruleStatsFunc, chipName=chipName )
   _populateMacRuleFilter( rule, ruleConfig )
   rule.text = ruleConfig.ruleStr( standard, rule.convertSymbols )

def _populateIp6Rule( rule, ruleid, ruleConfig, aclStatus, seqNo,
                      standard, aclType, ruleStatsFunc, chipName=None ):
   _populateRuleBase( rule, ruleid, ruleConfig, aclStatus, seqNo,
                      aclType, ruleStatsFunc, chipName=chipName )
   _populateIp6RuleFilter( rule, ruleConfig, standard )
   rule.text = ruleConfig.ruleStr( standard, rule.convertSymbols )

def _populateIpRule( rule, ruleid, ruleConfig, aclStatus, seqNo,
                     standard, aclType, ruleStatsFunc, chipName=None ):
   _populateRuleBase( rule, ruleid, ruleConfig, aclStatus, seqNo,
                      aclType, ruleStatsFunc, chipName=chipName )
   _populateIpRuleFilter( rule, ruleConfig, standard )
   rule.text = ruleConfig.ruleStr( standard, rule.convertSymbols )

def getIp4Rule( ruleid, ruleConfig, aclStatus, seqNo, standard,
                aclType, ruleStatsFunc, convertSymbols=True, chipName=None ):
   """ Returns the model for IPv4 rule. """

   rule = AclCliModel.IpRule()
   rule.convertSymbols = convertSymbols
   rule.ruleFilter = AclCliModel.IpFilter()
   _populateIpRule( rule, ruleid, ruleConfig, aclStatus, seqNo, standard,
                    aclType, ruleStatsFunc, chipName=chipName )
   return rule

def getIp6Rule( ruleid, ruleConfig, aclStatus, seqNo, standard,
                aclType, ruleStatsFunc, convertSymbols=True, chipName=None ):
   """ Returns the model for IPv6 rule. """

   rule = AclCliModel.Ipv6Rule()
   rule.convertSymbols = convertSymbols
   rule.ruleFilter = AclCliModel.Ipv6Filter()
   _populateIp6Rule( rule, ruleid, ruleConfig, aclStatus, seqNo, standard,
                     aclType, ruleStatsFunc, chipName=chipName )
   return rule

def getMacRule( ruleid, ruleConfig, aclStatus, seqNo, standard,
                aclType, ruleStatsFunc, convertSymbols=True, chipName=None ):
   """ Returns the model for Mac rule. """

   rule = AclCliModel.MacRule()
   rule.convertSymbols = convertSymbols
   rule.ruleFilter = AclCliModel.MacFilter()
   _populateMacRule( rule, ruleid, ruleConfig, aclStatus, seqNo, standard,
                     aclType, ruleStatsFunc, chipName=chipName )
   return rule

def _populateRuleBase( rule, ruleid, ruleConfig, aclStatus, seqNo,
                       aclType, ruleStatsFunc, chipName=None ):
   """ populates sequenceNumber, action, log, counterData, for a rule """

   rule.sequenceNumber = seqNo
   rule.action = ruleConfig.action
   rule.mirrorSession = ruleConfig.mirrorSession
   rule.log = ruleConfig.log
   rule.copyToCpuDst = AclCliLib.copyToCpuDstFromValue( ruleConfig.copyToCpuDst )
   aclCounter = AclCliModel.AclCounter()
   ( aclCounter.packetCount,
     aclCounter.connCount,
     aclCounter.checkpointPacketCount,
     aclCounter.checkpointConnCount,
     aclCounter.lastChangedTime ) = ruleStatsFunc( aclStatus, ruleConfig,
                                                   ruleid, aclType, chipName )
   rule.counterData = aclCounter

   def payloadElementList_( payloadString ):
      payloadList = AclCliLib.payloadStringToValue( payloadString )
      elemList = []
      for payload in payloadList:
         offset, pattern, mask, alias, patOverride, maskOverride = payload
         elemList.append( AclCliModel.PayloadElement( offset=offset,
                        pattern=pattern, patternMask=mask, alias=alias,
                        patternOverride=patOverride, maskOverride=maskOverride ) )
      return sorted( elemList, key=lambda payload: payload.offset )
   headerStart = True if ruleConfig.filter.payloadOpt.headerStart == \
         AclPayloadHdrStart.start else False
   rule.payload = AclCliModel.PayloadSpec( headerStart=headerStart,
                           payload=payloadElementList_(
                                 ruleConfig.filter.payloadOpt.offsetPattern ) )

def _populateMacRuleFilter( rule, ruleConfig ):

   macFilter = rule.ruleFilter
   macFilter.sourceAddr = ruleConfig.filter.getRawAttribute( "sourceAddr" )
   macFilter.sourceMask = ruleConfig.filter.getRawAttribute( "sourceMask" )
   macFilter.destinationAddr = ruleConfig.filter.getRawAttribute( "destinationAddr" )
   macFilter.destinationMask = ruleConfig.filter.getRawAttribute( "destinationMask" )
   macFilter.protocol = ruleConfig.filter.proto
   macFilter.vlan = AclCliModel.VlanSpec( id=ruleConfig.filter.vlan,
                                      mask=ruleConfig.filter.vlanMask,
                                      innerId=ruleConfig.filter.innerVlan,
                                      innerMask=ruleConfig.filter.innerVlanMask )
   macFilter.mpls = AclCliModel.MplsSpec( label=ruleConfig.filter.mplsLabel,
                                      labelMask=ruleConfig.filter.mplsLabelMask,
                                      innerLabel=ruleConfig.filter.mplsInnerLabel,
                                      innerLabelMask= \
                                         ruleConfig.filter.mplsInnerLabelMask )
   mplsFilter = AclCliModel.MplsFilter()
   labels = ruleConfig.filter.mplsFilter.label
   for key in labels:
      mplsLabelFilter = AclCliModel.MplsLabelFilter( 
         value=labels[ key ].value,
         mask=labels[ key ].mask ^ AclLib.MPLSLABEL_MAX )
      mplsFilter.labels[ key ] = mplsLabelFilter
   if ruleConfig.filter.mplsFilter.bos != Tac.Type("Acl::MplsBosEnum").bosDontCare:
      mplsFilter.bos = \
         Tac.enumValue("Acl::MplsBosEnum", ruleConfig.filter.mplsFilter.bos)
   macFilter.mplsFilter = mplsFilter

def _populateIpFilterBase( rule, ruleConfig, standard ):
   """ populates ip/ipv6 filter base info """

   ipFilter = rule.ruleFilter

   ipFilter.standard = standard
   ipFilter.protocol = ruleConfig.filter.proto
   ipFilter.innerProtocol = ruleConfig.filter.innerProto
   ipFilter.ttl = AclCliModel.TtlSpec( value=ruleConfig.filter.ttl.ttl,
                                    oper=ruleConfig.filter.ttl.oper )
   ipFilter.srcPort = AclCliModel.PortSpec( oper=ruleConfig.filter.sport.oper,
                               ports=_parsePorts( ruleConfig.filter.sport.ports ),
                               maxPorts=ruleConfig.filter.sport.maxPorts )
   ipFilter.dstPort = AclCliModel.PortSpec( oper=ruleConfig.filter.dport.oper,
                               ports=_parsePorts( ruleConfig.filter.dport.ports ),
                               maxPorts=ruleConfig.filter.dport.maxPorts )
   ipFilter.tracked = ruleConfig.filter.tracked
   ipFilter.tcpFlags = ruleConfig.filter.tcpFlag.value
   ipFilter.established = ruleConfig.filter.established
   ipFilter.icmp = AclCliModel.IcmpSpec( type=ruleConfig.filter.icmpType,
                                      code=ruleConfig.filter.icmpCode )

   ipFilter.vlan = AclCliModel.VlanSpec( id=ruleConfig.filter.vlan,
                                      mask=ruleConfig.filter.vlanMask,
                                      innerId=ruleConfig.filter.innerVlan,
                                      innerMask=ruleConfig.filter.innerVlanMask )
   ipFilter.userL4 = AclCliModel.UserL4Spec( pattern=ruleConfig.filter.userL4,
                                      patternMask=ruleConfig.filter.userL4Mask )
   ipFilter.nvgre = AclCliModel.NvgreSpec( tni=ruleConfig.filter.tni,
                                        tniMask=ruleConfig.filter.tniMask,
                                        protocol=ruleConfig.filter.greProto,
                                        protoMask=ruleConfig.filter.greProtoMask )
   ipFilter.gre = AclCliModel.GreSpec( protocol=ruleConfig.filter.greProto,
                                    protoMask=ruleConfig.filter.greProtoMask )
   ipFilter.vxlan = AclCliModel.VxlanSpec( vxlanValid=ruleConfig.filter.vxlanValid,
                                        vni=ruleConfig.filter.vni,
                                        vniMask=ruleConfig.filter.vniMask )
   ipFilter.nexthopGroup = ruleConfig.filter.nexthopGroup

   gtpVersion = gtpProto = None
   if ruleConfig.filter.gtpVersion != 0:
      gtpVersion = ruleConfig.filter.gtpVersion
      gtpProto = AclLib.gtpStrFromProto( ruleConfig.filter.gtpProto )
   ipFilter.gtp = AclCliModel.GtpSpec( version=gtpVersion,
                                    protocol=gtpProto,
                                    teid=ruleConfig.filter.teid,
                                    teidMask=ruleConfig.filter.teidMask )
   match = False
   ecnName = AclCliLib.ecnNameFromValue( AclLib.EcnValue.dontCare )

   if ruleConfig.filter.ecn != AclLib.EcnValue.dontCare:
      match = True
      ecnName = AclCliLib.ecnNameFromValue( ruleConfig.filter.ecn )
   rule.ruleFilter.ecn = AclCliModel.EcnSpec( name=ecnName, match=match )

def _populateIpRuleFilter( rule, ruleConfig, standard ):

   _populateIpFilterBase( rule, ruleConfig, standard )

   ipAddr = ruleConfig.filter.source.getRawAttribute( "address" )
   rule.ruleFilter.source = ArnetModel.IpAddrAndMask( ip=ipAddr,
                                            mask=ruleConfig.filter.source.mask )
   ipAddr = ruleConfig.filter.destination.getRawAttribute( "address" )
   rule.ruleFilter.destination = ArnetModel.IpAddrAndMask( ip=ipAddr,
                                   mask=ruleConfig.filter.destination.mask )
   if ruleConfig.filter.matchInnerIp:
      ipAddr = ruleConfig.filter.innerSource.getRawAttribute( "address" )
      rule.ruleFilter.innerSource = AclCliModel.IpAddrAndMask( ip=ipAddr,
                                   mask=ruleConfig.filter.innerSource.mask )
      ipAddr = ruleConfig.filter.innerDest.getRawAttribute( "address" )
      rule.ruleFilter.innerDest = AclCliModel.IpAddrAndMask( ip=ipAddr,
                                   mask=ruleConfig.filter.innerDest.mask )
   if ruleConfig.filter.matchInnerIp6:
      ipv6Addr = ruleConfig.filter.innerSource6.getRawAttribute( "address" )
      rule.ruleFilter.innerSource6 = AclCliModel.Ip6AddrAndMask( ip=ipv6Addr,
                                    mask=ruleConfig.filter.innerSource6.len )
      ipv6Addr = ruleConfig.filter.innerDest6.getRawAttribute( "address" )
      rule.ruleFilter.innerDest6 = AclCliModel.Ip6AddrAndMask( ip=ipv6Addr,
                                    mask=ruleConfig.filter.innerDest6.len )
 
   rule.ruleFilter.fragments = ruleConfig.filter.fragments

   dscpName = None
   dscpValue = ruleConfig.filter.dscp
   if ( ruleConfig.filter.dscpUseName ):
      dscpName = AclCliLib.dscpNameFromValue( dscpValue )
   dscpMask = ruleConfig.filter.dscpMask
   rule.ruleFilter.dscp = AclCliModel.DscpSpec( value=dscpValue,
                                      name=dscpName,
                                      mask=dscpMask,
                                      match=ruleConfig.filter.matchDscp )

   rule.ruleFilter.ipLen = AclCliModel.IpLenSpec(
      oper=ruleConfig.filter.ipLen.oper,
      ipLens=_parsePorts( ruleConfig.filter.ipLen.ipLens ) )

def _populateIp6RuleFilter( rule, ruleConfig, standard ):

   _populateIpFilterBase( rule, ruleConfig, standard )

   sourceFullMask = ruleConfig.filter.sourceFullMask
   ipv6Addr = sourceFullMask.getRawAttribute( "address" )
   if sourceFullMask.hasPrefixRepr():
      rule.ruleFilter.source = ArnetModel.Ip6AddrAndMask( ip=ipv6Addr,
                                             mask=sourceFullMask.prefixLen() )
   else:
      rule.ruleFilter.source = ArnetModel.Ip6AddrAndMask( ip=ipv6Addr,
                                             fullMask=sourceFullMask.mask )

   destinationFullMask = ruleConfig.filter.destinationFullMask
   ipv6Addr = destinationFullMask.getRawAttribute( "address" )
   if destinationFullMask.hasPrefixRepr():
      rule.ruleFilter.destination = ArnetModel.Ip6AddrAndMask( ip=ipv6Addr,
                                             mask=destinationFullMask.prefixLen() )
   else:
      rule.ruleFilter.destination = ArnetModel.Ip6AddrAndMask( ip=ipv6Addr,
                                             fullMask=destinationFullMask.mask )

   if ruleConfig.filter.matchInnerIp: 
      ipAddr = ruleConfig.filter.innerSource.getRawAttribute( "address" )
      rule.ruleFilter.innerSource = AclCliModel.IpAddrAndMask( ip=ipAddr,
                                   mask=ruleConfig.filter.innerSource.mask )
      ipAddr = ruleConfig.filter.innerDest.getRawAttribute( "address" )
      rule.ruleFilter.innerDest = AclCliModel.IpAddrAndMask( ip=ipAddr,
                                   mask=ruleConfig.filter.innerDest.mask )
   if ruleConfig.filter.matchInnerIp6:
      ipv6Addr = ruleConfig.filter.innerSource6.getRawAttribute( "address" )
      rule.ruleFilter.innerSource6 = AclCliModel.Ip6AddrAndMask( ip=ipv6Addr,
                                    mask=ruleConfig.filter.innerSource6.len ) 
      ipv6Addr = ruleConfig.filter.innerDest6.getRawAttribute( "address" )
      rule.ruleFilter.innerDest6 = AclCliModel.Ip6AddrAndMask( ip=ipv6Addr,
                                    mask=ruleConfig.filter.innerDest6.len )
   dscpName = None
   dscpValue = ruleConfig.filter.tc >> 2
   if ( ruleConfig.filter.dscpUseName ):
      dscpName = AclCliLib.dscpNameFromValue( dscpValue )
   dscpMask = ruleConfig.filter.tcMask >> 2
   rule.ruleFilter.dscp = AclCliModel.DscpSpec( value=dscpValue,
                                      name=dscpName,
                                      mask=dscpMask,
                                      match=( ruleConfig.filter.tcMask != 0 ) )     

   rule.ruleFilter.ipLen = AclCliModel.IpLenSpec(
      oper=ruleConfig.filter.ipLen.oper,
      ipLens=_parsePorts( ruleConfig.filter.ipLen.ipLens ) )

def _getIntfSummary( name, target=None, vrf=None ):
   """ Returns the model for an interface in the ACL summary. """

   model = AclCliModel.IntfSummary()
   model.name = name
   model.target = target
   model.vrf = vrf
   return model


def _populateAclModelBase( model, aclType, name, aclConfig, subConfig,
                           aclStatus ):
   """ Fills in attributes common to ACL and ACL summary models. """

   model.type = _aclTypeToEnum( aclType )
   model.name = name
   model.readonly = aclConfig.readonly
   model.standard = aclConfig.standard
   model.countersEnabled = subConfig.countersEnabled
   model.countersIncomplete = bool( aclStatus and aclStatus.countersIncomplete )
   model.dynamic = aclConfig.dynamic

def _parsePorts( ports ):
   """ Parses the TACC model for a port list. """

   return [ int( port ) for port in ports.split() ]


def _aclTypeToEnum( aclType ):
   """ Converts the aclType enum into the corresponding model enum. """

   if ( aclType == 'ip' ):
      return 'Ip4Acl'
   if ( aclType == 'ipv6' ):
      return 'Ip6Acl'
   if ( aclType == 'mac' ):
      return 'MacAcl'
   return None
