#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AclCliModelRender
from ArnetModel import IpAddrAndMask
from ArnetModel import Ip6AddrAndMask
from ArnetModel import MacAddress  
from CliModel import Bool
from CliModel import Enum
from CliModel import Float
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from CliModel import Dict
from AclCliLib import ecnAclNames
from AclCliLib import copyToCpuDestinations
import Tac

Action = Tac.Type( "Acl::Action" )
RangeOperator = Tac.Type( "Acl::RangeOperator" )

# This module defines the ACL CLI's model classes. We use a factory
# system here to generate specific top-level models, even though all
# their structure is very similar.

# The rendering code is in the AclCliModelRender module.
# Each model class either implements the render() method or
# derives from a subclass which does. The render() methods
# call the corresponding method in AclCliModelRender.
# This makes the model easier to read.

# AclCliModelImpl has the code which instantiates
# model class instances. AclCli.py invokes it.

class VlanSpec( Model ):
   """ VLAN is specified with a id and a mask. """

   id = Int( help="VLAN id" )
   mask = Int( help="VLAN mask" )
   innerId = Int( help="Inner VLAN id" )
   innerMask = Int( help="Inner VLAN mask" )

class NvgreSpec( Model ):
   """ Nvgre has protocol and optional tni with mask. """ 

   tni = Int( help="Tenant Network Id" )
   tniMask = Int( help="TNI mask" )
   protocol = Int( help="NVGRE protocol" )
   protoMask = Int( help="NVGRE protocol mask" )

class GreSpec( Model ):
   """ Gre has optional protocol mask. """

   protocol = Int( help="GRE protocol" )
   protoMask = Int( help="GRE protocol mask" )

class GtpSpec( Model ):
   """ Gtp has version, protocol and optional teid with mask. """

   version = Int( help="GTP version", optional=True )
   protocol = Enum( values=( 'gtp-u', 'gtp-c', 'gtp-prime' ),
                    help="GTP protocol type", optional=True )
   teid = Int( help="Tunnel Endpoint Id" )
   teidMask = Int( help="TEID Mask" )

class VxlanSpec( Model ):
   """ Vxlan has valid bit, and optional vni with mask. """

   vxlanValid = Bool( help="True if valid VXLAN packet" )
   vni = Int( help="Virtual Network Id" )
   vniMask = Int( help="VNI mask" )


class MplsLabelFilter ( Model ):
   """ Mpls label filter based on a value and a mask """

   value = Int( help="MPLS label value" )
   mask = Int( help="MPLS label mask" )

class MplsFilter ( Model ):
   """ Mpls filter based on labels and a BOS"""

   labels = Dict( keyType=int,
                  valueType=MplsLabelFilter,
                  help="MPLS labels filter")
   bos = Int( optional=True,
              help="MPLS BOS (Bottom Of Stack) bit of the largest specified label")

class MplsSpec( Model ):
   """ Mpls has label and inner labels with mask. """

   label = Int( help="MPLS label id" )
   labelMask = Int( help="MPLS label mask" )
   innerLabel = Int( help="MPLS inner label id" )
   innerLabelMask = Int( help="MPLS inner label mask" )

class PayloadElement( Model ):
   """ Acl can have multiple of these as part of payload match """

   offset = Int( help="Offset inside payload for DPI" )
   pattern = Int( help="Pattern to match at offset inside payload" )
   patternMask = Int( help="Mask for the pattern to match at offset inside payload" )
   patternOverride = Bool( help='True, if pattern is overridden in alias' )
   maskOverride = Bool( help='True, if mask is overridden in alias' )
   alias = Str( help="Alias of payload" )

class PayloadSpec( Model ):
   """ Payload is a list of offset, pattern, mask tuple """
   headerStart = Bool( help="True, if offset is from header start" )
   payload = List( help='List of offset,pattern,mask tuple',
                    valueType=PayloadElement )

class UdfAlias( Model ):
   """Alias for Udf Payload"""
   udfAliases = Dict( keyType=str, valueType=PayloadSpec,
                   help="Mapping of alias name to payload" )

   def render( self ):
      AclCliModelRender.UdfAliasRenderfn.render( self )

class UserL4Spec( Model ):
   """ When DPI is enabled, for unknown L4, first word doesn't go into
       Payload match but goes into UserL4 field. """
   pattern = Int( help="Pattern to match into first word payload for IP packet" )
   patternMask = Int( help="Mask for pattern to match" )

class TtlSpec( Model ):
   """ TTL is specified with a value and operator. """

   value = Int( help="TTL value" )
   oper = Enum( values=RangeOperator.attributes,
                help="TTL range operator" )

class PortSpec( Model ):
   """ An IP port specification is a list of port numbers,
   a maximum port count and a range operator. """

   oper = Enum( values=RangeOperator.attributes,
                help="Port number range operator" )
   ports = List( valueType=int,
                 help="Port numbers" )
   maxPorts = Int( help="Maximum number of ports" )


class IcmpSpec( Model ):
   """ An ICMP specification is composed of a type and a code. """

   type = Int( help="ICMP type" )
   code = Int( help="ICMP code" )


class DscpSpec( Model ):
   """ DSCP matching is specified with a numeric value and a flag. """

   value = Int( help="DSCP filter value" )
   name = Str( help="DSCP filter name", optional=True )
   mask = Int( help="DSCP mask" )
   match = Bool( help="True if filtered by DSCP value or name" )

class IpLenSpec( Model ):
   oper = Enum( values=RangeOperator.attributes,
                help="IP packet length range operator" )
   ipLens = List( valueType=int, help="IP packet lengths" )

class EcnSpec( Model ):
   """ ECN matching is specified with a keyword. """
   name = Enum( values=ecnAclNames.keys(), help="ECN filter value" )
   match = Bool( help="True if filtered by ECN value or name" )

class IntfSummary( Model ):
   """ summary of an interface."""

   name = Str( help="Name of interface" )
   target = Str( help="Where the ACL is applied", optional=True )
   vrf = Str( help="VRF name, since the summary is per vrf", optional=True )

aclTypes = [ 'Ip4Acl', 'Ip6Acl', 'MacAcl' ]

class BaseIpFilter( Model ):
   """ Filter attributes common to all IP rules """
   standard = Bool( help="True if this is an ACL in standard form" )
   protocol = Int( help="IP protocol number" )
   innerProtocol = Int( optional=True, help="IP inner protocol number" )
   ttl = Submodel( valueType=TtlSpec, optional=True, help="TTL specification" )
   srcPort = Submodel( valueType=PortSpec, optional=True,
               help="Source port number filter specification" )
   dstPort = Submodel( valueType=PortSpec, optional=True,
               help="Destination port number filter specification" )
   ipLen = Submodel( valueType=IpLenSpec, 
               help = "IP packet length specification" )
   tracked = Bool( help="Connection tracked" )
   tcpFlags = Int( optional=True, help="TCP flags" )
   established = Bool( optional=True,
               help="True if filtered by established TCP connection" )
   icmp = Submodel( optional=True, valueType=IcmpSpec, help="ICMP specification" )
   dscp = Submodel( optional=True, valueType=DscpSpec,
               help="DSCP filter specification" )
   ecn = Submodel( optional=True, valueType=EcnSpec,
               help="ECN filter specification" )
   vlan = Submodel( valueType=VlanSpec, optional=True,
               help="VLAN specification" )
   userL4 = Submodel( valueType=UserL4Spec,
                    help="DPI based first word payload filter for unknown L4" )
   nvgre = Submodel( valueType=NvgreSpec,
                     help="NVGRE filter speciication" )
   gre = Submodel( valueType=GreSpec,
                   help="GRE filter speciication" )
   gtp = Submodel( valueType=GtpSpec,
                   help="GTP filter specification" )
   vxlan = Submodel( valueType=VxlanSpec,
                     help="VXLAN filter specification" )
   nexthopGroup = Str( help="Nexthop-group name" )

class IpFilter( BaseIpFilter ):
   source = Submodel( valueType=IpAddrAndMask,
               help="Source IPv4 address and mask. " )
   destination = Submodel( valueType=IpAddrAndMask,
               help="Destination IPv4 address and mask" )
   innerSource = Submodel( optional=True, valueType=IpAddrAndMask,
               help="Inner source IPv4 address and mask. " )
   innerDest = Submodel( optional=True, valueType=IpAddrAndMask,
               help="Inner destination IPv4 address and mask" )
   innerSource6 = Submodel( optional=True, valueType=Ip6AddrAndMask,
               help="Inner source IPv6 address and mask length" )
   innerDest6 = Submodel( optional=True, valueType=Ip6AddrAndMask,
               help="Inner destination IPv6 address and mask length" )
   fragments = Bool( help="Match only fragmented packets." )

   def render( self ):
      pass

class Ipv6Filter( BaseIpFilter ):
   source = Submodel( valueType=Ip6AddrAndMask,
               help="Source IPv6 address and mask" )
   destination = Submodel( valueType=Ip6AddrAndMask,
               help="Destination IPv6 address and mask" )
   innerSource = Submodel( optional=True, valueType=IpAddrAndMask,
               help="Inner source IPv4 address and mask. " )
   innerDest = Submodel( optional=True, valueType=IpAddrAndMask,
               help="Inner destination IPv4 address and mask" )
   innerSource6 = Submodel( optional=True, valueType=Ip6AddrAndMask,
               help="Inner source IPv6 address and mask length" )
   innerDest6 = Submodel( optional=True, valueType=Ip6AddrAndMask,
               help="Inner destination IPv6 address and mask length" )

   def render( self ):
      pass

class MacFilter( Model ):
   sourceAddr = MacAddress( help="Source address" )
   sourceMask = MacAddress( help="Source mask" )
   destinationAddr = MacAddress( help="Destination address" )
   destinationMask = MacAddress( help="Destination mask" )
   vlan = Submodel( valueType=VlanSpec, optional=True, help="VLAN specification" )
   protocol = Int( help="Ethernet protocol" )
   mpls = Submodel( valueType=MplsSpec,
                    help="Two-label syntax MPLS filter specification" )
   mplsFilter = Submodel ( optional=True, valueType=MplsFilter,
                    help="MPLS filter specification" )

class AclCounter( Model ):
   packetCount = Int( help="Number of packets matching the rule",
                      optional=True )
   connCount = Int( help="Number of connections matching the rule",
                      optional=True )
   checkpointPacketCount = Int( help="Number of packets matching the "
                      "rule since last checkpoint",
                      optional=True )
   checkpointConnCount = Int( help="Number of connections matching the "
                      "rule since last checkpoint",
                      optional=True )
   lastChangedTime = Float( help="Last time the counters changed",
                      optional=True )

class _AclBase( Model ):
   """ Common to Acl and AclSummary models. """

   type = Enum( values=aclTypes, help="Type of ACL" )
   name = Str( help="Name" )
   readonly = Bool( help="True if read-only" )
   standard = Bool( help="True if a standard access control list "
                    "- standard ACLs filter only on the source address" )
   countersEnabled = Bool( help="True if counters are enabled" )
   countersIncomplete = Bool(
         help="True if the hardware could not allocate counters for every rule" )
   dynamic = Bool( help="True if a dynamic (non-persistent) access control list" )

class DeepSkipVal( Model ):
   l2SkipValue = Int( help = "deep-inspection payload skip value for L2" )
   l4SkipValue = Int( help = "deep-inspection payload skip value for L4" )
   def render( self ):
      AclCliModelRender.DpiRenderfn.DpiRender( self )

def ruleFactory( filterType, filterName ):
   class Rule( Model ):
      sequenceNumber = Int( help="Numeric sequence number" )
      text = Str( help="A textual representation of the rule, sans sequence "
               "number. If this rule is a remark, this field will contain the "
               "remark text." )
      action = Enum( values=Action.attributes, optional=True,
                  help="Action to take when the filter matches" )
      mirrorSession = Str( help="Target mirror session if this rule is matched",
                           optional=True )
      log = Bool( help="Log when this rule is matched", optional=True )
      copyToCpuDst = Enum( values=copyToCpuDestinations.keys(), optional=True,
                           help="Copy packet to CPU when this rule is matched" )
      convertSymbols = Bool( help="Auto-conversion from subnet and port numbers to "
                          "words in show commands", default=True )
      counterData = Submodel( valueType=AclCounter, optional=True,
                  help="counter data for packets matching this rule." )

      ruleFilter = Submodel( valueType=filterType, optional=True,
               help="The filter criteria to match this rule. Unset for "
               "remarks. When an optional filter attribute is not present "
               "in the output, this implies that there was no "
               "specification configured for that attribute, and any "
               "valid value is permitted." )
      payload = Submodel( valueType=PayloadSpec, optional=True,
               help="DPI based payload filter specification" )

      def render( self ):
         AclCliModelRender.RuleRenderer.render( self )

   Rule.__name__ = filterName
   return Rule

IpRule = ruleFactory( IpFilter, "IpRule" )
Ipv6Rule = ruleFactory( Ipv6Filter, "Ipv6Rule" )
MacRule = ruleFactory( MacFilter, "MacRule" )

def getRuleType( filterType ):
   if filterType == IpFilter:
      return IpRule
   elif filterType == Ipv6Filter:
      return Ipv6Rule
   else:
      return MacRule

class ServiceAclSummary( _AclBase ):
   """ summary of a service ACL """

   sequenceLength = Int( help="Total number of rules and remarks in the list" )

   configuredVrfs = List( help='Configured on vrfs', valueType=str )
   activeVrfs = List( help='Active on vrfs', valueType=str )

   def render( self ):
      AclCliModelRender.ServiceAclSummaryRenderer.render( self )

   def renderAfter( self ):
      AclCliModelRender.ServiceAclSummaryRenderer.renderAfter( self )

def aclListFactory( filterType, filterName ):
   """ Given a filtertype, this returns a top-level model (i.e. a list
   of ACLs that only accept ACLs and their corresponding rules that
   match that filter type """

   class Acl(_AclBase ):
      """ An access control list (ACL) is list of rules. """
      sequence = List( valueType=getRuleType( filterType ),
             help='List of rules, ordered by ascending sequence number' )
      # used by service acl show command
      noMatchCounter = Submodel( valueType=AclCounter, optional=True,
             help="counter of implicit deny for no rules match" )

      staleCounters = Bool( help="True if displaying stale counters" )
      chipName = Str( optional=True, help="Chip name if displaying detail" )
      summary = Submodel( valueType=ServiceAclSummary, optional=True,
                          help="Summary of this ACL" )

      def render( self ):
         AclCliModelRender.AclRenderer.render( self )

   class AclList( Model ):
      """ Returned by 'show * access-list' """
      aclList = List( help='List of ACLs', valueType=_AclBase )

      def newAcl( self ):
         return Acl()
      def render( self ):
         AclCliModelRender.AclListRenderer.render( self )

   AclList.__name__ = filterName + AclList.__name__
   Acl.__name__ = filterName + Acl.__name__
   return AclList

# The three top level models corresponding to "show ip/ipv6/mac
# access-lists"
IpAclList = aclListFactory( IpFilter, "Ip" )
Ipv6AclList = aclListFactory( Ipv6Filter, "Ipv6" )
MacAclList = aclListFactory( MacFilter, "Mac" )

class AllAclList( Model ):
   ipAclList = Submodel( valueType=IpAclList, help='IP access list',
                         optional=True )
   ipv6AclList = Submodel( valueType=Ipv6AclList, help='IPv6 access list',
                           optional=True )

   def render( self ):
      if self.ipAclList:
         self.ipAclList.render()
      if self.ipv6AclList:
         self.ipv6AclList.render()

class AclSummary( _AclBase ):
   """ summary of an ACL """

   sequenceLength = Int( help="Total number of rules and remarks in the list" )

   configuredIngressIntfs = List( help='Configured ingress interfaces',
                                  valueType=IntfSummary )
   configuredEgressIntfs = List( help='Configured egress interfaces',
                                  valueType=IntfSummary )
   activeIngressIntfs = List( help='Active ingress interfaces',
                              valueType=IntfSummary )
   activeEgressIntfs = List( help='Active egress interfaces',
                             valueType=IntfSummary )
   notFullySupportedByHardware = \
         Bool( help="True if some rules are not fully supported by hardware" )

   def render( self ):
      AclCliModelRender.AclSummaryRenderer.render( self )
