#!/usr/bin/env python
# Copyright (c) 2009-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import itertools
import os
import sys
import threading

import AclLib
import AclCliLib
import AclCliModel
import AclCliModelImpl
import BasicCli
import BasicCliSession
import BasicCliUtil
import Cell
import CliParser
import CliCommand
import CliCommon
import CliExtensions
import CliGlobal
import CliMatcher
import CliSession
import CliToken.Cli
import CliToken.Clear
import CliToken.Hardware
import CliToken.Ip
import CliToken.Ipv6
import CliToken.Mac
import CliToken.Service
import CliPlugin.AclCliRules as AclCliRules
import CliPlugin.IntfCli as IntfCli
import CliPlugin.VrfCli as VrfCli
from CliMode.Acl import BaseAclMode, CpMode
import ConfigMount
from eunuchs.in_h import IPPROTO_IP
import Intf
import Intf.Log
import LazyMount
import ShowCommand
import Tac
import Tracing

t0 = Tracing.trace0
t7 = Tracing.trace7
t8 = Tracing.trace8

# cli config instances
cliConfig = None
cliCpConfig = None

# aggregated config instances
config = None
cpConfig = None

paramConfig = None
intfConfigAll = None
intfConfigDir = None
intfConfigDirMergeSm = None
intfConfigAggregatorSm = None
hwConfig = None
status = None
statusDp = None
checkpoint = None
allVrfConfig = None
udfAlias = None
aclCmds = None
threadLocalData_ = threading.local()
aclConfigAggregatorSm = None
aclCpConfigAggregatorSm = None
aggregatorInitialized = False
hwEpoch = None

aclConfigValidCallbacks = []

# New global variables should be added here (old ones can be moved too).
gv = CliGlobal.CliGlobal(
   dict(
      # callbacks for mirroring
      mirrorSessionNameFn = None,
      mirrorValidator=None,
      # intfConfig
      intfConfigCli=None,
      intfConfigSecureMonitor=None,
      intfConfigInputDir=None,
      # acl/[cp]config/input dirs
      aclConfigDir = None,
      aclCpConfigDir=None,
   )
)

# different types of acl updated
aclUpdateTypeCreate = 1
aclUpdateTypeModify = 2
aclUpdateTypeDelete = 3

# -----------------------------------------------------------------------
#     CliHook - Undefined Reference Check
# This hook is invoked when an access-list is attempted to be deleted.
# The extension implementing this check must provide an interface
# that accepts the same arguments of function `_noAclConfigMode`
# @see _noAclConfigMode
aclUnconfReferencesHook = CliExtensions.CliHook()

def registerMirroringCallback( nameFn, validator ):
   gv.mirrorSessionNameFn = nameFn
   gv.mirrorValidator = validator

def dpAclDIParsingSupported( mode, token ):
   if status.dpAclDIParsingSupported:
      return None
   return CliParser.guardNotThisPlatform

def dpAclSctpPortMatchSupported( mode, token ):
   if status.dpAclSctpPortMatchSupported:
      return None
   return CliParser.guardNotThisPlatform

def countersPerChipEnabledGuard( mode, token ):
   # platform supports ACL counters per chip
   if status.dpCountersPerChipEnabled:
      return None
   return CliParser.guardNotThisPlatform

def serviceAclGuard( mode, token ):
   return None

def allHwStatusEnabled():
   for inst in statusDp.itervalues():
      if not inst.waitForHwStatus:
         return False
   return True

def _hwStatusPresent():
   return bool( len( statusDp ) )

def _sessionCheckpoint():
   # BUG193181: Avoid using thread local data here
   if getattr( threadLocalData_, 'sessionCheckpoint', None ) is None:
      sessionCheckpoint = Tac.newInstance( "Acl::CheckpointStatus",
                                           "acl-counter-checkpoint" )
      for t in AclLib.aclTypes:
         sessionCheckpoint.newStatus( t )
      threadLocalData_.sessionCheckpoint = sessionCheckpoint

   return threadLocalData_.sessionCheckpoint

def registerAclConfigValidCallback( callback ):
   # Any function that wants to make sure that after the 
   # updation of an ACL, the configuration is valid registers here.
   # On denial the updation of ACL will be reverted back.
   #
   # callback parameters:
   # @mode: mode for the Cli
   # @aclName: the acl name which is updated
   # @aclType: the acl type which is updated
   # @aclUpdateType : aclUpdateTypeCreate ==> when an ACL is created
   #                  aclUpdateTypeDelete ==> when an ACL is deleted
   #                  aclUpdateTypeModify ==> when an ACL is modified like
   #                                          rules added/deleted/modified
   # @timestamp : acl config operation timestamp. Used by policy Cli plugin.
   #
   # callback return:
   # ( status, errStr ):
   #    ==> On success, status == True.
   #    ==> On failure, status == False and 
   #                    errStr will describe reason for failure.
   aclConfigValidCallbacks.append( callback )

def aclApiForAcl( mode, aclName, aclType, apiType, standard=False, dynamic=False ):
   if mode.session.secureMonitor():
      intfConfig = gv.intfConfigSecureMonitor
   else:
      intfConfig = gv.intfConfigCli
   # Since C++ expects a fully mounted directory, mounts must be forced.
   aclApiSm = Tac.Type( "AclApi::ApiSm" )( aclName, aclType, standard, apiType,
                                           cliConfig.force(), paramConfig.force(),
                                           intfConfig.force(),
                                           hwConfig.force(), status.force(),
                                           statusDp.force(), udfAlias.force() )
   aclApiSm.inConfigSession = mode.session_.inConfigSession()
   aclApiSm.guardsEnabled = mode.session_.guardsEnabled()
   aclApiSm.dynamic = dynamic
   aclApiSm.secureMonitor = mode.session.secureMonitor()
   return aclApiSm

def isBothInnerIpv4Ipv6Configured( mode, currCfgAfterCommit, aclName, aclType ):
   innerIpConfigured = False
   innerIp6Configured = False
   ruleById = None
   if aclType == 'ip':
      ruleById = currCfgAfterCommit.ipRuleById
   elif aclType == 'ipv6':
      ruleById = currCfgAfterCommit.ip6RuleById
   if ruleById:
      for ruleId in ruleById:
         if ruleById[ ruleId ].filter.matchInnerIp:
            innerIpConfigured = True
         if ruleById[ ruleId ].filter.matchInnerIp6:
            innerIp6Configured = True
   return True if innerIpConfigured and innerIp6Configured else False

def tryWaitForApiStatus( aclApi ):
   try:
      # In runtime environment, sleep is set to True
      # In cohab breadth test, when ACL_COHAB_TEST is set, sleep is False
      #    so that Tac.waitFor() will call Tac.runActivities()
      # see Bug/159679
      sleep = False if os.environ.get( 'ACL_COHAB_TEST' ) else True
      Tac.waitFor( lambda: aclApi.apiState != 'waitingForHw', warnAfter=3.0,
                   sleep=sleep, maxDelay=0.1, timeout=AclCliLib.hwWaitTimeout,
                   description="Acl configuration to be applied" )
      return aclApi.apiState, aclApi.errString
   except Tac.Timeout:
      err = {}
      err[ 'error' ] = "Operation timed out"
      return err, err
   except KeyboardInterrupt:
      err = {}
      err[ 'error' ] = "Keyboard Interrupt"
      return err, err

seqnumMatcher = CliMatcher.IntegerMatcher( 1, AclLib.MAX_SEQ,
      helpdesc='Index in the sequence' )

# Acl name, auto-completes all existing Acls
ipAclNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: cliConfig.config[ 'ip' ].acl,
   'Access-list name', priority=CliParser.PRIO_LOW )
ip6AclNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: cliConfig.config[ 'ipv6' ].acl,
   'Access-list name', priority=CliParser.PRIO_LOW )
ipOrIp6AclNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: itertools.chain( cliConfig.config[ 'ip' ].acl,
                                 cliConfig.config[ 'ipv6' ].acl ),
   'Access-list name', priority=CliParser.PRIO_LOW )
macAclNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: cliConfig.config[ 'mac' ].acl,
   'Access-list name', priority=CliParser.PRIO_LOW )
# User-defined acl names
# Note: the namesFn is really o(n), so it's a bit more expensive. However,
# DynamicNameRule only calls namesFn() for completion, so it's not a real
# performance problem.
userIpAclNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: ( x for x, v in cliConfig.config[ 'ip' ].acl.iteritems( )
                  if not v.readonly ),
   'Access-list name' )
standardIpAclNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: ( x for x, v in cliConfig.config[ 'ip' ].acl.iteritems( )
                  if v.standard ),
   'Standard access-list name' )
notStandardIpAclNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: ( x for x, v in cliConfig.config[ 'ip' ].acl.iteritems( )
                  if not v.standard ),
   'Access-group name' )
userIp6AclNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: ( x for x, v in cliConfig.config[ 'ipv6' ].acl.iteritems( )
                  if not v.readonly ),
   'IPv6 access-list name' )
standardIp6AclNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: ( x for x, v in cliConfig.config[ 'ipv6' ].acl.iteritems( )
                  if v.standard ),
   'Standard IPv6 access-list name' )
notStandardIp6AclNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: ( x for x, v in cliConfig.config[ 'ipv6' ].acl.iteritems( )
                  if not v.standard ),
   'IPv6 access-group name' )
userMacAclNameMatcher = CliMatcher.DynamicNameMatcher(
   # no read-only MAC ACLs
   lambda mode: cliConfig.config[ 'mac' ].acl,
   'Access-list name' )
notStandardMacAclNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: ( x for x, v in cliConfig.config[ 'mac' ].acl.iteritems( )
                  if not v.standard ),
   'MAC access-group name' )

class AclNameMatcher( CliMatcher.DynamicNameMatcher ):
   ''' Accept standard ACL names, and allow for certain keywords
   to be excluded as a match.'''
   __slots__ = ()
   def __init__( self, excludeKeywords=() ):
      excludePattern = ''.join( BasicCliUtil.notAPrefixOf( k )
            for k in excludeKeywords )
      pattern = excludePattern + r'[A-Za-z0-9_:{}\[\]-]*'
      CliMatcher.DynamicNameMatcher.__init__( self,
            lambda mode: ( k for k, v in cliConfig.config[ 'ip' ].acl.items()
               if v.standard ),
            'Standard access-list name', pattern=pattern,
            partialPattern=pattern )

def _getVrfNames( mode=None ):
   vrfNames = allVrfConfig.vrf.members()
   vrfNames.append( AclLib.defaultVrf )
   return vrfNames

def clearCheckpoint( aclName, aclType ):
   # delete local/global checkpoint for aclName
   if aclName in checkpoint.status[ aclType ].acl:
      del checkpoint.status[ aclType ].acl[ aclName ]

   sessionCheckpoint = _sessionCheckpoint()
   if sessionCheckpoint and aclName in \
          sessionCheckpoint.status[ aclType ].acl:
      del sessionCheckpoint.status[ aclType ].acl[ aclName ]

def printUnsupportedRulesWarning( mode, name, aclType, errMsg="" ):
   mode.addWarning( AclCliLib.unsuppRulesWarning( name, aclType, errMsg ) )

def getUnsupportedQualifiersActionsForWarning( name, aclType, aclStatusDp,
                                               feature=None, linecard=None ):
   quals, actions = getUnsupportedQualifierActionList( name, aclType, aclStatusDp,
                                                       feature, linecard )
   warnings = [ 'Key field ' + val + ' not supported' for val in quals ] + \
              [ 'Action ' + val + ' not supported' for val in actions ]
   return ', '.join( warnings )

def getUnsupportedQualifierActionList( name, aclType, aclStatusDp,
                                       feature=None, linecard=None ):
   qualSet = set()
   actionSet = set()
   for inst in aclStatusDp.itervalues():
      if feature:
         unsuppComponent = inst.unsupportedComponent.get( feature )
         uComponents = [ unsuppComponent ] if unsuppComponent else []
      else:
         uComponents = inst.unsupportedComponent.values()
      for uComponent in uComponents:
         if linecard:
            lComponent = uComponent.linecardUComponent.get( linecard )
            uLComponents = [ lComponent ] if lComponent else []
         else:
            uLComponents = uComponent.linecardUComponent.values()
         for uLComponent in uLComponents:
            uAclTypeComponent = uLComponent.aclTypeUComponent.get( aclType )
            if uAclTypeComponent:
               components = uAclTypeComponent.unsupportedComponent.get( name )
               if components:
                  qualSet.update( components.qual.values() )
                  actionSet.update( components.action.values() )
   return sorted( list( qualSet ) ), sorted( list( actionSet ) )

def maybePrintUnsupportedRulesWarning( mode, aclName, aclType, aclStatusDp,
                                       feature=None, linecard=None ):
   unsupportedQualifiersAndActions = \
           getUnsupportedQualifiersActionsForWarning( aclName, aclType, aclStatusDp,
                                                      feature, linecard )
   if unsupportedQualifiersAndActions:
      printUnsupportedRulesWarning( mode, aclName, aclType,
                                    unsupportedQualifiersAndActions )

def printDpiRulesWarning( mode, name, aclType ):
   mode.addWarning( AclCliLib.dpiRulesWarning( name, aclType ) )

def getAclTypeConfig( aclType ):
   return config.config[ aclType ]

def printNotConfiguredError( mode, aclType, aclName, intfName ):
   mode.addWarning( AclCliLib.notConfiguredError( aclType, aclName, intfName ) )

def getAclConfig( aclType, cliConf=False ):
   conf = ( cliConfig if cliConf else config )
   return conf.config[ aclType ].acl

def getAclCurrConfig( aclType, aclName, cliConf=False ):
   aclConfig = getAclConfig( aclType, cliConf ) 
   if aclConfig:
      cfg = aclConfig.get( aclName )
      if cfg:
         return cfg.currCfg

   # probably ACL still doesn't exit
   return None

# Handlers invoked after ACL config changes are committed to Sysdb in
# CLI config sessions.
#
# There are two types of handlers. One is a global handler that is not
# dependent on ACL names. _aclSessionCommitHandler() simply waits for
# hardware status and see if all ACL configurations are committed
# successfully. We only need to invoke it once no matter how many ACL
# operations are performed. Then there are handlers that we need to 
# invoke per ACL (clearCheckpoint). 

def _aclSessionCommitHandler( mode, aclApi ):
   aclApi.inConfigSession = False
   aclApi.waitForHw()
   ret, errStr = tryWaitForApiStatus( aclApi )

   if ret != 'succeeded':
      mode.addWarning( AclCliLib.aclSessionCommitError( 
            ",".join( errStr.values() ) ) )
      # check 'all' status to see if system supports rollback
      # if rollback is supported return 'ACL_COMMIT_FAIL'
      if status.dpAclRollbackSupported:
         return "ACL_COMMIT_FAIL"
   return None

def _aclSessionCommitHandlerClearCheckpoint( mode, aclName, aclType ):
   clearCheckpoint( aclName, aclType )

def registerAclSessionOnCommitHandler( mode, aclApi, doClearCheckPoint ):
   # This handler covers all ACL operations. So if we made multiple ACL
   # operations in one session, only one handler will be registered.
   CliSession.registerSessionOnCommitHandler( 
      mode.session_.entityManager, 
      "ACL", 
      lambda m, onSessionCommit: _aclSessionCommitHandler( m, aclApi ) )
   if doClearCheckPoint:
      CliSession.registerSessionOnCommitHandler( 
      mode.session_.entityManager, 
      "ACL:%s.%s" % ( aclApi.aclType, aclApi.aclName ), 
      lambda m, onSessionCommit: 
      _aclSessionCommitHandlerClearCheckpoint( m, aclApi.aclName, 
                                               aclApi.aclType ) )

#------------------------------------------------------------
# config-acl-_aclNameExprGenerator mode
#------------------------------------------------------------
class BaseAclConfigMode( BaseAclMode, BasicCli.ConfigModeBase ):
   name = "ACL Configuration"
   modeParseTree = CliParser.ModeParseTree()
   autoConfigSessionAllowed = False
   showActiveCmdRegistered_ = True

   def __init__( self, parent, session, aclName, aclType,
                 standard=False, dynamic=False ):
      self.aclName = aclName
      self.aclType = aclType
      self.standard = standard
      self.dynamic = dynamic

      # We already have a previous implementation for 'show active'.
      BaseAclMode.__init__( self, aclType, standard, aclName, self.dynamic )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

      self.aclApi = aclApiForAcl( self, self.aclName, self.aclType, 'commit', 
                                  self.standard, dynamic=self.dynamic )

   def commitAcl( self ):
      """ This supports the initial implementation of the CLI Config Sessions
      feature (config-replace only, no config sessions). CommitAcl is factored
      into two parts. The first part updates the config, but does not wait for
      agent responses. Agents will not see the config changes if we are doing
      a config-replace. The second part is executed after the session is
      committed.

      This code must be refactored when the full CLI Config Sessions feature is
      enabled. """
      aclApi = self.aclApi
      if aclApi:
         currCfgBeforeCommit = getAclCurrConfig( self.aclType, self.aclName, 
                                                 cliConf=True )
         origHwStatusOK = _hwStatusOK()
         ret = aclApi.commitExt( False )

         if ret not in ( 'succeeded', 'waitingForHw', 'errHw' ):
            if ret == 'errSessionCreate':
               self.addError(
                  AclCliLib.aclCommitError( self.aclName, self.aclType, 
                                            'Created by another session' ) )
            elif ret == 'errSessionDelete':
               self.addError(
                  AclCliLib.aclCommitError( self.aclName, self.aclType, 
                                            'Deleted by another session' ) )
            elif ret == 'errSessionModify':
               self.addError(
                  AclCliLib.aclCommitError( self.aclName, self.aclType, 
                                            'Modified by another session' ) )
            else:
               errMsg = ",".join( aclApi.errString.values() )
               self.addError(
                  AclCliLib.aclCommitError( self.aclName, self.aclType,
                                            errMsg ) )
            return
         if not aclApi.dpAclOk():
            errMsg = ",".join( aclApi.errString.values() )
            printUnsupportedRulesWarning( self, self.aclName, self.aclType, errMsg )
         elif not aclApi.dpAclDpiOk():
            printDpiRulesWarning( self, self.aclName, self.aclType )

         if self.session.inConfigSession():
            registerAclSessionOnCommitHandler( self, aclApi, not aclApi.exists() )
            # we should register the aclConfigValidCallback() handler
            # after BUG77723 is resolved
            return

         if not aclApi.exists():
            clearCheckpoint( self.aclName, self.aclType )

         if not _hwStatusPresent():
            self.addWarning( AclCliLib.noHwWarningMsg )

         ret, errStr = tryWaitForApiStatus( aclApi )

         if ret != 'succeeded':
            if origHwStatusOK:
               self.addError(
                  AclCliLib.aclCommitError( self.aclName, self.aclType,
                                            ",".join( errStr.values() ) ) )
               t0( "Reverting..." )
               aclApi.revert()
               tryWaitForApiStatus( aclApi )
            else:
               # cleanup previous config
               t8( "Cleaning up previous subConfig" )
               aclApi.cleanupOldConfig()
         else:
            maybePrintUnsupportedRulesWarning( self, self.aclName, self.aclType,
                                               statusDp )

            currCfgAfterCommit = getAclCurrConfig( self.aclType, self.aclName,
                                                   cliConf=True )
            if getattr( currCfgBeforeCommit, 'version', None ) != \
                  getattr( currCfgAfterCommit, 'version', None ):
               if currCfgBeforeCommit:
                  updateType = aclUpdateTypeModify
               else:
                  updateType = aclUpdateTypeCreate

               t8( "ACL updated", self.aclName, self.aclType, updateType )

               # loop and go thru all registered callbacks
               for aclConfigValidCallback in aclConfigValidCallbacks:
                  ( cbStatus, err ) = aclConfigValidCallback( self,
                                                              self.aclName, 
                                                              self.aclType, 
                                                              updateType )
                  errStr = ",".join( err.values() ) if err else 'Unknown'
                  if 'Operation timed out' in errStr:
                     self.addWarning( AclCliLib.aclTimeoutWarning() )
                     return
                  if not cbStatus:
                     if origHwStatusOK:
                        errStr = ",".join( err.values() ) if err else 'Unknown'
                        self.addError( 
                           AclCliLib.aclCommitError( self.aclName, self.aclType, 
                                                     errStr ) )
                        #revert since someone failed due to a recent ACL change
                        t0( "Reverting..." )
                        aclApi.revert()
                        tryWaitForApiStatus( aclApi )
                        return
                  else:
                     if 'warning' in err:
                        self.addWarning( err[ 'warning' ] )

               # When inner ip is supported, we can configure either inner ipv4
               # or inner ipv6 in one Acl, but not both.
               if status.dpAclMplsOverGreSupported and \
                     isBothInnerIpv4Ipv6Configured( self,
                           currCfgAfterCommit,
                           self.aclName,
                           self.aclType ):
                  self.addError(
                     AclCliLib.aclCommitError( self.aclName, self.aclType,
                        "ACL contains both inner IPv4 and IPv6 qualifiers,"
                              " which is not supported" ) )
                  aclApi.revert()
                  tryWaitForApiStatus( aclApi )

            # cleanup previous config
            t8( "Cleaning up previous subConfig" )
            aclApi.cleanupOldConfig()

   def onExit( self ):
      self.commitAcl( )
      BasicCli.ConfigModeBase.onExit( self )

class IpAclConfigMode( BaseAclConfigMode ):
   name = "ACL Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, aclName, dynamic=False ):
      BaseAclConfigMode.__init__( self, parent, session, aclName, 'ip',
                                  dynamic=dynamic )

class IpStdAclConfigMode( BaseAclConfigMode ):
   name = "Standard ACL Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, aclName, dynamic=False ):
      BaseAclConfigMode.__init__( self, parent, session, aclName, 'ip',
                                  standard=True, dynamic=dynamic )

class Ip6AclConfigMode( BaseAclConfigMode ):
   name = "IPv6 ACL Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, aclName, dynamic=False ):
      BaseAclConfigMode.__init__( self, parent, session, aclName, 'ipv6',
                                  dynamic=dynamic )

class Ip6StdAclConfigMode( BaseAclConfigMode ):
   name = "Standard IPv6 ACL Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, aclName, dynamic=False ):
      BaseAclConfigMode.__init__( self, parent, session, aclName, 'ipv6',
                                  standard=True, dynamic=dynamic )

class MacAclConfigMode( BaseAclConfigMode ):
   name = "MAC ACL Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, aclName, dynamic=False ):
      BaseAclConfigMode.__init__( self, parent, session, aclName, 'mac',
                                  dynamic=dynamic )

allAclConfigModes = ( IpAclConfigMode, IpStdAclConfigMode,
                      Ip6AclConfigMode, Ip6StdAclConfigMode,
                      MacAclConfigMode )

#---------------------------------------------------------------------
# ip|ipv6|mac access-list name
#---------------------------------------------------------------------
def _gotoAclConfigMode( mode, aclName, aclType, filterType='extended',
                        dynamic=False ):
   standard = ( filterType == 'standard' )
   dynamic = bool( dynamic )
   secureMonitor = mode.session.secureMonitor()

   # Allow the same name to exist in both IP and MAC ACLs
   aclConfig = getAclConfig( aclType, cliConf=True )
   cfg = aclConfig.get( aclName )

   if cfg:
      # filter type has to be the same
      if cfg.standard != standard:
         mode.addError(
            AclCliLib.aclModifyError( aclName, aclType,
                                      'Not %s access-list' % 
                                      filterType ) )
         return
      if cfg.dynamic != dynamic:
         mode.addError(
            AclCliLib.aclModifyError( aclName, aclType,
                                      'Not dynamic access-list' ) )
         return
      if cfg.secureMonitor != secureMonitor:
         mode.addError(
            AclCliLib.aclModifyError( aclName, aclType,
                                      'Not secure-monitor access-list' ) )
         return
      # Check if current config is committed.
      # If it is not committed and was recently created, it's
      # likely being committed at the moment, so return.
      cfg = cfg.currCfg
      timestamp = long( Tac.now( ) )
      if cfg and ( not cfg.readonly ) and ( not cfg.committed ) and \
             ( timestamp - cfg.version.timestamp <= AclCliLib.hwWaitTimeout ):
         mode.addError(
            AclCliLib.aclModifyError( aclName, aclType,
                                      'Being committed by another session' ) )
         return

   if aclType == 'ip':
      cmode = IpStdAclConfigMode if standard else IpAclConfigMode
      childMode = mode.childMode( cmode, aclName=aclName, dynamic=dynamic )
   elif aclType == 'ipv6':
      cmode = Ip6StdAclConfigMode if standard else Ip6AclConfigMode
      childMode = mode.childMode( cmode, aclName=aclName, dynamic=dynamic )
   else:
      childMode = mode.childMode( MacAclConfigMode, aclName=aclName,
                                  dynamic=dynamic )
   mode.session_.gotoChildMode( childMode )

#---------------------------------------------------------------------
# no ip access-list name
#---------------------------------------------------------------------
@CliExtensions.executeCliHookBefore( aclUnconfReferencesHook )
def _noAclConfigMode( mode, aclName, aclType, filterType='extended' ):

   lastMode = mode.session_.modeOfLastPrompt()
   standard = ( filterType == 'standard' )
   if isinstance( lastMode, BaseAclConfigMode ) and lastMode.aclName == aclName \
          and lastMode.aclType == aclType and lastMode.standard == standard:
      # deleting self inside the config-acl mode, so discard the config
      # since we'll succeed
      lastMode.aclApi = None

   if not _hwStatusPresent():
      mode.addWarning( AclCliLib.noHwWarningMsg )

   currCfgBeforeCommit = getAclCurrConfig( aclType, aclName, cliConf=True )
   aclApi = aclApiForAcl( mode, aclName, aclType, 'remove', standard )
   ret = aclApi.remove()
   if ret == 'succeeded':

      if mode.session.inConfigSession():
         registerAclSessionOnCommitHandler( mode, aclApi, True )
         return

      clearCheckpoint( aclName, aclType )
      t8( "ACL updated ", aclName, aclType, aclUpdateTypeDelete )
      # we should register the aclConfigValidCallback() handler
      # after BUG77723 is resolved
      currCfgAfterCommit = getAclCurrConfig( aclType, aclName, cliConf=True )
      if getattr( currCfgBeforeCommit, 'version', None ) != \
                  getattr( currCfgAfterCommit, 'version', None ):
         for aclConfigValidCallback in aclConfigValidCallbacks:
            ( cbStatus, err ) = aclConfigValidCallback( mode,
                                                        aclName, 
                                                        aclType, 
                                                        aclUpdateTypeDelete )
            if not cbStatus:
               errStr = ",".join( err.values() ) if err else 'Unknown'
               errStr = 'Not %s access-list. %s' % ( filterType, errStr )
               mode.addWarning( AclCliLib.aclModifyError( aclName, aclType, 
                                                          errStr ) )
   elif ret == 'errStandardAcl':
      mode.addError( AclCliLib.aclModifyError( aclName, aclType,
                                   'Not %s access-list' % filterType ) )
   elif ret == 'errReadOnly':
      mode.addError(
         AclCliLib.aclModifyError( aclName, aclType, 'Readonly list' ) )

accessListMatcher = CliMatcher.KeywordMatcher( 'access-list',
      helpdesc='Named access-lists' )
accessListKwMatcherForServiceAcl = CliMatcher.KeywordMatcher( 'access-list',
      helpdesc='Named access-list' )
aclSummaryKwMatcher = CliMatcher.KeywordMatcher( 'summary',
      helpdesc='Access list summary' )
aclDynamicKwMatcher = CliMatcher.KeywordMatcher( 'dynamic',
      helpdesc='Dynamic (non-persistent) access-list' )
matcherDetail = CliMatcher.KeywordMatcher( 'detail', helpdesc='Access list detail' )
nodeDetail = CliCommand.Node( matcher=matcherDetail,
      guard=countersPerChipEnabledGuard )

#---------------------------------------------------------------------
# <ip|mac> access-list payload alias <alias> [ header <start|end> ]
# offset <0-31> pattern <pattern> mask <mask>
#---------------------------------------------------------------------
def handleAlias( mode, aliasPayload, aliasCollection, aliasType='ip' ):
   udfAliasPayload = {}
   aliasName = aliasPayload[ 'payload' ].alias
   update = False
   if aliasName in aliasCollection:
      udfAliasPayload = aliasCollection[ aliasName ]
      if aliasType == 'ip' and \
            udfAliasPayload.headerStart != aliasPayload[ 'headerStart' ]:
         raise CliCommon.InvalidInputError( "Header start/end cannot be modified" )
      update = True
   else:
      if aliasType == 'ip':
         udfAliasPayload = udfAlias.newIpAlias( aliasName )
      else:
         udfAliasPayload = udfAlias.newMacAlias( aliasName )
   udfAliasPayload.payload = aliasPayload[ 'payload' ]
   if aliasType == 'ip' and aliasPayload[ 'headerStart' ] != 'end':
      udfAliasPayload.headerStart = AclLib.AclPayloadHdrStart.start

   if update:
      for aclName in aliasCollection[ aliasName ].aceListForAcl:
         aclApi = aclApiForAcl( mode, aclName, aliasType, 'commit' )
         # pylint: disable-msg=maybe-no-member
         aclApi.updateUdfAliasInAcls( aliasName, udfAliasPayload.payload )
         _aclSessionCommitHandler( mode, aclApi )

def noAlias( mode, aliasName, udfAliasColl ):
   if aliasName in udfAliasColl:
      if len( udfAliasColl[ aliasName ].aceListForAcl ) == 0:
         del udfAliasColl[ aliasName ]
      else:
         mode.addError( 'Error: Alias %s is currently being used, cannot delete' \
                        % aliasName )

class IpAliasCmd( CliCommand.CliCommandClass ):
   syntax = 'ip access-list ALIAS_PAYLOAD'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'access-list': accessListMatcher,
      'ALIAS_PAYLOAD': AclCliRules.UdfAliasMatcher( status=status )
   }

   @staticmethod
   def handler( mode, args ):
      handleAlias( mode, args[ 'ALIAS_PAYLOAD' ], udfAlias.ipAlias, aliasType='ip' )

BasicCli.GlobalConfigMode.addCommandClass( IpAliasCmd )

class NoOrDefaultIpAliasCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ip access-list payload ALIAS'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'access-list': accessListMatcher,
      'payload': AclCliRules.nodePayloadFactory( status=status ),
      'ALIAS': AclCliRules.AliasNameMatcher()
   }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noAlias( mode, args[ 'ALIAS' ], udfAlias.ipAlias )

BasicCli.GlobalConfigMode.addCommandClass( NoOrDefaultIpAliasCmd )

class MacAliasCmd( CliCommand.CliCommandClass ):
   syntax = 'mac access-list ALIAS_PAYLOAD'
   data = {
      'mac': CliToken.Mac.macMatcherForConfig,
      'access-list': accessListMatcher,
      'ALIAS_PAYLOAD': AclCliRules.UdfAliasMatcher( status=status, macFilter=True )
   }

   @staticmethod
   def handler( mode, args ):
      handleAlias( mode, args[ 'ALIAS_PAYLOAD' ], udfAlias.macAlias,
            aliasType='mac' )

BasicCli.GlobalConfigMode.addCommandClass( MacAliasCmd )

class NoOrDefaultMacAliasCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'mac access-list payload ALIAS'
   data = {
      'mac': CliToken.Mac.macMatcherForConfig,
      'access-list': accessListMatcher,
      'payload': AclCliRules.nodePayloadFactory( status=status, macFilter=True ),
      'ALIAS': AclCliRules.AliasNameMatcher( macFilter=True )
   }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noAlias( mode, args[ 'ALIAS' ], udfAlias.macAlias )

BasicCli.GlobalConfigMode.addCommandClass( NoOrDefaultMacAliasCmd )

# Support both standard and extended IP ACLs. 'extended' is optional,
# but we keep it hidden for industry standard compatibility.
standardNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'standard',
         helpdesc='Standard access-list' ),
      alias='FILTER' )
extendedNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'extended',
         helpdesc='Extended access-list' ),
      alias='FILTER',
      hidden=True )

class IpAclConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'ip access-list [ standard | extended ] ACL_NAME [ dynamic ]'
   noOrDefaultSyntax = 'ip access-list [ standard | extended ] ACL_NAME ...'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'access-list': accessListMatcher,
      'standard':  standardNode,
      'extended': extendedNode,
      'ACL_NAME': userIpAclNameMatcher,
      'dynamic': aclDynamicKwMatcher,
   }
   autoConfigSessionAllowed = False

   handler = lambda mode, args: _gotoAclConfigMode( mode, args[ 'ACL_NAME' ],
         'ip', filterType=args.get( 'FILTER', 'extended' ),
         dynamic='dynamic' in args )
   noOrDefaultHandler = lambda mode, args: _noAclConfigMode( mode,
         args[ 'ACL_NAME' ], 'ip', filterType=args.get( 'FILTER', 'extended' ) )

BasicCli.GlobalConfigMode.addCommandClass( IpAclConfigCmd )

class Ipv6AclConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 access-list [ standard | extended ] ACL_NAME [ dynamic ]'
   noOrDefaultSyntax = 'ipv6 access-list [ standard | extended ] ACL_NAME ...'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
      'access-list': accessListMatcher,
      'standard':  standardNode,
      'extended': extendedNode,
      'ACL_NAME': userIp6AclNameMatcher,
      'dynamic': aclDynamicKwMatcher,
   }
   autoConfigSessionAllowed = False

   handler = lambda mode, args: _gotoAclConfigMode( mode, args[ 'ACL_NAME' ],
         'ipv6', filterType=args.get( 'FILTER', 'extended' ),
         dynamic='dynamic' in args )
   noOrDefaultHandler = lambda mode, args: _noAclConfigMode( mode,
         args[ 'ACL_NAME' ], 'ipv6', filterType=args.get( 'FILTER', 'extended' ) )

BasicCli.GlobalConfigMode.addCommandClass( Ipv6AclConfigCmd )

class MacAclConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'mac access-list [ extended ] ACL_NAME [ dynamic ]'
   noOrDefaultSyntax = 'mac access-list [ extended ] ACL_NAME ...'
   data = {
      'mac': CliToken.Mac.macMatcherForConfig,
      'access-list': accessListMatcher,
      'extended': extendedNode,
      'ACL_NAME': userMacAclNameMatcher,
      'dynamic': aclDynamicKwMatcher,
   }
   autoConfigSessionAllowed = False

   handler = lambda mode, args: _gotoAclConfigMode( mode, args[ 'ACL_NAME' ],
                                                    'mac',
                                                    dynamic='dynamic' in args )
   noOrDefaultHandler = lambda mode, args: _noAclConfigMode( mode,
                                                             args[ 'ACL_NAME' ],
                                                             'mac' )

BasicCli.GlobalConfigMode.addCommandClass( MacAclConfigCmd )

#------------------------------------------------------------
# [ no | default ] service configuration access-list symbols
#------------------------------------------------------------
class ConvertSymbolsCmd( CliCommand.CliCommandClass ):
   syntax = 'service configuration access-list symbols'
   noOrDefaultSyntax = syntax
   data = {
      'service': CliToken.Service.serviceMatcherForConfig,
      'configuration': CliToken.Service.configMatcherForAfterService,
      'access-list': 'Access list configuration related configuration',
      'symbols': 'Use symbols when printing ACL qualifier values'
   }

   @staticmethod
   def handler( mode, args ):
      paramConfig.convertSymbols = True

   @staticmethod
   def noHandler( mode, args ):
      paramConfig.convertSymbols = False

   defaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( ConvertSymbolsCmd )

from AclCmdChains import AclCfgCmdChains

class AclCfgCmdBinder( AclCfgCmdChains ):
   def __init__( self, status ):    # pylint: disable-msg=W0621
      AclCfgCmdChains.__init__( self, status )
      cls = self.__class__

      ########################################################
      #                       IP filters
      ########################################################
      class IpAclCommand( CliCommand.CliCommandClass ):
         syntax = '[ SEQ_NUM ] ACTION [ VLAN_EXPR ] IP_FILTER [ RULE_ACTION ]'
         noOrDefaultSyntax = 'ACTION [ VLAN_EXPR ] IP_FILTER [ RULE_ACTION ]'
         data = {
            'SEQ_NUM': seqnumMatcher,
            'ACTION': self.actionNode,
            'VLAN_EXPR': self.vlanExpression,
            'IP_FILTER': self.ipFilterExpression,
            'RULE_ACTION': self.ipRuleAction,
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            cls.adaptRuleCommonArgs( mode, args )
            cls._handleIpArgs( mode, args ) # pylint: disable-msg=protected-access

         @staticmethod
         def handler( mode, args ):
            if CliCommand.isNoOrDefaultCmd( args ):
               cls.adaptNoOrDefaultRuleCommonArgs( mode, args )
            cls.handleIpRule( mode, **args )

         noOrDefaultHandler = handler

      IpAclConfigMode.addCommandClass( IpAclCommand )

      ########################################################
      #             Standard IP filters
      ########################################################
      class IpStdAclCommand( CliCommand.CliCommandClass ):
         syntax = '[ SEQ_NUM ] ACTION [ VLAN_EXPR ] SOURCE [ RULE_ACTION ]'
         noOrDefaultSyntax = 'ACTION [ VLAN_EXPR ] SOURCE [ RULE_ACTION ]'

         data = {
            'SEQ_NUM': seqnumMatcher,
            'ACTION': self.actionNode,
            'VLAN_EXPR': self.vlanExpression,
            'SOURCE': self.ipSourceMatcher,
            'RULE_ACTION': self.ipRuleAction,
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            cls.adaptRuleCommonArgs( mode, args )
            source = args.pop( 'source' )
            # pylint: disable-msg=protected-access
            args[ 'ipFilter' ] = self._handleIpFilter( mode, IPPROTO_IP, source,
                  AclLib.anyIpAddr )
 
         @staticmethod
         def handler( mode, args ):
            if CliCommand.isNoOrDefaultCmd( args ):
               cls.adaptNoOrDefaultRuleCommonArgs( mode, args )
            cls.handleIpStdRule( mode, **args )

         noOrDefaultHandler = handler

      IpStdAclConfigMode.addCommandClass( IpStdAclCommand )

      ########################################################
      #                    IPv6 filters
      ########################################################
      class Ip6AclCommand( CliCommand.CliCommandClass ):
         syntax = '[ SEQ_NUM ] ACTION [ VLAN_EXPR ] IP6_FILTER [ RULE_ACTION ]'
         noOrDefaultSyntax = 'ACTION [ VLAN_EXPR ] IP6_FILTER [ RULE_ACTION ]'
         data = {
            'SEQ_NUM': seqnumMatcher,
            'ACTION': self.actionNode,
            'VLAN_EXPR': self.vlanExpression,
            'IP6_FILTER': self.ip6FilterExpression,
            'RULE_ACTION': self.ip6RuleAction
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            cls.adaptRuleCommonArgs( mode, args )
            self._handleIp6Args( mode, args ) # pylint: disable-msg=protected-access

         @staticmethod
         def handler( mode, args ):
            if CliCommand.isNoOrDefaultCmd( args ):
               cls.adaptNoOrDefaultRuleCommonArgs( mode, args )
            cls.handleIp6Rule( mode, **args )

         noOrDefaultHandler = handler

      Ip6AclConfigMode.addCommandClass( Ip6AclCommand )

      ########################################################
      #             Standard IPv6 filters
      ########################################################
      class Ip6StdAclCommand( CliCommand.CliCommandClass ):
         syntax = '[ SEQ_NUM ] ACTION [ VLAN_EXPR ] SOURCE [ RULE_ACTION ]'
         noOrDefaultSyntax = 'ACTION [ VLAN_EXPR ] SOURCE [ RULE_ACTION ]'
         data = {
            'SEQ_NUM': seqnumMatcher,
            'ACTION': self.actionNode,
            'VLAN_EXPR': self.vlanExpression,
            'SOURCE': self.ip6SourceMatcher,
            'RULE_ACTION': self.ip6RuleAction
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            cls.adaptRuleCommonArgs( mode, args )
            source = args.pop( 'source' )
            # pylint: disable-msg=protected-access
            args[ 'ipFilter' ] = self._handleIp6Filter( mode, IPPROTO_IP, source,
                  AclLib.anyIp6AddrWithFullMask )

         @staticmethod
         def handler( mode, args ):
            if CliCommand.isNoOrDefaultCmd( args ):
               cls.adaptNoOrDefaultRuleCommonArgs( mode, args )
            cls.handleIp6StdRule( mode, **args )

         noOrDefaultHandler = handler

      Ip6StdAclConfigMode.addCommandClass( Ip6StdAclCommand )

      ########################################################
      #                       MAC filters
      ########################################################
      class MacAclCommand( CliCommand.CliCommandClass ):
         syntax = '[ SEQ_NUM ] ACTION [ VLAN_EXPR ] MAC_FILTER [ RULE_ACTION ]'
         noOrDefaultSyntax = 'ACTION [ VLAN_EXPR ] MAC_FILTER [ RULE_ACTION ]'
         data = {
            'SEQ_NUM': seqnumMatcher,
            'ACTION': self.actionNode,
            'VLAN_EXPR': self.vlanExpression,
            'MAC_FILTER': self.macFilterExpression,
            'RULE_ACTION' : self.macRuleAction,
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            cls.adaptRuleCommonArgs( mode, args )

         @staticmethod
         def handler( mode, args ):
            cls.handleMacRule( mode, **args )

         @staticmethod
         def noOrDefaultHandler( mode, args ):
            cls.adaptNoOrDefaultRuleCommonArgs( mode, args ) 
            cls.handleMacRule( mode, **args )

      MacAclConfigMode.addCommandClass( MacAclCommand )

      ########################################################
      #                       remarks
      ########################################################
      class RemarkCliExpression( CliCommand.CliExpression ):
         expression = '''[ SEQ_NUM ] remark REMARK'''
         data = { 'SEQ_NUM': seqnumMatcher,
                  'remark': self.remarkMatcher,
                  'REMARK': self.commentMatcher }

      class IpRemarkCmd( CliCommand.CliCommandClass ):
         syntax = '''<remarkExpression>'''
         noOrDefaultSyntax = '''remark REMARK'''
         data = { '<remarkExpression>': RemarkCliExpression }
         handler = cls.handleIpRemark
         noOrDefaultHandler = cls.handleIpRemark

      for m in ( IpAclConfigMode, IpStdAclConfigMode ):
         m.addCommandClass( IpRemarkCmd )

      class Ip6RemarkCmd( CliCommand.CliCommandClass ):
         syntax = '''<remarkExpression>'''
         noOrDefaultSyntax = '''remark REMARK'''
         data = { '<remarkExpression>': RemarkCliExpression }
         handler = cls.handleIp6Remark
         noOrDefaultHandler = cls.handleIp6Remark

      for m in ( Ip6AclConfigMode, Ip6StdAclConfigMode ):
         m.addCommandClass( Ip6RemarkCmd )

      class MacRemarkCmd( CliCommand.CliCommandClass ):
         syntax = '''<remarkExpression>'''
         noOrDefaultSyntax = '''remark REMARK'''
         data = { '<remarkExpression>': RemarkCliExpression }
         handler = cls.handleMacRemark
         noOrDefaultHandler = cls.handleMacRemark
      MacAclConfigMode.addCommandClass( MacRemarkCmd )

      #-----------------------------------------------------------------
      #              [no] counters per-entry
      # legacy:
      #              [no] statistics per-entry
      #-----------------------------------------------------------------
      class CountersPerEntryCmd( CliCommand.CliCommandClass ):
         syntax = 'counters | statsDeprecated per-entry'
         noOrDefaultSyntax = syntax
         data = { 'counters': self.countersParserNode,
                  'statsDeprecated': self.statsParserNode,
                  'per-entry': self.perEntryMatcher }
         handler = cls.handleStatsPerEntry
         noOrDefaultHandler = cls.handleNoStatsPerEntry

      for m in allAclConfigModes:
         m.addCommandClass( CountersPerEntryCmd )

      #-----------------------------------------------------------------
      #              [no] fragment-rules
      #-----------------------------------------------------------------
      class FragmentRulesCmd( CliCommand.CliCommandClass ):
         syntax = 'fragment-rules'
         noSyntax = 'fragment-rules'
         defaultSyntax = 'fragment-rules'
         data = { 'fragment-rules': 'Add fragment rules' }
         handler = cls.handleFragmentRules
         defaultHandler = cls.handleFragmentRules
         noHandler = cls.handleNoFragmentRules

      for m in allAclConfigModes:
         m.addCommandClass( FragmentRulesCmd )

      #-----------------------------------------------------------------
      #              no sequence-number
      #-----------------------------------------------------------------
      class NoSequenceCmd( CliCommand.CliCommandClass ):
         noOrDefaultSyntax = 'SEQUENCES'
         data = {
            'SEQUENCES': AclCliRules.seqRangeMatcher
         }
         noOrDefaultHandler = cls.handleNoSeq

      for m in allAclConfigModes:
         m.addCommandClass( NoSequenceCmd )

      #-----------------------------------------------------------------
      #              resequence
      #-----------------------------------------------------------------
      class ResequenceCmd( CliCommand.CliCommandClass ):
         syntax = 'resequence [ START [ INCREMENT ] ]'
         data = { 'resequence': 'Resequence the list',
                  'START': self.startSeqMatcher,
                  'INCREMENT': self.incSeqMatcher }

         @staticmethod
         def handler( mode, args ):
            start = args.get( 'START', self.defaultSeqStart )
            inc = args.get( 'INCREMENT', self.defaultSeqInc )
            cls.handleResequence( mode, start, inc )

      for m in allAclConfigModes:
         m.addCommandClass( ResequenceCmd )

      #-----------------------------------------------------------------
      #              show [ active | diff | pending ]
      #-----------------------------------------------------------------
      class ShowActiveDiffPendingCmd( ShowCommand.ShowCliCommandClass ):
         syntax = 'show [ active | diff | pending ]'
         data = {
            'active': 'Show the list in the current running-config',
            'diff': 'Show the difference between active and pending list',
            'pending': 'Show pending list in this session'
         }

         @staticmethod
         def handler( mode, args ):
            if 'active' in args:
               return cls.showActive( mode )
            elif 'diff' in args:
               return cls.showDiff( mode )
            else:
               return cls.showCurrent( mode )

      for m in allAclConfigModes:
         m.addShowCommandClass( ShowActiveDiffPendingCmd )

      #-----------------------------------------------------------------
      #              abort
      #-----------------------------------------------------------------
      class AbortCmd( CliCommand.CliCommandClass ):
         syntax = 'abort'
         data = { 'abort': CliToken.Cli.abortMatcher, }

         @staticmethod
         def handler( mode, args ):
            cls.abort( mode )

      for m in allAclConfigModes:
         m.addCommandClass( AbortCmd )

   @classmethod
   def _handleRuleCommon( cls, mode, seqnum, action, vlanSpec, ruleFilter,
                          mirrorSession, log, remark, copyToCpuDst, **kwargs ):
      #t8( "_handleRuleCommon", seqnum, action, vlanSpec, ruleFilter, log, remark )
      # XXX_TAPAGG Consider pushing this in ruleFilter earlier
      if mirrorSession and gv.mirrorValidator:
         err = gv.mirrorValidator( mode, mirrorSession )
         if err:
            mode.addError( err )
            return

      if vlanSpec:
         ( ruleFilter.vlan, ruleFilter.vlanMask,
                     ruleFilter.innerVlan, ruleFilter.innerVlanMask ) = vlanSpec
      aclApi = mode.aclApi
      mirrorSession = mirrorSession if mirrorSession else ''
      log = bool( log )
      copyToCpuDst = AclCliLib.copyToCpuDstFromCli( mode, copyToCpuDst )

      add = True
      if seqnum is None:
         seqnum = 0

      if seqnum is True or seqnum == 'default':
         seqnum = 0
         add = False

      if action == 'remark':
         remarkConfig = Tac.Type( "Acl::RemarkConfig" )( remark=remark )
         if add:
            ret = aclApi.addRemark( remarkConfig, seqnum )
         else:
            ret = aclApi.removeRemark( remarkConfig, seqnum )
      else: # Add non remark rules
         # sequence number is assumed to be an int for now. Others are not supported.
         if mode.aclType == 'ip':
            ipRuleConfig = Tac.Type( "Acl::IpRuleConfig" )( 
                  filter=ruleFilter,
                  action=action,
                  mirrorSession=mirrorSession,
                  log=log,
                  copyToCpuDst=copyToCpuDst )
            if add:
               ret = aclApi.addIpRule( seqnum, ipRuleConfig )
            else:
               ret = aclApi.removeIpRuleByConfig( ipRuleConfig )

         elif mode.aclType == 'ipv6':
            ip6RuleConfig = Tac.Type( "Acl::Ip6RuleConfig" )(
                  filter=ruleFilter,
                  action=action,
                  mirrorSession=mirrorSession,
                  log=log,
                  copyToCpuDst=copyToCpuDst )
            if add:
               ret = aclApi.addIp6Rule( seqnum, ip6RuleConfig )
            else:
               ret = aclApi.removeIp6RuleByConfig( ip6RuleConfig )

         elif mode.aclType == 'mac':
            macRuleConfig = Tac.Type( "Acl::MacRuleConfig" )(
                  filter=ruleFilter,
                  action=action,
                  mirrorSession=mirrorSession,
                  log=log )
            if add:
               ret = aclApi.addMacRule( seqnum, macRuleConfig )
            else:
               ret = aclApi.removeMacRuleByConfig( macRuleConfig )

      if ret == 'errRuleExists':
         mode.addError( "Error: Duplicate sequence number" )
      elif ret == 'errSequenceOutOfRange':
         mode.addError( "Error: Sequence number out of range" )
      elif ret == 'errReadOnly':
         mode.addError(
            AclCliLib.aclModifyError( mode.aclName, mode.aclType,
                                      'Readonly list' ) )

   #-----------------------------------------------------------------
   #              show active|pending|diff
   #-----------------------------------------------------------------
   @classmethod
   def _showList( cls, name, aclType, subConfig, standard, output=None ):
      if subConfig is None:
         return
      if output is None:
         output = sys.stdout
      output.write( "%s%s Access List %s\n" % (
         'Standard ' if standard else '', aclType, name ) )
      if subConfig.countersEnabled:
         print "        counters per-entry\n"
      for seq in AclCliLib.sortedSequenceNumbers( subConfig ):
         rule = AclCliLib.getRuleValue( subConfig, seq, aclType, standard,
                                        paramConfig.convertSymbols )
         output.write( "        %d %s\n" % ( seq, rule ) )
      output.write( '\n' )

   @classmethod
   def showCurrent( cls, mode ):
      aclApi = mode.aclApi
      subConfig = aclApi.currentAcl()
      cls._showList( mode.aclName, mode.aclType, subConfig, mode.standard )

   @classmethod
   def showActive( cls, mode ):
      cfg = getAclConfig( mode.aclType, cliConf=True ).get( mode.aclName )
      if cfg:
         cls._showList( mode.aclName, mode.aclType, cfg.currCfg, cfg.standard )

   @classmethod
   def showDiff( cls, mode ):
      # generate diff between active and pending
      import difflib, cStringIO
      activeOutput = cStringIO.StringIO( )
      cfg = getAclConfig( mode.aclType, cliConf=True ).get( mode.aclName )
      if cfg:
         cls._showList( mode.aclName, mode.aclType,
                    cfg.currCfg, cfg.standard, output=activeOutput )
      aclApi = mode.aclApi
      subConfig = aclApi.currentAcl()
      pendingOutput = cStringIO.StringIO( )
      cls._showList( mode.aclName, mode.aclType, subConfig, mode.standard,
                 output=pendingOutput )

      diff = difflib.unified_diff( activeOutput.getvalue( ).splitlines( ),
                                   pendingOutput.getvalue( ).splitlines( ),
                                   lineterm='' )
      print '\n'.join( list( diff ) )

#------------------------------------------------------------
# config-cp mode
#------------------------------------------------------------
class CpConfigMode( CpMode, BasicCli.ConfigModeBase ):
   name = "Control-Plane Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CpMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

accessGroupKwMatcher = CliMatcher.KeywordMatcher( 'access-group',
      helpdesc='Configure access control list' )
accessListKwMatcher = CliMatcher.KeywordMatcher( 'access-list',
      helpdesc='Configure access control list' )
inKwMatcher = CliMatcher.KeywordMatcher( 'in',
      helpdesc='Inbound packets' )
ipKwForClearServiceAclMatcher = CliMatcher.KeywordMatcher( 'ip',
      helpdesc='Clear IP information' )
ipKwForServiceAclMatcher = CliMatcher.KeywordMatcher( 'ip',
      helpdesc='IP config commands' )
ipKwForShowServiceAcl = CliMatcher.KeywordMatcher( 'ip',
      helpdesc='Details related to IPv4' )
ipv4KwForServiceAclMatcher = CliMatcher.KeywordMatcher( 'ipv4',
      helpdesc='IPv4 config commands' )
ipv6KwForShowServiceAcl = CliMatcher.KeywordMatcher( 'ipv6',
      helpdesc='Details related to IPv6' )
ipv6KwMatcherForClearServiceAcl = CliMatcher.KeywordMatcher( 'ipv6',
      helpdesc='Clear IPv6 information' )
ipv6KwMatcherForServiceAcl = CliMatcher.KeywordMatcher( 'ipv6',
      helpdesc='IPv6 config commands' )
vrfExprFactoryForConfigAcl = VrfCli.VrfExprFactory(
      helpdesc='Configure the VRF in which to apply the access control list',
      inclDefaultVrf=True )
vrfKwAndNameExprFactory = vrfExprFactoryForConfigAcl

ipMatcherForConfigIf = CliToken.Ip.ipMatcherForConfigIf
ipv6MatcherForConfigIf = CliToken.Ipv6.ipv6MatcherForConfigIf
macMatcherForConfigIf = CliToken.Mac.macMatcherForConfigIf

#------------------------------------------------------------
# show ip/ipv6/mac access-lists [name] [summary|detail]
#------------------------------------------------------------
def waitForCounters( mode, aclConfig, aclStatus ):
   ''' 
   Wait for counters to be available.   
   Returns: true if counters were updated
            false if we timed out waiting for SysDB
    '''
   # If we are missing the aclStatus, then there are no counters to get.
   # If we are missing the aclConfig, then the ACL doesn't exist so there's
   # nothing to wait for.
   if not aclStatus or not aclConfig:
      return True

   if mode.session_.entityManager.locallyReadOnly():
      # This would cause a warning to be printed
      return False

   aclConfig.counterUpdateRequestTime = Tac.now( )
   try:
      # See BUG36450
      envTimeout = os.getenv( 'ACL_CLI_TESTS_TIMEOUT' )
      timeout = int( envTimeout ) if envTimeout else 10

      Tac.waitFor( lambda: aclConfig.counterUpdateRequestTime <= \
                           aclStatus.counterUpdateTime,
                   description='counters to be available',
                   warnAfter=None, sleep=True, maxDelay=0.1, timeout=timeout )
      return True
   except Tac.Timeout:
      return False

def _hwStatusOK( ):
   for inst in statusDp.itervalues():
      for v in inst.hwStatus.itervalues( ):
         if v:
            if inst.name in statusDp:
               return False
            else:
               # the module is removed
               break
   return True

def _hwStatusErr( ):
   # Return the first hw error, if any, 
   # otherwise return empty string.
   for inst in statusDp.itervalues():
      if inst.waitForHwStatus:
         for v in inst.hwStatus.itervalues( ):            
            if v:
               if inst.name in statusDp:
                  return v
               else:
                  # the module is removed
                  break
   return ""

def _getRuleStats( aclStatus, ruleConfig, ruleid, aclType, chipName=None ):
   """ Returns the 3-tuple (pkts, ckptPkts, lastChangedTime). """

   pkts = None
   ckptPkts = None
   lastChangedTime = None
   ruleStatus = None
   if aclStatus and ruleConfig.action != 'remark':
      if chipName:
         statusPerChip = aclStatus.ruleStatusPerChip.get( chipName )
         assert statusPerChip
         # leaving statusPerChip in a stale state, since statusPerChip is only
         # updated with counters enabled
         ruleStatus = statusPerChip.ruleStatus.get( ruleid )
      else:
         # no detail per chip
         ruleStatus = aclStatus.ruleStatus.get( ruleid )
      if ruleStatus and ruleStatus.pkts:
         # check both session and checkpoint counters
         ckptPkts = _getCheckpointPkts( aclStatus.name, ruleid,
                                        aclType, chipName )
         pkts = ruleStatus.pkts
         lastChangedTime = ruleStatus.lastChangedTime
   return ( pkts, None, ckptPkts, None, lastChangedTime )

def _getCheckpointPkts( name, ruleid, aclType, chipName=None ):
   sessTime, sessPkts = _getTimestampPkts( _sessionCheckpoint(),
                                          name, ruleid, aclType, chipName )
   globalTime, globalPkts = _getTimestampPkts( checkpoint,
                                              name, ruleid, aclType, chipName )
   # use the latest checkpoint
   return sessPkts if sessTime >= globalTime else globalPkts

def _getTimestampPkts( ckpt, name, ruleid, aclType, chipName=None ):
   # return timestamp and counter of a rule
   # if the acl status or rule status does not exist,
   # return 0
   timestamp = 0
   pkts = 0
   if ckpt is not None:
      acl = ckpt.status[ aclType ].acl.get( name )
      if acl is not None:
         timestamp = acl.timestamp
         rule = None
         if chipName:
            statusPerChip = acl.ruleStatusPerChip.get( chipName )
            if statusPerChip:
               rule = statusPerChip.ruleStatus.get( ruleid )
         else:
            rule = acl.ruleStatus.get( ruleid )
         if rule is not None:
            pkts = rule.pkts
   return ( timestamp, pkts )

def showIpAccessLists( mode, name ):
   aclListModel = AclCliModel.IpAclList()
   _showAccessListsByModel( mode, name, aclListModel, 'ip' )
   return aclListModel

def showIp6AccessLists( mode, name ):
   aclListModel = AclCliModel.Ipv6AclList()
   _showAccessListsByModel( mode, name, aclListModel, 'ipv6' )
   return aclListModel

def showMacAccessLists( mode, name ):
   aclListModel = AclCliModel.MacAclList()
   _showAccessListsByModel( mode, name, aclListModel, 'mac' )
   return aclListModel

# This is set by AclAgent's CliPlugin due to dependency (Lag -> Acl)
__lagToPortMapFunc__ = None

def _showAccessListsByModel( mode, name, aclListModel, aclType ):
   """ Composes an instance of AclCliModel.AclList. This will be a list
   of length one if an ACL name is given, else a list of all ACLs. """

   if not _hwStatusOK( ):
      mode.addWarning( "Not all ACLs are programmed in the hardware!" )
   
   aclName, summary, dynamic, detail = name
   # lagToPort map is only needed for 'summary' command and feature is enabled
   lagToPort = ( __lagToPortMapFunc__() # pylint: disable=not-callable
                 if __lagToPortMapFunc__ and
                 summary and status.dpAclDirStatusEnabled else {} )
   if aclName is None:
      aclConfig = getAclConfig( aclType )
      names = sorted( aclConfig.keys( ) )
      for n in names:
         model = _showAclByModel( mode, aclListModel, n, summary, aclType,
                                  lagToPort, dynamic=dynamic, detail=detail )
         if not model:
            continue
         elif detail and isinstance( model, list ):
            aclListModel.aclList += model
         else:
            aclListModel.aclList.append( model )
   else:
      model = _showAclByModel( mode, aclListModel, aclName, summary, aclType,
                               lagToPort, dynamic=dynamic, detail=detail )
      if model:
         if detail and isinstance( model, list ):
            aclListModel.aclList += model
         else:
            aclListModel.aclList.append( model )

def initAclIntfConfigAggregatorSm():
   global intfConfigAll, intfConfigDir, intfConfigDirMergeSm, intfConfigAggregatorSm
   if intfConfigAll:
      return
   intfConfigAll = Tac.newInstance( "Acl::IntfConfig", "intfConfigAll" )
   intfConfigAll.initialize( 0 )
   intfConfigDir = Tac.newInstance( "Sysdb::DirMerged", "aclIntfConfig" )
   intfConfigDir.addPtr( gv.intfConfigCli.force() )
   intfConfigDirMergeSm = Tac.newInstance( "Sysdb::DirMergeSm", intfConfigDir )
   intfConfigDirMergeSm.addInputDir( gv.intfConfigInputDir.force() )
   intfConfigAggregatorSm = Tac.newInstance( "AclAgent::IntfConfigSm",
                                             intfConfigDir, intfConfigAll )

def _showAclByModel( mode, aclListModel, name, summary, aclType,
                     lagToPort=None, dynamic=False, detail=False ):
   """ Gets the named ACL from the TACC model. """

   aclTypeConfig = cliConfig.config[ aclType ]
   aclConfig = aclTypeConfig.acl.get( name )
   aclTypeStatus = status.status[ aclType ]
   aclStatus = aclTypeStatus.acl.get( name )
   chipNames = sorted( aclStatus.ruleStatusPerChip.keys() ) \
               if aclStatus is not None else []
   countersSupported = status.dpAclCountersSupported
   countersEnabled = False
   if name in config.config[ aclType ].acl:
      cfg = config.config[ aclType ].acl[ name ]
      if cfg.secureMonitor != mode.session.secureMonitor():
         return None
      countersEnabled = cfg.currCfg.countersEnabled

   staleCounters = True
   if countersSupported and waitForCounters( mode, aclConfig, aclStatus ):
      staleCounters = False

   if summary:
      initAclIntfConfigAggregatorSm()
      return AclCliModelImpl.getAclSummaryModel( config, cpConfig, paramConfig, 
                                                 intfConfigAll, allVrfConfig,
                                                 status, name, aclType,
                                                 staleCounters,
                                                 ruleStatsFunc=_getRuleStats,
                                                 lagToPort=lagToPort,
                                                 dynamic=dynamic )
   elif detail and chipNames and countersEnabled:
      # if there aren't chips names then the ACL has not been applied
      # counters must be enabled for detail command to run
      modelList = []
      for chipName in aclStatus.ruleStatusPerChip:
         model = AclCliModelImpl.getAclModel( config, allVrfConfig, status,
                                              name, aclType, aclListModel,
                                              staleCounters,
                                              ruleStatsFunc=_getRuleStats,
                                              paramConfig=paramConfig,
                                              dynamic=dynamic,
                                              chipName=chipName )
         modelList.append( model )
      return modelList
   else:
      return AclCliModelImpl.getAclModel( config, allVrfConfig, status,
                                          name, aclType, aclListModel,
                                          staleCounters,
                                          ruleStatsFunc=_getRuleStats,
                                          paramConfig=paramConfig,
                                          dynamic=dynamic )

def _getServiceAclRuleStats( aclStatus, ruleConfig, ruleid, aclType, chipName=None ):
   """ Returns the 3-tuple (pkts, ckptPkts, lastChangedTime). """

   pkts = None
   connPkts = None
   lastChangedTime = None
   if aclStatus and ruleConfig.action != 'remark':
      ruleStatus = aclStatus.ruleStatus.get( ruleid )
      if ruleStatus and ruleStatus.pkts:
         pkts = ruleStatus.pkts
         lastChangedTime = ruleStatus.lastChangedTime

      connStatus = aclStatus.connRuleStatus.get( ruleid )
      if connStatus and connStatus.pkts:
         connPkts = connStatus.pkts
         if connStatus.lastChangedTime > lastChangedTime:
            lastChangedTime = connStatus.lastChangedTime
   # ckptPkts will be reset in AclCliModel.getServiceAclModel
   return ( pkts, connPkts, 0, 0, lastChangedTime )

def _showServiceAclByModel( mode,
                            serviceAclConfig, serviceAclStatus, serviceAclCkptStatus,
                            aclListModel, name, summary, aclType, supressVrf,
                            vrfUnaware=False, serviceName=None ):

   """ Gets the named ACL from the TACC model. """
   if summary:
      return AclCliModelImpl.getServiceAclSummaryModel( config,
                                          serviceAclConfig, serviceAclStatus,
                                          allVrfConfig, name, aclType, supressVrf,
                                          vrfUnaware, serviceName=serviceName )
   else:
      model = AclCliModelImpl.getServiceAclModel( config, serviceAclStatus,
                                          serviceAclCkptStatus,
                                          name, aclType, aclListModel,
                                          ruleStatsFunc=_getServiceAclRuleStats,
                                          paramConfig=paramConfig,
                                          serviceName=serviceName )
      if model:
         model.summary = AclCliModelImpl.getServiceAclSummaryModel( config,
                                          serviceAclConfig, serviceAclStatus,
                                          allVrfConfig, name, aclType, supressVrf,
                                          vrfUnaware, serviceName=serviceName )
      return model

def showServiceAcl( mode, serviceAclConfig, serviceAclStatus,
                    serviceAclCkpt, aclType, name, supressVrf=False,
                    vrfUnaware=False, serviceName=None ):
   # show both ip and ipv6 if aclType is None
   allAclList = AclCliModel.AllAclList()
   if not aclType:
      allAclList.ipAclList = showServiceAcl(
                     mode, serviceAclConfig, serviceAclStatus,
                     serviceAclCkpt, 'ip', name, supressVrf,
                     vrfUnaware, serviceName ).ipAclList
      allAclList.ipv6AclList = showServiceAcl(
                     mode, serviceAclConfig, serviceAclStatus,
                     serviceAclCkpt, 'ipv6', name, supressVrf,
                     vrfUnaware, serviceName ).ipv6AclList
   else:
      if aclType == 'ip':
         aclListModel = AclCliModel.IpAclList()
         allAclList.ipAclList = aclListModel
      else:
         aclListModel = AclCliModel.Ipv6AclList()
         allAclList.ipv6AclList = aclListModel

      # Name may contain a dynamic flag, but we don't need to unpack it as we
      # don't support dynamic service ACLs
      name, summary = name[ :2 ]
      aclNames = set()
      if not serviceName:
         for aclTypeAndVrfName, aclName in serviceAclConfig.aclName.iteritems():
            if ( aclTypeAndVrfName.aclType == aclType and
                 ( not name or aclName == name ) ):
               aclNames.add( aclName )
      else:
         # iptables based service ACL
         cpAclTypeConfig = serviceAclConfig.cpConfig[ aclType ]
         for serviceAclVrfConfig in cpAclTypeConfig.serviceAcl.itervalues():
            serviceAcl = serviceAclVrfConfig.service.get( serviceName )
            if serviceAcl and serviceAcl.aclName:
               aclNames.add( serviceAcl.aclName )
         if name:
            aclNames = { name } if name in aclNames else set()

      if serviceAclCkpt:
         serviceAclCkptStatus = serviceAclCkpt.status.get( aclType )
      for n in aclNames:
         model = _showServiceAclByModel( mode, serviceAclConfig,
                                         serviceAclStatus, serviceAclCkptStatus,
                                         aclListModel, n, summary, aclType,
                                         supressVrf, vrfUnaware, serviceName )
         if model:
            aclListModel.aclList.append( model )
   return allAclList

def clearServiceAclCounters( mode, serviceAclStatus, serviceAclCkpt, aclType ):
   if not aclType:
      clearServiceAclCounters( mode, serviceAclStatus, serviceAclCkpt, 'ip' )
      clearServiceAclCounters( mode, serviceAclStatus, serviceAclCkpt, 'ipv6' )
      return

   if serviceAclStatus and serviceAclStatus.status.get( aclType ):
      if aclType not in serviceAclCkpt.status:
         serviceAclCkpt.newStatus( aclType )
      ckptStatus = serviceAclCkpt.status[ aclType ].acl
      for aclName, aclStatus in serviceAclStatus.status[ aclType ].acl.iteritems():
         if aclName in ckptStatus:
            del ckptStatus[ aclName ]
         dest = ckptStatus.newMember( aclName )
         for r, v in aclStatus.ruleStatus.iteritems( ):
            value = Tac.Value( "Acl::RuleCheckpointStatus", pkts=v.pkts )
            dest.ruleStatus[ r ] = value
         for r, v in aclStatus.connRuleStatus.iteritems( ):
            value = Tac.Value( "Acl::RuleCheckpointStatus", pkts=v.pkts )
            dest.connRuleStatus[ r ] = value
         # no rule match counter
         value = Tac.Value( "Acl::RuleCheckpointStatus",
                            pkts=aclStatus.noRuleMatches.pkts )
         dest.noRuleMatches = value
         value = Tac.Value( "Acl::RuleCheckpointStatus",
                            pkts=aclStatus.noConnRuleMatches.pkts )
         dest.noConnRuleMatches = value
         dest.timestamp = Tac.now()

      # clean up stale checkpoint status which has no corresponding acl status
      for aclName in ckptStatus:
         if aclName not in serviceAclStatus.status[ aclType ].acl:
            del ckptStatus[ aclName ]

# aclName, summary and dynamic are optional, and when user enters
# 'show ip/mac access-lists summary/dynamic', we want it to match the keyword,
def _aclNameExprGenerator( aclType, nameMatcher ):
   aclName = 'ACL_NAME_%s' % aclType
   aclSummary = 'SUMMARY_%s' % aclType
   aclDynamic = 'DYNAMIC_%s' % aclType
   aclDetail = 'DETAIL_%s' % aclType

   class AclNameExpression( CliCommand.CliExpression ):
      expression = '[ %s ] [ %s ] [ %s ] [ %s ]' % ( aclName, aclSummary, 
            aclDynamic, aclDetail )
      data = {
               aclName: nameMatcher,
               aclSummary: aclSummaryKwMatcher,
               aclDynamic: aclDynamicKwMatcher,
               aclDetail: nodeDetail,
             }
      
      @staticmethod
      def adapter( mode, args, argsList ):
         if '<aclNameExpr>' not in args:
            # set some default for this
            args[ '<aclNameExpr>' ] = ( None, False, False, False )

         if ( aclName not in args and
              aclSummary not in args and
              aclDynamic not in args and
              aclDetail not in args):
            # nothing to do, because this isn't the right one, return
            return
         summary = bool( args.pop( aclSummary, False ) )
         dynamic = bool( args.pop( aclDynamic, False ) )
         detail = bool( args.pop( aclDetail, False ) )
         args[ '<aclNameExpr>' ] = ( args.pop( aclName, None ), summary, dynamic, 
               detail )

   return AclNameExpression

def aclNameExpression( aclType ):
   if aclType == 'ip':
      return _aclNameExprGenerator( aclType, ipAclNameMatcher )
   elif aclType == 'ipv6':
      return _aclNameExprGenerator( aclType, ip6AclNameMatcher )
   elif aclType == 'ipOrIpv6':
      return _aclNameExprGenerator( aclType, ipOrIp6AclNameMatcher )
   elif aclType == 'mac':
      return _aclNameExprGenerator( aclType, macAclNameMatcher )
   else:
      assert False, 'Invalid ACL Type: %s' % aclType
      return None

ipAclNameExpression = aclNameExpression( 'ip' )
ip6AclNameExpression = aclNameExpression( 'ipv6' )
ipOrIpv6AclNameExpression = aclNameExpression( 'ipOrIpv6' )
macAclNameExpression = aclNameExpression( 'mac' )

accessListsMatcher = CliMatcher.KeywordMatcher( 'access-lists',
      helpdesc='IP access-lists' )

class ShowIpAccessListsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip access-lists [ ACL_NAME ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'access-lists': accessListsMatcher,
      'ACL_NAME': ipAclNameExpression,
   }
   cliModel = AclCliModel.IpAclList
   privileged = True
   handler = lambda mode, args: showIpAccessLists( mode,
                                                   args.get( '<aclNameExpr>' ) )

BasicCli.addShowCommandClass( ShowIpAccessListsCmd )

class ShowIp6AccessListsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 access-lists [ ACL_NAME ]'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
      'access-lists': accessListsMatcher,
      'ACL_NAME': ip6AclNameExpression,
   }
   cliModel = AclCliModel.Ipv6AclList
   privileged = True
   handler = lambda mode, args: showIp6AccessLists( mode,
                                                    args.get( '<aclNameExpr>' ) )

BasicCli.addShowCommandClass( ShowIp6AccessListsCmd )

class ShowMacAccessListsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mac access-lists [ ACL_NAME ]'
   data = {
      'mac': CliToken.Mac.macMatcherForShow,
      'access-lists': accessListsMatcher,
      'ACL_NAME': macAclNameExpression,
   }
   cliModel = AclCliModel.MacAclList
   privileged = True
   handler = lambda mode, args: showMacAccessLists( mode,
                                                    args.get( '<aclNameExpr>' ) )

BasicCli.addShowCommandClass( ShowMacAccessListsCmd )

#------------------------------------------------------------
# show ( ip | mac ) access-list payload alias
#------------------------------------------------------------
class ShowPayloadAlias( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ( ip | mac ) access-list payload alias'
   data = {
      'ip': CliCommand.Node( CliToken.Ip.ipMatcherForShow, alias='ALIAS_TYPE' ),
      'mac': CliCommand.Node( CliToken.Mac.macMatcherForShow, alias='ALIAS_TYPE' ),
      'access-list': accessListsMatcher,
      'payload': 'Payload parameters',
      'alias': 'Show payload alias'
   }
   privileged = True
   cliModel = AclCliModel.UdfAlias

   @staticmethod
   def handler( mode, args ):
      aliasType = args[ 'ALIAS_TYPE' ]
      aclListModel = AclCliModel.UdfAlias()
      udfAliases = aclListModel.udfAliases
      if aliasType == 'mac':
         udfAliasColl = udfAlias.macAlias
      else:
         udfAliasColl = udfAlias.ipAlias

      for aliasName in udfAliasColl.iterkeys():
         alias = udfAliasColl[ aliasName ]
         payload = alias.payload
         headerStart = alias.headerStart == AclLib.AclPayloadHdrStart.start
         payloadElement = AclCliModel.PayloadElement( offset=payload.offset,
                                                      pattern=payload.pattern,
                                                      patternMask=payload.mask,
                                                      patternOverride=\
                                                            payload.patternOverride,
                                                      maskOverride=\
                                                            payload.maskOverride,
                                                      alias='' )
         udfAliases[ aliasName ] = AclCliModel.PayloadSpec( headerStart=headerStart,
                                 payload=[ payloadElement ] )
      return aclListModel

BasicCli.addShowCommandClass( ShowPayloadAlias )

#------------------------------------------------------------
# clear access-lists counters [session]
#------------------------------------------------------------
def clearOneAclCounters( mode, aclName, ckpt, aclType ):
   # Clear counters for one ACL
   aclConfig = getAclConfig( aclType, cliConf=True ).get( aclName )
   aclStatus = status.status[ aclType ].acl.get( aclName )

   if not aclConfig or not aclStatus:
      return
   aclConfig.clearCounter = True # ensure stale chips removed from RuleStatusPerChip

   if not waitForCounters( mode, aclConfig, aclStatus ):
      # somehow we timed out, but we'll show whatever we have
      mode.addWarning( "Using stale counters" )

   ckptStatus = ckpt.status[ aclType ].acl

   if aclName in ckptStatus:
      del ckptStatus[ aclName ]
   source = status.status[ aclType ].acl.get( aclName )
   if source is None:
      # source could be empty, in which case we do nothing
      return
   dest = ckptStatus.newMember( aclName )
   # copy counters into status
   for r, v in source.ruleStatus.iteritems( ):
      value = Tac.Value( "Acl::RuleCheckpointStatus",
                         pkts=v.pkts )
      dest.ruleStatus[ r ] = value
   # copy counters per rule into checkpoint status
   for chipName, statusPerChip in source.ruleStatusPerChip.iteritems():
      chipStatus = dest.ruleStatusPerChip.newMember( chipName )
      for r, v in statusPerChip.ruleStatus.iteritems():
         value = Tac.Value( "Acl::RuleCheckpointStatus",
                            pkts=v.pkts )
         chipStatus.ruleStatus[ r ] = value
   dest.timestamp = Tac.now( )
   aclConfig.clearCounter = False

def clearAclCounters( mode, args ):
   aclName = args.get( 'ACL_NAME' )
   session = 'session' in args
   aclType = args[ 'ACL_TYPE' ]
   if session:
      ckpt = _sessionCheckpoint()
      logAction = False
   else:
      ckpt = checkpoint
      logAction = True

   if aclName:
      if logAction:
         Intf.Log.logClearCounters( "access list",
                                     "%s access list %s" % ( aclType.lower(),
                                                             aclName ) )
      # clear counter for one Acl
      clearOneAclCounters( mode, aclName, ckpt, aclType )
   else:
      if logAction:
         Intf.Log.logClearCounters( "access list",
                                     "all %s access lists" % aclType.lower() )
      for name in status.status[ aclType ].acl.iterkeys( ):
         clearOneAclCounters( mode, name, ckpt, aclType )
      for name in ckpt.status[ aclType ].acl.iterkeys( ):
         if name not in status.status[ aclType ].acl:
            del ckpt.status[ aclType ].acl[ name ]

countersKwMatcher = CliMatcher.KeywordMatcher( 'counters',
                                helpdesc='Clear IP access list counters' )

#------------------------------------------------------------------------------------
# clear ip access-lists counters [ ACL_NAME ] [ session ]
#------------------------------------------------------------------------------------
class ClearIpAccessListsCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip access-lists counters [ ACL_NAME ] [ session ]'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'ip': CliCommand.Node( matcher=CliToken.Ip.ipMatcherForClear,
                              alias='ACL_TYPE' ),
      'access-lists': accessListsMatcher,
      'counters': countersKwMatcher,
      'ACL_NAME': ipAclNameMatcher,
      'session': 'Clear IP access list counters in current session',
   }

   handler = clearAclCounters

BasicCli.EnableMode.addCommandClass( ClearIpAccessListsCmd )

#------------------------------------------------------------------------------------
# clear ipv6 access-lists counters [ ACL_NAME ] [ session ]
#------------------------------------------------------------------------------------
class ClearIp6AccessListsCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ipv6 access-lists counters [ ACL_NAME ] [ session ]'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'ipv6': CliCommand.Node( matcher=CliToken.Ipv6.ipv6MatcherForClear,
                              alias='ACL_TYPE' ),
      'access-lists': 'IPv6 access-lists',
      'counters': 'Clear IPv6 access list counters',
      'ACL_NAME': ip6AclNameMatcher,
      'session': 'Clear IPv6 access list counters in current session',
   }

   handler = clearAclCounters

BasicCli.EnableMode.addCommandClass( ClearIp6AccessListsCmd )

#------------------------------------------------------------------------------------
# clear mac access-lists counters [ ACL_NAME ] [ session ]
#------------------------------------------------------------------------------------
class ClearMacAccessListsCmd( CliCommand.CliCommandClass ):
   syntax = 'clear mac access-lists counters [ ACL_NAME ] [ session ]'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'mac': CliCommand.Node( matcher=CliToken.Mac.macMatcherForClear,
                              alias='ACL_TYPE' ),
      'access-lists': 'MAC access-lists',
      'counters': 'Clear MAC access list counters',
      'ACL_NAME': macAclNameMatcher,
      'session': 'Clear MAC access list counters in current session',
   }

   handler = clearAclCounters

BasicCli.EnableMode.addCommandClass( ClearMacAccessListsCmd )

#-------------------------------------------------------------------------------
# Cleanup per-interface configuration
#-------------------------------------------------------------------------------
class IntfAclConfigCleaner( IntfCli.IntfDependentBase ):
   """This class is responsible for removing per-interface ACL config
   when the interface is deleted."""
   def setDefault( self ):
      # FIXME: not sure if this is the right thing to do, but we are not
      # clearing out secure-monitor config here.
      for v in gv.intfConfigCli.config.itervalues():
         for d in v.intf.itervalues():
            del d.intf[ self.intf_.name ]

def initAclConfigAggregatorSm():
   global aclConfigAggregatorSm, aclCpConfigAggregatorSm
   global aggregatorInitialized
   if not aggregatorInitialized:
      aggregatorInitialized = True
      aclConfigAggregatorSm = Tac.newInstance( "Acl::AclConfigAggregatorSm",
                                               gv.aclConfigDir.force(),
                                               cliConfig.sysdbObj(),
                                               config )
      aclCpConfigAggregatorSm = Tac.newInstance( "Acl::AclCpConfigAggregatorSm",
                                                 gv.aclCpConfigDir.force(),
                                                 cliCpConfig.sysdbObj(),
                                                 cpConfig )


#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global cliConfig, cliCpConfig, paramConfig, hwConfig
   global status, statusDp, checkpoint
   global allVrfConfig, udfAlias, hwEpoch
   global aclCmds
   global config, cpConfig

   config = Tac.newInstance( "Acl::Config" )
   cpConfig = Tac.newInstance( "Acl::CpConfig" )
   cliConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   cliCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                    "Acl::Input::CpConfig", "w" )
   gv.aclConfigDir = LazyMount.mount( entityManager, "acl/config/input",
                                      "Tac::Dir", "ri" )
   gv.aclCpConfigDir = LazyMount.mount( entityManager, "acl/cpconfig/input",
                                        "Tac::Dir", "ri" )
   paramConfig = ConfigMount.mount( entityManager, "acl/paramconfig",
                             "Acl::ParamConfig", "w" )
   gv.intfConfigInputDir = LazyMount.mount( entityManager, "acl/intf/config/input",
                                            "Tac::Dir", "ri" )
   gv.intfConfigSecureMonitor = LazyMount.mount( entityManager,
                                              "acl/intf/config/input/secure-monitor",
                                              "Acl::IntfConfig", "w" )
   gv.intfConfigCli = ConfigMount.mount( entityManager, "acl/intf/config/cli",
                                         "Acl::IntfConfig", "w" )

   hwConfig = LazyMount.mount( entityManager, "acl/hwconfig/cli",
                               "Acl::HwConfig", "w" )

   status = LazyMount.mount( entityManager,
                             "acl/status/all", "Acl::Status", "r" )
   if not aclCmds:
      aclCmds = AclCfgCmdBinder( status )

   statusDp = LazyMount.mount( entityManager, 
                               "cell/%d/acl/status/dp" % Cell.cellId(),
                               "Tac::Dir", "ri" )
   # ConfigMount since called from commitAcl().
   checkpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                 "Acl::CheckpointStatus", "w" )
   allVrfConfig = LazyMount.mount( entityManager, "ip/vrf/config",
                                   "Ip::AllVrfConfig", "r" )
   udfAlias = ConfigMount.mount( entityManager, "acl/udfalias",
                                 "Acl::UdfAlias", "w" )

   hwEpoch = LazyMount.mount( entityManager, "hwEpoch/status",
                              "HwEpoch::Status", "r" )

   # register interface-deletion handler
   IntfCli.Intf.registerDependentClass( IntfAclConfigCleaner, priority=10 )
   BasicCliSession.registerReadlineThreadWork( initAclConfigAggregatorSm ) 
   AclCliLib.initialize()
