#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from eunuchs.in_h import IPPROTO_ICMPV6

import AclLib
import BasicCliModes
import CliCommand
import CliMatcher
import CliPlugin.AclCli as AclCli
import CliPlugin.AclCliRules as AclCliRules
import CliPlugin.DpAclCli as DpAclCli
from CliToken.Hardware import hardwareMatcherForConfig

matcherAccessList = CliMatcher.KeywordMatcher( 'access-list',
      helpdesc='Hardware parameter related to access-list' )
matcherIn = CliMatcher.KeywordMatcher( 'in', helpdesc='Ingress' )
matcherKeyWidth = CliMatcher.KeywordMatcher( 'key-width', helpdesc='Key Width' )
matcherNarrow = CliMatcher.KeywordMatcher( 'narrow',
      helpdesc='Narrow mode. Allows only /64 IPv6' )
matcherPayload = CliMatcher.KeywordMatcher( 'payload',
      helpdesc='Payload parameters' )
matcherSharing = CliMatcher.KeywordMatcher( 'sharing',
      helpdesc='Sharing behaviour of hardware resources' )
matcherSkip = CliMatcher.KeywordMatcher( 'skip', helpdesc='Deep inspection offset' )
nodeDeepInspection = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'deep-inspection',
         helpdesc='Configure deep inspection parameters' ),
      guard=AclCli.dpAclDIParsingSupported )
nodeIpv6 = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'ipv6',
         helpdesc='IPV6 ACLS' ),
      guard=DpAclCli.dpIp6KeyWidthModeSupported )
matcherResource = CliMatcher.KeywordMatcher( 'resource',
      helpdesc='Hardware resources' )
nodeResource = CliCommand.Node( matcher=matcherResource,
      guard=DpAclCli.dpAclSharingSupported )
nodeVlan = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'vlan',
         helpdesc='Share hardware resources when access-list is attached to '
                  'VLAN interfaces' ),
      guard=DpAclCli.dpAclSharingSupported )

#--------------------------------------------------------------------------------
#  [ no | default ] deep-inspection payload l2 skip SKIP
#--------------------------------------------------------------------------------
class DeepInspectionPayloadL2SkipSkipCmd( CliCommand.CliCommandClass ):
   syntax = 'deep-inspection payload l2 skip SKIP'
   noOrDefaultSyntax = 'deep-inspection payload l2 skip ...'
   data = {
      'deep-inspection': nodeDeepInspection,
      'payload': matcherPayload,
      'l2': 'Deep inspection for L2 packets',
      'skip': matcherSkip,
      'SKIP': CliMatcher.DynamicIntegerMatcher( DpAclCli.deepInspMaxSkipL2Fn_,
         helpdesc='Words to skip ( in 32 bits )' ),
   }
   handler = DpAclCli.setDeepInspL2Offset
   noOrDefaultHandler = DpAclCli.noSetDeepInspL2Offset

BasicCliModes.GlobalConfigMode.addCommandClass( DeepInspectionPayloadL2SkipSkipCmd )

#--------------------------------------------------------------------------------
# [ no | default ] deep-inspection payload l4 skip SKIP
#--------------------------------------------------------------------------------
class DeepInspectionPayloadL4SkipSkipCmd( CliCommand.CliCommandClass ):
   syntax = 'deep-inspection payload l4 skip SKIP'
   noOrDefaultSyntax = 'deep-inspection payload l4 skip ...'
   data = {
      'deep-inspection': nodeDeepInspection,
      'payload': matcherPayload,
      'l4': 'Deep inspection for L4/unknown L4 packets',
      'skip': matcherSkip,
      'SKIP': CliMatcher.DynamicIntegerMatcher( DpAclCli.deepInspMaxSkipL4Fn_,
         helpdesc='Words to skip ( in 32 bits )' ),
   }
   handler = DpAclCli.setDeepInspL4Offset
   noOrDefaultHandler = DpAclCli.noSetDeepInspL4Offset

BasicCliModes.GlobalConfigMode.addCommandClass( DeepInspectionPayloadL4SkipSkipCmd )

#--------------------------------------------------------------------------------
# [ no | default ] hardware access-list ipv4 egress resource sharing
#                                                               routed-interfaces
#--------------------------------------------------------------------------------
class HwAclIpv4EgressSource( CliCommand.CliCommandClass ):
   syntax = 'hardware access-list ipv4 egress resource sharing routed-interfaces'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'ipv4': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'ipv4', helpdesc='IPv4' ),
         guard=DpAclCli.dpEgressRoutedInterfaceAclSharingSupported ),
      'egress': 'Egress',
      'resource': nodeResource,
      'sharing': matcherSharing,
      'routed-interfaces': 'Change behaviour on Routed Interfaces ACL',
   }
   hidden = True
   handler = DpAclCli.changeEgressRoutedInterfaceAclSharing
   noOrDefaultHandler = DpAclCli.changeNoEgressRoutedInterfaceAclSharing

BasicCliModes.GlobalConfigMode.addCommandClass( HwAclIpv4EgressSource )

#--------------------------------------------------------------------------------
# [ no | default ] hardware access-list ipv4-ipv6 mirroring-policy key-width
#                                                                           narrow in
#--------------------------------------------------------------------------------
class HwAclIpv4Ipv6MirroringPolicy( CliCommand.CliCommandClass ):
   syntax = 'hardware access-list ipv4-ipv6 mirroring-policy key-width narrow in'
   noOrDefaultSyntax = 'hardware access-list ipv4-ipv6 mirroring-policy key-width in'
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'ipv4-ipv6': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'ipv4-ipv6',
            helpdesc='Key width for Banks of IPv4 and IPv6 versions of feature' ),
         guard=DpAclCli.dpMirroringAclKeyWidthModeSupported ),
      'mirroring-policy': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'mirroring-policy',
            helpdesc='Mirroring Policy' ),
         guard=DpAclCli.dpMirroringAclKeyWidthModeSupported ),
      'in': matcherIn,
      'key-width': matcherKeyWidth,
      'narrow': matcherNarrow,
   }
   hidden = True
   handler = DpAclCli.gotoHardwareInMirroringAclKeyWidthModeNarrow
   noOrDefaultHandler = DpAclCli.gotoHardwareInMirroringAclKeyWidthModeNone

BasicCliModes.GlobalConfigMode.addCommandClass( HwAclIpv4Ipv6MirroringPolicy )

#--------------------------------------------------------------------------------
# ( no | default ) hardware access-list ipv6 implicit-permit
#                                               PROTOCOL ( all | neighbor-discovery )
#--------------------------------------------------------------------------------
class HwAclIpv6ImplicitPermitCmd( CliCommand.CliCommandClass ):
   syntax = ( 'hardware access-list ipv6 implicit-permit PROTOCOL '
              '( all | neighbor-discovery )' )
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'ipv6': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'ipv6', helpdesc='IPV6 ACLS' ),
         guard=DpAclCli.allIcmpTypesInAclSupportedGuard ),
      'implicit-permit': 'Add default icmpv6permit rules',
      'PROTOCOL': AclCliRules.ipProtocolMatcher( 'icmpv6', IPPROTO_ICMPV6,
         'Internet Control Message Protocol version 6' ),
      'all': 'Match on all ICMPv6 types',
      'neighbor-discovery': 'Match only on Icmpv6 Neighbor Discovery protocol types',
   }
   handler = DpAclCli.ipv6AclImcp6NeighborDiscoveryTypes
   noOrDefaultHandler = DpAclCli.noIpv6AclImcp6NeighborDiscoveryTypes

BasicCliModes.GlobalConfigMode.addCommandClass( HwAclIpv6ImplicitPermitCmd )

#--------------------------------------------------------------------------------
# hardware access-list ipv6 ( qos-policy | security-acl ) key-width narrow in
# ( no | default ) hardware access-list ipv6 ( qos-policy | security-acl )
#                                                                        key-width in
#--------------------------------------------------------------------------------
class HwAclIpv6KeyWidth( CliCommand.CliCommandClass ):
   syntax = ( 'hardware access-list ipv6 ( qos-policy | security-acl ) key-width '
              'narrow in' )
   noOrDefaultSyntax = ( 'hardware access-list ipv6 ( qos-policy | security-acl ) '
                         'key-width in' )
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'ipv6': nodeIpv6,
      'qos-policy': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'qos-policy', helpdesc='Qos PolicyMap' ),
         guard=DpAclCli.dpIp6QosAclKeyWidthModeSupported ),
      'security-acl': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'security-acl', helpdesc='Port ACL' ),
         guard=DpAclCli.dpIp6SecurityAclKeyWidthModeSupported ),
      'key-width': matcherKeyWidth,
      'narrow': matcherNarrow,
      'in': matcherIn,
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      if 'qos-policy' in args:
         DpAclCli.gotoHardwareInIpv6QosAclKeyWidthMode( mode,
            sharingMode=AclLib.TcamBankSharingMode.bankSharingModeNarrow )
      elif 'security-acl' in args:
         DpAclCli.gotoHardwareInIpv6SecurityAclKeyWidthMode( mode,
            sharingMode=AclLib.TcamBankSharingMode.bankSharingModeNarrow )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'qos-policy' in args:
         DpAclCli.gotoHardwareInIpv6QosAclKeyWidthMode( mode,
            sharingMode=AclLib.TcamBankSharingMode.bankSharingModeNone )
      elif 'security-acl' in args:
         DpAclCli.gotoHardwareInIpv6SecurityAclKeyWidthMode( mode,
            sharingMode=AclLib.TcamBankSharingMode.bankSharingModeNone )

BasicCliModes.GlobalConfigMode.addCommandClass( HwAclIpv6KeyWidth )

#--------------------------------------------------------------------------------
# [ no | default ] hardware access-list resource sharing ipv4-ipv6
#                                 ( mirroring-policy | qos-policy | security-acl ) in
#--------------------------------------------------------------------------------
class HwAclResourceSharingIpv4Ipv6( CliCommand.CliCommandClass ):
   syntax = ( 'hardware access-list resource sharing ipv4-ipv6 '
              '( mirroring-policy | qos-policy | security-acl ) in' )
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'resource': matcherResource,
      'sharing': matcherSharing,
      'ipv4-ipv6': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'ipv4-ipv6',
            helpdesc='Share TCAM Banks for IPv4 and IPv6 versions of feature' ),
         guard=DpAclCli.dpIpv4Ip6BankSharingSupported ),
      'qos-policy': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'qos-policy', helpdesc='Qos PolicyMap' ),
         guard=DpAclCli.dpIpv4Ip6QosAclBankSharingSupported ),
      'security-acl': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'security-acl', helpdesc='Port ACL' ),
         guard=DpAclCli.dpIpv4Ip6SecurityAclBankSharingSupported ),
      'mirroring-policy': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'mirroring-policy',
            helpdesc='Mirroring ACL' ),
         guard=DpAclCli.dpIpv4Ip6MirroringAclBankSharingSupported ),
      'in': matcherIn,
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      if 'mirroring-policy' in args:
         DpAclCli.gotoHardwareInIpv4Ipv6MirroringAclSharing( mode, True )
      elif 'qos-policy' in args:
         DpAclCli.gotoHardwareInIpv4Ipv6QosAclSharing( mode, True )
      elif 'security-acl' in args:
         DpAclCli.gotoHardwareInIpv4Ipv6SecurityAclSharing( mode, True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'mirroring-policy' in args:
         DpAclCli.gotoHardwareInIpv4Ipv6MirroringAclSharing( mode, False )
      elif 'qos-policy' in args:
         DpAclCli.gotoHardwareInIpv4Ipv6QosAclSharing( mode, False )
      elif 'security-acl' in args:
         DpAclCli.gotoHardwareInIpv4Ipv6SecurityAclSharing( mode, False )

BasicCliModes.GlobalConfigMode.addCommandClass( HwAclResourceSharingIpv4Ipv6 )

#--------------------------------------------------------------------------------
# [ no | default ] hardware access-list resource sharing vlan in
#--------------------------------------------------------------------------------
class HwAclResourceSharingVlanInCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware access-list resource sharing vlan in'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'resource': matcherResource,
      'sharing': matcherSharing,
      'vlan': nodeVlan,
      'in': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'in', helpdesc='Inbound packets' ),
         guard=DpAclCli.dpIngressRaclSharingSupported ),
   }
   handler = DpAclCli.gotoHardwareInAclSharing
   noOrDefaultHandler = DpAclCli.gotoNoHardwareInAclSharing

BasicCliModes.GlobalConfigMode.addCommandClass( HwAclResourceSharingVlanInCmd )

#--------------------------------------------------------------------------------
# [ no | default ] hardware access-list resource sharing vlan ( ipv4 | ipv6 ) out
#--------------------------------------------------------------------------------
class HwAclResourceSharingVlanIpv4OutCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware access-list resource sharing vlan ( ipv4 | ipv6 ) out'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'resource': matcherResource,
      'sharing': matcherSharing,
      'vlan': nodeVlan,
      'ipv4': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'ipv4', helpdesc='Ipv4' ),
         guard=DpAclCli.dpEgressRaclSharingSupported ),
      'ipv6': CliCommand.Node( 
         matcher=CliMatcher.KeywordMatcher( 'ipv6', helpdesc='Ipv6' ),
         guard=DpAclCli.dpEgressRaclSharingSupported ),
      'out': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'out', helpdesc='Outbound packets' ),
         guard=DpAclCli.dpAclSharingSupported ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'ipv4' in args:
         DpAclCli.gotoHardwareIpv4OutAclSharing( mode, args )
      else:
         DpAclCli.gotoHardwareIpv6OutAclSharing( mode, args )

   @staticmethod
   def noHandler( mode, args ):
      if 'ipv4' in args:
         DpAclCli.gotoNoHardwareIpv4OutAclSharing( mode, args )
      else:
         DpAclCli.gotoNoHardwareIpv6OutAclSharing( mode, args )

   @staticmethod
   def defaultHandler( mode, args ):
      if 'ipv4' in args:
         DpAclCli.gotoHardwareIpv4OutAclSharing( mode, args )
      else:
         DpAclCli.gotoNoHardwareIpv6OutAclSharing( mode, args )

BasicCliModes.GlobalConfigMode.addCommandClass( HwAclResourceSharingVlanIpv4OutCmd )

#--------------------------------------------------------------------------------
# [ no | default ] hardware access-list router-acl exclude mlag peer-link
#--------------------------------------------------------------------------------
class HwAclRouterAcl( CliCommand.CliCommandClass ):
   syntax = 'hardware access-list router-acl exclude mlag peer-link'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'router-acl': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'router-acl',
            helpdesc='Change behaviour on Router ACL' ),
         guard=DpAclCli.dpRouterAclExcludeMlagPeerLink ),
      'exclude': 'Exclude application of access-list on particular interfaces',
      'mlag': 'MLAG interfaces',
      'peer-link': 'Peer-link interfaces',
   }
   handler = DpAclCli.excludeMlagPeerLink
   noOrDefaultHandler = DpAclCli.noExcludeMlagPeerLink

BasicCliModes.GlobalConfigMode.addCommandClass( HwAclRouterAcl )

#--------------------------------------------------------------------------------
# [ no | default ] hardware access-list update default-result permit
#--------------------------------------------------------------------------------
class HwAclUpdate( CliCommand.CliCommandClass ):
   syntax = 'hardware access-list update default-result permit'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'update': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'update',
            helpdesc='Hardware behavior while access-list is being updated' ),
         guard=DpAclCli.dpPermitDuringAclUpdate ),
      'default-result': ( 'Default action on packets while access-list is '
                          'being updated' ),
      'permit': 'Accept the packets when access-list is being updated',
   }
   handler = DpAclCli.gotoHardwareAclUpdate
   noHandler = DpAclCli.gotoNoHardwareAclUpdate
   defaultHandler = DpAclCli.gotoDefaultHardwareAclUpdate

BasicCliModes.GlobalConfigMode.addCommandClass( HwAclUpdate )
