# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli 
import BasicCliModes
import CliParser
import ConfigMgmtMode
import CliCommand
import CliMatcher
import ConfigMount
import Tac 
from CliMode.Accounts import AccountsUserMode
from LocalUserLib import usernameRe

accountsConfig = None
sessionLimitType = Tac.Type( "Mgmt::Account::SessionLimit" )
unlimitedSessionLimit = sessionLimitType.unlimited
minSessionLimit = sessionLimitType.min
maxSessionLimit = sessionLimitType.max

# --------------------------------------------------------------------------
# Mode "management accounts" 
# --------------------------------------------------------------------------
class AccountsConfigMode( ConfigMgmtMode.ConfigMgmtMode ): 

   name = "Accounts configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ): 
      ConfigMgmtMode.ConfigMgmtMode.__init__( self, parent, session, "accounts" )

   def config( self ): 
      return accountsConfig

   def setSessionLimitGlobal( self, limit ):
      self.config().sessionLimitGlobal = limit

   # This is the common handler for [no|default] cmd on global limit.
   def noSessionLimitGlobal( self ): 
      self.config().sessionLimitGlobal = self.config().sessionLimitGlobalDefault

   def gotoAccountsConfigUserMode( self, account ):
      childMode = self.childMode( UserConfigMode, userName=account )
      self.session_.gotoChildMode( childMode )

   def noUser( self, account ): 
      del self.config().account[ account ]

   def noAccounts( self ):
      self.noSessionLimitGlobal()
      self.config().account.clear()

def gotoAccountsConfigMode( mode, args ):
   childMode = mode.childMode( AccountsConfigMode )
   mode.session_.gotoChildMode( childMode )

def defaultAccountsConfig( mode, args ): 
   mode = mode.childMode( AccountsConfigMode )
   mode.noAccounts()

#--------------------------------------------------------------------------------
# [ no | default ] management accounts
#--------------------------------------------------------------------------------
class ManagementAccountsCmd( CliCommand.CliCommandClass ):
   syntax = 'management accounts'
   noOrDefaultSyntax = syntax
   data = {
      'management': ConfigMgmtMode.managementKwMatcher,
      'accounts': 'Configure user accounts',
   }
   handler = gotoAccountsConfigMode
   noOrDefaultHandler = defaultAccountsConfig

BasicCliModes.GlobalConfigMode.addCommandClass( ManagementAccountsCmd )

# --------------------------------------------------------------------------
# [no|default] session default-limit
# --------------------------------------------------------------------------
class SessionDefaultLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'session default-limit LIMIT'
   noOrDefaultSyntax = 'session default-limit ...'
   data = {
      'session': 'Configure session parameters',
      'default-limit': 'Default limit for maximum sessions per user',
      'LIMIT': CliMatcher.IntegerMatcher( minSessionLimit, maxSessionLimit,
                                          helpdesc='Set the limit value' ),

   }

   @staticmethod
   def handler( mode, args ):
      mode.setSessionLimitGlobal( args[ 'LIMIT' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noSessionLimitGlobal()

AccountsConfigMode.addCommandClass( SessionDefaultLimitCmd )

#--------------------------------------------------------------------------------
# [ no | default ] account USERNAME
#--------------------------------------------------------------------------------
class AccountUsernameCmd( CliCommand.CliCommandClass ):
   syntax = 'account USERNAME'
   noOrDefaultSyntax = syntax
   data = {
      'account': 'Configure user accounts',
      'USERNAME': CliMatcher.PatternMatcher( pattern=usernameRe,
                  helpname="WORD", helpdesc="Account name string" ),

   }

   @staticmethod
   def handler( mode, args ):
      mode.gotoAccountsConfigUserMode( args[ 'USERNAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noUser( args[ 'USERNAME' ] )

AccountsConfigMode.addCommandClass( AccountUsernameCmd )

# --------------------------------------------------------------------------
# SubMode "account <user>"
# --------------------------------------------------------------------------
class UserConfigMode( AccountsUserMode, BasicCli.ConfigModeBase ): 

   name = "Account specific configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, userName ): 
      self.name = userName 
      self.session_ = session
      AccountsUserMode.__init__( self, userName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.config().account.newMember( self.name )

   def config( self ): 
      return accountsConfig

   def setUserSessionLimit( self, limit ): 
      self.config().account[ self.name ].sessionLimit = limit

   def noUserSessionLimit( self ): 
      self.config().account[ self.name ].sessionLimit = unlimitedSessionLimit

   def defaultUserSessionLimit( self ): 
      self.config().account[ self.name ].sessionLimit = \
            self.config().account[ self.name ].sessionLimitUserDefault

#--------------------------------------------------------------------------------
# [ no | default ] session limit LIMIT
#--------------------------------------------------------------------------------
class SessionLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'session limit LIMIT'
   noOrDefaultSyntax = 'session limit ...'
   data = {
      'session': 'Configure session parameters',
      'limit': 'Set the limit for maximum login sessions',
      'LIMIT': CliMatcher.IntegerMatcher( minSessionLimit, maxSessionLimit,
                                          helpdesc='Set the limit value' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.setUserSessionLimit( args[ 'LIMIT' ] )

   @staticmethod
   def noHandler( mode, args ):
      mode.noUserSessionLimit()

   @staticmethod
   def defaultHandler( mode, args ):
      mode.defaultUserSessionLimit()

UserConfigMode.addCommandClass( SessionLimitCmd )

def Plugin( entityManager ):
   global accountsConfig
   accountsConfig = ConfigMount.mount( entityManager, "mgmt/acct/config",
                                       "Mgmt::Account::Config", "w" )
