# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
import CliCommon
from CliModel import Bool
from CliModel import Dict
from CliModel import Float
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Enum
from CliModel import Submodel
import SecretCli
import Tac
import TableOutput
import datetime
import time
import math

ACCT_NAME_LEN = 8

NonInteractiveSessionsEnumToStr = {
      "commandApi" : "command-api",
      "ssh" : "ssh",
      "xmpp" : "xmpp",
      "jsonApi" : "json-api",
}

NonInteractiveSessionsMapping = {
      "command-api" : "commandApi",
      "ssh" : "ssh",
      "xmpp" : "xmpp",
      "json-api" : "jsonApi",
}

class Users( Model ):

   class User( Model ):
      username = Str( help="The username of this account" )
      role = Str( help="What role the account has been given"
                  " (e.g. 'network-admin')", optional=True,
                  default="network-operator" )
      privLevel = Int( help="The configured privilege level for this user"
                       " (between %d and %d inclusive)"
                       % ( CliCommon.MIN_PRIV_LVL, CliCommon.MAX_PRIV_LVL ) )
      sshAuthorizedKey = Str( help="SSH public key for this user", optional=True )

      def render( self ):
         print "user: %s" % self.username
         print "       role: %s" % self.role
         print "       privilege level: %d" % self.privLevel
         if self.sshAuthorizedKey:
            print "       ssh public key: %s" % self.sshAuthorizedKey

   users = Dict( valueType=User, help="Maps a username to details for this user" )

   def render( self ):
      for _, user in sorted( self.users.iteritems() ):
         user.render()

class AaaUsers( Model ):
   class AaaUser( Model ):
      password = Str( "The encrypted password of this account" )

   enablePassword = Str( help="The encrypted enable password", optional=True )
   users = Dict( valueType=AaaUser, help="Maps a username to details for this user" )

   def render( self ):
      if not self.enablePassword:
         self.enablePassword = "(None set)"
      print "Enable password (encrypted): %s" % self.enablePassword
      
      if not self.users:
         print "No logins configured"
      else: 
         passwdLen = len( SecretCli.md5EncryptedPassword( "A" ) )
         fmtStr = "%-" + str( ACCT_NAME_LEN ) + "s" + "  %-" + str( passwdLen ) + "s"
         print fmtStr % ( "Username", "Encrypted passwd" )
         print fmtStr % ( "-" * ACCT_NAME_LEN, "-" * passwdLen )
         for user in sorted( self.users ):
            print  fmtStr % ( user, self.users[ user ].password )

class AaaLockedUser( Model ):
   lockoutEndTime = Float( help="Time that user's account will be unlocked" )
   lockoutStartTime = Float( help="Time that user's account was locked" )

class AaaLockedUsers( Model ):
   lockedUsers = Dict( valueType=AaaLockedUser,
                       help="Maps a username to details for this user" )

   def render( self ):
      headings = ( 'User', 'Start Time', 'End Time', 'Expires In' )
      table = TableOutput.createTable( headings )
      for user, userInfo in sorted( self.lockedUsers.iteritems() ):
         startTime = time.ctime( userInfo.lockoutStartTime )
         endTime = time.ctime( userInfo.lockoutEndTime )
         # math.floor removes microseconds
         expirationTime = math.floor( userInfo.lockoutEndTime - Tac.utcNow() )
         expirationTimeStr = datetime.timedelta( seconds=expirationTime )
         table.newRow( user, startTime, endTime, expirationTimeStr )
      # 'table.output()' adds an empty line to terminate the table. The
      # tests don't expect this line, so remove it.
      print "\n".join( table.output().split( "\n" )[ : -1 ] )
           
class AaaCounters( Model ):
   authenticationSuccess = Int( help= "Number of successful authentication "
                                      "requests" )
   authenticationFail = Int( help= "Number of failed authentication requests" )
   authenticationUnavailable = Int( help= "Number of unavailable authentication "
                                          "requests" )
   authorizationAllowed = Int( help= "Number of allowed authorization requests" )
   authorizationDenied = Int( help= "Number of denied authorization requests" )
   authorizationUnavailable = Int( help= "Number of unavailable authorization "
                                         "requests" )
   accountingSuccess = Int( help= "Number of successful accounting requests" )
   accountingError = Int( help= "Number of error accounting requests" )
   pendingAccountingRequests = Int( help= "Number of pending accounting requests" )
   counterResetTimestamp = Float( help= "Timestamp of last counter reset", 
                                  optional=True )
   
   def render( self ):
      fmt = "%20s:%11d"
      def _printAttr( description, attr ):
         print fmt % ( description, attr )

      print "Authentication"
      _printAttr( "Successful", self.authenticationSuccess )
      _printAttr( "Failed", self.authenticationFail )
      _printAttr( "Service unavailable", self.authenticationUnavailable )      
      print
      print "Authorization"
      _printAttr( "Allowed", self.authorizationAllowed )
      _printAttr( "Denied", self.authorizationDenied )
      _printAttr( "Service unavailable", self.authorizationUnavailable )   
      print
      print "Accounting"
      _printAttr( "Successful", self.accountingSuccess )
      _printAttr( "Error", self.accountingError )
      _printAttr( "Pending", self.pendingAccountingRequests )
      if self.counterResetTimestamp:
         self.counterResetTimestamp = ( self.counterResetTimestamp + 
                                        Tac.now() - Tac.utcNow() )
      print "\nLast time counters were cleared:", \
             Ark.timestampToStr( self.counterResetTimestamp )
             
class AaaMethodListsMethods( Model ):
   methods = List( valueType=str, 
                   help="List of methods for a particular mode" )
   
   def renderMethods( self, name ):
      print "  name=%s methods=%s" % ( name, ", ".join( self.methods ) )
      
class AaaMethodListsAcctMethods( Model ):
   defaultAction = Str( help="Present if accounting for default actions is enabled", 
                        optional=True )
   defaultMethods = List( valueType=str, 
                          help="List of methods for default accounting" )
   _consoleUseOwnMethod = Bool( help="If console method is possible, private" )
   consoleAction = Str( help="Present if accounting for console actions is enabled", 
                        optional=True )
   consoleMethods = List( valueType=str, 
                          help="List of methods for a console accounting" )
   
   def renderMethods( self, name ):
      if not self.defaultAction:
         print "  name=%s default-action=none" % name
      else:
         print ( "  name=%s default-action=%s default-methods=%s" %
                 ( name, self.defaultAction, ", ".join( self.defaultMethods ) ) )
             
      if self._consoleUseOwnMethod:
         if not self.consoleAction:
            print " " * len( "  name=" + name ), "console-action=none"
         else:
            print ( "%s console-action=%s console-methods=%s" %  
                    ( " " * len( "  name=" + name ), self.consoleAction, 
                      ", ".join( self.consoleMethods ) ) )
             
class AaaMethodListsAuthen( Model ):
   loginAuthenMethods = Dict( valueType=AaaMethodListsMethods, 
                              help="A Mapping between login authentication type and "
                                   "methods configured" )
   enableAuthenMethods = Dict( valueType=AaaMethodListsMethods,
                                help="A Mapping between enable authentication type "
                                     "and methods configured" )
   dot1xAuthenMethods = Dict( valueType=AaaMethodListsMethods, 
                              help="A Mapping between dot1x authentication type and "
                                   "methods configured" )
   
   def renderAuthenType( self, listType, authType, authenMethodInfo ):
      print "Authentication method %s for %s:" % ( listType, authType )
      for name, methods in sorted( authenMethodInfo.items(), key=lambda x: x[ 0 ] ):
         methods.renderMethods( name )
   
   def render( self ):
      self.renderAuthenType( "lists", "LOGIN", self.loginAuthenMethods )
      self.renderAuthenType( "list", "ENABLE", self.enableAuthenMethods )
      self.renderAuthenType( "list", "DOT1X", self.dot1xAuthenMethods )

class AaaMethodListsAuthz( Model ):
   commandsAuthzMethods = Dict( valueType=AaaMethodListsMethods, 
                                help="A Mapping between command authorization type "
                                     "and methods configured" )
   execAuthzMethods = Dict( valueType=AaaMethodListsMethods,
                            help="A Mapping between exec authorization type "
                                 "and methods configured" )
   
   def renderAuthzType( self, listType, authzType, authzMethodInfo ):
      print "Authorization method %s for %s:" % ( listType, authzType )
      for name, methods in sorted( authzMethodInfo.items(), key=lambda x: x[ 0 ] ):
         methods.renderMethods( name )
   
   def render( self ):
      self.renderAuthzType( "lists", "COMMANDS", self.commandsAuthzMethods )
      self.renderAuthzType( "list", "EXEC", self.execAuthzMethods )
      
class AaaMethodListsAcct( Model ):
   commandsAcctMethods = Dict( valueType=AaaMethodListsAcctMethods, 
                               help="A Mapping between command accounting type "
                                     "and methods configured" )
   execAcctMethods = Dict( valueType=AaaMethodListsAcctMethods,
                           help="A Mapping between exec accounting type "
                                 "and methods configured" )
   systemAcctMethods = Dict( valueType=AaaMethodListsAcctMethods, 
                             help="A Mapping between system accounting type "
                                  "and methods configured" )
   dot1xAcctMethods = Dict( valueType=AaaMethodListsAcctMethods,
                            help="A Mapping between dot1x accounting type "
                                 "and methods configured" )
   
   def renderAcctType( self, listType, acctType, acctMethodInfo ):
      print "Accounting method %s for %s:" % ( listType, acctType )
      for name, methods in sorted( acctMethodInfo.items(), key=lambda x: x[ 0 ] ):
         methods.renderMethods( name )
   
   def render( self ):
      self.renderAcctType( "lists", "COMMANDS", self.commandsAcctMethods )
      self.renderAcctType( "list", "EXEC", self.execAcctMethods )
      self.renderAcctType( "list", "SYSTEM", self.systemAcctMethods )
      self.renderAcctType( "list", "DOT1X", self.dot1xAcctMethods )
      
class AaaMethodLists( Model ):
   authentication = Submodel( valueType=AaaMethodListsAuthen, 
                              help="AAA Authentication information" )
   authorization = Submodel( valueType=AaaMethodListsAuthz, 
                              help="AAA Authorization information" )
   accounting = Submodel( valueType=AaaMethodListsAcct, 
                              help="AAA Accounting information" )
   
   def render( self ):
      self.authentication.render()
      self.authorization.render()
      self.accounting.render()

def convertFromSeconds( idleTime ):
   days = idleTime / 86400
   hours = ( idleTime - ( days * 86400 ) ) / 3600
   minutes = (idleTime - ( ( days * 86400 ) + ( hours * 3600 ) ) ) / 60
   seconds = idleTime - ( ( days * 86400 ) + ( hours * 3600 ) + ( minutes * 60 ) )
   if days > 7:
      return '%dw%dd' % ( ( days / 7 ), ( days % 7 ) )
   elif days > 0:
      return '%dd%dh' % ( days, seconds / ( 60 * 60 ) )
   else:
      return "%02d:%02d:%02d" % ( hours, minutes, seconds )

class ShowAaaUsersList( Model ):
   sessionTty = Bool( help="Indicates if this is the terminal used by "
                           "the session that just ran this command" )
   user = Str( help="Login name of the user of this session" )
   idleTime = Int( help="Duration of session's inactivity in seconds" )
   location = Str( help="Source IP address or hostname for vty sessions "
                        "and absent for serial connections", optional=True )

class ShowAaaUsers( Model ):
   serials = Dict( valueType=ShowAaaUsersList,
                  help="Terminal information for serial connection" )
   vtys = Dict( valueType=ShowAaaUsersList,
                help="Terminal information for ssh or telnet connections" )

   def render( self ):
      headings = ( ' ', 'Line', 'User', 'Host(s)', 'Idle', 'Location' )
      table = TableOutput.createTable( headings )

      # All entries will be left justified, and all columns have the same
      # format.
      fmt_star = TableOutput.Format( justify="left", minWidth=2, maxWidth=2 )
      fmt_star.padLimitIs( True )
      fmt_line = TableOutput.Format( justify="center" )
      fmt_line.padLimitIs( True )
      fmt = TableOutput.Format( justify="left" )

      table.formatColumns( fmt_star, fmt_line, fmt, fmt, fmt, fmt )
       
      n = 1
      sessionType = 'con'
      for session in [ self.serials, self.vtys ]:
         for key, value in session.iteritems():
            table.newRow( '*' if value.sessionTty else ' ', '%d %s %s'
                           % ( n, sessionType, key ), value.user, 'idle',
                           convertFromSeconds( value.idleTime ), 
                           '-' if sessionType == 'con' else value.location )
            n += 1
         sessionType = 'vty'
      lines = table.output().split( '\n' )
      print lines[ 0 ]
      for line in lines[ 2: ]:
         print line

class ShowAaaSessions( Model ):
   class Sessions( Model ):
      username = Str( help="Login name of the user of this session" )
      role = Str( help="Role assigned to the user" )
      terminal = Str( help="Maps to terminal type and terminal number",
                 optional=True )
      state = Enum( values=( "established", "pending" ),
              help="State of the session either established or pending" )
      sessionStartTime = Int( help="UTC time at which session was started" )
      authMethod = Str( help="Authentication method of the session" )
      remoteAddress = Str( help="Remote ip address", optional=True )
      remoteHost = Str( help="Remote hostname", optional=True )
      service = Enum( values=NonInteractiveSessionsEnumToStr.keys(),
                help="Non-interactive sessions", optional=True )

   serials = Dict( keyType=int, valueType=Sessions,
                   help="Session information for serial connections" )
   vtys = Dict( keyType=int, valueType=Sessions,
                help="Session information for ssh or telnet connections" )
   nonInteractives = Dict( keyType=int, valueType=Sessions, 
         help="Session information for nonInteractive connections" )
   def render( self ):
      headings = tuple( ( h, "l" ) for h in (
         "Session",
         "Username",
         "Roles",
         "TTY",
         "State",
         "Duration",
         "Auth",
         "Remote Host" ) )

      table = TableOutput.createTable( headings )

      # All entries will be left justified, and all columns have the same
      # format.
      f = TableOutput.Format( justify="left" )
      f.noPadLeftIs( True )
      table.formatColumns( *( [ f ] * len( headings ) ) )

      for session in [ self.serials, self.vtys, self.nonInteractives ]:
         for key, value in session.iteritems():
            remote = ( value.remoteHost + '@' + value.remoteAddress
                    if value.remoteHost and value.remoteAddress else
                    ( value.remoteHost if value.remoteHost else
                    value.remoteAddress ) )
            durationStr = int( Tac.utcNow() - value.sessionStartTime )        

            table.newRow( key, value.username, value.role,
                          ( value.terminal.strip('dev/').replace( 'pts/', 'vty' )
                          if value.terminal else
                          NonInteractiveSessionsEnumToStr[ value.service ] ),
                          "P" if value.state == 'pending' else 'E',
                          str( datetime.timedelta( seconds=durationStr ) ),
                          str( value.authMethod ),
                          remote if remote else " " )
      
      # 'table.output()' adds an empty line to terminate the table. The
      # tests don't expect this line, so remove it.
      print "\n".join( table.output().split( "\n" )[ : -1 ] )

class ShowRolesList( Model ):
   sequenceNumber = Int( help="Sequence number designates a rule's placement "
                              "in the role, value ranges from 1 to 256" )
   cmdPermission = Enum( values=( "permit", "deny" ), help="If match is found "
                   "against a rule, decision is based on that rule and all other "
                   "rules followed are ignored. If no match is found, the command "
                   "is denied by default" )
   mode = Str( help="The regex of a cli mode name for which this rule applies, "
               "if absent, rule is valid in all modes", optional=True )
   cmdRegex = Str( help="Regular expression that corresponds to one "
                        "or more CLI commands" )

class Rules( Model ):
   rules = List( valueType=ShowRolesList, help="A mapping of an authorization "
                     "role name to its rules" )

class ShowRoles( Model ):
   roles = Dict( valueType=Rules, help="Display the contents of built-in "
                 "and user-defined roles" )
   defaultRole = Str( help="Default role" )
   def render( self ):
      print 'The default role is ' + self.defaultRole + '\n'
      for role, info in sorted( self.roles.iteritems() ):
         print 'role:', role
         for rule in info.rules:
            print '       ', rule.sequenceNumber, rule.cmdPermission, ( 'mode %s '
                  % rule.mode if rule.mode else '' ) + 'command', rule.cmdRegex

