#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import os
import signal
import sys
from Syscall import gettid, tgkill
import Tac
import TacSigint
import Tracing
import CliThreadCtrl

th = Tracing.Handle( "CliPatchSigint" )
t0 = th.trace0

# Monkey patch SIGINT handler in Tac.py:
#    enableSigintKeyboardInterrupt
#    disableSigintKeyboardInterrupt
#    setManualSigintHandler
#
# So they can work in a multi-threading environment.
#
# The new mechanism:
#
# A global per-thread KeyboardInterrupt state is maintained, which contains
#
# 1. enabled (set by enable/disableKeyboardInterrupt)
# 2. interrupted (causes Tac.checkInterrupt to raise KeyboardInterrupt)
# The per thread interrupted state is maintained in c code, so access to it is done
# via the CliThreadCtrl python extension. That's because it also needs to be
# accessed from C code, see the CliPrint feature.
#
# First of all, setManualSigintHandler() installs its own global interrupt
# handler which just does nothing.
#
# enable/disableSigintKeyboardInterrupt would set enabled
# for the thread.
#
# in CliServer's signal forwarding code, when it gets a SIGINT, it
# checks the current thread's enabled state.
#
# 1) if it's enabled, it sends a signal (*) to the thread. This would trigger
# -EINTR for any blocking calls and interrupt the caller. However, it would
# not be a KeyboardInterrupt (Python code expecting it would need to handle
# OSError).
#
# 2) if it's disabled, just set the interrupted flag for the thread.
#
# checkInterrupt() would check and clear the per-thread interrupted flag.
#
# (*) We can use SIGINT, but for now we use SIGUSR2 as it'd be nice to be able
# to interrupt CliServer when doing interactive testing.

useSignum = signal.SIGUSR2

class KeyboardInterruptState( object ):
   def __init__( self, tid ):
      self.tid = tid
      self.enabled = False
      self.interruptedIs( False )

   def checkInterrupt( self ):
      # pylint: disable-msg=c-extension-no-member
      interrupted = CliThreadCtrl.isInterrupted( self.tid )
      # pylint: disable-msg=c-extension-no-member
      CliThreadCtrl.resetInterrupted( self.tid )
      if interrupted:
         raise KeyboardInterrupt

   def clearKeyboardInterrupt( self ):
      # pylint: disable-msg=c-extension-no-member
      CliThreadCtrl.resetInterrupted( self.tid )

   def enabledIs( self, enabled ):
      assert gettid() == self.tid
      Tac.threadsigmask( useSignum, not enabled )
      self.enabled = enabled
      if not enabled:
         self.checkInterrupt()

   def interruptedIs( self, interrupted ):
      # pylint: disable-msg=c-extension-no-member
      if interrupted:
         CliThreadCtrl.setInterrupted( self.tid )
      else:
         CliThreadCtrl.resetInterrupted( self.tid )

   def isInterrupted( self ):
      # pylint: disable-msg=c-extension-no-member
      return CliThreadCtrl.isInterrupted( self.tid )

   def __str__( self ):
      return "KeyboardInterrupt<%s,%s>" % ( self.enabled, self.isInterrupted() )

threadState_ = dict()

def getState( tid=None ):
   if tid is None:
      tid = gettid()
   state = threadState_.get( tid, None )
   if not state:
      state = KeyboardInterruptState( tid )
      threadState_[ tid ] = state
   return state

def cleanupState( tid=None ):
   if tid is None:
      tid = gettid()
   threadState_.pop( tid, None )
   CliThreadCtrl.delInterrupted( tid ) # pylint: disable-msg=c-extension-no-member

def enableSigintKeyboardInterrupt():
   sys.stdout.flush()
   sys.stderr.flush()
   getState().enabledIs( True )

def disableSigintKeyboardInterrupt():
   getState().enabledIs( False )

def checkInterrupt():
   getState().checkInterrupt()

def clearKeyboardInterrupt():
   getState().clearKeyboardInterrupt()

def kill( tid ):
   state = getState( tid )
   t0( "kill", tid, str( state ) )
   state.interruptedIs( True )
   if state.enabled:
      tgkill( os.getpid(), tid, useSignum )

def sigintHandler( signum, frame ):
   pass

def init():
   signal.signal( useSignum, sigintHandler )
   signal.siginterrupt( useSignum, True )
   TacSigint.check = checkInterrupt
   TacSigint.clear = clearKeyboardInterrupt
   # pylint: disable-msg=W0212
   TacSigint._setImmediate = enableSigintKeyboardInterrupt
   # pylint: disable-msg=W0212
   TacSigint._unsetImmediate = disableSigintKeyboardInterrupt
