#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import os
import re
import string
import sys
import RegexParser

# We will only parse deprecated commands marked by deprecatedByCmd in startup-config.
DEPRECATED_CMD_ALLOWED = False

# Special pipe tokens
PIPE_TOKENS = ( '|', '>>', '>' )

def getModelet( mode, valueFunc ):
   if hasattr( valueFunc, "im_self" ) and ( valueFunc.im_self is None ):
      # Unbound method.  Try the modelet map.  If not in there, just pass
      # the root mode.
      if not valueFunc.im_class in mode.modeletMap:
         raise GrammarError(
            "In mode %s, I tried to call %s.  It is a member of "
            "class %s, but that modelet is not active." %
            ( mode, debugFuncName( valueFunc ), valueFunc.im_class ) )
      return mode.modeletMap[ valueFunc.im_class ]
   else:
      return mode

def stripFilter( tokens ):
   r = []
   for x in tokens:
      for p in PIPE_TOKENS:
         i = x.find( p )
         if i >= 0:
            # found a filter
            if i > 0:
               r.append( x[ : i ] )
            return r
      r.append( x )
   return r

def splitOnPipe( tokens ):
   # Find the first token that contains any of the pipeTokens, and
   # split the token up in the list. For example:
   # [ 'show', 'interface|nz' ] -> [ 'show', 'interface', '|', 'nz' ]
   # returns None if no pipe is found.
   for i, token in enumerate( tokens ):
      for p in PIPE_TOKENS:
         if token == p:
            return None # standalone pipe, already handled
         t = token.split( p, 1 )
         if len( t ) > 1:
            # found the pipe
            newTokens = tokens[ : i ]
            if t[ 0 ]:
               newTokens.append( t[ 0 ] )
            newTokens.append( p )
            if t[ 1 ]:
               newTokens.append( t[ 1 ] )
            newTokens += tokens[ i + 1 : ]
            return newTokens
   return None

def deprecatedCmdGuardCode( deprecatedByCmd, startupConfig ):
   if ( deprecatedByCmd and not startupConfig and not DEPRECATED_CMD_ALLOWED ):
      return "This command is deprecated by '%s'" % deprecatedByCmd
   return None

# The directory into which CLI exception logfiles are written.
CLI_LOG_DIR = '/var/log/cli'

class _NoMatchClass( object ):
   __slots__ = ()

   def __str__( self ):
      return "noMatch"

noMatch = _NoMatchClass()

class _PartialMatchClass( object ):
   __slots__ = ()

   def __str__( self ):
      return "partialMatch"

partialMatch = _PartialMatchClass()

class MatchResult( object ):
   __slots__ = ( 'result', 'aaaTokens', 'more' )

   def __init__( self, result, aaaTokens, more=False ):
      self.result = result
      # aaaTokens is only needed for valid results; for multi-token matcher
      # return a list.
      self.aaaTokens = aaaTokens
      # whether this matcher can accept more tokens
      self.more = more

   def __str__( self ):
      s = str( self.result )
      if self.more:
         s += " (+)"
      return s

#-------------------------------------------------------------------------------
# Debugging functions.
#-------------------------------------------------------------------------------
def debugFuncName( f ):
   name = getattr( f, 'func_name', str( f ) ) + '()'
   code = getattr( f, 'func_code', None )
   if code is None:
      filename = lineno = 'Unknown'
   else:
      filename = os.path.basename( code.co_filename )
      lineno = code.co_firstlineno
   return '%s[%s:%s]' % ( name, filename, lineno )

commentAppendStr = '!!'

#-------------------------------------------------------------------------------
# Cli Exceptions.
#-------------------------------------------------------------------------------
class ParserError( Exception ):
   """Base class for all exceptions originating in this module during parsing."""
   def __init__( self, *args, **kwargs ):
      Exception.__init__( self, *args )
      self.index = kwargs.get( 'index' )
      self.token = kwargs.get( 'token' )

   def __str__( self ):
      if self.index is None:
         return "%s" % Exception.__str__( self )
      else:
         return "%s (at token %d: %r)" % (
            Exception.__str__( self ),
            self.index,
            self.token )

class GrammarError( ParserError ):
   pass

class ParseError( ParserError ):
   pass

class InvalidInputError( ParseError ):
   def __init__( self, msg="Invalid input", **kwargs ):
      if "Invalid input" not in msg:
         msg = "Invalid input %s" % msg
      ParseError.__init__( self, msg, **kwargs )

class AmbiguousCommandError( ParseError ):
   def __init__( self, msg="Ambiguous command", **kwargs ):
      ParseError.__init__( self, msg, **kwargs )

class IncompleteCommandError( ParseError ):
   def __init__( self, msg="Incomplete command", **kwargs ):
      ParseError.__init__( self, msg, **kwargs )

class IncompleteTokenError( ParseError ):
   def __init__( self, msg="Incomplete token", **kwargs ):
      ParseError.__init__( self, msg, **kwargs )

class GuardError( ParseError ):
   def __init__( self, guardCode, **kwargs ):
      self.guardCode = guardCode
      ParseError.__init__( self, "Unavailable command (%s)" % guardCode, **kwargs )

class AlreadyHandledError( Exception ):
   TYPE_NONE = 0
   TYPE_ERROR = 1
   TYPE_WARNING = 2
   TYPE_INFO = 3
   """A command handler can raise this exception to abort immediately with an
   optional message/warning/error. TYPE_ERROR will cause CAPI to report error.
   """
   def __init__( self, msg=None, msgType=TYPE_NONE ):
      if msgType == self.TYPE_NONE:
         assert not msg, "no msgType is specified"
      else:
         assert msg, "no msg is specified"
      self.msgType = msgType
      Exception.__init__( self, msg )

class InformationalShowCmdError( AlreadyHandledError ):
   """
   This is a wrapper of AlreadyHandledError and should be deprecated.
   """
   def __init__( self, msg, warning=False ):
      msgType = self.TYPE_WARNING if warning else self.TYPE_INFO
      AlreadyHandledError.__init__( self, msg=msg, msgType=msgType )

class AuthzDeniedError( Exception ):
   def __init__( self, command, message="" ):
      self.command = command
      s = "Authorization denied for command '%s'" % command
      if message:
         s += ": %s" % message
      self.msg = s
      Exception.__init__( self, s )

class IncompatibleModelError( Exception ):
   def __init__( self, msg ):
      self.msg = msg
      Exception.__init__( self, msg )

   def __str__( self ):
      return "%s. If this is an intentional change please " \
            "change the revision on the model and write " \
            "a degrade function. See https://docs.google.com/a/arista.com/presen" \
            "tation/d/1g_JeUwqGFTz3xK7JQTWBANmtoI3aUJa33VHeU4uNNOI/edit#slide=id" \
            ".g61d0b4f2a_023 or " \
            "https://docs.google.com/a/aristanetworks.com/document/d/1rRTVsAZGx-" \
            "58xN9p1mZIN9erBFNOfEpJvAIpKkaoNiE/edit#heading=h.mfpmf4skthdf for " \
            "more details." % self.msg

class CliModelNotDegradable( Exception ):
   def __init__( self, msg ):
      self.msg = msg
      Exception.__init__( self, msg )

   def __str__( self ):
      return "This model cannot be downgraded because the changes in the feature "\
             "affected more than syntax, thus you have to understand the new "\
             "output to be able to make correct sense of it. %s" % self.msg

class CliModelChangedError( Exception ):
   def __init__( self, msg ):
      self.msg = msg
      Exception.__init__( self, msg )

   def __str__( self ):
      return "You have registered a new model. This model needs to be registered. " \
             "Please run Cli/CliReferenceModelUpdater.py. %s." % self.msg

class ApiError( ParserError ):
   """Base class for exceptions that Cli functions can return.

   If a Cli receives invalid arguments, or encounter another error related to
   the user's input that isn't an internal error (i.e. unexpected problem),
   then the Cli function should return or raise an instance of a subclass of
   ApiError.

   When the Cli command is executed interactively and an exception of this
   type is raised, the exception will be caught and its message will be
   printed to stdout.
   """
   ERR_CODE = 400  # Bad Request.

   def __str__( self ):
      return Exception.__str__( self )

class ModelUnsupportedError( ApiError ):
   """ The show command that was run did not return a model """
   ERR_CODE = 501 # Not Implemented

class CommandIncompatibleError( ApiError ):
   """ The command that was run cannot has been marked as
   incompatible. See the EapiIncompatible decorator. """
   ERR_CODE = 403 # Forbidden

class LoadConfigError( Exception ):
   """Errors generated by loading config from a file or URL."""
   def __init__( self, msg ):
      self.msg = msg
      Exception.__init__( self, msg )

class MD5Error( Exception ):
   """A content did not match the expected MD5 hash."""
   def __init__( self, msg ):
      self.msg = msg
      Exception.__init__( self, msg )

def printMessage( msg, prefix=None, cmd=None, lineNo=None ):
   if lineNo is None:
      if prefix is None:
         print( msg )
      else:
         print( '%s %s' % ( prefix, msg ) )
   else:
      assert cmd is not None
      print()
      print( '> %s' % cmd )
      print( '%s %s at line %d' % ( prefix, msg, lineNo ) )
   # In case the printed was a question and the code would do readline next, we need
   # to flush or sometimes the question would be posed after the answer is given.
   sys.stdout.flush()

def printWarningMessage( msg, cmd=None, lineNo=None ):
   printMessage( msg, "!", cmd, lineNo )

def printErrorMessage( msg, cmd=None, lineNo=None ):
   printMessage( msg, "%", cmd, lineNo )

def getCliLogDir():
   return os.environ.get( 'FILESYSTEM_ROOT', '' ) + CLI_LOG_DIR

def safeInt( n ):
   '''
   This is a safe version of int. Instead of letting
   process crash, it will retun None. The user needs
   to check the returned value for None and take
   appropriate action.
   '''
   try:
      v = int( n )
   except ValueError:
      return None
   return v

def safeFloat( n ):
   '''
   This is a safe version of float. Instead of letting
   process crash, it will retun None. The user needs
   to check the returned value for None and take
   appropriate action.
   '''
   try:
      v = float( n )
   except ValueError:
      return None
   return v

DEPRECATED_HELP_STRINGS = {
 '32-bit tag value': 'Number',
 '48-bit hardware address of arp entry': 'ARP entry MAC address',
 'aaa group definitions': 'Group configuration',
 'address family ipv4': 'IPv4 address family',
 'address family ipv6': 'IPv6 address family',
 'address family modifier': 'Unicast sub-address family',
 'administratively shut down this neighbor': 'Disable the neighbor',
 'advertising router (as an ip address)':
   'IP address of a router advertising the LSA',
 'advertising router link states': 'Router advertising the LSA',
 'all modules': 'Display information for all of the modules in the system',
 'always advertise default route': 'Advertise a default route unconditionally',
 'an ordered list as a regular-expression': 'A regular expression',
 'arp table': 'ARP entries',
 'arp type arpa': 'ARPA-type ARP entry',
 'as number': 'Autonomous system number',
 'asbr summary link states': 'ASBR summary route LSAs',
 'assign policy-map to the input of an interface':
   'Apply the policy map to ingress packets',
 'assign policy-map to the output of an interface':
   'Apply the policy map to egress packets',
 'authentication parameters for the user': 'User authentication values',
 'begin with the line that matches': 'Start output at the first matching line',
 'bfd information': 'Status for the BFD protocol',
 'bgp distance': 'Administrative distance for the BGP protocol',
 'bgp timers': 'Timers for the BGP protocol',
 'border and boundary router information': 'Status for border routers',
 'border routers': 'Status for border routers',
 'brief interface information': 'Condensed interface status',
 'brief output': 'Condensed output',
 'change current directory': 'Move to another directory',
 'class of service': 'COS value',
 'clear platform information': 'Clear platform specific state',
 'configure a local or remote snmpv3 engineid': 'SNMPv3 engine ID configuration',
 'configure from the terminal': 'Config mode',
 'configure ip address summaries': 'Summary route configuration',
 'configure load balancing': 'Load balancing configuration',
 'configure logging for interface': 'Interface-related logging configuration',
 'configure qos class map': 'Class map configuration',
 'configure qos policy map': 'Policy map configuration',
 'configure qos service policy': 'Service policy configuration',
 'connected routes': 'Connected interface routes',
 'contents of startup configuration': 'Configuration used at boot',
 'context name': 'SNMP group context identifier',
 'control distribution of default information':
   'Default route related configuration',
 'copy from current system configuration': 'System operating configuration',
 'copy from one file to another': 'Copy files',
 'current operating configuration': 'System running configuration',
 'database summary': 'LSA database',
 'datagram size': 'Ping packet size',
 'debug counters': 'Internal counters for debugging',
 'default domain name': 'DNS domain name',
 'default vrf': 'Default virtual routing and forwarding instance',
 'define a user security model group': 'SNMP USM group',
 'define a user who can access the snmp engine': 'SNMP user configuration',
 'define an administrative distance': 'Administrative distance configuration',
 'define an snmpv2 mib view': 'SNMPv2 view configuration',
 'delay value (seconds)': 'Number of seconds',
 'delete a file': 'Remove a file',
 'delete all multicast routes': 'All routes',
 'description of the interactive help system': 'Help',
 'destination address translation': 'Translation of destination addresses',
 'destination file path': 'Name of the file to write',
 'destination ip address': 'IP address for the destination',
 'destination mac address': 'MAC address for the destination',
 'detailed information': 'Display information in detail',
 'detailed interface information': 'Display information in detail',
 'detailed output': 'Display information in detail',
 'differentiated services code point (dscp)': 'Packet DSCP value',
 'directory or file name': 'The file or directory or interest',
 'disable the interface': 'Administratively shut off the interface',
 'display current working directory': 'Show the CLI shell current directory',
 'display detailed information': 'Display information in detail',
 'display hardware inventory with serial numbers':
   'Hardware components of the system',
 'display ospf router ids as dns names': 'Show DNS-resolved router names',
 'display the contents of a file': 'Print file contents',
 'display the session command history': 'CLI shell history of entered commands',
 'display the system clock': 'System time',
 'distance for external routes': 'BGP external route admin distance',
 'distance for internal routes': 'BGP internal route admin distance',
 'distance for local routes': 'BGP local route admin distance',
 'distribute a default route': 'Source a default route',
 'enable authentication': 'Use authenticatication',
 'enable ip routing': 'Routing for IP packets',
 'enable local proxy arp': 'Proxy ARP for subnet local addresses',
 'enable logging to all supported destinations': 'Turn on logging',
 'enable proxy arp': 'Proxy ARP for off-subnet addresses',
 'enable snmp; set community string and access privs':
    'SNMP community string configuration',
 'end ip address': 'Last IP address in range',
 'end of range': 'Interface list end',
 'engineid of a remote agent': 'Remote SNMP agent',
 'engineid of the local agent': 'Local SNMP agent',
 'enter configuration mode': 'Config mode',
 'entry index': 'Index value for the entry',
 'exclude lines that match': 'Do not print lines matching the given pattern',
 'exit from configure mode': 'Leave config mode',
 'external link state': 'External LSAs',
 'external link states': 'External LSAs',
 'file to be deleted': 'Name of file being removed',
 'file to display': 'Name of file to print',
 'filter by interface name': 'Interface name database selector',
 'forwarding router\'s address': 'Address of the nexthop router',
 'global ipv6 configuration commands': 'Configuration for IPv6',
 'group number': 'Multicast group ID number',
 'group to which the user belongs': 'SNMP group for the user',
 'group using the user security model (snmpv3)': 'SNMP USM group',
 'group using the v1 security model': 'SNMP v1 security group',
 'group using the v2c security model': 'SNMP v2c security group',
 'halt and perform a cold restart': 'Reboot the system',
 'hello interval': 'Time between hello packets',
 'hello interval value': 'Time between hello packets',
 'hello multiplier value': 'Hold time multiplier of the hello interval',
 'hold time': 'Number of seconds',
 'host name': 'Name of the host',
 'icmp message code': 'Message code for ICMP packets',
 'icmp message type': 'Message type for ICMP packets',
 'identification of the contact person for this managed node':
    'Person to contact about this system',
 'igmp host query interval': 'Time between IGMP queries',
 'igmp max query response value': 'Time to wait for IGMP query responses',
 'igmp static multicast group': 'Configured IGMP group membership',
 'include lines that match': 'Print lines matching the given pattern',
 'interface events': 'Events related to the interface',
 'interface filter': 'Interface selector',
 'interface information': 'Interface-specific details',
 'interface name': 'Name of the interface',
 'interface specific description':
    'Description string to associate with the interface',
 'interface status and configuration': 'Details related to interfaces',
 'interval in milliseconds': 'Interval in units of milliseconds',
 'interval in seconds': 'Interval in units of seconds',
 'ip address of arp entry': "ARP entry's IPv4 address",
 'ip address or hostname of a remote system': 'Remote hostname or IP address',
 'ip arp table': 'Address Resolution Protocol table',
 'ip group address': 'Multicast group address',
 'ip information': 'Details related to IPv4',
 'ip interface status and configuration': 'Details related to IPv4 interfaces',
 'ip routing table': 'Entries used for routing packets',
 'ip source address': 'IPv4 address used as a source',
 'ip subnet mask': "Subnet's mask value",
 'ipv4 address family': 'IPv4 related',
 'ipv4 echo': 'IPv4 ping',
 'ipv4 trace': 'IPv4 traceroute',
 'ipv6 address family': 'IPv6 related',
 'ipv6 echo': 'IPv6 ping',
 'ipv6 information': 'Details related to IPv6',
 'ipv6 interface status and configuration': 'Details related to IPv6 interfaces',
 'ipv6 trace': 'IPv6 traceroute',
 'is-is instance name': 'Name of the IS-IS protocol instance',
 'keepalive interval': 'Time between BGP keepalive messages in seconds',
 'key number': 'Key identifier',
 'label value': 'Value of the MPLS label',
 'link state id (as an ip address)': 'IP address used as the link state ID',
 'list file information': 'Details about the file',
 'list files on a filesystem': 'Disply details about files',
 'load interval delay in seconds': 'Number of seconds',
 'loopback interface': 'Hardware interface used for looping packets',
 'mac address': 'Ethernet address',
 'management address tlv': 'TLV for the switch management address',
 'metric value': 'Value of the route metric',
 'mib family is excluded from the view': 'Exclude the named MIB from the named view',
 'mib family is included in the view': 'Include the named MIB in the named view',
 'mib view family name': 'MIB name',
 'mib view to which this community has access': 'MIB view name',
 'mirroring information': 'Details on mirroring',
 'modify enable password parameters': 'Enable-privilege related configuration',
 'modify message logging facilities': 'Logging configuration',
 'modify system boot parameters': 'System boot configuration',
 'mtu (bytes)': 'Maximum transmission unit in bytes',
 'multicast source address': 'Source IP address',
 'multicast source discovery protocol (msdp)': 'MSDP protocol commands',
 'name of the group': 'Group name',
 'name of the next hop': 'String naming the next hop',
 'name of the user': 'SNMP user name',
 'name of the view': 'SNM view name',
 'negate a command or set its defaults': 'Disable the command that follows',
 'neighbor filter': 'Limit to specified PIM neighbors',
 'neighbor information': 'Protocol neighbor details',
 'neighbor list': 'Protocol neighbors',
 'network link states': 'OSPF network LSA entries',
 'network summary link states': 'OSPF summary LSA entries',
 'network time protocol': 'NTP',
 'next hop': 'Route next hop IP address',
 'next hop address': 'Next hop IP address for forwarding',
 'no accounting': 'Disable accounting',
 'notify view name': 'Name of the notifying view',
 'nssa external link states': 'OSPF NSSA LSA entries',
 'ntp associations': 'Associations with NTP peers',
 'ntp status': 'Details of NTP operation',
 'ntp version number': 'Version of the NTP protocol',
 'number of lines on screen (0 for no pausing)':
    'Terminal pager lines, disable the pager with value 0',
 'number of mac addresses': 'MAC address limit',
 'number of probes': 'Probe packets to send per interval',
 'number of retries to this server for a transaction': 'Retry limit',
 'object name': 'String name',
 'opaque area link states': 'OSPF area-local opaque LSA entries',
 'opaque as link states': 'OSPF AS-wide opaque LSA entries',
 'opaque link-local link states': 'OSPF link-local LSA entries',
 'open a telnet connection': 'TELNET client',
 'open a terminal connection': 'TELNET client',
 'open shortest path first (ospf)': 'OSPF protocol',
 'originate default route to this neighbor':
    'Advertise a default route to this peer',
 'ospf router-id in ip address format': 'OSPF 32-bit router ID as an IP address',
 'output modifiers': 'Command output pipe filters',
 'packet counters': 'counters',
 'ping destination address or hostname':
    'Ping IPv4 address, IPv6 address, or hostname',
 'port description tlv': 'TLV for the port description string',
 'port id': 'OpenStack identity string for the port',
 'port name': 'Name for the port',
 'port number': 'Number of the port to use',
 'port-channel interface': 'Link Aggregation Group (LAG)',
 'prefix length': 'Length of the prefix in bits',
 'prefix list name': 'Name of the prefix list',
 'priority level': 'VRRP router priority value',
 'process id': 'Process identifier',
 'query interval in seconds': 'Time between queries in units of seconds',
 'radius configuration': 'Commands for RADIUS',
 'radius server attributes': 'Details of RADIUS operation',
 'rate in kbps': 'The rate expressed in units of kilobits per second',
 'read view name': 'Name of the view for reading',
 'read-only access with this community string': 'Only reads are permitted',
 'reason for reload': 'String to display explaining the purpose of the reload',
 'redistribute ospf external routes': 'OSPF routes learned from external sources',
 'redistribute ospf internal routes': 'OSPF routes learned from internal sources',
 'redistribute ospf nssa external routes':
    'OSPF routes learned from external NSSA sources',
 'redistribution of ospf routes': 'Routes learned by the OSPF protocol',
 'rename a file': 'Move the file',
 'repeat count': 'Count',
 'reset a terminal line': 'A terminal session',
 'restrict this community to a named mib view': 'Limits community access to a view',
 'route distinguisher': 'BGP route distinguisher',
 'route map reference': 'Name a route map',
 'router link states': 'OSPF router LSA entries',
 'rpf across equal-cost paths': 'Multiple reverse paths',
 'rule number': 'Rule index',
 'select an interface to configure': 'Interface configuration',
 'self-originated link states': 'OSPF locally originated LSA entries',
 'send community attribute to this neighbor': 'Enable sending communities',
 'send echo messages': 'Ping remote systems',
 'send inform messages to this host': 'Use SNMP inform messages',
 'send trap messages to this host': 'Use SNMP trap messages',
 'sequence number': 'Index in the sequence',
 'session information': 'Details about a session',
 'set a static arp entry': 'Configured ARP table data',
 'set advertised ns retransmission interval':
    'Time between neighbor solicitation retransmissions',
 'set authentication list for enable': 'Enable-privilege related configuration',
 'set authentication lists for logins': 'Login related configuration',
 'set buffered logging parameters': 'Logging buffer configuration',
 'set ip dscp (diffserv codepoint)': 'DSCP values',
 'set ipv6 router advertisement interval': 'Time between RA messages',
 'set key string': 'Authentication key',
 'set priority for designated router election': 'DR election priority',
 'set syslog server logging level': 'Severity of messages sent to the syslog server',
 'set tacacs+ encryption key': 'Key string',
 'set the ip address of an interface': 'Interface address',
 'show all enabled dot1q-tunnel ports': 'Details for 802.1q tunnels',
 'show controller information': 'Details of CVX operation',
 'show detailed information': 'More comprehensive output',
 'show detailed output': 'More comprehensive output',
 'show diagnostic tests': 'Diag outputs',
 'show filesystem information': 'Details about files',
 'show interface description': 'Details on description strings of interfaces',
 'show interface flowcontrol information': 'Details on flow control',
 'show interface line status': 'Details on the state of interfaces',
 'show interface switchport information': 'Details on switchports',
 'show interface transceiver': 'Details on transceivers',
 'show interface vlan information': 'Details on VLAN interfaces',
 'show ptp interface information': 'Details on PTP',
 'show qos class map': 'Details on class maps',
 'show qos policy map': 'Details on policy maps',
 'show running system information': 'Display details of switch operation',
 'show snmpv3 groups': 'Details of groups in SNMPv3',
 'show snmpv3 users': 'Details of users in SNMPv3',
 'show summary information': 'Summarized output',
 'show switch version information': 'Software and hardware versions',
 'show the contents of logging buffers': 'Details of the system log',
 'show vlan status': 'Details on VLAN operation',
 'snmp community string': 'Community name',
 'snmp statistics': 'Details on SNMP operation',
 'snmp version to use for notification messages':
    'Notification message SNMP version',
 'snmpv1/v2c community string or snmpv3 user name': 'Community or user name',
 'source address translation': 'Source',
 'source file path': 'Name of the file',
 'source ip address': 'IP address of the source',
 'source mac address': 'MAC address of the source',
 'spanning tree subsystem': 'Spanning tree protocol',
 'spanning tree topology': 'Details on STP',
 'specifies that an unencrypted key will follow':
    'Indicates that the key string is not encrypted',
 'specify a notify view for the group': 'View to restrict notifications',
 'specify a radius server': 'RADIUS server configuration',
 'specify a read view for the group': 'View to restrict read access',
 'specify a remote snmp entity to which the user belongs':
    'System where an SNMPv3 user is hosted',
 'specify a tacacs+ server': 'TACACS+ server configuration',
 'specify a write view for the group': 'View to restrict read access',
 'specify hosts to receive snmp notifications': 'Notification destinations',
 'specify interface': 'Interface configuration',
 'specify interval for load calculation for an interface':
    'Time used in calculating interface utilization',
 'specify name of the next hop': 'Next hop name',
 'stamp logger messages with a sequence number': 'Number log messages',
 'start ip address': 'First address',
 'string to uniquely identify this chassis': 'SNMP chassis identifier',
 'summary of database': 'Link state database summary',
 'suppress routing updates in an interface':
    'Include interface but without actively running OSPF',
 'suppress routing updates on this interface':
    'Include interface but without actively running IS-IS',
 'system capabilities tlv': 'TLV for advertising system capabilities',
 'system description tlv': 'TLV for the system description string',
 'system name tlv': 'TLV for the system name string',
 'tacacas+ server attributes': 'Details of TACACS+ operation',
 'tcp protocol': 'TCP',
 'text for mib object syscontact': 'The sysContact string',
 'text for mib object syslocation': 'The sysLocation string',
 'the hidden shared key': 'Obfuscated key string',
 'the notification host\'s udp port number': 'UDP port for notification messages',
 'the physical location of this node': 'The location string',
 'the remote snmp entity\'s udp port number':
    'UDP port used by the remote SNMP system',
 'the unencrypted (cleartext) shared key': 'Unobfuscated key string',
 'threshold value': 'Percentage from 0 to 100',
 'time in minutes': 'Number of minutes',
 'time in seconds': 'Number of seconds',
 'time interval in seconds': 'Number of seconds',
 'time to live value': 'Tunnel encapsulation TTL value',
 'timeout in seconds': 'Number of seconds',
 'topology information': 'Details of the STP topology',
 'trace route to destination': 'Traceroute command',
 'trace route to destination address or hostname': 'DNS name or IP address',
 'track an interface': 'Name an interface',
 'traffic class': 'Internal class of traffic',
 'transmission control protocol': 'TCP',
 'tunnel id': 'Tunnel identifier',
 'turn off privileged commands': 'Disable commands for a specified privilege level',
 'turn on privileged commands': 'Enable commands for a specified privilege level',
 'udp protocol': 'UDP',
 'unique id string': 'The chassis identifier string',
 'update (merge with) current system configuration': 'System running configuration',
 'use hmac md5 algorithm for authentication': 'User authenticated using HMAC MD5',
 'use hmac sha algorithm for authentication': 'User authenticated using HMAC SHA',
 'use the snmpv3 authnopriv security level': 'SNMPv3 security level - authNoPriv',
 'use the snmpv3 noauthnopriv security level':
   'SNMPv3 security level - noAuthNoPriv',
 'user datagram protocol': 'UDP',
 'user name': 'Account name string',
 'user using the v1 security model': 'SNMPv1 security',
 'user using the v2c security model': 'SNMPv2 security',
 'user using the v3 security model': 'SNMPv3 security',
 'verify a file': 'Check SWI signature',
 'version number': 'Version',
 'virtual terminal': 'Terminal',
 'vlan id': 'Identifier for a Virtual LAN',
 'vlan interface': 'Logical interface into a VLAN',
 'vpn routing/forwarding instance': 'VRF name',
 'wait time (default 5 seconds)': 'Number of seconds',
 'write view name': 'Name of the view for writing'
}

# pylint: disable-msg=redefined-builtin
class Completion( object ):
   __slots__ = ( 'name', 'help', 'literal', 'partial', 'guardCode', 'common' )

   def __init__( self, name, help, literal=True, partial=False, guardCode=None,
                  common=False ):
      assert name and isinstance( name, str )
      if name.lower() in DEPRECATED_HELP_STRINGS:
         # these are help string that aren't allowed on our system. So instead we
         # dynamically replace the help strings.
         name = DEPRECATED_HELP_STRINGS[ name.lower() ]
      if help and help.lower() in DEPRECATED_HELP_STRINGS:
         # these are help string that aren't allowed on our system. So instead we
         # dynamically replace the help strings.
         help = DEPRECATED_HELP_STRINGS[ help.lower() ]
      self.name = name
      self.help = help
      self.literal = literal
      self.partial = partial
      self.guardCode = guardCode
      self.common = common

   def __cmp__( self, other ):
      # '<cr>' should always appear as the last item in the list of completions,
      # and so should compare as greater than any other name.
      if self.name == '<cr>':
         return 1
      elif other.name == '<cr>':
         return -1
      elif self.name != other.name:
         # Always put non-alphanum characters at the end.
         r = other.name[ 0 ].isalnum( ) - self.name[ 0 ].isalnum( )
         if r:
            return r
         return cmp( self.name, other.name )
      # The two completions have the same name. We consider them different
      # if one is guarded but the other is not. However, if both are guarded
      # with different codes, we still treat them the same (so only one is left),
      # otherwise we might get AmbiguousCommandError if there are no unguarded
      # completions.
      return int( other.guardCode is None ) - int( self.guardCode is None )

   def __hash__( self ):
      return hash( self.name )

   def __repr__( self ):
      return ( '<Completion: name=%s, help=%r%s%s%s%s>' %
               ( self.name, self.help,
                 "" if self.literal else ", nonliteral",
                 ", partial" if self.partial else "",
                 ", common" if self.common else "",
                 ", guardCode=%r" % self.guardCode if self.guardCode else "" ) )

eolCompletion = Completion( '<cr>', '', False )

PRIO_HIGH = 0
PRIO_NORMAL = 1
PRIO_LOW = 2
PRIO_KEYWORD = PRIO_HIGH

def makePartialPattern( pattern ):
   """Convert a regular expression pattern into a pattern that matches any prefix of
   a valid match for the original pattern."""

   # To do this correctly for all patterns would be difficult, so we just do
   # something that will work for most cases, and raise an exception at the first
   # sign of trouble.

   try:
      regex = RegexParser.parse( pattern )
      partialRegex = regex.partial()
      partialPattern = str( partialRegex )
      return partialPattern
   except Exception:
      raise Exception( "Unable to automatically create partial pattern from pattern "
                       "'%s'.  You need to manually create a partial pattern and "
                       "pass it to the PatternRule constructor." % pattern )

SPACE_RE = re.compile( '[' + string.whitespace + ']+' )

def ignoredComment( mode, line ):
   '''A line is an ignored comment if it starts with '!' and
   1. it does not start with '!!', or
   2. the mode does not support comment
   '''
   if not line.startswith( '!' ):
      return False
   if line[ : 3 ].rstrip() != commentAppendStr:
      return True
   return not ( mode and mode.commentSupported )

def textToTokens( text, mode=None ):
   text = text.strip()
   if text == '':
      return []
   if ignoredComment( mode, text ):
      # comments that we should ignore
      return []
   return re.split( SPACE_RE, text )

# namePattern is intended to be used as a (relatively) safe pattern token for user
# visible names throughout the system. It avoids characters that might be troublesome
# in show commands, where pipe or redirection might be a problem, or where shell
# shell substitution would be problematic. This is not appropriate for passwords or
# regular expressions.
namePattern = r'[^\s|>&;`$()\\]+'
excludePipeTokensPattern = '(?!(' + '|'.join( re.escape( x ) for x in PIPE_TOKENS ) \
                           + ')).+'

def isPrefixOfValueInRange( partial, minValue, maxValue, base=10 ):
   """Check if 'partial' may be in the range of [ minValue, maxValue ] with
   possibly more digits input. Only works for non-negative values."""
   if minValue <= partial and partial <= maxValue:
      return True

   if partial > maxValue:
      return False

   return isPrefixOfValueInRange( partial, minValue // base, maxValue // base, base )

# Default lower and upper bounds for sub interface range rule.
subLowerBound = 1
subUpperBound = 4094

class ParserOptions( object ):
   def __init__( self, autoComplete=True, disableGuards=False, startupConfig=False ):
      self.autoComplete = autoComplete
      self.disableGuards = disableGuards
      self.startupConfig = startupConfig
