# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import collections
import copy
import os
import sys

import CliMatcher
import CliParserCommon
from CliParserStructs import ParserCtx, ProcessCtxResult # pylint: disable-msg=E0611
import Tracing

# Disabling the tracing during parse for performance gain
PARSE_TREE_TRACING_ALLOWED = ( os.environ.get( 'CLI_PARSER_DEBUG', '' ) == '1' )

def traceGenerator( traceHandle ):
   def traceFunc( *args, **kwargs ):
      return traceHandle( *args, **kwargs ) if PARSE_TREE_TRACING_ALLOWED else None
   return traceFunc

th = Tracing.defaultTraceHandle()
t0 = traceGenerator( th.trace0 )
t2 = traceGenerator( th.trace2 )
t3 = traceGenerator( th.trace3 )
t7 = traceGenerator( th.trace7 )
t8 = traceGenerator( th.trace8 )

class CliParseTree( object ):
   def __init__( self ):
      self.rootNode_ = []
      self.rootKwNodeMap_ = collections.defaultdict( list )

   def getRootNodes( self ):
      return self.rootNode_

   def _guarded( self, ctx, raiseError=False ):
      # check if there is any context with a guard error
      while ctx:
         if ctx.guardError:
            if raiseError:
               raise ctx.guardError
            return True
         ctx = ctx.nextCtx
      return False

   def _essentiallySameNode( self, node1, node2 ):
      # Sometimes we create two nodes out of the same data, so when we resolve
      # conflicting contexts, we should treat them the same. This isn't precise,
      # but should be good enough.
      t2( 'checking if', node1.name_, 'and', node2.name_,
          'are essentially the same' )
      return node1.matcher_ == node2.matcher_ and node1.name_ == node2.name_

   def _resolveNextCtxs( self, ctx1, ctx2 ):
      # ctx1 and ctx2 are essentially the same, so try to walk down the nextCtx
      # chain to see if there is a higher priority one. This can happen if we
      # have the following:
      #
      # no foo ...
      # no foo bar
      #
      # and there are two nodes of foo.
      t0( "resolve essentially same nodes", ctx1.node.name_, ctx2.node.name_ )
      next1 = ctx1.nextCtx
      next2 = ctx2.nextCtx
      while next1 or next2:
         if next1 and next2:
            prio1 = next1.node.matcher_.priority_
            prio2 = next2.node.matcher_.priority_
            t2( next1.node.name_, "priority", prio1 )
            t2( next2.node.name_, "priority", prio2 )
            if prio1 < prio2:
               return ctx1
            elif prio1 > prio2:
               return ctx2
            if not self._essentiallySameNode( next1.node, next2.node ):
               raise CliParserCommon.GrammarError(
                  "node %r and %r match simultaneously" % ( next1.node.name_,
                                                            next2.node.name_ ) )
            next1 = next1.nextCtx
            next2 = next2.nextCtx
         elif next1:
            return ctx1
         else:
            return ctx2

      # keep the original by default
      return ctx2

   def _resolveCtxs( self, ctx1, ctx2, guarded1 ):
      t2( "resolving ctx for nodes", ctx1.node.name_, ctx2.node.name_ )
      # Given two ctx, return one that is "higher priority"
      if self._essentiallySameNode( ctx1.node, ctx2.node ):
         return self._resolveNextCtxs( ctx1, ctx2 )

      # first compare matcher priority, if same, pick one without guard error
      prio1 = ctx1.node.matcher_.priority_
      prio2 = ctx2.node.matcher_.priority_
      t2( ctx1.node.name_, "priority", prio1 )
      t2( ctx2.node.name_, "priority", prio2 )

      if prio1 < prio2:
         return ctx1
      elif prio1 > prio2:
         return ctx2
      elif guarded1:
         return ctx2
      else:
         if self._guarded( ctx2 ):
            return ctx1
         else:
            raise CliParserCommon.GrammarError(
               "node %r and %r match simultaneously" % ( ctx1.node.name_,
                                                         ctx2.node.name_ ) )

   def _processFinalCtxs( self, terminalMatches, guardedMatches ):
      # This function does two things:
      #
      # 1. Return the root context by traversing contexts backwards, and setup
      #    forward pointers.
      # 2. If we find two ctxs share the same parent context, pick the one with
      #    the higher priority (lower priority value).
      #
      # If we have more than one terminal matches with the same priority, we
      # raise GrammarError.
      #
      # Example: if we have ( kw1 kw2 | STRING )
      # where kw1 kw2 are keywords and STRING is a StringMatcher. If we pass in
      # 'kw1 kw2', it'll have two matches. If STRING is lower priority, we'll pick
      # the match with 'kw1 kw2'. If STRING is the same priority, we'll have
      # GrammarError.
      t0( "matches: terminal", len( terminalMatches ),
          "guarded", len( guardedMatches ) )
      rootTerminalMatches = []
      for i, ctx in enumerate( terminalMatches + guardedMatches ):
         t2( "process ctx", ctx.node.name_ )
         # get the root context
         skipRootCtx = False
         rootCtx = ctx
         rootCtxGuarded = ( i >= len( terminalMatches ) )
         while rootCtx.prevCtx:
            prevCtx = rootCtx.prevCtx
            t2( "process previous ctx", prevCtx.node.name_ )
            if prevCtx.nextCtx:
               t2( 'previous ctx', prevCtx.node.name_, 'already has a nextCtx',
                   prevCtx.nextCtx.node.name_, 'trying to resolve' )
               # somehow previous context has more than one nextCtx,
               # pick the highest priority
               skipRootCtx = True
               candidate = self._resolveCtxs( rootCtx, prevCtx.nextCtx,
                                              rootCtxGuarded )
               prevCtx.nextCtx = candidate
               break
            # continue
            prevCtx.nextCtx = rootCtx
            rootCtx = prevCtx
         if not skipRootCtx:
            rootTerminalMatches.append( rootCtx )

      t0( "root terminal matches:", len( rootTerminalMatches ) )
      if len( rootTerminalMatches ) > 1:
         # pick one with highest priority
         priorityMatches = []
         highestPriority = sys.maxint
         for m in rootTerminalMatches:
            priority = m.node.matcher_.priority_
            if priority < highestPriority:
               priorityMatches = [ m ]
               highestPriority = priority
            elif priority == highestPriority:
               priorityMatches.append( m )

         # if more than one, remove guarded ones
         if len( priorityMatches ) > 1:
            guarded = []
            unguarded = []
            for m in priorityMatches:
               if self._guarded( m ):
                  guarded.append( m )
               else:
                  unguarded.append( m )
            rootTerminalMatches = unguarded or guarded
         else:
            rootTerminalMatches = priorityMatches

         # We shouldn't have more than 1 terminal match. This means that we have
         # a bug in command syntax definition. But we may create multiple nodes
         # out of the same name in a CommandClass data.
         match = rootTerminalMatches[ 0 ]
         if not all( self._essentiallySameNode( match.node, x.node )
                     for x in rootTerminalMatches[ 1 : ] ):
            raise CliParserCommon.GrammarError(
               "More than 1 terminal matches: '%s'"
               % ','.join( x.node.name_ for x in rootTerminalMatches ) )
         rootTerminalMatches = [ match ]
      # check guard error
      if rootTerminalMatches:
         self._guarded( rootTerminalMatches[ 0 ], raiseError=True )
      return rootTerminalMatches

   def addCommandClass( self, commandClass ):
      commandClass.initialize()
      self._mergeTrees( commandClass.rootNodes(), self.rootNode_,
                        self.rootKwNodeMap_ )

   def addCommandClassRootNode( self, commandClass, rootNodeName ):
      commandClass.initialize()
      rootNodes = getattr( commandClass, rootNodeName )()
      if rootNodes:
         self._mergeTrees( rootNodes, self.rootNode_, self.rootKwNodeMap_ )

   def _maybeModifyHiddenAttr( self, srcNode, dstNode ):
      # This function looks to see if we need to modify the hidden attribute of
      # a node. Even if 2 nodes' hidden paramter differ we are still able to merge
      # the nodes. However if one of the nodes is hidden, we have to unhide that
      # node, a the hidden attribute pushed to it's children. We shouldn't have to
      # worry about cylcial structures because those are already considered
      # non-mergeable.
      if dstNode.hidden_ == srcNode.hidden_:
         # don't need to do anything before merging
         return

      for node in ( srcNode, dstNode ):
         if not node.hidden_:
            continue

         # this node is hidden, so mark it's nextNodes as hidden, but this node
         # then becomes non-hidden
         for nextNode in node.nextNodes_:
            nextNode.hidden_ = True
         node.hidden_ = False

   def _checkGuards( self, srcNode, dstNode ):
      if srcNode.guard_ == dstNode.guard_:
         return True
      if srcNode.guard_ is not None and dstNode.guard_ is not None:
         return False

      return False

   def _maybeMergeCmdHandler( self, srcNode, dstNode ):
      # Generally speaking we can merge CmdHandler. If both nodes have a different
      # command handler set, that is an error
      if srcNode.cmdHandler_ == dstNode.cmdHandler_:
         return

      if dstNode.cmdHandler_ is not None:
         assert srcNode.cmdHandler_ is None, 'Ambiguous commands being registered'
      else:
         dstNode.cmdHandler_ = srcNode.cmdHandler_

   def _mergeTrees( self, srcNodes, dstNodes, dstNodesMap=None ):
      # So in order to merge we look at all of the dstNodes and find which
      # ones can be merged with srcNodes, and go ahead and merge them.
      # Whichever of the next nodes aren't mergeable  we will just add those to
      # the dstNodes as is
      unmergedNodes = []
      for srcNode in srcNodes:
         for dstNode in dstNodes:
            if not dstNode.canMergeNode( srcNode ):
               continue
            if not self._checkGuards( srcNode, dstNode ):
               continue
            self._maybeModifyHiddenAttr( srcNode, dstNode )
            self._maybeMergeCmdHandler( srcNode, dstNode )
            self._mergeTrees( srcNode.nextNodes_, dstNode.nextNodes_ )
            break
         else: # (unbroken, like a poorly trained puppy).
            # loop fell through without finding a match
            unmergedNodes.append( srcNode )
            if dstNodesMap is not None and isinstance( srcNode.matcher_,
                                                       CliMatcher.KeywordMatcher ):
               dstNodesMap[ srcNode.matcher_.keyword_ ].append( srcNode )
               if srcNode.matcher_.alternates_:
                  for k in srcNode.matcher_.alternates_:
                     dstNodesMap[ k ].append( srcNode )

      dstNodes.extend( unmergedNodes )

   def _getCmdHandlerInfo( self, mode, rootCtx ):
      aaaTokens = []
      kwargs = {}
      argsList = []

      lastCtx = iterCtx = rootCtx
      cmdDeprecatedBy = None
      while iterCtx:
         currCtx = iterCtx
         node = iterCtx.node
         if node.deprecatedByCmd_:
            cmdDeprecatedBy = node.deprecatedByCmd_
         result = iterCtx.result
         tokens = iterCtx.aaaTokens
         lastCtx = iterCtx
         iterCtx = iterCtx.nextCtx

         # make a copy of the aaaTokens so that each ctx has its own aaaTokens
         if tokens is not None:
            if node.sensitive_:
               aaaTokens.append( '*' )
            elif isinstance( tokens, str ):
               aaaTokens.append( tokens )
            else:
               assert isinstance( tokens, list )
               aaaTokens.extend( tokens )

         if node.noResult_:
            continue

         # if a matcher has a value function, call it on the result
         f = node.matcher_.valueFunction( currCtx )
         if f:
            result = f( mode, result )

         # if a node is repeatable this means that it can be matched multiple times.
         # Since it can be repeated multiple times the argument should be a list.
         # in general it doesn't make sense to make it a list, so we special case
         # repeatable nodes
         if node.isRepeatable_ and node.maxMatches_ != 1:
            if node.alias_ not in kwargs:
               kwargs[ node.alias_ ] = []
            kwargs[ node.alias_ ].append( result )
         else:
            assert node.alias_ not in kwargs, "node %s in kwargs %r" % \
               ( node.alias_, kwargs )
            kwargs[ node.alias_ ] = result

         argsList.append( ( node.alias_, result ) )

      cmdHandler = lastCtx.node.cmdHandler_
      adapters = lastCtx.node.adapters_

      if adapters:
         seenAdapters = set()
         for adapter in lastCtx.node.adapters_:
            if adapter in seenAdapters:
               continue
            seenAdapters.add( adapter )
            adapter( mode, kwargs, argsList )

      originalCls = cmdHandler.originalCls
      if ( getattr( originalCls, 'rootNodesCacheEnabled', False ) and
           mode and mode.session ):
         t0( "save lastRootNodes", mode )
         mode.session.sessionDataIs( 'lastRootNodes',
                                    ( mode, originalCls.cachedRootNodes() ) )

      # if this is a command that has a filter (ie a show command)
      # then we need to strip out any filters. This is because the pipes need to be
      # authorized sepreately and it is handled by the show command handler.
      if cmdHandler.cmdFilter:
         authzTokens = CliParserCommon.stripFilter( list( aaaTokens ) )
      else:
         authzTokens = list( aaaTokens )
      aaa = { 'requireAuthz': cmdHandler.authz,
              'requireAcct': cmdHandler.acct,
              'requireSyncAcct': cmdHandler.syncAcct,
              'authzTokens': authzTokens,
              'acctTokens': aaaTokens,
              'authzFunc': cmdHandler.authzFunc,
              'acctFunc': cmdHandler.acctFunc }
      return { 'valueFunc': cmdHandler.handler,
               'autoConfigSessionAllowed': cmdHandler.autoConfigSessionAllowed,
               'allowCache': cmdHandler.allowCache,
               'kargs': { 'args': kwargs },
               'aaa': aaa,
               'cmdDeprecatedBy': cmdDeprecatedBy }

   def _matchObj( self, node ):
      return node.sharedMatchObj_ if node.sharedMatchObj_ else node

   def _maxMatched( self, node, matchedNodes ):
      matchObj = self._matchObj( node )
      return matchedNodes.get( matchObj, 0 ) >= node.maxMatches_

   def _processCurrCtxs( self, index, token, mode, currCtxs, parserOptions,
                         finalToken ):
      ''' Our parser does a breadth first search. This function will take the all of
      the contexts for our current depth and returns a ctxResult which contains the
      results for the next level (or terminal matches). We have added optimization
      at root-level, where this function takes only keyword matcher based contexts
      for given token by doing dictionary lookup. If successful, we will skip
      iterating over other contexts at root-level. This means it will not detect what
      could have been ambiguous scenario earlier because of both keyword and non-kw
      nodes matching for a token at root-level.'''
      t0( "process context for", token, "final", finalToken )
      ctxResult = ProcessCtxResult()
      parentMultiTokenCtx = None
      guardedCtx = None

      for ctx in currCtxs:
         if parentMultiTokenCtx:
            if ctx.prevCtx is parentMultiTokenCtx:
               # We have a parent ctx that already matched this token,
               # so we bail out
               t2( "skip node", ctx.node, "as parent", parentMultiTokenCtx.node,
                   "matched" )
               continue
            else:
               parentMultiTokenCtx = None
         node = ctx.node
         matcher = node.matcher_

         # We need to check if we have seen this node before and if it has a cap on
         # the number of times it can match. If we've exceeded the number of times
         # it can be matched lets not even consider this match
         matchedNodes = ctx.matchedNodes
         if ( node.maxMatches_ and matchedNodes and
              self._maxMatched( node, matchedNodes ) ):
            continue

         # we try to match the node. If it does not match this ctx is a deadend

         # This is kinda ugly, but needed for some matcher's optimizations.
         # Since it's very specialized, we add it to the context instead of passing
         # through every match() function.
         ctx.finalToken = finalToken
         guardError = None
         try:
            mr = matcher.match( mode, ctx, token )

            # this node didn't match, so lets move on with our lives
            if mr is CliParserCommon.noMatch:
               continue

            t0( "node", node, "match result", mr )
            # Check for deprecated commands
            guardCode = CliParserCommon.deprecatedCmdGuardCode(
               node.deprecatedByCmd_, parserOptions.startupConfig )
            if ( guardCode is None and node.guard_ is not None and
                 not parserOptions.disableGuards ):
               guardCode = node.guard_( mode, token )
            if guardCode is not None:
               guardError = CliParserCommon.GuardError( guardCode,
                                                        index=index,
                                                        token=token )
         except CliParserCommon.GuardError as e:
            # The matcher might have raised a GuardError (it's rare)
            guardError = e
            guardError.index = index
            guardError.token = token

         if guardError is not None:
            t0( "node", node, "raised guard error:", guardError.guardCode )
            ctx.guardError = guardError
            if ( guardedCtx is None or
                 ctx.node.matcher_.priority_ <
                 guardedCtx.node.matcher_.priority_ ):
               # save the guard error in the ctx but no more further parsing
               guardedCtx = ctx
            continue

         # I matched, so all my children should be discarded
         parentMultiTokenCtx = ctx

         if mr is CliParserCommon.partialMatch:
            # we do not have a match, but we should continue to include this
            # context in the next match.
            ctxResult.nextCtxs.append( ctx ) # pylint: disable=no-member
            ctxResult.matchFound = True
            ctxResult.matchHasNextNodes = True
            continue

         # If this node is special and can only be a certain amount of time
         # we have to increment the cnt of the how many times it has matched.
         # if we have to take care not to touch matchedNodes directly as it
         # belong to this ctx, instead we would make a copy that we modify
         # and will place into our new ctx.
         if node.maxMatches_ is not None:
            if matchedNodes is None:
               matchedNodes = {}
            else:
               matchedNodes = dict( matchedNodes )
            matchObj = self._matchObj( node )
            if matchObj not in matchedNodes:
               matchedNodes[ matchObj ] = 0
            matchedNodes[ matchObj ] += 1

         nextNodes = node.nextNodes_
         if not matchedNodes:
            # this is an optimization. If we have seen any of the 'special' nodes
            # that require us to count the max number of uses don't create a new list
            # since we'll never filter anything out
            availableNextNodes = nextNodes
         else:
            availableNextNodes = []
            for n in nextNodes:
               if n.maxMatches_ and self._maxMatched( n, matchedNodes ):
                  continue
               availableNextNodes.append( n )

         if node.storeSharedResult_:
            t2( "store shared result", mr.result, "for", node.name_ )
            if ctx.sharedResult is None:
               ctx.sharedResult = {}
            if not node.isRepeatable_:
               assert node.name_ not in ctx.sharedResult
               ctx.sharedResult[ node.name_ ] = mr.result
            else:
               if node.name_ in ctx.sharedResult:
                  ctx.sharedResult[ node.name_ ].append( mr.result )
               else:
                  ctx.sharedResult[ node.name_ ] = [ mr.result ]

         ctx.result = mr.result
         if mr.aaaTokens:
            ctx.aaaTokens = mr.aaaTokens
         ctxResult.matchHasNextNodes |= len( availableNextNodes ) > 0
         ctxResult.matchFound = True

         if not finalToken:
            if mr.more:
               t2( "ctx returns more, continue in next" )
               ctxResult.nextCtxs.append( ctx ) # pylint: disable=no-member
            # We have to make a new ctx for the next set of children of
            # this node because we still have more to parse
            #
            # Note if the current ctx is multi-token, creating those child ctxs
            # is only opportunistic as if the next token can be matched by the
            # current ctx again, those child ctxs should be abandoned (parent
            # is greedy). This is handled by the parentMultiTokentCtx logic at
            # the beginning of this function.
            for nextNode in availableNextNodes:
               t2( "add next node", nextNode )
               ctxResult.nextCtxs.append( # pylint: disable=no-member
                  ParserCtx( nextNode, matchedNodes, ctx ) )

         # this is the end of the line for this tree branch.
         if node.cmdHandler_:
            ctxResult.terminalMatches.append( ctx ) # pylint: disable=no-member

      if guardedCtx:
         # only add guardedMatches if we do not have any other matches with
         # at least the same priority
         guardedPriority = guardedCtx.node.matcher_.priority_
         t2( "Guard error priority", guardedPriority,
             "terminal", len( ctxResult.terminalMatches ),
             "nextCtxs", len( ctxResult.nextCtxs ) )
         # pylint: disable=not-an-iterable
         if ( not ctxResult.matchFound or
              not ( any( c.node.matcher_.priority_ <= guardedPriority
                         for c in ctxResult.terminalMatches ) or
                    any( c.node.matcher_.priority_ <= guardedPriority
                         for c in ctxResult.nextCtxs ) ) ):
            ctxResult.matchFound = True
            ctxResult.guardedMatch = guardedCtx
         # pylint: enable=not-an-iterable
      return ctxResult

   def _availableNextNodes( self, nextNodes, matchedNodes ):
      if not matchedNodes:
         # this is an optimization. If we have seen any of the 'special' nodes
         # that require us to count the max number of uses don't create a new list
         # since we'll never filter anything out
         return nextNodes

      availableNextNodes = []
      for node in nextNodes:
         if node.maxMatches_ and matchedNodes and node in matchedNodes:
            if matchedNodes[ node ] >= node.maxMatches_:
               continue
         availableNextNodes.append( node )
      return availableNextNodes

   def _callCompletionFuncs( self, mode, ctx, partialToken, parserOptions ):
      ''' Will run a node's completion function with a given partial token.
      Returns all of the completions the given node can do'''
      node = ctx.node
      if node.hidden_:
         # if it is hidden we don't try to run the completer on it
         return set( [] )

      matcher = node.matcher_
      completions = matcher.completions( mode, ctx, partialToken )
      if not completions:
         return set( [] )

      deprecatedCmdGuardCode = CliParserCommon.deprecatedCmdGuardCode(
         node.deprecatedByCmd_,
         parserOptions.startupConfig )
      guard = node.guard_
      if ( deprecatedCmdGuardCode is not None or
           guard and not parserOptions.disableGuards ):
         # Check if the completions need to be guarded
         results = set()
         for completion in completions:
            if deprecatedCmdGuardCode is not None:
               guardCode = deprecatedCmdGuardCode
            else:
               guardCode = guard( mode, completion.name )
            if guardCode is not None:
               # Make a copy as we might not be able to modify the
               # original completion.
               completion = copy.copy( completion )
               completion.guardCode = guardCode
            results.add( completion )
      else:
         results = set( completions )
      return results

   def _errorToken( self, tokens, index ):
      return None if index is None else tokens[ index ]

   def _getCompletions( self, token, mode, currCtxs, parserOptions ):
      ''' Runs through all of the currCtxs and get completions for the
      next token within the context'''
      completions = set()
      for ctx in currCtxs:
         completion = self._callCompletionFuncs( mode, ctx, token,
                                                 parserOptions )
         completions.update( completion )
      return completions

   def _parseTokens( self, mode, tokens, parserOptions, forCompletion ):
      # Given a set of tokens this will go through the parse tree. These values:
      # 1) Terminal matches found
      # 2) guarded matches found
      # 3) Remaining contexts
      # 4) the last ctxResult seen
      # 5) last token index processed (for ParseError)

      if mode and mode.session and not forCompletion:
         # Optimization: if we saved root nodes from previous command handler,
         # use it for parsing first. If it fails, fall back to general root nodes.
         parserOptionsCopy = copy.copy( parserOptions )
         parserOptionsCopy.autoComplete = False
         lastRootNodeMode, lastRootNodes = mode.session.sessionData( 'lastRootNodes',
                                                                     ( None, None ) )
         if mode is lastRootNodeMode and lastRootNodes:
            t0( "use cached root nodes for mode", mode )
            initialCtxs = [ ParserCtx( node, None, None ) for node in lastRootNodes ]
            try:
               result = self._parseTokensWithCtxs( mode, tokens, parserOptionsCopy,
                                                   forCompletion,
                                                   initialCtxs )
               if result[ 0 ]: # we have terminal matches
                  return result
            except CliParserCommon.ParserError:
               pass

            t0( 'clear cached root nodes for cache miss' )
            mode.session.sessionDataIs( 'lastRootNodes', ( None, None ) )

      if tokens and tokens[ 0 ].lower() in self.rootKwNodeMap_:
         firstToken = tokens[ 0 ].lower()
         initialCtxs = [ ParserCtx( node, None, None )
                         for node in self.rootKwNodeMap_[ firstToken ] ]
      else:
         initialCtxs = [ ParserCtx( node, None, None ) for node in self.rootNode_ ]

      return self._parseTokensWithCtxs( mode, tokens, parserOptions, forCompletion,
                                        initialCtxs )

   def _parseTokensWithCtxs( self, mode, tokens, parserOptions, forCompletion,
                             initialCtxs ):
      currCtxs = initialCtxs
      processCtxResult = None
      terminalMatches = []
      guardedMatches = []
      tokensLen = len( tokens )
      i = None
      for i, token in enumerate( tokens ):
         finalToken = i == tokensLen - 1
         try:
            processCtxResult = self._inhaleCtxs( i, token, mode, currCtxs,
                                                 parserOptions,
                                                 False if forCompletion
                                                 else finalToken )
         except CliParserCommon.ParseError as e:
            e.index = i
            e.token = token
            raise
         if finalToken:
            terminalMatches.extend( processCtxResult.terminalMatches )
         if processCtxResult.guardedMatch:
            guardedMatches.append( processCtxResult.guardedMatch )
         currCtxs = processCtxResult.nextCtxs
         if not currCtxs:
            break
      return terminalMatches, guardedMatches, currCtxs, processCtxResult, i

   def getCompletions( self, mode, tokens, partialToken, parserOptions ):
      t0( 'getCompletions', tokens, 'partial', partialToken )
      result = self._parseTokens( mode, tokens, parserOptions, True )
      terminalMatches, guardedMatches, currCtxs, _, index = result
      if not terminalMatches and not guardedMatches and not currCtxs:
         raise CliParserCommon.InvalidInputError(
            index=index, token=self._errorToken( tokens, index ) )

      completions = set()
      for ctx in currCtxs:
         completions.update( self._callCompletionFuncs( mode, ctx, partialToken,
                                                        parserOptions ) )

      try:
         terminalMatches = self._processFinalCtxs( terminalMatches,
                                                   guardedMatches )
      except CliParserCommon.GuardError:
         if not completions:
            # raise GuardError if no other completions
            raise
         terminalMatches = []

      if partialToken == '' and len( terminalMatches ) == 1:
         # this means that we are at the end of the line
         completions.add( CliParserCommon.eolCompletion )
      return completions

   def _inhaleCtxs( self, index, token, mode, currCtxs, parserOptions, finalToken ):
      ''' Our parser does a breadth first search. This function will take the all of
      the contexts for our current depth and returns a ctxResult which contains the
      result for the next level. If not matches are found, this function will then
      auto-complete to try to get an exact match'''
      t2( "inhale", token )
      processCtxResult = self._processCurrCtxs( index, token, mode, currCtxs,
                                                parserOptions, finalToken )
      if processCtxResult.matchFound:
         # this means we found some matches
         return processCtxResult

      # we couldn't match anything, lets try to complete the command
      # to see if we find any matches
      completions = self._getCompletions( token, mode, currCtxs, parserOptions )
      t2( "completions for", token, "is:", completions )
      if not completions:
         return processCtxResult

      # If we have a mix of guard-blocked and guard-allowed completions,
      # we only care about the guard-allowed ones at this point.
      unguardedCompletions = [ c for c in completions if c.guardCode is None ]
      if unguardedCompletions:
         # for autocomplete, we really only care about literal ones.
         completions = [ c for c in unguardedCompletions if c.literal ]
      else:
         # All of my completions are blocked by a guard.
         # Fall through, with completions being entirely guarded.
         completions = list( completions )

      t2( "completions are", completions )
      # At this point, 'completions' is either all guard-allowed, or
      # all guard-blocked.
      # The code below implements the following logic:
      # * If I have exactly one literal unguarded completion, then accept it.
      # * If I have more than one literal unguarded completion, ambiguous.
      # * If I have zero unguarded completions and one guard-blocked, then
      #   pass the guard's complaint along.
      # * If I have zero unguarded completions and more than one
      #   guard-blocked, then ambiguous.
      if len( completions ) != 1:
         # This could be problematic for multi-token matcher, as both the matcher
         # and its children might have completions, where only the parent should
         # win if it has a liberal completion. But we do not have a multi-token
         # matcher with liberal completion in the middle, so let's not worry about
         # it for now.
         raise CliParserCommon.AmbiguousCommandError()
      if not parserOptions.autoComplete:
         raise CliParserCommon.IncompleteTokenError()

      token = completions[ 0 ].name
      return self._processCurrCtxs( index, token, mode, currCtxs,
                                    parserOptions, finalToken )

   def parse( self, mode, tokens, parserOptions ):
      t0( "parse", tokens )
      # This is a BREADTH first search of our parse tree.
      lastCtxResult = None
      result = self._parseTokens( mode, tokens, parserOptions, False )
      terminalMatches, guardedMatches, _, lastCtxResult, index = result

      # See if we didn't find any terminal matches. If we don't have any terminal
      # matches we could raise 2 different errors,
      # IncompleteCommand and InvalidInput. The difference being that
      # IncompleteCommand means that we if we had more input we *could* match
      # more, but with InvalidInput we could never match this subset of tokens.
      if not terminalMatches:
         errorToken = self._errorToken( tokens, index )
         if ( lastCtxResult and lastCtxResult.matchFound and
              lastCtxResult.matchHasNextNodes ):
            raise CliParserCommon.IncompleteCommandError( index=index,
                                                          token=errorToken )
         elif not guardedMatches:
            raise CliParserCommon.InvalidInputError( index=index,
                                                     token=errorToken )

      terminalMatches = self._processFinalCtxs( terminalMatches,
                                                guardedMatches )

      # at this point we should only 1 match, so lets assert that that's the case
      # and return that one match
      assert len( terminalMatches ) == 1
      return self._getCmdHandlerInfo( mode, terminalMatches[ 0 ] )

   def _getCommandHandlerNodes( self, node, seenNodes, cmdHandlers ):
      if node in seenNodes:
         return
      seenNodes.add( node )

      if node.cmdHandler_:
         cmdHandlers.add( node.cmdHandler_ )

      for childNode in node.nextNodes_:
         self._getCommandHandlerNodes( childNode, seenNodes, cmdHandlers )

   def getCommandHandlerNodes( self ):
      seenNodes = set()
      cmdHandlers = set()
      for node in self.getRootNodes():
         self._getCommandHandlerNodes( node, seenNodes, cmdHandlers )
      return cmdHandlers
