# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import collections
import itertools
import sys
from types import FunctionType

import CliCommon
import Tac
import Tracing

INFINITY = float( "inf" )
NEGATIVE_INFINITY = float( "-inf" )

t0 = Tracing.trace0
t1 = Tracing.trace1
t2 = Tracing.trace2
t3 = Tracing.trace3
__defaultTraceHandle__ = Tracing.Handle( "CliModel" )

allowedBuiltInTypes = frozenset( [ int, long, float, str, bool ] )

GLOBAL_ATTRIBUTE_ORDER = 0

class _AttributeType( object ):
   """Base class for model attribute types."""
   realType = None  # To be overridden by subclasses.

   # pylint: disable-msg=W0622
   def __init__( self, help=None, optional=False, default=None, sinceRevision=1,
                 order=None ):
      assert help, "No help string given"
      self.help = help
      self.optional = optional
      self.sinceRevision = sinceRevision
      if default is not None:
         typeOk, valueOk, default = self.check( default )
         assert typeOk, ( "Invalid default value of type %s, expected %s"
                          % ( type( default ), self.realType ) )
         assert valueOk, "Invalid default value: %r" % ( default, )
      self.default = default

      # This helps is keep track of the order in which this attribute was created
      # later attributes get larger numbers
      global GLOBAL_ATTRIBUTE_ORDER
      self.order = GLOBAL_ATTRIBUTE_ORDER
      GLOBAL_ATTRIBUTE_ORDER += 1

   def check( self, value ):
      """Checks whether the given value can be assigned to this attribute.

      Returns a tuple ( typeOk, valueOk, newValue ) where:
        - typeOk is True if value has an acceptable type, False otherwise.
        - valueOk is True if the value is acceptable, False otherwise.
        - newValue is a possibly transformed version of value that must be
          used for storage (for instance when we convert a string into its
          corresponding TAC type, or a plain collection into a type one).
      """
      ok = type( value ) is self.realType
      return ok, ok, value

   def __repr__( self ):
      cls = type( self )
      classAttr = dir( cls )
      instAttr = dir( self )
      # Things defined on the instance and not on the class:
      attrs = set( instAttr ) - set( classAttr )
      return "%s(%s)" % ( cls.__name__, ", ".join( "%s=%r"
         % ( attr, getattr( self, attr ) ) for attr in attrs ) )

class Int( _AttributeType ):
   """Equivalent to a Python integer."""
   realType = ( int, long )

   def check( self, value ):
      cls = type( value )
      ok = cls is int or cls is long
      # if not ok and cls is str:
      #   try:
      #      value = int( value )
      #      ok = True
      #   except:
      #      pass
      return ok, ok, value

class Float( _AttributeType ):
   """Equivalent to a Python floating point value."""
   realType = float

   def check( self, value ):
      if type( value ) != float:
         return False, False, value

      # We can't convert NaN and +-infinity to JSON
      if value == INFINITY or value != value:
         value = sys.float_info.max
      elif value == NEGATIVE_INFINITY:
         value = sys.float_info.min
      return True, True, value

class Str( _AttributeType ):
   """Equivalent to a Python string. At this time we do not support
   unicode strings as valid values."""
   realType = str

   def check( self, value ):
      if type( value ) == unicode:
         return True, True, value.encode( 'utf8' )
      return _AttributeType.check( self, value )

class Bool( _AttributeType ):
   """Equivalent to a Python boolean."""
   realType = bool

class Enum( _AttributeType ):
   """A string attribute that can only take on certain values."""
   realType = str

   def __init__( self, values=None, **kwargs ):
      try:
         assert not isinstance( values, str )
         # Convert before checking values, in case we're given a generator.
         self.values = frozenset( values )
      except ( TypeError, AssertionError ):
         err = "Invalid type for `values`. Expected an iterable (except str); got %r"
         raise TypeError( err % type( values ) )

      for v in self.values:
         assert isinstance( v, str ), (
               "Invalid value type. Expected a str; got: %r" % type( v ) )

      super( Enum, self ).__init__( **kwargs )

   def check( self, value ):
      typeOk = type( value ) is str
      valueOk = value in self.values or ( value is None and self.optional )
      return typeOk, valueOk, value

def tacAttributeTypeSubclasses():
   return _TacAttributeType.__subclasses__() # pylint: disable-msg=E1101

class _TacAttributeType( _AttributeType ):
   """An attribute that wraps a TAC type.

   Subclasses need to implement _createTacValue() so that the TAC value can be
   created from a string.  The TAC value is also expected to have a stringValue
   attribute to allow itself to be converted to a string.

   For instance, with MAC addresses:
     >> mac = Tac.Value( "Arnet::EthAddr" )
     >> mac
     Value('Arnet::EthAddr', ** {'word1': 0, 'word0': 0, 'word2': 0})
     >> mac.stringValue
     '00:00:00:00:00:00'
     >> mac.stringValue = "01:02:03:04:05:06"  # This is _createTacValue()
     >> mac.stringValue
     '01:02:03:04:05:06'
     >> mac
     Value('Arnet::EthAddr', ** {'word1': 772, 'word0': 258, 'word2': 1286})
   """
   tacType = None  # To be set by subclass

   def _createTacValue( self, value ):
      raise NotImplementedError

   def check( self, value ):
      cls = type( value )
      if cls == unicode:
         value = value.encode( 'utf8' )
         cls = str
      if cls not in self.realType:
         return False, False, value
      valueOk = True
      if cls is str:
         # Check that it's a valid string.
         try:
            value = self._createTacValue( value )
         except ( IndexError, TypeError ):
            valueOk = False
      return True, valueOk, value

class _Collection( _AttributeType ):
   """An attribute that is a collection (dict or list)."""
   builtinType = None  # To be overridden by subclasses.

   def __init__( self, valueType=None, **kwargs ):
      if ( valueType in allowedBuiltInTypes
           or ( isinstance( valueType, type )
                and issubclass( valueType, ( Model, _TacAttributeType ) ) ) ):
         self.valueType = valueType
      else:
         assert False, "Invalid value type %r" % ( valueType, )
      # We need to call the parent ctor only here, once self.valueType is set.
      super( _Collection, self ).__init__( **kwargs )

   # The following methods are implemented just so pylint
   # would stop panicking and get on with its life.

   # Get rid of 'unsubscriptable-object' (E1136).
   # Get rid of 'unsupported-membership-test' (E1135) for `Dict`.
   def __getitem__( self, item ):
      raise NotImplementedError()

   # Get rid of 'unsupported-assignment-operation' (E1137).
   def __setitem__( self, item, value ):
      raise NotImplementedError()

   # Get rid of 'unsupported-membership-test' (E1135) for `List`.
   def __contains__( self, item ):
      raise NotImplementedError()

   # Get rid of 'not-an-iterable' (E1133).
   def __iter__( self ):
      raise NotImplementedError()

def _TacTypeCheck( tacType, tacValue,
                   variableType='value', valueOptional=False ):
   if type( tacValue ) == unicode:
      tacValue = tacValue.encode( 'utf-8' )
   if not isinstance( tacValue, tacType ):
      if issubclass( tacType, _TacAttributeType ):
         checker = tacType( help="<internal>" )
         typeOk, valueOk, newValue = checker.check( tacValue )
         if typeOk:
            if not valueOk:
               raise ValueError( "Invalid %s: %r" % ( variableType, tacValue, ) )
            # Make sure we use the string representation of the key because we
            # can't guarantee that the TAC value can be used as a key in a dict
            # (e.g. TypeError: non-const Arnet::IntfId objects are unhashable).
            tacValue = newValue.stringValue
      elif tacType is long and isinstance( tacValue, int ):
         typeOk = True # Accept int if tacType is long
      else:
         typeOk = False
      if not typeOk and not valueOptional:
         raise TypeError( "Invalid %s of type %s (must be %s): %r"
                          % ( variableType, type( tacValue ).__name__,
                              tacType.__name__, tacValue ) )
   return tacValue

class _TypedList( collections.MutableSequence ):
   """A list that does type checking of values."""

   def __init__( self, *args, **kwargs ):
      collections.MutableSequence.__init__( self )
      self.__underlying = list()
      self.__valueType = kwargs.pop( "__valueType", None )
      assert not kwargs, "got unexpected keyword argument(s): %r" % kwargs
      assert isinstance( self.__valueType, type ), ( "_TypedList doesn't have a"
                                                     " valueType: %r, %r" %
                                                     ( args, kwargs ) )
      if args:
         assert len( args ) == 1, "too many arguments: %r" % ( args, )
         for item in args[ 0 ]:
            self.append( item )

   @staticmethod
   def typeCheck( valueType, value ):
      _TacTypeCheck( valueType, value )

   def __delitem__( self, index ):
      del self.__underlying[ index ]

   def __setitem__( self, index, value ):
      _TypedList.typeCheck( self.__valueType, value )
      self.__underlying[ index ] = value

   def __getitem__( self, index ):
      return self.__underlying[ index ]

   def __len__( self ):
      return len( self.__underlying )

   def insert( self, index, value ):
      _TypedList.typeCheck( self.__valueType, value )
      self.__underlying.insert( index, value )

   def __eq__( self, other ):
      if isinstance( other, _TypedList ):
         other = other.__underlying  # pylint: disable-msg=W0212
      return self.__underlying == other

   def __ne__( self, other ):
      return not self == other

   def __repr__( self ):
      return repr( self.__underlying )

class List( _Collection ):
   """Equivalent to a Python list."""

   builtinType = list
   realType = _TypedList

   def check( self, value ):
      cls = type( value )
      if cls is list:
         # pylint: disable-msg=E1102
         value = _TypedList( value, __valueType=self.valueType )
      elif cls is not _TypedList:
         return False, False, value
      return True, True, value

   def defaultGenerator( self, default ):
      """Returns a lambda that can be used to generate a default value."""
      if default is None:
         default = ()
      return lambda: _TypedList( default, __valueType=self.valueType )

raiseAlreadyUsedGeneratorError = True

class _TypedGeneratorList( collections.Iterator ):
   """A dynamically generated list that does type checking of its
   values"""

   def __init__( self, *args, **kwargs ):
      collections.Iterator.__init__( self )
      self.__valueType = kwargs.pop( "__valueType", None )
      assert not kwargs, "got unexpected keyword argument(s): %r" % kwargs
      assert isinstance( self.__valueType, type ), (
         "_TypedGeneratorList doesn't have a valueType: %r, %r" %
         ( args, kwargs ) )
      assert len( args ) == 1, "Must specify an underlying iterator: %r" % ( args, )
      self.__underlying = args[ 0 ]
      self.__hasBeenUsed = False # True if this colletions has been iterated over.

   def __iter__( self ):
      """ Returns an iterator over the underlying collection elements,
      if the collection has not been accessed before. If this
      collection has already been iterated over, raises an
      AlreadyUsedGeneratorError. """
      if self.__hasBeenUsed:
         if raiseAlreadyUsedGeneratorError:
            raise AlreadyUsedGeneratorError()
      self.__hasBeenUsed = True
      return self

   def next( self ):
      val = self.__underlying.next()
      _TypedList.typeCheck( self.__valueType, val )
      return val

class GeneratorList( _Collection ):
   """Equivalent to a Python list, however the value items are
   generated just-in-time"""

   builtinType = list
   realType = _TypedGeneratorList

   def __init__( self, *args, **kwargs ):
      # We need to call the parent ctor only here, once self.keyType is set.
      if "default" in kwargs:
         raise TypeError( "GeneratorLists do not support default values" )
      super( GeneratorList, self ).__init__( **kwargs )

   def check( self, value ):
      if isinstance( value, collections.Iterator ):
         value = _TypedGeneratorList( value, __valueType=self.valueType )
         return True, True, value
      return False, False, value

   def defaultGenerator( self, default ):
      """Returns a lambda that can be used to generate a default value."""
      if default is None:
         default = iter( () )
      return lambda: _TypedGeneratorList( default, __valueType=self.valueType )

class _TypedDict( collections.MutableMapping ):
   """A dictionary that does type checking of keys and values."""

   def __init__( self, *args, **kwargs ):
      collections.MutableMapping.__init__( self )
      self.__underlying = {}
      self.__valueType = None
      self.__keyType = None
      self.__valueOptional = False
      if args:
         self.__keyType = args[ 0 ].get( "__keyType", self.__keyType )
         self.__valueType = args[ 0 ].get( "__valueType", self.__valueType )
         self.__valueOptional = args[ 0 ].get( "__valueOptional",
                                               self.__valueOptional )

      self.__keyType = kwargs.get( "__keyType", self.__keyType )
      self.__valueType = kwargs.get( "__valueType", self.__valueType )
      self.__valueOptional = kwargs.get( "__valueOptional", self.__valueOptional )
      if "__order" in kwargs:
         # Use an ordered Dict
         self.__underlying = collections.OrderedDict()
         del kwargs[ "__order" ]
         constructorArgs = collections.OrderedDict( *args, **kwargs )
      else:
         constructorArgs = dict( *args, **kwargs )

      assert isinstance( self.__keyType, type ), ( "_TypedDict doesn't have a"
                                                   " keyType: %r, %r" %
                                                   ( args, kwargs ) )
      assert isinstance( self.__valueType, type ), ( "_TypedDict doesn't have a"
                                                     " valueType: %r, %r" %
                                                     ( args, kwargs ) )
      del constructorArgs[ "__valueType" ]
      del constructorArgs[ "__keyType" ]
      if "__valueOptional" in constructorArgs:
         del constructorArgs[ "__valueOptional" ]
      self.update( constructorArgs )

   def __getitem__( self, key ):
      try:
         key = _TacTypeCheck( self.__keyType, key, variableType='key' )
      except ( ValueError, TypeError ):
         pass
      return self.__underlying[ key ]

   @staticmethod
   def typeCheck( keyType, key, valueType, value, valueOptional ):
      """ Checks that the provided key and value match the specified
      types. In some cases, this method passes only in the case of a
      transformed key (i.e. _TacAttributeType). Thus, this method
      always returns the key as it should be used in the dict."""
      newKey = _TacTypeCheck( keyType, key, variableType='key' )
      _TacTypeCheck( valueType, value, valueOptional=valueOptional )
      return newKey

   def __setitem__( self, key, value ):
      newKey = _TypedDict.typeCheck( self.__keyType, key, self.__valueType, value,
                                     self.__valueOptional )
      self.__underlying[ newKey ] = value

   def __delitem__( self, key ):
      try:
         key = _TacTypeCheck( self.__keyType, key, variableType='key' )
      except ( ValueError, TypeError ):
         pass
      del self.__underlying[ key ]

   def __iter__( self ):
      return iter( self.__underlying )

   def __len__( self ):
      return len( self.__underlying )

   def __repr__( self ):
      return repr( self.__underlying )

class _TypedOrderedDict( _TypedDict ):
   """ Dictionaries where the output is printed in the order the entries
       were added. """
   def __init__( self, *args, **kwargs ):
      kwargs[ '__order' ] = True
      super( _TypedOrderedDict, self ).__init__( *args, **kwargs )

class _DictBase( _Collection ):
   def __init__( self, keyType, validJsonObjectKeyTypes, builtinType, realType,
                 **kwargs ):
      if keyType in validJsonObjectKeyTypes or issubclass( keyType,
         _TacAttributeType ):
         self.keyType = keyType
         self.builtinType = builtinType
         self.realType = realType
         if "valueOptional" in kwargs:
            # ValueOptional just makes sense for dictionaries so we remove it from
            # processing in the Collections class
            self.valueOptional = kwargs[ "valueOptional" ]
            del kwargs[ "valueOptional" ]
         else:
            self.valueOptional = False

         # We need to call the parent ctor only here, once self.keyType is set.
         super( _DictBase, self ).__init__( **kwargs )
      else:
         raise TypeError( "Invalid key type %r. Must be one of %r"
            % ( keyType, validJsonObjectKeyTypes ) )

   def check( self, value ):
      cls = type( value )
      if cls is self.builtinType:
         # integer/long keys can be provided as strings, so convert the dict as
         # needed so that json -> model works (used in revision tests, see
         # unmarshalModel) (json keys are strings)
         if ( self.keyType in ( int, long ) and value and
               type( value.iterkeys().next() ) == str ):
            newValue = {}
            for k, v in value.iteritems():
               try:
                  newValue[ self.keyType( k ) ] = v
               except ValueError:
                  newValue[ k ] = v
            value = newValue
         # pylint: disable-msg=E1102
         value = self.realType( value, __keyType=self.keyType,
                                __valueType=self.valueType,
                                __valueOptional=self.valueOptional )
      elif cls is not self.realType:
         return False, False, value
      return True, True, value

   def defaultGenerator( self, default ):
      """Returns a lambda that can be used to generate a default value."""
      if default is None:
         default = self.builtinType()
      return lambda: self.realType( default,
                                 __keyType=self.keyType,
                                 __valueType=self.valueType,
                                 __valueOptional=self.valueOptional )

class Dict( _DictBase ):
   """ Equivalent to a Python dict, except the keys must be of
   type( str ) unless otherwise specified via the keyType parameter """

   builtinType = dict
   realType = _TypedDict

   validJsonObjectKeyTypes = frozenset( [ str, int, long ] )

   def __init__( self, keyType=str, **kwargs ):
      # We need to call the parent ctor only here, once self.keyType is set.
      super( Dict, self ).__init__( keyType,
                           Dict.validJsonObjectKeyTypes,
                           Dict.builtinType,
                           Dict.realType, **kwargs )

class OrderedDict( _DictBase ):
   """ Equivalent to a Python OrderedDict, except the keys must be of
   type( str ) unless otherwise specified via the keyType parameter.
   Please don't use this unless you have received permission from capi-dev,
   it probably doesn't do what you want it to do."""

   builtinType = collections.OrderedDict
   realType = _TypedOrderedDict

   validJsonObjectKeyTypes = frozenset( [ str, int, long ] )

   def __init__( self, keyType=str, **kwargs ):
      super( OrderedDict, self ).__init__( keyType,
                     OrderedDict.validJsonObjectKeyTypes,
                     OrderedDict.builtinType,
                     OrderedDict.realType, **kwargs )

class _TypedGeneratorDict( collections.Iterator ):
   """A generated dictionary that does type checking of keys and
   values."""

   def __init__( self, *args, **kwargs ):
      collections.Iterator.__init__( self )
      self.__keyType = kwargs.get( "__keyType", None )
      self.__valueType = kwargs.get( "__valueType", None )
      assert isinstance( self.__keyType, type ), ( "_TypedDict doesn't have a"
                                                   " keyType: %r, %r" %
                                                   ( args, kwargs ) )
      assert isinstance( self.__valueType, type ), ( "_TypedDict doesn't have a"
                                                     " valueType: %r, %r" %
                                                     ( args, kwargs ) )

      assert len( args ) == 1, "Must specify an underlying iterator: %r" % ( args, )
      self.__underlying = args[ 0 ]
      self.__hasBeenUsed = False # True if this colletions has been iterated over.
      self.__valueOptional = False
      self.__valueOptional = kwargs.get( "__valueOptional", self.__valueOptional )

   @staticmethod
   def _typeCheck( keyType, valueType, item, valueOptional ):
      if type( item ) is not tuple:
         raise TypeError( "Invalid iterator return type (must be tuple): %r"
                          % ( item, ) )
      if len( item ) != 2:
         raise TypeError( "Invalid iterator return type, tuple only contain a "
                          "key and value: %r" % ( item, ) )
      key, value = item
      newKey = _TypedDict.typeCheck( keyType, key, valueType, value, valueOptional )
      return newKey

   def __iter__( self ):
      """ Returns an iterator over the underlying collection elements,
      if the collection has not been accessed before. If this
      collection has already been iterated over, raises an
      AlreadyUsedGeneratorError. """
      if self.__hasBeenUsed:
         if raiseAlreadyUsedGeneratorError:
            raise AlreadyUsedGeneratorError()
      self.__hasBeenUsed = True
      return self

   def next( self ):
      """ Returns a key-value pair representing a single instance of a
      dictionary """
      item = self.__underlying.next()
      key = _TypedGeneratorDict._typeCheck( self.__keyType, self.__valueType, item,
             self.__valueOptional )
      return key, item[ 1 ] # return the updated key and the original value.

class GeneratorDict( _Collection ):
   """ Equivalent to a Python dict, except elements are generated
   just-in-time and the keys must be of type( str ) unless otherwise
   specified via the keyType parameter"""

   builtinType = dict
   realType = _TypedGeneratorDict

   _validJsonObjectKeyTypes = Dict.validJsonObjectKeyTypes

   def __init__( self, keyType=str, **kwargs ):
      assert ( keyType in Dict.validJsonObjectKeyTypes
               or issubclass( keyType, _TacAttributeType ) ), (
            "Invalid key type %r. Must be one of %r"
            % ( keyType, Dict.validJsonObjectKeyTypes ) )
      if "default" in kwargs:
         raise TypeError( "GeneratorDicts do not support default values" )
      self.keyType = keyType
      if "valueOptional" in kwargs:
         # ValueOptional just makes sense for dictionaries so we remove it from
         # processing in the Collections class
         self.valueOptional = kwargs[ "valueOptional" ]
         del kwargs[ "valueOptional" ]
      else:
         self.valueOptional = False
      # We need to call the parent ctor only here, once self.keyType is set.
      super( GeneratorDict, self ).__init__( **kwargs )

   def check( self, value ):
      if isinstance( value, collections.Iterator ):
         value = _TypedGeneratorDict( value, __keyType=self.keyType,
                                      __valueType=self.valueType,
                                      __valueOptional=self.valueOptional )
         return True, True, value
      return False, False, value

   def defaultGenerator( self, default ):
      """Returns a lambda that can be used to generate a default value."""
      if default is None:
         default = iter( () )
      return lambda: _TypedGeneratorDict( default,
                                          __keyType=self.keyType,
                                          __valueType=self.valueType,
                                          __valueOptional=self.valueOptional )

class Submodel( _AttributeType ):
   """An object that is a subclass of Model."""
   def __init__( self, valueType=None, **kwargs ):
      self.realType = valueType
      # be consistent with other types, plus we want to serialize (for the reference
      # models) an attribute called 'valueType', since required by the ctor (called
      # when re-creating the model from the reference to validate degraded instances)
      self.valueType = valueType
      super( Submodel, self ).__init__( **kwargs )
      assert issubclass( valueType, Model ), ( "Invalid object type: %r, must be"
                                               " a subclass of CliModel.Model"
                                               % ( valueType, ) )
      assert valueType != Model, "type must be a subclass of CliModel.Model"

   def check( self, value ):
      """
      Overrides check function from parent class to allow inheritance in
      submodels.
      """
      ok = isinstance( value, self.realType )
      return ok, ok, value

class _SubmodelCallbackFunc( object ):
   """A wrapper for a function and the Submodel type it will evaluate to."""
   def __init__( self, func, valueType ):
      self.func = func
      self.called = False
      self.realType = valueType
      self.value = None

   def evaluate( self ):
      assert not self.called, ( 'GeneratorSubmodel function %s has already been '
                                'called. The result has been stored in '
                                '`instance.value`' % self.func )
      self.called = True

      value = self.func()
      valueType = type( value )
      if not isinstance( value, self.realType ):
         raise TypeError( "Callback function returned wrong Submodel type. Expected "
                          "%s, but got % s." % ( self.realType, valueType ) )
      self.value = value
      return value

   def toDict( self, revision=None, includePrivateAttrs=False, streaming=False ):
      value = self.value if self.called else self.evaluate()
      return value.toDict( revision, includePrivateAttrs, streaming )

class GeneratorSubmodel( Submodel ):
   """
   An object that is a subclass of Model, except that its attributes are not
   populated until serialization.
   """
   def check( self, value ):
      ok = callable( value )
      newValue = _SubmodelCallbackFunc( value, self.valueType )
      return ok, ok, newValue

# Model attributes must be one of the following types.
_allowedAttributeTypes = frozenset( [ Int, Float, Str, Bool, Enum,
                                      List, Dict, Submodel, GeneratorSubmodel,
                                      GeneratorList, GeneratorDict, OrderedDict ] )

class ModelMetaClass( type ):
   """Meta-class class for Cli API objects.  Do not use directly, HBD."""

   _passThroughAttributes = frozenset( [ "__module__", "__doc__", "__revision__",
                                         "__revisionMap__", "__public__",
                                         "__streamable__" ] )
   _reservedAttributes = frozenset( [ "errors", "warnings", "_meta" ] )

   # See pylint bug #4014.
   # pylint: disable-msg=C0203
   def __new__( cls, name, bases, fields ):
      create = super( ModelMetaClass, cls ).__new__
      if name == "Model":
         # This is the base class from which actual API objects inherit.
         # Don't fiddle with it, create it as-is.
         return create( cls, name, bases, fields )
      # Make sure that we don't have multiple inheritance as we don't support it.
      assert len( bases ) == 1, "Class %s must not use multiple inheritance" % name
      assert issubclass( bases[ 0 ], Model ), (
            "Class %s must inherit from Model" % name )

      attrs = dict( attr for attr in bases[ 0 ].__attributes__.iteritems() )
      newFields = {}
      for attr, desc in fields.iteritems():
         assert attr not in ModelMetaClass._reservedAttributes, (
            "Class %r is not allowed to name an attribute %r" % ( name, attr ) )
         if attr.startswith( "__" ):
            assert attr in ModelMetaClass._passThroughAttributes, (
                  "Class %r is not allowed to define %r" % ( name, attr ) )
            newFields[ attr ] = desc  # Carry these over as-is.
            continue
         elif ( callable( desc ) or
                # See https://bugs.python.org/issue20309
                isinstance( desc, ( staticmethod, classmethod ) ) ):
            # Allow methods.
            newFields[ attr ] = desc
            continue
         assert ( type( desc ) in _allowedAttributeTypes
                  or isinstance( desc, _TacAttributeType ) ), (
               "Invalid attribute %r in class %r: type %s not allowed"
               % ( attr, name, type( desc ) ) )

         assert attr not in attrs, ( "Cannot override attribute %r in class %r,"
                                     " attribute already defined in %r as %r"
                                     % ( attr, name, bases[ 0 ], attrs[ attr ] ) )
         helpText = desc.help.strip()
         if helpText[ -1 ] not in ".?!;":
            helpText += "."
         desc.help = helpText
         attrs[ attr ] = desc
         newFields[ attr ] = desc
      newFields.setdefault( "__revision__", 1 )
      newFields.setdefault( "__revisionMap__", [ ( 1, 1 ) ] )
      newFields.setdefault( "__public__", True ) # models are public by default
      newFields.setdefault( "__streamable__", False )
      newFields[ "__attributes__" ] = attrs
      newFields[ "__defaults__" ] = { attr: ModelMetaClass._defaultOf( desc )
                                      for attr, desc in attrs.iteritems() }
      newFields[ "__doc__" ] = ( fields.get( "__doc__", "" ) + "\n"
                                 + ModelMetaClass._genHelp( attrs ) )
      ModelMetaClass._validateRevisionMap( newFields[ "__revisionMap__" ] )
      return create( cls, name, bases, newFields )

   @staticmethod
   def _validateRevisionMap( revMap ):
      """ Validates the revision map is properly specified. See
      CliModel.Model's docstring for details. """
      assert revMap, "__revisionMap__ must be an array with at least one tuple"
      curGlobalVersion, curRevision = 0, 0
      for globalVer, rev in revMap:
         assert globalVer > curGlobalVersion, (
            "Tuple (%r, %r) must have a later global version than the previous entry"
            " (at global version %r)" % ( globalVer, rev, curGlobalVersion ) )
         assert rev > curRevision, (
            "Tuple (%r, %r) must have a later revision than the previous entry"
            " (at revision %r)" % ( globalVer, rev, curRevision ) )
         curGlobalVersion = globalVer
         curRevision = rev

   @staticmethod
   def _genHelp( attrs ):
      """Generates a useful docstring for the class based on its attributes."""
      def prettyType( desc ):
         if isinstance( desc, _Collection ):
            return "%s of %s" % ( desc.builtinType.__name__,
                                  prettyType( desc.valueType ) )
         elif isinstance( desc, _AttributeType ):
            if isinstance( desc.realType, tuple ):
               return " or ".join( type.__name__ for type in desc.realType )
            return desc.realType.__name__
         elif isinstance( desc, tuple ):
            desc = desc[ 0 ]
         assert isinstance( desc, type ), "Unexpected type: %r" % ( desc, )
         return desc.__name__

      def mkHelp( attr, desc ):
         return "  - %s (%s): %s" % ( attr, prettyType( desc ), desc.help )

      return ( "Attributes defined here:\n%s"
               % "\n".join( mkHelp( attr, d ) for attr, d in attrs.iteritems() ) )

   @staticmethod
   def _defaultOf( attr ):
      """Returns a lambda that creates the default value of the given attribute."""
      if isinstance( attr, _Collection ):
         return attr.defaultGenerator( attr.default )
      return lambda: attr.default

class Model( object ):
   """Base class for Cli API objects.

   Class level attributes which can be set by Model definitions:
   __public__: bool, defaults to True
      - If set to False, this Model will not be documented and
        revisions will not be tracked. Use sparingly and consult
        capi-dev before setting a Model to be private.

   __revision__: number, defaults to 1
      - The current revision of the Model. Only increment this
        variable if an incompatible change to the Model definition has
        been made.

   __revisionMap__: list of tuples, defaults to [ (1, 1) ]
      - A data structure which maps from a global version number to
        the corresponding revision of this particular model. The
        format of this structure is a list of tuples, with the first
        element of the tuple indicating the global version and the
        second item indicating the revision that should be used for
        that revision. The list should have the oldest (smallest)
        global version first with each subsequent tuple having a
        larger global version. For example:
           [ (1, 1), (2, 3), (4, 8) ]
        indicates that at global version 1 we should use revision 1,
        at global version 2 we should use model revision 3, and at
        global version 4 we should use revision 8. Because this map
        doesn't define anything for global version 3, we'll fall back
        to using the revision at global version 2, which in this case
        is revision 3. For as-of-yet undefined global versions, we
        return the last revision specified by the map (in this
        example, global version 5 would yield revision 8.

   __streamable__: bool, defaults to False
      - If set to True, any attributes of type GeneratorList or GeneratorDict on this
        Model will be serialized to JSON using a Python generator (not in
        Model#toDict) *when writing JSON to an interactive CLI*. Thus, consuming
        those generators must *not* generate any messages, warnings, or errors, as
        these will not be included in the CAPI response. Setting this attribute True
        is useful for high-scale show commands which do not generate errors.
   """
   __metaclass__ = ModelMetaClass
   __attributes__ = {}

   def __init__( self, **attrs ):
      """Constructs a new instance.

      Args:
         attrs: If passed, with be used to assign values to the attributes of
         this instance.  For instance:
            m = MyModel( foo=42 )
         is short for:
            m = MyModel()
            m.foo = 42
         Other attributes are assigned a default value based on their types.
      """
      dikt = self.__dict__  # Save a local copy as an optimization.
      dikt[ "__printed" ] = False  # Can't access self.__printed directly
      for attr, default in self.__defaults__.iteritems():
         dikt[ attr ] = default()
      # Assign the values passed to the constructor.
      setAttr = self.__setattr__  # Save a local copy as an optimization.
      for attr, value in attrs.iteritems():
         setAttr( attr, value )

   def __getattr__( self, name ):
      """Returns the attribute of the given name."""
      if name not in self.__attributes__:
         raise AttributeError( "No %r in %s" % ( name, type( self ) ) )
      return self.__dict__[ name ]

   def __getitem__( self, name ):
      """Returns the attribute of the given name (dict-like interface)."""
      if name not in self.__attributes__:
         raise KeyError( "No %r in %s" % ( name, type( self ) ) )
      return self.__dict__[ name ]

   def __setattr__( self, name, value ):
      """Sets the given attribute to its a new value.

      Raises:
         TypeError if the type of `value' doesn't match the expected type of
         the attribute being changed.
      """
      desc = self.__attributes__.get( name )
      if desc is None:
         raise AttributeError( "No %r in %s" % ( name, type( self ) ) )
      if value is None:
         typeOk = valueOk = True
         newValue = None
      else:
         typeOk, valueOk, newValue = desc.check( value )
      if not typeOk:
         expectedType = FunctionType if type( desc ) is GeneratorSubmodel \
                        else desc.realType
         raise TypeError( "Cannot set %r to argument of type %s, expected a"
                          " %s instead, with value %r" % ( name, type( value ),
                                                           expectedType, value ) )
      if not valueOk:
         raise ValueError( "Invalid value for %r: %r" % ( name, value ) )
      self.__dict__[ name ] = newValue

   def __delattr__( self, name ):
      """Resets the given attribute to its default value."""
      default = self.__defaults__.get( name )
      if default is None:
         raise AttributeError( "No %r in %s" % ( name, type( self ) ) )
      self.__dict__[ name ] = default()

   def __repr__( self ):
      return "%s(%s)" % ( self.__class__.__name__,
                          ", ".join( "%s=%r" % ( attr, getattr( self, attr ) )
                                     for attr in self.__attributes__ ) )

   def currentRevision( self ):
      """ Returns current revision number this model is at """
      return self.__revision__

   def __eq__( self, other ):
      if isinstance( other, self.__class__ ):
         return self.__dict__ == other.__dict__
      else:
         return False

   def __ne__( self, other ):
      return not self == other

   def checkOptionalAttributesHelper( self, warnings, stack=None ):
      """ implementation of checkOptionalAttributes. Adds extra args for keeping
      track of nested model names during recursion in order to be able to
      create fully qualified names in the warnings for the missing attributes.
      """
      stack = stack or []
      attributes = self.__attributes__
      for a in attributes:
         stack.append( a )
         v = getattr( self, a )
         cls = type( v )
         if cls is _TypedDict:
            if issubclass( attributes[ a ].valueType, Model ):
               for val in v.itervalues():
                  if not val:
                     if not attributes[ a ].valueOptional:
                        fqn = ".".join( stack )
                        warnMsg = CliCommon.SHOW_OUTPUT_MISSING_ATTRIBUTE_WARNING
                        warnings.add( "%s %s" % ( warnMsg, fqn ) )
                  else:
                     val.checkOptionalAttributesHelper( warnings, stack )
         elif cls is _TypedList:
            if issubclass( attributes[ a ].valueType, Model ):
               for val in v:
                  val.checkOptionalAttributesHelper( warnings, stack )
         elif isinstance( v, Model ):
            v.checkOptionalAttributesHelper( warnings, stack )
         elif v is None and not attributes[ a ].optional and a[ 0 ] != "_":
            fqn = ".".join( stack )
            warnMsg = CliCommon.SHOW_OUTPUT_MISSING_ATTRIBUTE_WARNING
            warnings.add( "%s %s" % ( warnMsg, fqn ) )
         stack.pop()

   def checkOptionalAttributes( self ):
      """ Validates that all non-optional attributes are set in
      non-dynamically generated attributes. If we discover a mandatory
      attribute with a value of None, we add a warning. Test clients
      will convert such warning into an assert. """
      warnings = set()
      self.checkOptionalAttributesHelper( warnings )
      return warnings

   def degrade( self, dictRepr, revision ):
      """ This function should be overridden in subclasses to provide
      a previous revision of the model. Two arguments are provided:
      - dictRepr: a representation of the model as a dict. This simple
        representation contains attributes as keys and values which may
        be a str, bool, int, float, long, list, dict, or NoneType.
      - revision: the requested revision of the model, which may be
        this revision or a previous one.

      For example:
         class ExampleModel( Model ):
            __revision__ = 2
            myInt = Int( help="In revision 1 this was a complex string" )

            def degrade( self, dictRepr, revision ):
               if revision < 2:
                  # Old model was some complex string
                  dictRepr[ "myInt" ] = "This is %d units" % dictRepr[ "myInt" ]
               return dictRepr

      Further examples can be seen in Cli/test/ModelTest.py.
      By default this function does nothing.
      """
      return dictRepr

   def toDict( self, revision=None, includePrivateAttrs=False,
               streaming=False ):
      """ Emits a sequence of (key, value) pairs for this object's
      attributes.
      - Accepts an optional numeric 'revision' parameter, which if
        specified, determines which revision of this model should be
        returned. For models wishing to take advantage of versioning,
        they should override the degrade() method locally.  If an
        unknown revision is passed in (i.e. revision 8 when the model
        is only at revision 3), the model will just return the most
        up-to-date version of itself.
      - If 'includePrivateAttrs' is True, this method will include
        private variables (vars starting with a '_') in the dictionary
        output. Otherwise, those attributes will be excluded.
      - If 'streaming' is True, the resulting dictionary is being streamed directly
        to a file descriptor. This implies this method should try to stream
        GeneratorDict and GeneratorList types if the 'self' is __streamable__.
      """
      def convert( v ):
         if isinstance( v, Model ):
            return v.toDict( revision, includePrivateAttrs,
                             streaming=streaming )
         elif Tac.isValue( v ):
            v = v.stringValue
         return v

      class StreamIterable():
         """
         Base class for a streaming JSON object which is parsable by an
         unmodified simplejson.JSONEncoder() to stream GeneratorDict and
         GeneratorList model types. The class implements the basic Python iterable
         requirements (defining __iter__), but must also implement a dummy __len__ as
         JSONEncoder will not attempt to serialize an iterable if the length is 0.

         As calling len() on a generator exhausts it (defeating the purpose of
         streaming), we implement a dummy method which returns 1 if the generator has
         at least 1 element and 0 otherwise. This requires storing the head of the
         generator and reforming the whole generator once more (self.iterator) via
         itertools.chain().

         Finally, we redefine 'next' to call 'convert' on each item. Derived classes
         must implement 'convert'.

         @param iterable - any Python iterable (implements '__iter__'), such as a
         list, dict, or generator
         """
         def __init__( self, iterable ):
            self.iterable = iterable
            self.length = None
            self.iterator = None

         def _setIterator( self ):
            head = []
            tail = []
            try:
               # Note: We must not call 'next' in  __init__, as the generator may not
               # be ready to be read until __iter__ (or __len__) is called.
               head.append( next( self.iterable ) )
               tail.append( self.iterable )
               self.length = 1
            except StopIteration:
               self.length = 0
            # Chain on the first element of 'tail', or [] if tail is the empty list
            self.iterator = itertools.chain( head, self.iterable )

         def __iter__( self ):
            if self.iterator is None:
               self._setIterator()
            return self

         def __len__( self ):
            if self.iterator is None:
               # It may seem odd to set iterator in __len__, but the JSON parser may
               # check the iterator prior to calling iter(), as in:
               #
               # if not myGenerator:
               #    yield {}
               #
               # This calls __len__, and thus forces us to decide if the generator
               # has a length of at least 1, or if it's empty.
               self._setIterator()
            assert self.length is not None
            return self.length

         def next( self ):
            return self.convert( self.iterator.next() )

         def convert( self, obj ):
            raise NotImplementedError

      # Both StreamList and StreamDict derive from 'list' and 'dict', respectively,
      # to appease simplejson.JSONEncoder(). Without any additional work, the encode
      # methods will automatically recognize these two types and try to iterate (and
      # encode) them (interpreting them as 'list' and 'dict', respectively).
      class StreamList( StreamIterable, list ):
         def convert( self, obj ):
            return convert( obj )

      class StreamDict( StreamIterable, dict ):
         def convert( self, obj ):
            ( key, value ) = obj
            return ( str( convert( key ) ), convert( value ) )

         def iteritems( self ):
            return self

      attributes = self.__attributes__

      def get( a ):
         v = getattr( self, a )
         cls = type( v )
         if cls is _TypedDict:
            v = { str( convert( k ) ): convert( x ) for k, x in v.iteritems() }
         elif cls is _TypedOrderedDict:
            v = collections.OrderedDict( [ ( str( convert( k ) ), convert( x )
                                           ) for k, x in v.iteritems() ] )
         elif cls is _TypedGeneratorDict:
            if self.__streamable__ and streaming:
               v = StreamDict( v )
            else:
               v = { str( convert( k ) ): convert( x ) for k, x in v }
         elif cls is _TypedList:
            v = [ convert( x ) for x in v ]
         elif cls is _TypedGeneratorList:
            if self.__streamable__ and streaming:
               v = StreamList( v )
            else:
               v = [ convert( x ) for x in v ]
         elif isinstance( v, Model ) or cls is _SubmodelCallbackFunc:
            v = v.toDict( revision, includePrivateAttrs, streaming=streaming )
         elif Tac.isValue( v ):
            v = v.stringValue

         return v

      ret = {}
      for a, _ in sorted( attributes.items(), key=lambda x: x[ 1 ].order ):
         if not includePrivateAttrs and isPrivateAttr( a ):
            continue
         # this is an "auto-degrade" case
         if revision and revision < attributes[ a ].sinceRevision:
            continue
         v = get( a )
         if v is None and attributes[ a ].optional:
            # Don't include optional attributes in the output if the value is None
            continue
         ret[ a ] = v
      if revision < self.__revision__:
         ret = self.degrade( ret, revision )
         assert isinstance( ret, dict ), (
            "No dict returned when degrading to revision %d" % revision )
      return ret

   def _renderOnce( self ):
      """Calls render() but only once."""
      if self.__dict__[ "__printed" ]:  # Can't access self.__printed directly
         return
      self.__dict__[ "__printed" ] = True
      self.render()

   def render( self ):
      """Transforms this object into a human-readable output.

      Subclasses have to implement this method to transform this object into
      a human readable output and print it to stdout.
      """
      return None  # was 'raise NotImplementedError()', which makes pylint unhappy.

   def setAttrsFromDict( self, data ):
      """Convenience function that many models utilize."""
      for key, value in data.items():
         setattr( self, key, value )

   @classmethod
   def getRevision( cls, requestedGlobalVersion=None, requestedRevision=None ):
      """ Accepts a numeric requestedGlobalVersion and numeric
      requestedRevision and returns a revision of this model based on
      the following criteria:

      - If neither requestedGlobalVersion nor requestedRevision is
      set, this returns the current revision of the model.

      - If just a requestedRevision is supplied, we return the
        requested reversion unless this model has not yet reached that
        requested revision. Otherwise we return the current revision.

      - If just the requestedGlobalVersion is set, this accesses the
      __revisionMap__ to calculate the appropriate revision it should
      revert to.

      - If both are set, this only pays attention to the
      requestedRevision attribute.

      At this time there is no support for deprecated Model versions.
      """
      # pylint: disable-msg=E1101
      if requestedRevision is not None:
         if requestedRevision == 0:
            return cls.__revision__
         if requestedRevision < 1:
            raise InvalidRevisionRequestError(
               cls, "revision 1 is the earliest possible revision" )
         if requestedRevision > cls.__revision__:
            raise InvalidRevisionRequestError(
               cls, "no such revision %d, most current is %d" %
               ( requestedRevision, cls.__revision__ ) )
         return requestedRevision

      if requestedGlobalVersion == CliCommon.JSONRPC_VERSION_LATEST:
         return cls.__revision__

      if requestedGlobalVersion:
         if requestedGlobalVersion < cls.__revisionMap__[ 0 ][ 0 ]:
            raise InvalidRevisionRequestError(
               cls, "invalid version %d, the earliest supported "
               "global version is %d" % ( requestedGlobalVersion,
                                          cls.__revisionMap__[ 0 ][ 0 ] ) )
         lastAcceptableRevision = None
         for globalVer, revision in cls.__revisionMap__:
            if requestedGlobalVersion >= globalVer:
               lastAcceptableRevision = revision
            else:
               break
         if lastAcceptableRevision:
            return lastAcceptableRevision

      return cls.__revision__

   def __iter__( self ):
      return iter( self.__attributes__ )

def isPrivateAttr( attrName ):
   """ Given the name of an attribute, return True if the provided
   attribute is private and should not be included in the returned
   dictionary """
   return attrName[ 0 ] == '_'

class UncheckedModel( Model ):
   """ All show commands have a corresponding model associated with them. The Cli
   enforces that the returned model correctly corresponds to registered model.
   However sometimes we are unable to do this (ie. IntfCli) and this check is
   done at a lower level. The Cli will not do its typecheck if it is a subclass of
   this model. """

class DeferredModel( Model ):
   """ All capi-enabled show commands have a model associated with them.
   If that model is not instantiated (thus buffered) before being rendered,
   then it must inherit from this one. This is used for scaled show commmands
   where the memory cost and cpu overhead would otherwise be too taxing.
   Used when printing in c code (generator models are not of this class).
   Maybe should call it NoModelModel, or AlreadySerializedModel.
   """

class BadRequest( CliCommon.ApiError ):
   """The Cli command was didn't make sense or had an invalid argument."""
   ERR_CODE = 400  # Forbidden.

class PermissionDenied( CliCommon.ApiError ):
   """The Cli command couldn't be executed due to insufficient privileges."""
   ERR_CODE = 401  # Unauthorized.

class UnknownEntityError( CliCommon.ApiError ):
   """The Cli command attempted to use an entity (e.g. interface) that does
   not exist."""
   ERR_CODE = 404  # Not Found.

class UnsetModelAttributeError( CliCommon.ApiError ):
   """The Cli command was didn't make sense or had an invalid argument."""
   ERR_CODE = 500  # Internal Server Error.

   def __init__( self, modelName, attributeName ):
      msg = "Non-optional attribute of %s not set: %r" % ( modelName, attributeName )
      CliCommon.ApiError.__init__( self, msg )

class ModelMismatchError( CliCommon.ApiError ):
   """ The Cli command return a model that differed from what was registered """
   ERR_CODE = 500   # Internal Server Error.

   def __init__( self, model, registeredModel ):
      msg = ( "Seen model %s did not match the registered model %s" %
              ( type( model ), registeredModel ) )
      CliCommon.ApiError.__init__( self, msg )

class CliModelError( Exception ):
   """ Module level exception for non-ApiError exceptions to subclass """

class AlreadyUsedGeneratorError( CliModelError ):
   """ Exception to protect against using GeneratorLists and
   GeneratorDicts multiple times. """
   def __init__( self ):
      CliModelError.__init__( self, "Iterator has already been used" )

class InvalidRevisionRequestError( CliModelError ):
   def __init__( self, modelInstance, msg ):
      CliModelError.__init__( self, "Invalid requested version for %s: %s" %
                              ( modelInstance.__class__.__name__, msg ) )

class ModelTranslationError( CliModelError ):
   """ Exception while attempting to turn a dictionary into a given
   Model. """

class ModelUnknownAttribute( CliModelError ):
   """ Exception while attempting to turn a dictionary into a given
   Model. """

class ModelMissingAttribute( CliModelError ):
   """ Exception while attempting to turn a dictionary into a given
   Model. """

def unmarshalModel( modelClass, dikt, degraded=True ):
   """ Utility method to turn a parsed json object into an instance of
   modelClass. Expects a subclass of Model in modelClass, and a
   dictionary representation in dokt, and returns an instance of
   modelClass. This function cannot deal with Model inheritance, and
   will likely fail in a strange way if you try."""
   t0( "Unmarshalling model: %s" % modelClass )
   assert issubclass( modelClass, Model ), "Invalid type: %r" % modelClass

   if type( dikt ) != dict:
      raise ModelTranslationError( "Invalid model representation (%s): %r " %
                                   ( type( dikt ).__name__, dikt ) )

   result = modelClass()

   # check for garbage attributes (normally checked during toDict)
   # if model instance was degraded, expect the unexpected: skip check
   if not degraded:
      for attrName, _ in dikt.iteritems():
         try:
            attrClass = modelClass.__attributes__[ attrName ]
         except KeyError as e:
            raise ModelUnknownAttribute( "'%s' not an attribute of %s" % (
                                            attrName, modelClass ) )

   attributes = modelClass.__attributes__
   for attrName, attrDesc in attributes.iteritems():
      if attrName not in dikt:
         if attrDesc.optional or attrName.startswith( '_' ):
            # explicitly set it to None, in case the attr has a default value
            setattr( result, attrName, None )
            continue
         else:
            raise ModelTranslationError( "Mandatory attribute '%s' missing " %
                                         attrName )
      attrVal = dikt[ attrName ]
      if isinstance( attrDesc, _Collection ):
         elemClass = attrDesc.valueType
         if issubclass( elemClass, Model ):
            # We're dealing with a collection of models, so we'll need
            # to unmarshal each of the sub-elements of this collection
            # before putting them in the modelClass.
            if isinstance( attrDesc, List ):
               setattr( result, attrName,
                        [ unmarshalModel( elemClass, e, degraded )
                           for e in attrVal ] )
            elif isinstance( attrDesc, Dict ):
               setattr( result, attrName,
                        { e: unmarshalModel( elemClass, attrVal[ e ], degraded )
                          for e in attrVal } )
            elif isinstance( attrDesc, GeneratorDict ):
               # TODO: skip the tee stuff: use iterators instead of generators
               def dictGen():
                  for e in attrVal:
                     v = unmarshalModel( elemClass, attrVal[ e ], degraded )
                     yield ( e, v )
               setattr( result, attrName, dictGen() )
            elif isinstance( attrDesc, GeneratorList ):
               def listGen():
                  for e in attrVal:
                     v = unmarshalModel( elemClass, attrVal[ e ], degraded )
                     yield v
               setattr( result, attrName, listGen() )
            else:
               assert False, "Unknown collection type for %s: %s" % ( attrName,
                                                                      attrClass )
         elif issubclass( elemClass, _Collection ):
            assert False, "Cannot unmarshal a collection of collections %s: %s" % (
               attrName, attrVal )
         else:
            # A collection of basic items. Just assign the entire
            # collection at one time, since there is no need to
            # convert the individual elements.
            setattr( result, attrName, attrVal )
      elif isinstance( attrDesc, Submodel ):
         submodel = unmarshalModel( attrDesc.realType, attrVal, degraded )
         setattr( result, attrName, submodel )

      else:
         setattr( result, attrName, attrVal )
   t0( "Reconstructed model", result )
   return result

# For cli handlers to tell cli infra that it chose to 'cliprint' the json, that is,
# the returned model instance is a dummy, the actual output was already delivered
# as json text straight to stdout (streamed, no intermediate python model instance).
def cliPrinted( ModelClass ):
   emptyInstance = ModelClass()
   emptyInstance.__dict__[ "__cliPrinted__" ] = True
   return emptyInstance

def noValidationModel( ModelClass ):
   emptyInstance = ModelClass()
   emptyInstance.__dict__[ "__noValidationModel__" ] = True
   return emptyInstance

def shouldValidateModel( result ):
   if ( result and hasattr( result, '__dict__' ) and
        result.__dict__.get( "__noValidationModel__" ) ):
      return False
   else:
      return True
