#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliMode
from CliMode.Classification import ( ClassificationConfigModeBase,
                                     FieldSetBasePrefixConfigMode )
import BasicCli
import CliParser
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'TrafficPolicies' )
t0 = __defaultTraceHandle__.trace0

FEATURE = 'traffic-policy'
class TrafficPoliciesModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return "traffic-policies"

   def __init__( self ):
      self.modeKey = "traffic"
      self.longModeKey = "traffic-policies"
      CliMode.ConfigMode.__init__( self, None,
                                   parentMode=CliMode.ConfigMode )

class TrafficPoliciesConfigMode( TrafficPoliciesModeBase, BasicCli.ConfigModeBase ):
   """ Configuration mode for traffic-policies """
   name = "traffic-policies"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      TrafficPoliciesModeBase.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def abort( self ):
      # need empty method to allow for code sharing with Classification field-sets
      pass

   def commit( self ):
      # need empty method to allow for code sharing with Classification field-sets
      pass

class TrafficPolicyModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return "traffic-policy %s" % self.trafficPolicyName

   def __init__( self, param ):
      self.feature, self.parentMode, self.trafficPolicyName = param
      self.modeKey = "traffic-policy"
      self.longModeKey = "%s-%s" % ( self.feature,
                                     self.trafficPolicyName )
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=self.parentMode )
      self.trafficPolicyContext = None

   def getContext( self ):
      return self.trafficPolicyContext

   def setContext( self, value ):
      self.trafficPolicyContext = value

class TrafficPolicyConfigModeBase( TrafficPolicyModeBase,
                                   ClassificationConfigModeBase ):
   def __init__( self, parent, session, context, feature ):
      TrafficPolicyModeBase.__init__( self, ( feature,
                                              type( parent ),
                                              context.pmapName() ) )
      ClassificationConfigModeBase.__init__( self, parent, session )
      self.trafficPolicyContext = context
      self.trafficPolicy = context.currentPolicy()
      context.modeIs( self )

class TrafficPolicyConfigMode( TrafficPolicyConfigModeBase,
                               ClassificationConfigModeBase ):
   """ Configuration mode for traffic-policy """
   name = "traffic-policy configuration"
   modeParseTree = CliParser.ModeParseTree()

class MatchRuleModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      cmd = "match %s %s" % ( self.ruleName, self.matchType )
      return cmd

   def __init__( self, param ):
      if len( param ) == 6:
         ( self.feature, self.parentMode, self.trafficPolicyName,
           self.ruleName, self.matchType, self.prio ) = param
      else:
         ( self.feature, self.parentMode, self.trafficPolicyName,
           self.ruleName, self.matchType ) = param
      self.modeKey = "match-%s" % self.matchType
      self.longModeKey = "%s-match-%s-%s-%s" % ( self.feature,
                                                 self.trafficPolicyName,
                                                 self.ruleName, self.matchType )
      CliMode.ConfigMode.__init__( self, param, parentMode=self.parentMode )

class MatchRuleBaseConfigMode( MatchRuleModeBase, ClassificationConfigModeBase ):
   """ Base configuration mode for match rule"""
   def __init__( self, parent, session, trafficPolicyContext, matchRuleContext,
                 feature ):
      param = ( feature, type( parent ),
                trafficPolicyContext.pmapName_, matchRuleContext.ruleName,
                matchRuleContext.matchType )
      MatchRuleModeBase.__init__( self, param )
      ClassificationConfigModeBase.__init__( self, parent, session )
      self.trafficPolicy = trafficPolicyContext.currentPolicy()
      self.trafficPolicyContext = trafficPolicyContext
      self.context = matchRuleContext
      self.feature = feature

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def getContext( self ):
      """ Used to allow each config mode to name their own context """
      return self.context

   def setContext( self, value ):
      self.context = value

   def abort( self ):
      if self.context.matchRuleAction is not None:
         self.context.matchRuleAction.abort()
      self.context = None
      self.session_.gotoParentMode()
      self.session_.mode.abort() # call abort of parent

   def commitContext( self ):
      if self.context is None:
         return
      context = self.context
      self.context = None
      context.commit()
      if self.trafficPolicyContext.shouldResequence:
         self.trafficPolicyContext.resequence( 10, 10 )
         lastseq = self.trafficPolicyContext.lastSequence()
         context.seqnum = self.trafficPolicyContext.lookupRuleToSeq(
                     context.matchRuleAction )

         # Ensure we've restored the sequence numbers to even intervals of 10.
         assert context.seqnum % 10 == 0

         # We must commit the context a second time to preserve the changes from
         # resequencing.
         context.commit()
         # context.commit() overwrites lastseq with the rule we're committing,
         # so we have to reset it again here.
         self.trafficPolicyContext.lastSequenceIs( lastseq )

   def commit( self ):
      self.commitContext()
      self.session_.gotoParentMode()
      self.session_.mode.commit() # call commit of parent

class MatchRuleDefaultConfigMode( MatchRuleBaseConfigMode ):
   """
   Configuration mode for match rule ipv{4,6}-all-default ipv{4,6}. Note we only need
   one as all actions are shared between IPV4 and IPV6 match statements.
   """
   name = "default match rule configuration"
   modeParseTree = CliParser.ModeParseTree()

class MatchRuleIpv4ConfigMode( MatchRuleBaseConfigMode ):
   """ Configuration mode for match rule ipv4"""
   name = "match rule configuration ipv4"
   modeParseTree = CliParser.ModeParseTree()

class MatchRuleIpv6ConfigMode( MatchRuleBaseConfigMode ):
   """ Configuration mode for match rule ipv6 """
   name = "match rule configuration ipv6"
   modeParseTree = CliParser.ModeParseTree()

class ActionsModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      cmd = "actions"
      return cmd

   def __init__( self, param ):
      self.feature, self.trafficPolicyName, self.ruleName = param
      self.modeKey = "actions"
      self.longModeKey = "%s-actions-%s-%s" % ( self.feature,
                                                self.trafficPolicyName,
                                                self.ruleName )
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=TrafficPolicyConfigMode )
      self.session_ = None
      self.context = None

   def commit( self ):
      self.session_.gotoParentMode()
      self.session_.mode.commit() # call commit of parent

   def abort( self ):
      if self.context.matchRuleAction is not None:
         self.context.matchRuleAction.abort()
      self.session_.gotoParentMode()
      self.session_.mode.abort() # call abort of parent

class ActionsConfigModeBase( ActionsModeBase, BasicCli.ConfigModeBase ):
   def __init__( self, parent, session, trafficPolicyContext, matchRuleContext,
                 feature ):
      param = ( feature, trafficPolicyContext.pmapName_,
                matchRuleContext.ruleName )
      ActionsModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.trafficPolicy = trafficPolicyContext.currentPolicy()
      self.trafficPolicyContext = trafficPolicyContext
      self.context = matchRuleContext
      self.feature = feature

class ActionsConfigMode( ActionsConfigModeBase, BasicCli.ConfigModeBase ):
   """ Configuration mode for actions sub-prompt """
   name = "actions configuration"
   modeParseTree = CliParser.ModeParseTree()

class FieldSetIpPrefixConfigMode( FieldSetBasePrefixConfigMode ):
   name = "traffic-policy field-set ipv4 prefix configuration"
   modeParseTree = CliParser.ModeParseTree()

class FieldSetIpv6PrefixConfigMode( FieldSetBasePrefixConfigMode ):
   name = "traffic-policy field-set ipv6 prefix configuration"
   modeParseTree = CliParser.ModeParseTree()
