# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliMode
from CliMode.Te import GlobalTeMode
from RsvpLib import bandwidthHumanReadable
import MultiRangeRule
import Tac
from TypeFuture import TacLazyType
import Toggles.RsvpToggleLib

IpGenAddr = TacLazyType( 'Arnet::IpGenAddr' )
RsvpLerConstants = TacLazyType( 'Rsvp::RsvpLerConstants' )
RsvpLerPathSpecId = TacLazyType( 'Rsvp::RsvpLerPathSpecId' )
RsvpLerPathSpecType = TacLazyType( 'Rsvp::RsvpLerPathSpecType' )

pathSpecDynamicType = RsvpLerPathSpecType.pathSpecDynamicType
pathSpecExplicitType = RsvpLerPathSpecType.pathSpecExplicitType

class RsvpTeMode( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'rsvp'

   def __init__( self, param ):
      self.modeKey = "rsvp"
      self.longModeKey = "te-rsvp"
      CliMode.ConfigMode.__init__( self, param, parentMode=GlobalTeMode )

   def commentKey( self ):
      return self.longModeKey

   def skipIfEmpty( self ):
      return True

def bitMaskToStr( bitMask ):
   '''
   Convert an integer representing a bitmask to a string representing each bit
   set to 1 in this bitmask.
   ie:
   0x0 -> 0000 -> ''
   0x1 -> 0001-> '0'
   0x3 -> 0011 -> '0-1'
   0x13 -> 0001 0011 -> '0-1,4'
   etc...
   '''
   if not bitMask:
      return ''
   maskList = []
   i = 0
   while bitMask:
      if bitMask % 2:
         maskList.append( i )
      i += 1
      bitMask >>= 1

   return MultiRangeRule.multiRangeToCanonicalString( maskList )

class RsvpPathSpecMode( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'path %s %s' % ( self.pathSpecName, self.pathSpecTypeStr )

   def commitContext( self ):
      raise NotImplementedError

   def pathSpecToCliCmds( self, pathSpec, saveAll=False ):
      cmds = []
      if not pathSpec:
         return cmds

      if Toggles.RsvpToggleLib.toggleRsvpLerIncludeHopEnabled():
         # No Hops
         if not pathSpec.includeHop and not pathSpec.excludeHop and saveAll:
            cmds.append( 'no hop' )

         # Include Hops
         for hop in pathSpec.includeHop.values():
            cmd = 'hop ' + str( hop.hopIp ) + ( ' loose' if hop.loose else '' )
            cmds.append( cmd )
      else:
         if self.pathSpecType == pathSpecDynamicType:
            # No Hops
            if not pathSpec.excludeHop and saveAll:
               cmds.append( 'no hop' )

         elif self.pathSpecType == pathSpecExplicitType:
            # No Hops
            if not pathSpec.includeHop and saveAll:
               cmds.append( 'no hop' )

            # Include Hops
            for hop in pathSpec.includeHop.values():
               cmd = 'hop ' + str( hop.hopIp )
               cmds.append( cmd )

      if self.pathSpecType == pathSpecExplicitType:
         # Explicit Mode doesn't have exclude hops or administrative groups
         return cmds

      # Dynamic mode only from now on
      # Exclude Hops
      for excludeHop in pathSpec.excludeHop:
         cmds.append( 'hop ' + str( excludeHop ) + ' exclude' )

      # Administrative Groups
      includeAllStr = 'include all {}'.format( bitMaskToStr(
          pathSpec.includeAllAdminGroup ) ) if pathSpec.includeAllAdminGroup else ''
      includeAnyStr = 'include any {}'.format( bitMaskToStr(
          pathSpec.includeAnyAdminGroup ) ) if pathSpec.includeAnyAdminGroup else ''
      excludeStr = 'exclude {}'.format( bitMaskToStr(
         pathSpec.excludeAdminGroup ) ) if pathSpec.excludeAdminGroup else ''
      if pathSpec.includeAllAdminGroup or pathSpec.includeAnyAdminGroup or \
            pathSpec.excludeAdminGroup:
         cmd = ' '.join( filter( None,
                               [ 'administrative-group',
                                 includeAllStr, includeAnyStr, excludeStr ] ) )
         cmds.append( cmd )
      elif saveAll:
         cmd = 'no administrative-group'
         cmds.append( cmd )

      return cmds

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def __init__( self, param ):
      self.pathSpecName = param[ 0 ]
      self.pathSpecType = param[ 1 ]
      if self.pathSpecType == pathSpecExplicitType:
         self.pathSpecTypeStr = 'explicit'
         self.shortPathTypeStr = 'expl'
      else:
         self.pathSpecTypeStr = 'dynamic'
         self.shortPathTypeStr = 'dyn'

      self.modeKey = "path-%s-%s" % ( self.shortPathTypeStr, self.pathSpecName )
      self.pathSpec = None
      self.longModeKey = "te-rsvp-" + self.modeKey
      CliMode.ConfigMode.__init__( self, param )

   def skipIfEmpty( self ):
      # Even if a pathSpec is 'empty' (i.e no configuration command issued for it),
      # we still need to display it, as it is saved in the pathSpec collection
      return False

class RsvpTunnelSpecMode( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'tunnel %s' % self.tunnelSpecName

   def commitContext( self ):
      raise NotImplementedError

   def tunnelSpecToCliCmds( self, tunnelSpec, saveAll=False ):
      cmds = []
      if not tunnelSpec:
         return cmds

      # Destination ip
      if tunnelSpec.dstIp != IpGenAddr():
         cmd = 'destination ip ' + str( tunnelSpec.dstIp )
         cmds.append( cmd )
      elif saveAll:
         cmd = 'no destination ip'
         cmds.append( cmd )

      # no shutdown
      if tunnelSpec.enabled:
         cmd = 'no shutdown'
         cmds.append( cmd )
      elif saveAll:
         cmd = 'shutdown'
         cmds.append( cmd )

      # primary and/or secondary paths
      if tunnelSpec.primaryPath or tunnelSpec.secondaryPath:
         # Primary path
         if tunnelSpec.primaryPath:
            cmd = 'path ' + tunnelSpec.primaryPath.pathSpecName
            cmds.append( cmd )
         # Secondary path and secondary pre-signaled
         if tunnelSpec.secondaryPath:
            secondaryPath = tunnelSpec.secondaryPath.pathSpecName
            secondaryPathPreSignaled = tunnelSpec.secondaryPathPreSignaled
            cmdSuffix = ' pre-signaled' if secondaryPathPreSignaled else ''
            cmd = 'path ' + secondaryPath + ' secondary' + cmdSuffix
            cmds.append( cmd )
      elif saveAll:
         cmd = 'no path'
         cmds.append( cmd )

      if tunnelSpec.autoBw:
         # Autobandwidth and autobandwidth parameters
         # convert bytes-per-sec to bits-per-sec
         minBw, minBwUnit = bandwidthHumanReadable(
                               tunnelSpec.autoBwParam.minBandwidth * 8 )
         maxBw, maxBwUnit = bandwidthHumanReadable(
                               tunnelSpec.autoBwParam.maxBandwidth * 8 )
         adjustmentPeriod = int( tunnelSpec.autoBwParam.adjustmentPeriod )
         if adjustmentPeriod:
            cmd = 'bandwidth auto min %.2f %s max %.2f %s adjustment-period %s' % \
                  ( minBw, minBwUnit, maxBw, maxBwUnit, adjustmentPeriod )
         else:
            cmd = 'bandwidth auto min %.2f %s max %.2f %s' % \
                  ( minBw, minBwUnit, maxBw, maxBwUnit )
         cmds.append( cmd )
      elif tunnelSpec.explicitBandwidth:
         # Explicit bandwidth
         # convert bytes-per-sec to bits-per-sec
         bw, unit = bandwidthHumanReadable( tunnelSpec.explicitBandwidth * 8 )
         cmd = 'bandwidth %.2f %s' % ( bw, unit )
         cmds.append( cmd )
      elif saveAll:
         cmd = 'no bandwidth'
         cmds.append( cmd )

      if tunnelSpec.setupPriority < RsvpLerConstants.defaultSetupPriority or \
            tunnelSpec.holdPriority > RsvpLerConstants.defaultHoldPriority:
         cmd = 'priority setup %d hold %d' % ( tunnelSpec.setupPriority,
                                               tunnelSpec.holdPriority )
         cmds.append( cmd )
      elif saveAll:
         cmd = 'priority setup %d hold %d' % ( RsvpLerConstants.defaultSetupPriority,
                                               RsvpLerConstants.defaultHoldPriority )
         cmds.append( cmd )

      if Toggles.RsvpToggleLib.toggleLdpOverRsvpEnabled():
         if tunnelSpec.eligibleForLdpTunneling:
            cmd = 'tunneling ldp'
            cmds.append( cmd )
         elif saveAll:
            cmd = 'no tunneling ldp'
            cmds.append( cmd )

      if Toggles.gatedToggleLib.toggleIsisIgpShortcutEnabled():
         if tunnelSpec.eligibleForIgpShortcut:
            cmd = 'igp shortcut'
            cmds.append( cmd )
         elif saveAll:
            cmd = 'no igp shortcut'
            cmds.append( cmd )

      if Toggles.RsvpToggleLib.toggleRsvpLerReoptimizationEnabled():
         if tunnelSpec.optimizationExplicitlyDisabled:
            cmd = 'optimization disabled'
            cmds.append( cmd )
         elif tunnelSpec.optimizationInterval != \
               RsvpLerConstants.optimizationIntervalDisabled:
            cmd = 'optimization interval %d seconds' % \
                  tunnelSpec.optimizationInterval
            cmds.append( cmd )
         elif saveAll:
            cmd = 'no optimization'
            cmds.append( cmd )

      return cmds

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def __init__( self, param ):
      self.tunnelSpecName = param
      self.modeKey = "tunnel-%s" % self.tunnelSpecName
      self.tunnelSpec = None
      self.longModeKey = "te-rsvp-" + self.modeKey
      CliMode.ConfigMode.__init__( self, param )

   def skipIfEmpty( self ):
      # Even if a tunnelSpec is 'empty' (i.e no configuration command issued for it),
      # we still need to display it, as it is saved in the tunnelSpec collection
      return False
