#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliMode
import QosLib

class IntfTxQueueMode( CliMode.ConfigMode ):
   
   def enterCmd( self ):
      if self.modeKey == 'txq':
         first = 'tx-queue'
      elif self.modeKey == 'uc-txq':
         first = 'uc-tx-queue'
      elif self.modeKey == 'mc-txq':
         first = 'mc-tx-queue'
      return '%s %d' % ( first, self.txQueueId ) 

   # This is for a particular interface and txQueueId pair to create a mode
   def __init__( self, param ):
      # Caller decides if child mode must be created. For intf range intf-txq pair
      # is not a child mode. But instead a set of them.
      # param = tuple ( parentIntfShortName, queueType, txQueueId, childMode )
      # parentIntfShortName should be prepended with 'if' for interfaces 
      ( parentIntfShortName, self.modeKey, self.txQueueId, parentMode ) = param
      # The longModeKey generated here is to just create the mode. It's not used
      # elsewhere.
      self.longModeKey = "%s-%s-%d" % ( parentIntfShortName, 
                                        self.modeKey, 
                                        self.txQueueId )
      CliMode.ConfigMode.__init__( self, self.longModeKey, parentMode=parentMode )

class QosProfileModeBase( CliMode.ConfigMode ):
   autoConfigSessionAllowed = False

   def enterCmd( self ):
      return 'qos profile %s ' % ( self.profileName )

   def __init__( self, param ):
      ( self.profileName ) = param
      self.modeKey = 'qos-profile'
      self.longModeKey = 'qos-profile-%s' % ( self.profileName )
      CliMode.ConfigMode.__init__( self, self.longModeKey )

class PolicingModeBase( CliMode.ConfigMode ):
   autoConfigSessionAllowed = False

   def enterCmd( self ):
      return 'policing'

   def __init__( self ):
      self.modeKey = 'policing'
      self.longModeKey = 'policing'
      CliMode.ConfigMode.__init__( self, self.longModeKey )

class DscpToTcModeBase( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'qos map dscp to traffic-class name %s' % self.mapName

   def __init__( self, param ):
      self.mapName = param
      self.modeKey = 'dscp-map'
      self.longModeKey = 'dscp-map-%s' % ( self.mapName )
      CliMode.ConfigMode.__init__( self, param )

class DscpRewriteModeBase( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'qos map traffic-class to dscp %s ' % ( self.mapName )

   def __init__( self, param ):
      self.mapName = param
      self.modeKey = 'map'
      self.longModeKey = 'map-%s' % ( self.mapName )
      CliMode.ConfigMode.__init__( self, param )

class TcToCosModeBase( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'qos map traffic-class to cos name %s' % self.mapName

   def __init__( self, param ):
      self.mapName = param
      self.modeKey = 'tc-map'
      self.longModeKey = 'tc-map-%s' % ( self.mapName )
      CliMode.ConfigMode.__init__( self, param )

class CosToTcProfileModeBase( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'qos map cos to traffic-class name %s' % self.profileName

   def __init__( self, param ):
      self.profileName = param
      self.modeKey = 'cos-map'
      self.longModeKey = 'cos-map-%s' % ( self.profileName )
      CliMode.ConfigMode.__init__( self, param )

class ClassMapModeBase( CliMode.ConfigMode ):
   autoConfigSessionAllowed = False
   
   def enterCmd( self ):
      return 'class-map type %s match-any %s' % ( self.mapTypeString, self.cmapName )

   def __init__( self, param ):
      ( self.mapType, self.cmapName ) = param
      mapType = QosLib.mapTypeFromEnum( self.mapType )
      if self.mapType == QosLib.coppMapType:
         mapType = 'copp'
      self.mapTypeString = mapType
      self.modeKey = 'cmap'
      self.longModeKey = 'cmap-%s-%s' % ( mapType, self.cmapName )
      CliMode.ConfigMode.__init__( self, self.longModeKey )

class PolicyMapModeBase( CliMode.ConfigMode ):
   autoConfigSessionAllowed = False

   def enterCmd( self ):
      return 'policy-map type %s %s%s' % ( self.mapTypeString,
                                           "shared " if self.shared else "",
                                           self.pmapName )

   def __init__( self, param ):
      ( self.mapType, self.pmapName, self.shared ) = param
      mapType = QosLib.mapTypeFromEnum( self.mapType )
      if self.mapType == QosLib.coppMapType:
         mapType = 'copp'
      elif self.mapType == QosLib.qosMapType:
         mapType = 'quality-of-service'
      self.mapTypeString = mapType
      self.modeKey = 'pmap'
      self.longModeKey = 'pmap-%s-%s' % ( mapType, self.pmapName )
      CliMode.ConfigMode.__init__( self, self.longModeKey )

class PolicyMapClassModeBase( CliMode.ConfigMode ):
   autoConfigSessionAllowed = False

   def enterCmd( self ):
      return 'class %s%s' % ( "built-in " if self.mapType == QosLib.pdpMapType \
                              and self.cmapName != 'class-default'
                              else "", self.cmapName )

   def __init__( self, param ):
      ( self.mapType, self.pmapName, self.cmapName ) = param
      mapType = QosLib.mapTypeFromEnum( self.mapType )
      if self.mapType == QosLib.coppMapType:
         mapType = 'copp'
      elif self.mapType == QosLib.qosMapType:
         mapType = 'quality-of-service'
      self.modeKey = 'pmap-c'
      self.longModeKey = 'pmap-c-%s-%s-%s' % \
          ( mapType, self.pmapName, self.cmapName )
      CliMode.ConfigMode.__init__( self, self.longModeKey, 
                                   parentMode=PolicyMapModeBase )

class QosSchedulingModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'qos scheduling'

   def __init__( self, param ):
      self.modeKey = 'qos-scheduling'
      self.longModeKey = 'qos-scheduling'
      CliMode.ConfigMode.__init__( self, param )

class QosSchedulingIntfModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'interface %s' % self.param_

   def __init__( self, param ):
      self.intf = param
      self.modeKey = 'qos-scheduling-%s' % self.intf
      self.longModeKey = 'qos-scheduling-interface-%s' % self.intf
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=QosSchedulingModeBase )

class QosSchedulingIntfGroupModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'scheduling group %s' % self.param_[ 1 ]

   def __init__( self, param ):
      self.intf, self.group = param
      self.modeKey = 'qos-scheduling-%s-%s' % ( self.intf, self.group )
      self.longModeKey = 'qos-scheduling-interface-%s-group-%s' % (
         self.intf, self.group )
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=QosSchedulingIntfModeBase )

class QosSchedulingPolicyModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'scheduling policy %s' % self.param_

   def __init__( self, param ):
      self.modeKey = 'qos-scheduling-policy-%s' % param
      self.longModeKey = 'qos-scheduling-policy-%s' % param
      self.policy = param
      CliMode.ConfigMode.__init__( self,
                                   param, parentMode=QosSchedulingModeBase )
