# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliMode
import BasicCli
import CliParser
import Tac

Constants = Tac.Type( "Dps::DpsConstants" )
constants = Constants()
DEFAULT_PRIORITY = constants.defaultPriority
DEFAULT_JITTER = constants.defaultJitter
DEFAULT_LATENCY = constants.defaultLatency
DEFAULT_LOSSRATE = constants.defaultLossRate
DEFAULT_UDP_PORT = constants.defaultUdpPort
# loss rate is specified in percentage in Cli. ( number of
# packets per 100 ) But Its measures it in number of packets
# per million.
LOSS_RATE_SCALE = constants.lossRateScale

# granularity adjustment
# Add granularity adjustment to loss rate. loss rate granularity is 0.01%,
# as input in the cli. However, Its granularity is 0.0001%. So, paths with
# differences between 0.0000 to 0.0099 should be considered as same
# loss-rate paths
# it is the same value as the default loss rate:
# default_loss_rate = 0 + loss_rate_adjustment
LOSS_RATE_ADJUSTMENT = DEFAULT_LOSSRATE

# latency and jitter are specified in ms in Cli. But Its measures
# it in micro second.
LATENCY_SCALE = 1000
JITTER_SCALE = 1000

class RouterPathSelectionModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'router path-selection'

   def __init__( self ):
      self.modeKey = 'dps'
      self.longModeKey = 'dynamic-path-selection'
      CliMode.ConfigMode.__init__( self, None, parentMode=CliMode.ConfigMode )

class RouterPathSelectionConfigMode( RouterPathSelectionModeBase,
                                     BasicCli.ConfigModeBase ):
   ''' Configuration mode for router path-selection '''
   name = 'router path-selection'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      RouterPathSelectionModeBase.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def commentKey( self ):
      return self.longModeKey

class DpsPathGroupModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'path-group %s' % self.pathGroupName

   def __init__( self, param ):
      self.pathGroupName = param
      self.modeKey = 'path-group'
      self.longModeKey = 'path-group-%s' % self.pathGroupName
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=RouterPathSelectionConfigMode )

class DpsPathGroupConfigMode( DpsPathGroupModeBase, BasicCli.ConfigModeBase ):
   ''' Configuration mode for path-group '''
   name = 'path-group configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context ):
      self.context = context
      self.pgCfg = context.currentPgCfg()
      context.modeIs( self )
      DpsPathGroupModeBase.__init__( self, context.pathGroupName() )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class DpsPathGroupRemoteRouterBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'peer static router-ip %s' % self.routerIp

   def __init__( self, routerIp, pgName ):
      self.routerIp = routerIp
      self.modeKey = 'peer-router-ip'
      self.longModeKey = 'peer-router-ip-%s-%s' % ( self.routerIp, pgName )
      CliMode.ConfigMode.__init__( self, routerIp,
                                   parentMode=DpsPathGroupConfigMode )

class DpsPathGroupRemoteRouterConfigMode( DpsPathGroupRemoteRouterBase,
                                          BasicCli.ConfigModeBase ):
   ''' Configuration mode for remote router-ip of DPS Path Group '''
   name = 'remote router configuration for path-group'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context ):
      self.context = context
      self.pgCfg = context.currentPgCfg()
      self.routerIp = context.currentRouterIp()
      context.modeIs( self )
      DpsPathGroupRemoteRouterBase.__init__( self, context.currentRouterIp(),
                                             context.pathGroupName() )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class DpsPolicyModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'policy %s' % self.policyName

   def __init__( self, param ):
      self.policyName = param
      self.modeKey = 'policy'
      self.longModeKey = 'policy-%s' % self.policyName
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=RouterPathSelectionConfigMode )

class DpsPolicyConfigMode( DpsPolicyModeBase, BasicCli.ConfigModeBase ):
   ''' Configuration mode for policy '''
   name = 'policy configuration'
   modeParseTree = CliParser.ModeParseTree()
   rulesChanged = False

   def __init__( self, parent, session, context ):
      self.context = context
      self.policyCfg = context.currentPolicyCfg()
      context.modeIs( self )
      DpsPolicyModeBase.__init__( self, context.policyName() )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      # TBD: optimize mode exit without any config changes
      if self.rulesChanged:
         self.policyCfg.configVersion = self.policyCfg.configVersion + 1

class DpsPolicyRuleKeyBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return '%s application-profile %s' % ( self.ruleKey, self.appProfile )

   def __init__( self, policyName, ruleKey, appProfile ):
      self.policyName = policyName
      self.ruleKey = ruleKey
      self.appProfile = appProfile
      self.modeKey = 'policy-rule-key'
      self.longModeKey = 'policy-rule-key-%s-%s' % ( self.ruleKey,
            self.policyName )
      CliMode.ConfigMode.__init__( self, self.policyName,
                                   parentMode=DpsPolicyConfigMode )

class DpsPolicyRuleKeyConfigMode( DpsPolicyRuleKeyBase,
                                          BasicCli.ConfigModeBase ):
   ''' Configuration mode for rule key of policy '''
   name = 'rule key configuration of policy'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context ):
      self.context = context
      self.policyCfg = context.currentPolicyCfg()
      self.ruleKey = context.currentRuleKey()
      self.appProfile = context.currentAppProfile()
      context.modeIs( self )
      DpsPolicyRuleKeyBase.__init__( self, context.policyName(),
            context.currentRuleKey(), context.currentAppProfile() )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class DpsPolicyDefaultRuleBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'default-match'

   def __init__( self, policyName, defaultRuleCfgd ):
      self.policyName = policyName
      self.defaultRuleCfgd = defaultRuleCfgd
      self.modeKey = 'policy-default-rule'
      self.longModeKey = 'policy-default-rule-%s' % self.policyName
      CliMode.ConfigMode.__init__( self, self.defaultRuleCfgd,
                                   parentMode=DpsPolicyConfigMode )

class DpsPolicyDefaultRuleConfigMode( DpsPolicyDefaultRuleBase,
                                          BasicCli.ConfigModeBase ):
   ''' Configuration mode for default rule of policy '''
   name = 'default matching rule configuration of policy'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context ):
      self.context = context
      self.policyCfg = context.currentPolicyCfg()
      self.defaultRuleCfgd = context.currentDefaultRuleCfgd()
      context.modeIs( self )
      DpsPolicyDefaultRuleBase.__init__( self, context.policyName(),
            context.currentDefaultRuleCfgd() )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class DpsLoadBalanceProfileConfigModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'load-balance policy %s' % self.profileName

   def __init__( self, param ):
      self.profileName = param
      self.modeKey = 'load-balance-policy'
      self.longModeKey = 'load-balance-policy-%s' % self.profileName
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=RouterPathSelectionConfigMode )

class DpsLoadBalanceProfileConfigMode( DpsLoadBalanceProfileConfigModeBase,
                                       BasicCli.ConfigModeBase ):
   ''' Configuration mode for load-balance policy '''
   name = 'load balance policy configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context ):
      self.context = context
      self.profile = context.currentProfile()
      context.modeIs( self )
      DpsLoadBalanceProfileConfigModeBase.__init__( self, context.profileName() )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class DpsVrfConfigModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'vrf %s' % self.vrfName

   def __init__( self, param ):
      self.vrfName = param
      self.modeKey = 'vrf'
      self.longModeKey = 'vrf-%s' % self.vrfName
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=RouterPathSelectionConfigMode )

class DpsVrfConfigMode( DpsVrfConfigModeBase,
                        BasicCli.ConfigModeBase ):
   ''' Configuration mode for setting vrf path selection policy '''
   name = 'vrf path selection policy configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context ):
      self.context = context
      self.vrfCfg = context.currentVrfCfg()
      context.modeIs( self )
      DpsVrfConfigModeBase.__init__( self, context.vrfName() )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
