#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import CliMode

# pylint: disable-msg=W0105
"""
Mode Structure
   dhcp server mode
      subnet mode (IPv4)
         reservation mode
            mac-address mode
      subnet mode (IPv6)
      vendor-option mode
"""

class DhcpServerBaseMode( CliMode.ConfigMode ):
   """
   CLI mode base class for:
      dhcp server mode
   """

   def enterCmd( self ):
      return "dhcp server"

   def __init__( self ):
      self.modeKey = "dhcp"
      self.longModeKey = "dhcp-server"
      CliMode.ConfigMode.__init__( self, None )

class DhcpServerSubnetBaseMode( CliMode.ConfigMode ):
   """
   CLI mode base class for:
      subnet mode
   """

   def enterCmd( self ):
      return "subnet %s" % self.subnet

   def __init__( self, param ):
      self.subnet, self.af = param
      self.modeKey = "subnet"
      self.longModeKey = "dhcp-subnet-%s" % self.af
      CliMode.ConfigMode.__init__( self, param )

   def commentKey( self ):
      return "dhcp-subnet-%s" % self.subnet

class DhcpServerVendorOptionBaseMode( CliMode.ConfigMode ):
   """
   CLI mode base class for:
      vendor-option mode
   """

   def enterCmd( self ):
      return "vendor-option %s %s" % ( self.af, self.vendorId )

   def __init__( self, param ):
      self.vendorId, self.af = param
      self.modeKey = "vendor"
      self.longModeKey = "dhcp-vendor-%s" % self.af
      CliMode.ConfigMode.__init__( self, param )

   def commentKey( self ):
      return "dhcp-vendor-%s %s" % ( self.af, self.vendorId )

class DhcpServerReservationsBaseMode( CliMode.ConfigMode ):
   """
   CLI mode base class for:
      reservations mode

   This is an umbrella mode for:
      mac-address mode
   """

   def enterCmd( self ):
      return "reservations"

   def __init__( self, param ):
      self.af = param
      self.modeKey = "reservations"
      self.longModeKey = "dhcp-reservations-%s" % self.af
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=DhcpServerSubnetBaseMode )

   def commentKey( self ):
      return self.longModeKey

class DhcpServerReservationsMacAddressBaseMode( CliMode.ConfigMode ):
   """
   CLI mode base class for:
      mac-address mode
   """

   def enterCmd( self ):
      return "mac-address %s" % self.macAddr

   def __init__( self, param ):
      self.macAddr, self.af = param
      self.modeKey = "mac-address"
      self.longModeKey = "dhcp-mac-address-%s" % self.af
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=DhcpServerReservationsBaseMode )

   def commentKey( self ):
      return "dhcp-mac-address-%s" % self.macAddr
