#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliMode
import CliParser
import ReversibleSecretCli
import Tac
from Tracing import t0

cloudAwsStr = 'cloud-aws'
cloudAzureStr = 'cloud-azure'
cloudGcpStr = 'cloud-gcp'
cloudHaPeer = 'cloud-ha-peer'
cloudHaStr = 'cloud-ha'
cloudProxyStr = 'cloud-proxy'
cloudHaPeerStr = cloudHaPeer + '-%s'

awsCloudConfig = None
azureCloudConfig = None
gcpCloudConfig = None
proxyConfig = None
cloudConfig = None

class CloudAwsMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'cloud provider aws'

   def __init__( self, param ):
      self.modeKey = cloudAwsStr
      self.longModeKey = cloudAwsStr
      CliMode.ConfigMode.__init__( self, param )

class CloudHaAwsMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return  'aws'

   def __init__( self, param ):
      self.modeKey = cloudHaPeer + "-%s-aws" % param
      self.longModeKey = cloudHaPeer + "-%s-aws" % param
      CliMode.ConfigMode.__init__( self, param )

class CloudAzureMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'cloud provider azure'

   def __init__( self, param ):
      self.modeKey = cloudAzureStr
      self.longModeKey = cloudAzureStr
      CliMode.ConfigMode.__init__( self, param )

class CloudHaAzureMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'azure'

   def __init__( self, param ):
      self.modeKey = cloudHaPeer + "-%s-azure" % param
      self.longModeKey = cloudHaPeer + "-%s-azure" % param
      CliMode.ConfigMode.__init__( self, param )

class CloudGcpMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'cloud provider gcp'

   def __init__( self, param ):
      self.modeKey = cloudGcpStr
      self.longModeKey = cloudGcpStr
      CliMode.ConfigMode.__init__( self, param )

class CloudHaGcpMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'gcp'

   def __init__( self, param ):
      self.modeKey = cloudHaPeer + '-%s-gcp' % param
      self.longModeKey = cloudHaPeer + '-%s-gcp' % param
      CliMode.ConfigMode.__init__( self, param )

class CloudProxyMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'cloud proxy %s' % self.param_

   def __init__( self, param ):
      self.modeKey = cloudProxyStr
      self.longModeKey = cloudProxyStr + "-%s" % param
      CliMode.ConfigMode.__init__( self, param )

class CloudHaMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'cloud high-availability'

   def __init__( self, param ):
      self.modeKey = cloudHaStr
      self.longModeKey = cloudHaStr
      CliMode.ConfigMode.__init__( self, param )

class CloudHaPeerMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'peer %s' % self.param_

   def __init__( self, param ):
      cloudHaPeerKey = cloudHaPeerStr % param
      self.modeKey = cloudHaPeerKey
      self.longModeKey = cloudHaPeerKey
      CliMode.ConfigMode.__init__( self, param )

#-------------------------------------------------------------------------------
# config-cloud-provider-aws mode
#-------------------------------------------------------------------------------
class ConfigCloudAwsMode( CloudAwsMode, BasicCli.ConfigModeBase ):
   name = "AWS configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CloudAwsMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.awsConfig = Tac.nonConst( awsCloudConfig.accessConfig )

   def onExit( self ):
      t0(" Exiting ConfigCloudAwsMode " )
      # Test case only w/o guard
      if not awsCloudConfig:
         return
      if self.awsConfig != awsCloudConfig.accessConfig:
         t0( "Changed AWS specific config, updating it" )
         awsCloudConfig.accessConfig = self.awsConfig
         awsCloudConfig.lastChangeTime = Tac.now()

#-------------------------------------------------------------------------------
#
# config-cloud-ha-peer-aws mode
#
#-------------------------------------------------------------------------------
class ConfigCloudHaAwsMode( CloudHaAwsMode, BasicCli.ConfigModeBase ):
   name = "AWS-HA configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CloudHaAwsMode.__init__( self, parent.param_ )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.localRoutes = awsCloudConfig.localRoutes
      self.peerRoutes = awsCloudConfig.peerRoutes
      self.changed = False
      self.parent = parent

   def onExit( self ):
      t0( "Exiting Cloud ha AWS submode " )
      if self.changed:
         t0( 'Updating parent submode as routes has changed' )
         self.parent.changed = True

#-------------------------------------------------------------------------------
#
# config-cloud-ha-peer-gcp mode
#
#-------------------------------------------------------------------------------
class ConfigCloudHaGcpMode( CloudHaGcpMode,
                            BasicCli.ConfigModeBase ):
   name = "GCP-HA configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CloudHaGcpMode.__init__( self, parent.param_ )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.localRoutes = gcpCloudConfig.localRoutes
      self.peerRoutes = gcpCloudConfig.peerRoutes
      self.changed = False
      self.parent = parent

   def onExit( self ):
      t0( "Exiting Cloud ha GCP submode " )
      if self.changed:
         t0( 'Updating parent submode as routes has changed' )
         self.parent.changed = True

#-------------------------------------------------------------------------------
# config-cloud-proxy mode
#-------------------------------------------------------------------------------
class ConfigCloudProxyMode( CloudProxyMode, BasicCli.ConfigModeBase ):
   name = "Proxy configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, proxyName ):
      CloudProxyMode.__init__( self, proxyName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      # Get the proxy object if there
      self.proxy = proxyConfig.proxy.get( proxyName )
      if not self.proxy:
         t0( 'Creating a new proxy for', proxyName )
         self.proxy = Tac.newInstance( 'Cloud::HA::Proxy', proxyName )
         # This is different object from the above object  as nominal.
         # But we still  add to proxyConfig as the user may just enter
         # the mode and doesn't configure anything at all.
         # Later we may change this sysdb collection based on submode
         # commands.
         proxyConfig.addProxy( self.proxy )

   def onExit( self ):
      def equivalentProxies( proxy1, proxy2 ):
         '''Checks if the two  proxies are basically same
            as the password can be encoded differently'''
         t0(' Checking proxies for equivalence', proxy1, proxy2 )
         proxy1 = Tac.nonConst( proxy1 )
         proxy2 = Tac.nonConst( proxy2 )
         httpPassword1 = proxy1.httpPassword
         httpPassword2 = proxy2.httpPassword
         httpsPassword1 = proxy1.httpsPassword
         httpsPassword2 = proxy2.httpsPassword
         proxy1.httpPassword = ''
         proxy1.httpsPassword = ''
         proxy2.httpsPassword = ''
         proxy2.httpPassword = ''

         decode = ReversibleSecretCli.decodeKey
         result = ( proxy1 == proxy2 and
                    decode( httpPassword1 ) == decode( httpPassword2 ) and
                    decode( httpsPassword1 ) == decode( httpsPassword2 ) )
         t0( 'Proxies', proxy1, proxy2, 'are found to be same:', result )
         return result

      t0(" Exiting ConfigCloudProxyMode " )
      cfgProxy = proxyConfig.proxy[ self.proxy.name ]
      proxyConfig.addProxy( self.proxy )
      if not equivalentProxies( cfgProxy, self.proxy ):
         t0( 'Notifying cloud providers as proxy', self.proxy.name, 'has changed' )
         t0( 'configured proxy ', cfgProxy, ' mode proxy ', self.proxy )
         import CliPlugin.CloudHaCliLib
         if CliPlugin.CloudHaCliLib.updateCloudHaProxy( self.proxy.name ):
            t0( 'Successfully notified cloud providers' )
      t0( 'onExit from proxy', proxyConfig.proxy[ self.proxy.name ] )

class ConfigCloudHaPeerMode( CloudHaPeerMode, BasicCli.ConfigModeBase ):
   name = "HA peer configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, peerName ):
      CloudHaPeerMode.__init__( self, peerName )
      BasicCli.ConfigModeBase.__init__( self, parent, session, peerName )
      self.parent = parent
      self.changed = False
      self.sysdbConfig = cloudConfig
      peer = self.sysdbConfig.peer.get( peerName, None )
      if not peer:
         self.peer = Tac.newInstance( 'Cloud::HA::HaPeer', peerName )
         self.sysdbConfig.addPeer( self.peer )
      else:
         self.peer = Tac.nonConst( peer )

   def onExit( self ):
      t0( "Exiting Cloud ha peer submode" )
      if self.changed:
         # Underlying child submode mode may have changed. trigger our parent
         t0( "Peer config changed by child modes. propogating changeTimeStamp" )
         self.parent.changed = True
      if self.sysdbConfig:
         peer = self.sysdbConfig.peer.get( self.peer.name, None )
         if peer != self.peer:
            t0( "Peer config changed. Updating changeTimeStamp" )
            self.parent.changed = True
         self.sysdbConfig.addPeer( self.peer )
         for i, v  in self.sysdbConfig.peer.items():
            t0( i, '--->', v )

class ConfigCloudHaAzureMode( CloudHaAzureMode,
                              BasicCli.ConfigModeBase ):
   name = "AZURE-HA configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CloudHaAzureMode.__init__( self, parent.param_ )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.azureLocalRoutes = azureCloudConfig.localRoutes
      self.azurePeerRoutes = azureCloudConfig.peerRoutes
      self.changed = False
      self.parent = parent

   def onExit( self ):
      t0( "Exiting Cloud ha Azure submode " )
      if self.changed:
         t0( 'Updating parent submode as routes has changed' )
         self.parent.changed = True

class ConfigCloudHaMode( CloudHaMode, BasicCli.ConfigModeBase ):
   name = "HA configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CloudHaMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.changed = False
      self.enable = cloudConfig.enable
      self.config = cloudConfig
      self.peer = None

   def onExit( self ):
      t0( "Exiting Cloud ha submode " )
      if self.changed or ( self.config and \
            self.config.enable != self.enable ):
         self.config.enable = self.enable
         t0( "Updating changeTimeStamp" )
         self.config.lastChangeTime = Tac.now()

class ConfigCloudAzureMode( CloudAzureMode, BasicCli.ConfigModeBase ):
   name = "AZURE configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CloudAzureMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.accessConfig = Tac.nonConst( azureCloudConfig.accessConfig )

   def onExit( self ):
      t0(" Exiting ConfigCloudAzureMode " )
      # Test case only w/o guard
      if not azureCloudConfig:
         return
      ## This internally compares each field of the nominal.
      if self.accessConfig != azureCloudConfig.accessConfig:
         t0( "Changed Azure specific config, updating it" )
         azureCloudConfig.accessConfig = self.accessConfig
         azureCloudConfig.lastChangeTime = Tac.now()
      t0( 'azureCloudConfig is: ', azureCloudConfig.accessConfig )

class ConfigCloudGcpMode( CloudGcpMode, BasicCli.ConfigModeBase ):
   name = "GCP configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CloudGcpMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.gcpConfig = Tac.nonConst( gcpCloudConfig.accessConfig )

   def onExit( self ):
      t0( 'Exiting ConfigCloudGcpMode' )
      if self.gcpConfig != gcpCloudConfig.accessConfig:
         t0( 'Changed GCP specific config, updating it' )
         gcpCloudConfig.accessConfig = self.gcpConfig
         gcpCloudConfig.lastChangeTime = Tac.now()
