#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliMode
import CliParser
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'Classification' )
t0 = __defaultTraceHandle__.trace0

#-------------------------------------------------------------------------------
# Base mode to be used by any CliPlugin using common Classification CLI code
#-------------------------------------------------------------------------------
class ClassificationConfigModeBase( BasicCli.ConfigModeBase ):
   def getContext( self ):
      raise NotImplementedError

   def setContext( self, value ):
      raise NotImplementedError

   def abort( self ):
      context = self.getContext()
      if context is not None:
         context.abort()
         self.setContext( None )
      self.session_.gotoParentMode()

   def commitContext( self ):
      context = self.getContext()
      if context is not None:
         context.commit()
         self.setContext( None )

   def commit( self ):
      self.commitContext()
      self.session_.gotoParentMode()

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

#-------------------------------------------------------------------------------
# The "application traffic recognition" mode.
#-------------------------------------------------------------------------------
class AppTrafficRecModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'application traffic recognition'

   def __init__( self ):
      self.modeKey = 'app-rec'
      self.longModeKey = 'app-recognition'
      CliMode.ConfigMode.__init__( self, None,
                                   parentMode=CliMode.ConfigMode )

class AppTrafficRecConfigMode( AppTrafficRecModeBase, ClassificationConfigModeBase ):
   """Configuration mode for application traffic recognition"""

   name = "application traffic recognition"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context ):
      self.appRecogContext = context
      AppTrafficRecModeBase.__init__( self )
      self.appRecogContext.modeIs( self )
      ClassificationConfigModeBase.__init__( self, parent, session )

   def getContext( self ):
      return self.appRecogContext

   def setContext( self, value ):
      self.appRecogContext = value

#-------------------------------------------------------------------------------
# The "field-set ( (ipv4 | ipv6) | l4-port ) <field-set-name>" mode
#-------------------------------------------------------------------------------
class FieldSetL4PortModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return "field-set l4-port %s" % self.fieldSetName

   def __init__( self, param ):
      self.feature, self.setType, self.fieldSetName, parentMode = param
      self.modeKey = 'field-set-l4-port'
      self.longModeKey = '%s-field-set-l4-port-%s' % ( self.feature,
                                                       self.fieldSetName )
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=parentMode )

class FieldSetL4PortConfigMode( FieldSetL4PortModeBase,
                                ClassificationConfigModeBase ):
   '''Configuration mode for L4 port field set'''

   name = "field-set l4-port configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context, feature ):
      self.l4PortContext = context
      self.fieldSetL4PortName = self.l4PortContext.fieldSetName
      self.l4PortContext.modeIs( self )
      param = ( feature, "l4-port", self.fieldSetL4PortName,
                AppTrafficRecConfigMode )
      FieldSetL4PortModeBase.__init__( self, param )
      ClassificationConfigModeBase.__init__( self, parent, session )

   def getContext( self ):
      return self.l4PortContext

   def setContext( self, value ):
      self.l4PortContext = value

   def abort( self ):
      super( FieldSetL4PortConfigMode, self ).abort()
      self.session_.mode.abort() # call abort of parent

   def commit( self ):
      super( FieldSetL4PortConfigMode, self ).commit()
      self.session_.mode.commit() # call commit of parent

#-------------------------------------------------------------------------------
# The "field-set ipv4|ipv6 prefix" mode
#-------------------------------------------------------------------------------
class FieldSetPrefixModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return "field-set %s prefix %s" % ( self.setType, self.fieldSetName )

   def __init__( self, param ):
      self.feature, self.setType, self.fieldSetName, parentMode = param
      self.modeKey = "field-set-%s-prefix" % self.setType
      self.longModeKey = "%s-field-set-%s-prefix-%s" % ( self.feature, self.setType,
                                                         self.fieldSetName )
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=parentMode )

class FieldSetBasePrefixConfigMode( FieldSetPrefixModeBase,
                                    ClassificationConfigModeBase ):
   '''Base onfiguration mode for field-set prefix'''
   def __init__( self, parent, session, context, feature ):
      self.ipPrefixContext = context
      self.setType = self.ipPrefixContext.setType
      self.fieldSetIpPrefixName = self.ipPrefixContext.fieldSetName
      self.ipPrefixContext.modeIs( self )
      param = ( feature, self.setType,
                self.fieldSetIpPrefixName, AppTrafficRecConfigMode )
      FieldSetPrefixModeBase.__init__( self, param )
      ClassificationConfigModeBase.__init__( self, parent, session )

   def getContext( self ):
      return self.ipPrefixContext

   def setContext( self, value ):
      self.ipPrefixContext = value

   def abort( self ):
      super( FieldSetBasePrefixConfigMode, self ).abort()
      self.session_.mode.abort() # call abort of parent

   def commit( self ):
      super( FieldSetBasePrefixConfigMode, self ).commit()
      self.session_.mode.commit() # call commit of parent

class FieldSetIpPrefixConfigMode( FieldSetBasePrefixConfigMode ):
   """Configuration mode for field-set ipv4 prefix"""
   name = "field-set ipv4 prefix configuration"
   modeParseTree = CliParser.ModeParseTree()

class FieldSetIpv6PrefixConfigMode( FieldSetBasePrefixConfigMode ):
   """Configuration mode for field-set ipv4 prefix"""
   name = "field-set ipv6 prefix configuration"
   modeParseTree = CliParser.ModeParseTree()

#-------------------------------------------------------------------------------
# The "application-profile <name>" mode
#-------------------------------------------------------------------------------
class AppProfileModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return "application-profile %s" % self.appProfileName

   def __init__( self, param ):
      self.appProfileName = param
      self.modeKey = 'app-profile'
      self.longModeKey = 'app-profile-%s' % self.appProfileName
      CliMode.ConfigMode.__init__( self, param,
                                   parentMode=AppTrafficRecConfigMode )

class AppProfileConfigMode( AppProfileModeBase, BasicCli.ConfigModeBase ):
   '''Configuration mode for application-profile'''

   name = "application-profiel configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context ):
      self.appProfileContext = context
      self.appProfileName = self.appProfileContext.appProfileName
      self.appProfileContext.modeIs( self )
      AppProfileModeBase.__init__( self, self.appProfileContext.appProfileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getContext( self ):
      return self.appProfileContext

   def onExit( self ):
      t0( "Exit application-profile %s" % self.appProfileName )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      t0( "Abort application-profile %s" % self.appProfileName )
      if self.appProfileContext is not None:
         self.appProfileContext.abort()
      self.appProfileContext = None
      self.session_.gotoParentMode()
      self.session_.mode.abort() # call abort of parent

   def commitContext( self ):
      t0( "Commit application-profile %s" % self.appProfileName )
      if self.appProfileContext:
         self.appProfileContext.commit()
         self.appProfileContext = None

   def commit( self ):
      self.commitContext()
      self.session_.gotoParentMode()
      self.session_.mode.commit() # call commit of parent

#-------------------------------------------------------------------------------
# The "application ipv4 <name>" mode
#-------------------------------------------------------------------------------
class AppModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return "application ipv4 %s" % self.appName

   def __init__( self, param ):
      self.appName = param
      self.modeKey = 'app-ipv4'
      self.longModeKey = 'app-ipv4-%s' % self.appName
      CliMode.ConfigMode.__init__( self, param, parentMode=AppTrafficRecConfigMode )

class AppConfigMode( AppModeBase, BasicCli.ConfigModeBase ):
   '''Configuration mode for application ipv4'''

   name = "application ipv4 configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context ):
      self.appContext = context
      self.appName = self.appContext.appName
      self.appContext.modeIs( self )
      AppModeBase.__init__( self, self.appContext.appName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getContext( self ):
      return self.appContext

   def onExit( self ):
      t0( "Exit application %s" % self.appName )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      t0( "Abort application %s" % self.appName )
      if self.appContext is not None:
         self.appContext.abort()
      self.appContext = None
      self.session_.gotoParentMode()
      self.session_.mode.abort() # call abort of parent

   def commitContext( self ):
      t0( "Commit application %s" % self.appName )
      if self.appContext:
         self.appContext.commit()
         self.appContext = None

   def commit( self ):
      self.commitContext()
      self.session_.gotoParentMode()
      self.session_.mode.commit() # call commit of parent
