#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliMode

class RoutingBgpBaseMode( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'router bgp %s' % self.param_

   def __init__( self, param ):
      # The industry standard cli has "config-router" as the prompt
      # for this mode.
      self.modeKey = 'router'
      self.longModeKey = 'router-bgp'
      self.addrFamily = 'all'
      CliMode.ConfigMode.__init__( self, param )

   def commentKey( self ):
      return self.longModeKey

class RoutingBgpVrfMode( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'vrf %s' % self.param_

   def __init__( self, param ):
      # The industry standard cli has "config-router" as the prompt
      # for this mode.
      self.modeKey = 'router'
      self.longModeKey = 'router-bgp-vrf-%s' % param
      self.addrFamily = 'all'
      CliMode.ConfigMode.__init__( self, param, parentMode=RoutingBgpBaseMode )

   def commentKey( self ):
      return self.longModeKey

class RoutingBgpBaseAfMode( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'address-family %s' % self.param_

   def __init__( self, param, modeKey=None ):
      # The industry standard cli has "config-router-af" as the prompt
      # for this mode (i.e. no '-v4'/'-v6' suffix).
      self.modeKey = 'router'
      if modeKey:
         self.longModeKey = modeKey
      else:
         self.longModeKey = 'router-bgp-af'
      self.addrFamily = param
      CliMode.ConfigMode.__init__( self, param, parentMode=RoutingBgpBaseMode )

   def commentKey( self ):
      # For labeled-unicast or srte mode we want to append the 'ipv4' or 'ipv6',
      # which is the first part of the addrFamily.
      if 'labeled-unicast' in self.addrFamily:
         return "router-bgp-af-label-%s" % self.addrFamily.split()[ 0 ]
      elif 'sr-te' in self.addrFamily:
         return "router-bgp-af-srte-%s" % self.addrFamily.split()[ 0 ]
      return "router-bgp-af-%s" % self.addrFamily.replace( ' ', '-' )

class RoutingBgpVrfAfMode( CliMode.ConfigMode ):

   def enterCmd( self ):
      return 'address-family %s' % self.param_

   def __init__( self, param, vrfName ):
      self.vrfName_ = vrfName
      # The industry standard cli has "config-router-af" as the prompt
      # for this mode (i.e. no '-v4'/'-v6' suffix).
      self.modeKey = 'router'
      self.longModeKey = 'router-bgp-vrf-%s-af' % vrfName
      self.addrFamily = param
      CliMode.ConfigMode.__init__( self, param, parentMode=RoutingBgpVrfMode )

   def commentKey( self ):
      return "router-bgp-vrf-%s-af-%s" % ( self.vrfName_, self.addrFamily )
