# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import re
import types

import CliParserCommon
from CliParserCommon import MatchResult, noMatch, partialMatch, debugFuncName
from CliParserCommon import Completion

def _checkDeprecatedHelpDesc( helpdesc ):
   if helpdesc and helpdesc.lower() in CliParserCommon.DEPRECATED_HELP_STRINGS:
      newHelpMessage = CliParserCommon.DEPRECATED_HELP_STRINGS[ helpdesc.lower() ]
      msg = ( 'This helpdesc %r is deprecated. Use helpdesc=%r' %
              ( helpdesc, newHelpMessage ) )
      raise ValueError( msg )

class Matcher( object ):
   __slots__ = ( 'helpname_', 'helpdesc_', 'common_', 'priority_',
                 'value_' )

   def __init__( self, helpdesc=None, helpname=None, common=False,
                 priority=CliParserCommon.PRIO_NORMAL, value=None ):
      err = '`%s` must be a string. Got %s instead.'
      if not isinstance( helpdesc, ( str, types.NoneType ) ):
         raise TypeError( err % ( 'helpdesc', type( helpdesc ) ) )
      if not isinstance( helpname, ( str, types.NoneType ) ):
         raise TypeError( err % ( 'helpname', type( helpdesc ) ) )

      self.helpname_ = helpname
      self.helpdesc_ = helpdesc
      _checkDeprecatedHelpDesc( helpdesc )
      _checkDeprecatedHelpDesc( helpname )
      self.common_ = common
      self.priority_ = priority
      self.value_ = value

   def match( self, mode, context, token ):
      # match one token, and return MatchResult.
      #
      # There are two special result values: noMatch meaning this token cannot
      # match; partialMatch meaning we need more tokens.
      #
      # For single-token, return noMatch or MatchResult( result, aaaToken ).
      #
      # For multi-token, we can set context.state (which is initially None)
      # to whatever state necessary to help matching additional tokens or
      # getting completions.
      raise NotImplementedError

   def completions( self, mode, context, token ):
      raise NotImplementedError

   def valueFunction( self, context ):
      return self.value_

   def canMerge( self, otherMatcher ):
      return self.__eq__( otherMatcher )

   def __eq__( self, otherMatcher ):
      return id( self ) == id( otherMatcher )

class KeywordMatcher( Matcher ):
   __slots__ = ( 'keyword_', 'alternates_', 'autoCompleteMinChars_',
                 'completion_' )

   def __init__( self, keyword, helpdesc, alternates=None,
                 autoCompleteMinChars=0, **kargs ):
      kargs.setdefault( 'priority', CliParserCommon.PRIO_KEYWORD )
      super( KeywordMatcher, self ).__init__( helpname=keyword,
                                              helpdesc=helpdesc,
                                             **kargs )
      self.keyword_ = keyword
      self.alternates_ = tuple( alternates ) if alternates else None
      self.autoCompleteMinChars_ = autoCompleteMinChars
      assert self.helpdesc_ is not None, 'KeywordMatcher must have a helpdesc'
      self.completion_ = CliParserCommon.Completion( keyword, self.helpdesc_,
                                             common=self.common_ )

   def match( self, mode, context, token ):
      token = token.lower()
      if token == self.keyword_.lower():
         return MatchResult( self.keyword_, self.keyword_ )

      if self.alternates_:
         for k in self.alternates_:
            if token == k.lower():
               return MatchResult( self.keyword_, self.keyword_ )
      return noMatch

   def completions( self, mode, context, token ):
      token = token.lower()
      if self.keyword_.lower().startswith( token ):
         if len( token ) < self.autoCompleteMinChars_:
            return []
         return [ self.completion_ ]
      return []

   def __str__( self ):
      return self.keyword_

   def canMerge( self, otherMatcher ):
      result = self.__eq__( otherMatcher )
      if result:
         assert self.helpdesc_ == otherMatcher.helpdesc_, \
               ( 'Keyword \'%s\' has 2 different helpdesc: \'%s\' and \'%s\'' %
                     ( self.keyword_, self.helpdesc_, otherMatcher.helpdesc_ ) )
         assert self.common_ == otherMatcher.common_, \
               ( 'Keyword \'%s\' has 2 different common flags: \'%s\' and \'%s\'' %
                     ( self.keyword_, self.common_, otherMatcher.common_ ) )
      return result

   def __eq__( self, otherMatcher ):
      return ( isinstance( otherMatcher, KeywordMatcher ) and
                self.keyword_ == otherMatcher.keyword_ and
                self.alternates_ == otherMatcher.alternates_ and
                self.autoCompleteMinChars_ == otherMatcher.autoCompleteMinChars_ and
                self.priority_ == otherMatcher.priority_ )

class DynamicKeywordMatcher( Matcher ):
   __slots__ = ( 'keywordsFn_', 'emptyTokenCompletion_',
                 'alwaysMatchInStartupConfig_', 'passContext_' )

   def __init__( self, keywordsFn, emptyTokenCompletion=None,
                 alwaysMatchInStartupConfig=False,
                 passContext=False,
                 **kargs ):
      kargs.setdefault( 'priority', CliParserCommon.PRIO_KEYWORD )
      super( DynamicKeywordMatcher, self ).__init__(
         helpdesc='', **kargs )
      self.keywordsFn_ = keywordsFn
      self.passContext_ = passContext
      # Note that emptyTokenCompletion sets literal to False, so that cli doesn't
      # allow the generic token as a valid tab-completion when user hasn't started
      # entering the word yet ( the empty-token case )
      if emptyTokenCompletion is not None:
         for x in emptyTokenCompletion:
            assert x.literal is False
      self.emptyTokenCompletion_ = emptyTokenCompletion
      self.alwaysMatchInStartupConfig_ = alwaysMatchInStartupConfig

   def _getKeywords( self, mode, context ):
      if self.passContext_:
         return self.keywordsFn_( mode, context )
      else:
         return self.keywordsFn_( mode )

   def match( self, mode, context, token ):
      if self.alwaysMatchInStartupConfig_ and mode.session_.startupConfig():
         return MatchResult( token, token )
      keywords = self._getKeywords( mode, context )
      if token in keywords:
         return MatchResult( token, token )
      else:
         # We can't find an exact match, but it may be due to different cases
         # so iterate through all the keys and match again. It could be expensive
         # if there are a lot of keywords.
         token = token.lower()
         for k in keywords:
            if token == k.lower():
               return MatchResult( k, k )
      return noMatch

   def completions( self, mode, context, token ):
      if self.emptyTokenCompletion_ is None:
         completions = []
         # keywordsFn must return a dictionary with help descriptions as value
         for ( k, v ) in self._getKeywords( mode, context ).iteritems():
            _checkDeprecatedHelpDesc( v )
            if k.lower().startswith( token.lower() ):
               completions.append( CliParserCommon.Completion( k, v,
                  common=self.common_ ) )
         return completions
      else: # only care about keywords, not help descriptions
         if token == '':
            return self.emptyTokenCompletion_
         return [ CliParserCommon.Completion( k, k, common=self.common_ )
                  for k in self._getKeywords( mode, context )
                  if k.lower().startswith( token.lower() ) ]

   def __str__( self ):
      return '<DynamicKeywordMatcher(%s)>' % debugFuncName( self.keywordsFn_ )

# This is a multi-token matcher. A sort of dynamic keyword matcher that accepts
# keywords that can contain spaces. It matches a list of keyword list. Thus the
# name of KeywordLists (note the plural on lists). We call that list of list
# "paths" below, and must be an iterable that returns a list of keyword/tokens.
# So paths could be [ ["one", "two"], ["one", "two", "three"] ] if both "one two"
# as well as "one two three" can match. In that case, paths is the unrolling of the
# syntax "one two [three]"...
# For making those keywordlists dynamic, since creating an iterator is more annoying
# that creating a function, "paths" can also be a function that returns the lists
class KeywordListsMatcher( Matcher ):

   def __init__( self, paths, **kargs ):
      if "helpdesc" not in kargs:
         kargs[ 'helpdesc' ] = '%s'
      super( KeywordListsMatcher, self ).__init__( **kargs )
      self.allPaths = paths

   def __str__( self ):
      return '(%s)' % "|".join( [ " ".join( p ) for p in self.allPaths ] )

   def getState( self, mode, context ):
      if not context.state: # first token (of potentially multiple): init state
         index = 0 # iteration of the matcher call (current parse index in path)
         paths = self.allPaths # initial paths, will shrink at each iteration
         if callable( paths ):
            paths = paths( mode )
         tokens = [] # holds the full resulting match, grows at each iteration
         return ( index, paths, tokens )
      else: # multi-token, second round: pick up the last state from the context
         return context.state

   def match( self, mode, context, token ):
      ( index, paths, tokens ) = self.getState( mode, context )
      # now look for matches in our remaining paths at the current index
      matches = []
      canEatMore = False # when at least 1 matching path can consume even more tokens
      terminal = False # when at least 1 matched token is terminal in its path
      toRm = [] # paths with terminal matches will be removed from next iteration
      for path in paths:
         if path[ index ] == token:
            matches.append( path )
            if len( path ) == index + 1:
               terminal = True
               toRm.append( path ) # no longer needed in next round
            else:
               canEatMore = True
      if not matches:
         return CliParserCommon.noMatch
      # Hurray, we got a match: figure out what to return and store state
      # grow our tokens by the extra token of this iteration
      tokens.append( token )
      for path in toRm: # remove all terminal matches (consumed now)
         matches.remove( path )
      # remember our state
      context.state = ( index + 1, matches, tokens )
      # now return results
      if not canEatMore: # we consumed last token in all remaining paths
         # the "False" below means "cannot accept more tokens"
         return CliParserCommon.MatchResult( tokens, tokens, False )
      else: # we can accept more tokens beyond this one
         # Unless one of our matches is terminal, we must say "partialMatch",
         # (if there are no further tokens, that's a syntax error)
         # otherwise we can return a match with a qualifier of "can eat more"
         if not terminal: # more tokens are mandatory
            return CliParserCommon.partialMatch
         else: # that could be it, but a longer match is also still possible
            # say "one two" and "one two three" are registered and user input is
            # "one two" -> we have a match AND we can take more tokens ("three")
            return CliParserCommon.MatchResult( tokens, tokens, True )

   def completions( self, mode, context, token ):
      cs = []
      ( index, paths, _ ) = self.getState( mode, context )
      for p in paths:
         if p[ index ].startswith( token ):
            c = CliParserCommon.Completion( p[ index ],
                                            self.helpdesc_ % p[ index ] )
            if c not in cs:
               cs.append( c )
      return cs

class EnumMatcher( Matcher ):
   __slots__ = ( 'keywords_', 'allKeywords_', 'completions_' )

   def __init__( self, keywordsHelpDescMapping, hiddenHelpDescMapping=None,
         **kargs ):
      super( EnumMatcher, self ).__init__( priority=CliParserCommon.PRIO_HIGH,
                                           helpdesc='', **kargs )
      if hiddenHelpDescMapping is None:
         hiddenHelpDescMapping = {}
      self.keywords_ = keywordsHelpDescMapping.keys()
      self.allKeywords_ = self.keywords_ + hiddenHelpDescMapping.keys()
      self.completions_ = []
      for name, helpdesc in keywordsHelpDescMapping.iteritems():
         _checkDeprecatedHelpDesc( helpdesc )
         self.completions_.append( CliParserCommon.Completion( name, helpdesc ) )

      for name, helpdesc in hiddenHelpDescMapping.iteritems():
         _checkDeprecatedHelpDesc( helpdesc )

   def getValidKeywords( self, context ):
      return self.allKeywords_

   def getValidCompletions( self, context ):
      return self.completions_

   def match( self, mode, context, token ):
      keywords = self.getValidKeywords( context )
      if token in keywords:
         return MatchResult( token, token )
      else:
         # We can't find an exact match, but it may be due to different cases
         # so iterate through all the keys and match again. It could be expensive
         # if there are a lot of keywords.
         token = token.lower()
         for k in keywords:
            if token == k.lower():
               return MatchResult( k, k )
      return noMatch

   def completions( self, mode, context, token ):
      token = token.lower() # TODO: we should remove this call
      completions = self.getValidCompletions( context )
      return [ c for c in completions if c.name.lower().startswith( token ) ]

   def __str__( self ):
      return '|'.join( self.keywords_ )

class PatternMatcher( Matcher ):
   __slots__ = ( 're_', 'rawResult_', 'partialRe_', 'completion_' )

   def __init__( self, pattern, partialPattern=None,
                 rawResult=False, **kargs ):
      super( PatternMatcher, self ).__init__( **kargs )
      assert 'helpname' in kargs, 'PatternMatcher requires a helpname'
      # Note that it is important to append '$' to these patterns rather than simply
      # testing whether a returned match object matches the entire token, as the
      # latter doesn't work correctly when the pattern contains '|' characters (since
      # '|' characters are not greedy).
      self.re_ = re.compile( '(?:%s)$' % pattern )
      self.rawResult_ = rawResult
      partialPattern = ( partialPattern or
                         CliParserCommon.makePartialPattern( pattern ) )
      self.partialRe_ = re.compile( '(%s)$' % partialPattern )
      self.completion_ = CliParserCommon.Completion( self.helpname_, self.helpdesc_,
                                                     False, common=self.common_ )

   def match( self, mode, context, token ):
      m = self.re_.match( token )
      if m:
         return MatchResult( m if self.rawResult_ else token, token )
      return noMatch

   def completions( self, mode, context, token ):
      m = self.partialRe_.match( token )
      if m:
         return [ self.completion_ ]
      return []

   def __str__( self ):
      return self.re_.pattern

class DynamicNameMatcher( PatternMatcher ):
   __slots__ = ( 'namesFn_', 'passContext_' )
   """Accept a name, and do autocompletion for existing names."""
   def __init__( self, namesFn, helpdesc, pattern=r'[A-Za-z0-9_:{}\[\]-]+',
                 partialPattern=None, helpname='WORD', passContext=False,
                 **kargs ):
      super( DynamicNameMatcher, self ).__init__( pattern=pattern,
                                                  partialPattern=partialPattern,
                                                  helpdesc=helpdesc,
                                                  helpname=helpname,
                                                  **kargs )
      self.namesFn_ = namesFn
      self.passContext_ = passContext

   def completions( self, mode, context, token ):
      patternCompletions = PatternMatcher.completions( self, mode, context,
                                                       token )
      if not ( patternCompletions and token ):
         # no completion for '?'
         return patternCompletions
      # Add existing matching names to completions (case-sensitive)
      return [ CliParserCommon.Completion( k, k ) for k in
               ( self.namesFn_( mode, context ) if self.passContext_ else
                 self.namesFn_( mode ) )
               if k.startswith( token ) ] + patternCompletions

   def __str__( self ):
      return '<DynamicNameMatcher(%s)>' % debugFuncName( self.namesFn_ )

class StringMatcher( Matcher ):
   __slots__ = ( 're_', 'partialRe_', 'completion_' )

   def __init__( self, pattern='.+', helpname='LINE', helpdesc=None, **kargs ):
      assert helpdesc is not None, "helpdesc required for StringMatcher"
      kargs[ 'priority' ] = kargs.get( 'priority', CliParserCommon.PRIO_LOW )
      Matcher.__init__( self, helpname=helpname, helpdesc=helpdesc, **kargs )
      self.re_ = re.compile( r'(?:%s)$' % pattern ) # "(?:...)" means non-grouping.
      partialPattern = CliParserCommon.makePartialPattern( pattern )
      self.partialRe_ = re.compile( '(%s)$' % partialPattern )
      self.completion_ = CliParserCommon.Completion( self.helpname_,
                                                     self.helpdesc_,
                                                     False,
                                                     common=self.common_ )

   def match( self, mode, context, token ):
      # with a string rule we always consume all of the tokens.
      if token:
         # token can be empty if passed in from QuotedStringMatcher
         m = self.re_.match( token )
         if not m:
            return noMatch
      if not context.state:
         context.state = []
      context.state.append( token )
      return MatchResult( ' '.join( context.state ), context.state,
                          more=True )

   def completions( self, mode, context, token ):
      if self.partialRe_.match( token ):
         return [ self.completion_ ]
      return []

   def __str__( self ):
      return '<string>'

class QuotedStringMatcher( StringMatcher ):
   __slots__ = ( 'requireQuote_', )
   # Matches one word or quoted multi-word

   def __init__( self, pattern='.+', helpname='QUOTED LINE',
                 helpdesc='A quoted string or a single unquoted word',
                 requireQuote=False, **kargs ):
      StringMatcher.__init__( self, pattern=pattern,
                              helpname=helpname, helpdesc=helpdesc,
                              **kargs )
      self.requireQuote_ = requireQuote

   def _parseToken( self, context, token ):
      # return ( token, finished ). token can be noMatch.
      finished = False
      if context.state is None:
         # first token, check if we have quotes
         if token.startswith( '"' ):
            if token.endswith( '"' ) and len( token ) > 1:
               token = token[ 1 : -1 ]
               finished = True
            else:
               token = token[ 1 : ]
         elif self.requireQuote_:
            return noMatch, True
         else:
            # unquoted token, we are done
            finished = True
      else:
         if token.endswith( '"' ):
            token = token[ : -1 ]
            finished = True

      return token, finished

   def match( self, mode, context, token ):
      token, finished = self._parseToken( context, token )
      if token is noMatch:
         return noMatch

      # call StringMatcher
      r = StringMatcher.match( self, mode, context, token )
      if r is noMatch:
         return r

      assert r.more

      if finished:
         r.more = False
         # AAA tokens are quoted
         r.aaaTokens[ 0 ] = '"' + r.aaaTokens[ 0 ]
         r.aaaTokens[ -1 ] = r.aaaTokens[ -1 ] + '"'
         return r
      else:
         return partialMatch

   def completions( self, mode, context, token ):
      if token:
         token, _ = self._parseToken( context, token )
      if token is not noMatch:
         return StringMatcher.completions( self, mode, context, token )
      return []

   def __str__( self ):
      return '<QuotedString>'

class _RangeMatcherBase( Matcher ):
   def completions( self, mode, context, token ):
      lbound, ubound = self._computeRange( mode )
      if lbound > ubound:
         return []

      if token == '' or ( self.match( mode, context, token ) is not noMatch ):
         helpname = self.helpname_ or self._getFmtStr() % ( lbound, ubound )
         return [ CliParserCommon.Completion( helpname, self.helpdesc_,
                                              False, common=self.common_ ) ]
      return []

   def _computeRange( self, mode ):
      raise NotImplementedError

   def _getFmtStr( self ):
      raise NotImplementedError

   def toValue( self, token ):
      raise NotImplementedError

   def match( self, mode, context, token ):
      val = self.toValue( token )
      if val is not None:
         lbound, ubound = self._computeRange( mode )
         if lbound <= val <= ubound:
            return MatchResult( val, str( val ) )
      return noMatch

class _IntegerMatcherBase( _RangeMatcherBase ):
   # Regular expressions for the formats of integers we accept.
   decRegEx = re.compile( r'^-?[0-9]+$' )
   hexRegEx = re.compile( r'^0[xX][0-9a-fA-F]+$' )

   def toValue( self, token ):
      # Convert our token to base-10 or base-16 integer.
      if self.decRegEx.match( token ):
         return int( token )

      if self.hexRegEx.match( token ):
         return int( token, 16 )

      return None

   def _getFmtStr( self ):
      return '<%d-%d>'

   # Note that we assume that calling self._computeRange() may be an expensive
   # operation, and therefore we go to some effort within this class to minimize the
   # number of times it is called.
   def _computeRange( self, mode ):
      raise NotImplementedError

class _FloatMatcherBase( _RangeMatcherBase ):
   def __init__( self, *args, **kargs ):
      self.precisionString_ = kargs.pop( 'precisionString' )
      err = "Expected format string such as '%%.1f', but got %r instead."
      if not isinstance( self.precisionString_, str ):
         raise TypeError( err % type( self.precisionString_ ) )
      try:
         self.precisionString_ % 3.1415
      except TypeError: # Not all arguments converted or not enough arguments.
         raise ValueError( err % self.precisionString_ )
      super( _FloatMatcherBase, self ).__init__( *args, **kargs )

   def _getFmtStr( self ):
      return '<%s-%s>' % ( self.precisionString_, self.precisionString_ )

   def toValue( self, token ):
      try:
         return float( token )
      except ValueError:
         return None

   def _computeRange( self, mode ):
      raise NotImplementedError

def NumberMatcherFactory( baseClass ):
   class StaticMatcher( baseClass ):
      def __init__( self, lbound, ubound, **kargs ):
         if lbound > ubound:
            raise ValueError( 'Empty range. %s > %s.' % ( lbound, ubound ) )
         super( StaticMatcher, self ).__init__( **kargs )
         self.lbound_ = lbound
         self.ubound_ = ubound

      def _computeRange( self, mode ):
         return self.lbound_, self.ubound_

      def __str__( self ):
         return self._getFmtStr() % ( self.lbound_, self.ubound_ )

   class DynamicMatcher( baseClass ):
      def __init__( self, rangeFn, **kargs ):
         if not callable( rangeFn ):
            raise TypeError( '`rangeFn` must be callable' )
         super( DynamicMatcher, self ).__init__( **kargs )
         self.rangeFn_ = rangeFn
         self.template_ = '<Dynamic%sMatcher>(%%s)'
         self.template_ %= 'Float' if 'precisionString' in kargs else 'Integer'

      def _computeRange( self, mode ):
         return self.rangeFn_( mode )

      def __str__( self ):
         return self.template_ % debugFuncName( self.rangeFn_ )

   return StaticMatcher, DynamicMatcher

class WrapperMatcher( Matcher ):
   # matcher wrapping another matcher with a different priority or value function
   __slots__ = ( 'matcher_', )

   def __init__( self, matcher, priority=None, value=None ):
      # by default use the underlying matcher's priority and value function
      self.matcher_ = matcher
      if priority is None:
         priority = matcher.priority_
      Matcher.__init__( self, priority=priority, value=value )

   def match( self, mode, context, token ):
      return self.matcher_.match( mode, context, token )

   def completions( self, mode, context, token ):
      return self.matcher_.completions( mode, context, token )

   def valueFunction( self, context ):
      return self.value_ or self.matcher_.valueFunction( context )

class DottedIntegerMatcher( Matcher ):
   """Type of matcher that matches an integer-with-sub format, e.g.,
      x.y, where x is between 'lbound' and 'ubound', and y is between
      'subLbound' and 'subUbound'. This is not a float number, but a
      number format for subinterfaces for example. All ranges are
      inclusive.
   """

   __slots__ = ( 'lbound_', 'ubound_', 'rangeFn_', 'subLbound_', 'subUbound_' )

   def __init__( self, lbound, ubound, helpname, helpdesc, rangeFn=None,
                 subLbound=CliParserCommon.subLowerBound,
                 subUbound=CliParserCommon.subUpperBound,
                 **kargs ):
      super( DottedIntegerMatcher, self ).__init__( helpname=helpname,
            helpdesc=helpdesc, **kargs )
      assert ( ( rangeFn is None and lbound is not None and ubound is not None ) or
               ( rangeFn is not None and lbound is None and ubound is None ) ),\
             ( "DottedIntegerMatcher takes either rangeFn or lbound and ubound, "
               "but not both" )
      self.lbound_ = lbound
      self.ubound_ = ubound
      self.rangeFn_ = rangeFn
      self.subLbound_ = subLbound
      self.subUbound_ = subUbound

   def _computeRange( self, mode ):
      if self.rangeFn_:
         return self.rangeFn_( mode )
      return self.lbound_, self.ubound_

   def match( self, mode, context, token ):
      if token.count( '.' ) != 1:
         return noMatch

      # In case of multi-range like et1.1-2,1.4, this rule should not accept.
      ( v1, v2 ) = token.split( '.' )
      try:
         ( v1, v2 ) = ( int( v1 ), int( v2 ) )
      except ValueError:
         return noMatch
      lbound, ubound = self._computeRange( mode )
      if ( lbound <= v1 <= ubound and self.subLbound_ <= v2 <= self.subUbound_ ):
         return MatchResult( token, token )
      return noMatch

   def completions( self, mode, context, token ):
      try:
         val = int( token.split( '.' )[ 0 ] ) if token != '' else 0
      except ValueError:
         return []
      lbound, ubound = self._computeRange( mode )
      if token == '' or lbound <= val <= ubound:
         if '.' not in token:
            return [ Completion( self.helpname_ or self.__str__(),
                                 self.helpdesc_, False ) ]
         elif ( token[ -1 ] == '.' or
                   ( self.match( mode, context, token ) is not noMatch ) ):
            return [ Completion( self.helpname_ or self.__subRangeStr__(),
                                 self.helpdesc_, False ) ]

      return []

   def __str__( self ):
      if self.rangeFn_:
         return '<DynamicDottedIntegerMatcher(%s)>' % debugFuncName( self.rangeFn_ )
      return '<%d-%d>.<%d-%d>' % ( self.lbound_, self.ubound_,
                                   self.subLbound_, self.subUbound_ )

   def __subRangeStr__( self ):
      return '<%d-%d>' % ( self.subLbound_, self.subUbound_ )

IntegerMatcher, DynamicIntegerMatcher = NumberMatcherFactory( _IntegerMatcherBase )
FloatMatcher, DynamicFloatMatcher = NumberMatcherFactory( _FloatMatcherBase )
