#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

__writeEnabled__ = True

def enableWrite( enabled ):
   global __writeEnabled__
   __writeEnabled__ = enabled

class CliGlobal( object ):
   """This class allows defining global CLI variables that are initialized
   at the beginning (before plugins are fully loaded). Instead of

      var = None

      def foo(): # typically called by Plugin entry function
         global var
         var = ...

      # and have to add to the allowed list in CliGlobalChecker.py

   you do

      gv = CliGlobal.CliGlobal( dict( var=None ) )

      def foo():
         gv.var = ...

   There are two benefits:
   1. CliGlobal cannot be modified after CLI initialization is completed.
      This avoids race conditions caused by multithreading in ConfigAgent.
   2. Avoid having to add to the allowed list it in EosImage/test/CliGlobalChecker.py

   """

   def __init__( self, attrs ):
      for k in attrs:
         assert not k.startswith( '__' ), "name cannot start with '__'"
      self.__globalAttrs__ = attrs

   def __getattr__( self, attr ):
      if attr != '__globalAttr__' and attr in self.__globalAttrs__:
         return self.__globalAttrs__[ attr ]
      raise AttributeError( attr )

   def __setattr__( self, attr, val ):
      if attr.startswith( '__' ) or attr in self.__dict__:
         object.__setattr__( self, attr, val )
      elif attr in self.__globalAttrs__:
         assert __writeEnabled__, \
            "CliGlobal variables are only writable during init time"
         self.__globalAttrs__[ attr ] = val
      else:
         raise AttributeError( val )
