#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliExtensionMatchers
import EbnfParser

# vendor fields
REQUIRED_VENDOR_FIELDS = ( 'name', 'email' )
ALLOWED_VENDOR_FIELDS = ( 'name', 'email', 'address', 'phoneNumber' )

# daemon fields
REQUIRED_DAEMON_FIELDS = ( 'exe', )
ALLOWED_DAEMON_FIELDS = ( 'exe', 'heartbeatPeriod' )

# mode fields
REQUIRED_MODE_FIELDS = ( 'command', 'modeKey' )
ALLOWED_MODE_FIELDS = ( 'command', 'modeKey', 'longModeKey', 'daemon' )

# cli commands fields
ALLOWED_CMD_FIELDS = ( ( 'syntax', ( str, ) ),
                       ( 'noSyntax', ( str, ) ),
                       ( 'mode', ( str, ) ), # TODO: Support list of modes?
                       ( 'outputSchema', ( str, dict ) ),
                       ( 'hidden', ( bool, ) ),
                       ( 'data', ( dict, ) ) )
REQUIRED_SHOW_FIELDS = ( 'syntax', 'mode' )
REQUIRED_CONFIG_FIELDS = ( 'mode', )
VERBOTEN_SHOW_FIELDS = ( 'noSyntax', )
VERBOTEN_CONFIG_FIELDS = ( 'outputSchema', )
EXEC_MODES = ( 'Unprivileged', 'Privileged' )

class CliExtensionValidationError( Exception ):
   pass

class CliExtensionVendorError( CliExtensionValidationError ):
   pass

class CliExtensionDaemonError( CliExtensionValidationError ):
   def __init__( self, daemonName, msg ):
      message = 'Daemon %s: %s' % ( daemonName, msg )
      super( CliExtensionDaemonError, self ).__init__( message )
      self.daemonName_ = daemonName

   @property
   def daemon( self ):
      return self.daemonName_

class CliExtensionModeError( CliExtensionValidationError ):
   def __init__( self, modeName, msg ):
      message = 'Mode %s: %s' % ( modeName, msg )
      super( CliExtensionModeError, self ).__init__( message )
      self.modeName_ = modeName

   @property
   def mode( self ):
      return self.modeName_

class CliExtensionCmdError( CliExtensionValidationError ):
   def __init__( self, command, msg ):
      message = 'Command %s: %s' % ( command, msg )
      super( CliExtensionCmdError, self ).__init__( message )
      self.command_ = command

   @property
   def command( self ):
      return self.command_

def _checkForOnlyAllowedFields( name, cmdInfo ):
   for field in cmdInfo:
      for allowedField in ALLOWED_CMD_FIELDS:
         if field != allowedField[ 0 ]:
            continue
         expectedType = allowedField[ 1 ]
         if not isinstance( cmdInfo[ field ], expectedType ):
            raise CliExtensionCmdError( name,
                  'Field "%s" saw type "%s", expected "%s"' %
                  ( field, type( cmdInfo[ field ] ), expectedType ) )
         break
      else:
         raise CliExtensionCmdError( name, 'Field "%s" not allowed' % field )

def _checkShowCommandFields( name, cmdInfo ):
   # validate for show commands
   for field in REQUIRED_SHOW_FIELDS:
      if field in cmdInfo:
         continue
      raise CliExtensionCmdError( name,
            'Required field "%s" is not present' % field )

   for field in VERBOTEN_SHOW_FIELDS:
      if field not in cmdInfo:
         continue
      raise CliExtensionCmdError( name,
            'Field "%s" not allowed in a show command' % field )

   mode = cmdInfo[ 'mode' ]
   if mode not in EXEC_MODES:
      raise CliExtensionCmdError( name,
            'Show commands are not supported in "%s" mode' % mode )

   cmdData = cmdInfo.get( 'data', {} )
   if 'show' in cmdData:
      raise CliExtensionCmdError( name,
            'Please omit the leading "show" in the data field' )

   # TODO: how do we valdiate the schema? Maybe do nothing with the schema for now?

def _checkNonShowCommandFields( name, cmdInfo, additionalModes ):
   # validate for config commands
   for field in REQUIRED_CONFIG_FIELDS:
      if field in cmdInfo:
         continue
      raise CliExtensionCmdError( name,
            'Required field "%s" is not present' % field )

   for field in VERBOTEN_CONFIG_FIELDS:
      if field not in cmdInfo:
         continue
      raise CliExtensionCmdError( name,
            'Field "%s" not allowed in a non-show command' % field )

   if ( 'syntax' not in cmdInfo and 'noSyntax' not in cmdInfo ):
      raise CliExtensionCmdError( name,
            'Command requires a "syntax" or "noSyntax" field' )

   mode = cmdInfo[ 'mode' ]
   if cmdInfo[ 'mode' ] in EXEC_MODES and 'noSyntax' in cmdInfo:
      raise CliExtensionCmdError( name,
            'noSyntax is not supported in "%s" mode' % mode )

   if 'noSyntax' in cmdInfo and cmdInfo[ 'noSyntax' ].lower().startswith( 'no ' ):
      # this this command also doubles as the default version so we should omit the
      # leading no.
      raise CliExtensionCmdError( name,
            'Please omit the leading "no " the noSyntax' )

   # TODO: need to validate that it's a valid mode

def _checkMatcherType( name, token, tokenType, args ):
   matcherInfo = CliExtensionMatchers.MATCHERS[ tokenType ]
   requiredArgs = matcherInfo[ 'requiredArgs' ]
   optionalArgs = matcherInfo[ 'optionalArgs' ]
   for requiredArg in requiredArgs:
      if requiredArg not in args:
         raise CliExtensionCmdError( name,
               'Required arg "%s" missing for token "%s"' % ( requiredArg, token ) )

   if args is None:
      # this means that no args were parsed from the customer config.
      # This is only allowed when there were no required args
      return

   for arg in args:
      if arg not in requiredArgs and arg not in optionalArgs:
         raise CliExtensionCmdError( name,
               'Unknown arg "%s" for token "%s"' % ( arg, token ) )
      requiredType = requiredArgs.get( arg, optionalArgs.get( arg ) )
      if not isinstance( args[ arg ], requiredType ):
         raise CliExtensionCmdError( name,
               ( 'Arg "%s" saw type "%s" expected "%s" for token "%s"' %
                  ( arg, type( args[ arg ] ), requiredType, token ) ) )

def _checkDataField( name, cmdInfo ):
   cmdData = cmdInfo.get( 'data', {} )
   tokens = set()
   for syntaxType in ( 'syntax', 'noSyntax' ):
      if syntaxType not in cmdInfo:
         continue
      tokens = tokens.union( EbnfParser.tokenize( cmdInfo[ syntaxType ] ) )

   for token, tokenInfo in cmdData.iteritems():
      if token not in tokens:
         raise CliExtensionCmdError( name,
               'Data field "%s" not present in the syntax' % token )
      if len( tokenInfo ) != 1:
         raise CliExtensionCmdError( name,
               'Data field "%s" should have exactly 1 value' % token )

      tokenType = tokenInfo.keys()[ 0 ]
      if tokenType not in CliExtensionMatchers.MATCHERS:
         raise CliExtensionCmdError( name,
               'Data field "%s" has unknown token type "%s"' % ( token, tokenType ) )

      _checkMatcherType( name, token, tokenType, tokenInfo[ tokenType ] )

def validateCmd( name, cmdInfo, additionalModes=None ):
   if additionalModes is None:
      additionalModes = {}
   _checkForOnlyAllowedFields( name, cmdInfo )

   syntax = cmdInfo.get( 'syntax' )
   if syntax and syntax.lower().startswith( 'show ' ):
      _checkShowCommandFields( name, cmdInfo )
   else:
      _checkNonShowCommandFields( name, cmdInfo, additionalModes )

   _checkDataField( name, cmdInfo )

def valdiateVendorInfo( vendorInfo ):
   for field in vendorInfo:
      if field not in ALLOWED_VENDOR_FIELDS:
         raise CliExtensionVendorError( 'Vendor Field "%s" not allowed' % field )

   for field in REQUIRED_VENDOR_FIELDS:
      if field not in vendorInfo:
         raise CliExtensionVendorError(
               'Vendor Required field "%s" is not present' % field )

def validateMode( modeName, modeInfo ):
   for field in modeInfo:
      if field not in ALLOWED_MODE_FIELDS:
         raise CliExtensionModeError( modeName, 'Field "%s" not allowed' % field )

   for field in REQUIRED_MODE_FIELDS:
      if field not in modeInfo:
         raise CliExtensionModeError( modeName,
               'Required field "%s" is not present' % field )

   # TODO: need to check the command field

def validateEosSdkDaemon( daemonName, daemonInfo ):
   for field in daemonInfo:
      if field not in ALLOWED_DAEMON_FIELDS:
         raise CliExtensionDaemonError( daemonName,
               'Field "%s" not allowed' % field )

   for field in REQUIRED_DAEMON_FIELDS:
      if field not in daemonInfo:
         raise CliExtensionDaemonError( daemonName,
               'Required field "%s" is not present' % field )

   # TODO check that the exe is executable
