# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import sys
import types

import CliMatcher
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.IpGenAddrMatcher as IpGenAddrMatcher
import CliPlugin.MacAddr as MacAddr

# pylint: disable=redefined-builtin

MATCHERS = {}

class CliExtensionMatcherError( Exception ):
   pass

def _checkArgs( args ):
   if not isinstance( args, dict ):
      raise CliExtensionMatcherError( 'Args %r was expected to be a dict' % args )
   for arg, requiredType in args.items():
      if not isinstance( arg, str ):
         raise CliExtensionMatcherError( 'Arg %r was expected to be a string' % arg )
      if not isinstance( requiredType, ( type, tuple ) ):
         raise CliExtensionMatcherError(
               'Arg %r value was expected to be a type or tuple' % arg )
      if isinstance( requiredType, tuple ):
         for i in requiredType:
            if not isinstance( i, type ):
               raise CliExtensionMatcherError(
                     'Arg %r value %r was expected to be a type' % ( arg, i ) )

def registerMatcher( matcherType, matcherGenFunc, requiredArgs=None,
      optionalArgs=None ):
   if requiredArgs is None:
      requiredArgs = {}
   if optionalArgs is None:
      optionalArgs = {}

   for arg in requiredArgs:
      if arg in optionalArgs:
         raise CliExtensionMatcherError(
               'Argument %r is both in requiredArgs and optionalArgs' % arg )

   _checkArgs( requiredArgs )
   _checkArgs( optionalArgs )

   if not callable( matcherGenFunc ):
      raise CliExtensionMatcherError(
            'matcherGenFunc %r is not callable' % matcherGenFunc )

   if matcherType in MATCHERS:
      raise CliExtensionMatcherError(
            'Matcher %r has already been defined' % matcherType )

   MATCHERS[ matcherType ] = {
      'func': matcherGenFunc,
      'requiredArgs': requiredArgs,
      'optionalArgs': optionalArgs
   }

#-------------------------------------------------------------------------------
# keyword matcher
#-------------------------------------------------------------------------------
def _generateKeywordMatcher( token, help=None ):
   helpdesc = token if help is None else help
   return CliMatcher.KeywordMatcher( token, helpdesc=helpdesc )

registerMatcher(
   matcherType='keyword',
   matcherGenFunc=_generateKeywordMatcher,
   requiredArgs=None,
   optionalArgs={
      'help': str
   }
)

#-------------------------------------------------------------------------------
# options matcher
#-------------------------------------------------------------------------------
# TODO: Implement and test

#-------------------------------------------------------------------------------
# setOptions matcher
#-------------------------------------------------------------------------------
# TODO: Implement and test

#-------------------------------------------------------------------------------
# regex matcher
#-------------------------------------------------------------------------------
def _generateRegexMatcher( token, regex=None, help=None ):
   helpdesc = token if help is None else help
   return CliMatcher.PatternMatcher( regex, helpname=token, helpdesc=helpdesc )

registerMatcher(
   matcherType='regex',
   matcherGenFunc=_generateRegexMatcher,
   requiredArgs=None,
   optionalArgs={
      'help': str,
      'regex': str
   }
)

#-------------------------------------------------------------------------------
# integer matcher
#-------------------------------------------------------------------------------

# we define a seperate context to send to the dynamic function so that customers
# don't directly use the mode object. This will allow us to extend what is sent
# to the function over time without causing any incompatabilities
class DynamicNumberContext( object ):
   def __init__( self, mode ):
      self.mode_ = mode

def _getBoundFunc( lbound, ubound ):
   def getBound( ctx, bound ):
      return bound( ctx ) if callable( bound ) else bound

   def rangeFn( mode ):
      ctx = DynamicNumberContext( mode )
      return ( getBound( ctx, lbound ),
               getBound( ctx, ubound ) )
   return rangeFn

def _generateIntegerMatcher( token, help='Integer value', min=None, max=None ):
   if callable( min ) or callable( max ):
      return CliMatcher.DynamicIntegerMatcher( _getBoundFunc( min, max ),
            helpdesc=help )

   lbound = min if min is not None else -sys.maxint - 1
   ubound = max if max is not None else sys.maxint
   return CliMatcher.IntegerMatcher( lbound, ubound, helpdesc=help )

registerMatcher(
   matcherType='integer',
   matcherGenFunc=_generateIntegerMatcher,
   requiredArgs=None,
   optionalArgs={
      'help': str,
      'min': ( int, types.FunctionType ),
      'max': ( int, types.FunctionType )
   }
)

#-------------------------------------------------------------------------------
# float matcher
#-------------------------------------------------------------------------------
def _generateFloatMatcher( token, help='Floating-point value', min=None, max=None ):
   if callable( min ) or callable( max ):
      return CliMatcher.DynamicFloatMatcher( _getBoundFunc( min, max ),
            helpdesc=help, precisionString='%.2g' )

   lbound = min if min is not None else -sys.float_info.max
   ubound = max if max is not None else sys.float_info.max
   return CliMatcher.FloatMatcher( lbound, ubound, helpdesc=help,
         precisionString='%.2g' )

registerMatcher(
   matcherType='float',
   matcherGenFunc=_generateFloatMatcher,
   requiredArgs=None,
   optionalArgs={
      'help': str,
      'min': ( float, types.FunctionType ),
      'max': ( float, types.FunctionType )
   }
)

#-------------------------------------------------------------------------------
# ipv4Address matcher
#-------------------------------------------------------------------------------
def _generateIpv4AddressMatcher( token, help='IPv4 Address' ):
   return IpAddrMatcher.IpAddrMatcher( helpdesc=help )

registerMatcher(
   matcherType='ipv4Address',
   matcherGenFunc=_generateIpv4AddressMatcher,
   requiredArgs=None,
   optionalArgs={
      'help': str,
   }
)

#-------------------------------------------------------------------------------
# ipv6Address matcher
# TODO: This currently returns a tac type, should return a string
#-------------------------------------------------------------------------------
def _generateIpv6AddressMatcher( token, help='IPv6 Address' ):
   return Ip6AddrMatcher.Ip6AddrMatcher( helpdesc=help )

registerMatcher(
   matcherType='ipv6Address',
   matcherGenFunc=_generateIpv6AddressMatcher,
   requiredArgs=None,
   optionalArgs={
      'help': str,
   }
)

#-------------------------------------------------------------------------------
# ipAddress matcher
# TODO: This currently returns a tac type, should return a string
#-------------------------------------------------------------------------------
def _generateIpAddressMatcher( token, help='IPv4 or IPv6 Address' ):
   return IpGenAddrMatcher.IpGenAddrMatcher( helpdesc=help )

registerMatcher(
   matcherType='ipAddress',
   matcherGenFunc=_generateIpAddressMatcher,
   requiredArgs=None,
   optionalArgs={
      'help': str,
   }
)

#-------------------------------------------------------------------------------
# macAddress matcher
#-------------------------------------------------------------------------------
def _generateMacAddressMatcher( token, help='Ethernet address' ):
   return MacAddr.MacAddrMatcher( helpdesc=help )

registerMatcher(
   matcherType='macAddress',
   matcherGenFunc=_generateMacAddressMatcher,
   requiredArgs=None,
   optionalArgs={
      'help': str,
   }
)
