# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import fnmatch
import inspect
import os
import traceback
import yaml

import BasicCli
import CliCommand
import CliExtensionConsts
import CliExtensionMatchers
import CliExtensionValidator
import CliGlobal
import CliMatcher
import CliMode
import CliModel
import CliParser
import CliSave
import ConfigMount
import EbnfParser
import LazyMount
import LauncherDaemonConstants
import LauncherLib
import LauncherUtil
import ShowCommand
from TypeFuture import TacLazyType

AGENT_TYPE = TacLazyType( 'GenericAgent::AgentTypeEnum' )
registeredCommands = {}
loadedDaemons = {}
loadedModes = {}
loadedCommands = {}

DEFAULT_EOSSDK_DAEMON_HEARTBEAT = 0

gv = CliGlobal.CliGlobal( {
   'extensionsLoaded': False,
   'useLocalMounts': False,
   'cliExtensionConfig': None,
   'eossdkConfigDir': None,
   'eossdkStatusDir': None,
   'aclConfigDir': None,
   'launcherConfifgDir': None,
} )

class CliExtensionLoadError( Exception ):
   pass

class CliCommandClass( object ):
   pass

class ShowCommandClass( object ):
   def handler( self, ctx ):
      raise NotImplementedError( 'Handler function must be defined' )

   def render( self, data ):
      print( data )

class CliExtensionCtx( object ):
   def __init__( self, mode, args ):
      self.mode_ = mode
      self.args_ = args

   @property
   def args( self ):
      return self.args_

   def addWarning( self, warning ):
      self.mode_.addWarning( warning )

   def addError( self, error ):
      self.mode_.addError( error )

   def isStartupConfig( self ):
      return self.mode_.startupConfig()

   def isNoOrDefaultCmd( self ):
      return CliCommand.isNoOrDefaultCmd( self.args_ )

   def isNoCmd( self ):
      return CliCommand.isNoCmd( self.args_ )

   def isDefaultCmd( self ):
      return CliCommand.isDefaultCmd( self.args_ )

   @property
   def config( self ):
      # first try to get the config from a daemon
      daemon = self.daemon
      if daemon is not None:
         return self.daemon.config

      if not hasattr( self.mode_, 'getModeName' ):
         return None
      modeName = self.mode_.getModeName()
      return self.getModeConfig( modeName )

   @property
   def status( self ):
      # first try to get the config from a daemon
      daemon = self.daemon
      if daemon is not None:
         return self.daemon.status

      # no other status for source so return None
      return None

   def getModeConfig( self, modeName ):
      if modeName not in gv.cliExtensionConfig.config:
         return None
      return ConfigAccessor( gv.cliExtensionConfig.config, modeName )

   @property
   def daemon( self ):
      if not hasattr( self.mode_, 'getDaemonName' ):
         return None
      daemonName = self.mode_.getDaemonName()
      return self.getDaemon( daemonName )

   def getDaemon( self, daemonName ):
      if daemonName not in loadedDaemons:
         return None
      return DaemonAccessor( daemonName )

class ConfigAccessor( object ):
   def __init__( self, configDir, name ):
      self.configDir_ = configDir
      self.name_ = name

   def configSet( self, key, value='' ):
      config = self.configDir_.get( self.name_ )
      if config is None:
         return
      config.option[ key ] = str( value )

   def configDel( self, key=None ):
      config = self.configDir_.get( self.name_ )
      if config is None:
         return
      del config.option[ key ]

   def config( self, key, defaultVal=None ):
      config = self.configDir_.get( self.name_ )
      if config is None:
         return None

      return config.option.get( key, defaultVal )

   def configIter( self, pattern='*' ):
      """Returns an iterator over the config for matching entries"""
      config = self.configDir_.get( self.name_ )
      if config is None:
         return

      for k, v in config.option.items():
         if fnmatch.fnmatch( k, pattern ):
            yield k, v

   def enable( self ):
      config = self.configDir_.get( self.name_ )
      if config is None:
         return
      config.enabled = True

   def disable( self ):
      config = self.configDir_.get( self.name_ )
      if config is None:
         return
      config.enabled = False

class StatusAccessor( object ):
   def __init__( self, statusDir, name ):
      self.statusDir_ = statusDir
      self.name_ = name

   def status( self, key, defaultVal=None ):
      status = self.statusDir_.get( self.name_ )
      if status is None:
         return None

      return status.data.get( key, defaultVal )

   def statusIter( self, pattern='*' ):
      """Returns an iterator over the status for matching entries"""
      status = self.statusDir_.get( self.name_ )
      if status is None:
         return

      for k, v in status.data.items():
         if fnmatch.fnmatch( k, pattern ):
            yield k, v

class DaemonAccessor( object ):
   def __init__( self, daemonName ):
      self.daemonName_ = daemonName

   @property
   def config( self ):
      return ConfigAccessor( gv.eossdkConfigDir, self.daemonName_ )

   @property
   def status( self ):
      return StatusAccessor( gv.eossdkStatusDir, self.daemonName_ )

def _generateData( syntax, cmdData, isShow=False ):
   data = {}
   for token in EbnfParser.tokenize( syntax ):
      if isShow and token.lower() == 'show':
         continue

      if token == '...':
         # this treated as trailing garbage by the parser we shouldn't
         # try to populate it
         continue

      if token in data:
         # this means that a token is repeated twice in the syntax
         continue

      if token not in cmdData:
         # this as a keyword
         matcher = CliMatcher.KeywordMatcher( token, helpdesc=token )
      else:
         # this assert is validated at a higher level
         assert len( cmdData[ token ] ) == 1
         matcherType = cmdData[ token ].keys()[ 0 ]
         matcherArgs = cmdData[ token ][ matcherType ]
         if matcherArgs is None:
            matcherArgs = {}
         matcherGenFunc = CliExtensionMatchers.MATCHERS[ matcherType ][ 'func' ]
         matcher = matcherGenFunc( token, **matcherArgs )

      # we set canMerge=False because when we merge we also ensure that the
      # helpdescs are the same. If they are not we throw an error. However
      # for customer plugins we don't want them to be beholden to our
      # potentially changing helpdesc. This does lower performance a bit,
      # but customers won't be adding a ton of commands.
      data[ token ] = CliCommand.Node( matcher=matcher, canMerge=False )

   return data

def _registerShowCommand( name, cmdInfo, cmdClass ):
   cmdClassInstance = cmdClass()

   class CustomerShowCmdModel( CliModel.Model ):
      def toDict( self, revision=None, includePrivateAttrs=False, streaming=False ):
         return self.__dict__[ '__data' ]

      def render( self ):
         cmdClassInstance.render( self.__dict__[ '__data' ] )

   hasSchema = 'outputSchema' in cmdInfo

   class CliExtensionShowCmd( ShowCommand.ShowCliCommandClass ):
      syntax = cmdInfo[ 'syntax' ]
      data = _generateData( cmdInfo[ 'syntax' ], cmdInfo.get( 'data', {} ),
            isShow=True )
      privileged = cmdInfo[ 'mode' ] != 'Unprivileged'
      hidden = cmdInfo.get( 'hidden', False )
      cliModel = CustomerShowCmdModel if hasSchema else None

      @staticmethod
      def handler( mode, args ):
         ctx = CliExtensionCtx( mode, args )
         # TODO: should we capture output if we a schema?
         data = cmdClassInstance.handler( ctx )
         if hasSchema:
            if data is None:
               mode.addError( 'Handler has returned no data, but data was '
                              'expected because a schema is registered' )
               return None
            else:
               # TODO: we should validate the data against the schema provided
               model = CustomerShowCmdModel()
               model.__dict__[ '__data' ] = data
               return model
         else:
            # handle a command that doesn't have a defined schema
            if data is not None:
               mode.addError( 'Handler has no output schema but return data' )
               return None
         return None

   BasicCli.addShowCommandClass( CliExtensionShowCmd )

def _getHandlerFn( name, cmdClassInstance ):
   def wrapHandler( handler ):
      def wrappedHandler( mode, args ):
         ctx = CliExtensionCtx( mode, args )
         return handler( ctx )
      return wrappedHandler

   if hasattr( cmdClassInstance, name ):
      return wrapHandler( getattr( cmdClassInstance, name ) )
   return None

def _registerNonShowCommand( name, cmdInfo, cmdClass ):
   cmdClassInstance = cmdClass()
   cmdHandler = _getHandlerFn( 'handler', cmdClassInstance )
   cmdNoHandler = _getHandlerFn( 'noHandler', cmdClassInstance )
   cmdDefaultHandler = _getHandlerFn( 'defaultHandler', cmdClassInstance )

   # if we have a no handler, but a default handler wasn't supplied
   # then we assume that the default handler is the same as the no handler
   if cmdNoHandler and cmdDefaultHandler is None:
      cmdDefaultHandler = cmdNoHandler

   if 'syntax' in cmdInfo and cmdHandler is None:
      raise CliExtensionLoadError(
            'Command Definition %s: Syntax field was defined but no '
            'command handler was defined' % name )
   if 'noSyntax' in cmdInfo and cmdNoHandler is None:
      raise CliExtensionLoadError(
            'Command Definition %s: NoSyntax field was defined but no '
            'command handler was defined' % name )

   class CliExtensionCmd( CliCommand.CliCommandClass ):
      syntax = cmdInfo.get( 'syntax' )
      noOrDefaultSyntax = cmdInfo.get( 'noSyntax' )
      handler = cmdHandler
      noHandler = cmdNoHandler
      defaultHandler = cmdDefaultHandler
      # TODO: we shouldn't rely on syntax always being there
      data = _generateData( cmdInfo[ 'syntax' ], cmdInfo.get( 'data', {} ) )
      hidden = cmdInfo.get( 'hidden', False )

   mode = loadedModes[ cmdInfo[ 'mode' ] ]
   mode.addCommandClass( CliExtensionCmd )

def _createModeCliSavePlugin( modeName, ExtensionCliSaveMode ):
   @CliSave.saver( 'CliExtension::Config', 'cli/extension' )
   # pylint: disable-msg=unused-variable
   def saveExtensionCliConfig( entity, root, sysdbRoot, options ):
      extensionConfig = entity.config.get( modeName )
      if extensionConfig is None:
         return

      if not extensionConfig.option and not extensionConfig.enabled:
         # if the mode is empty don't add any commands
         return

      CliSave.GlobalConfigMode.addChildMode( ExtensionCliSaveMode )
      ExtensionCliSaveMode.addCommandSequence( 'ExtensionCli.config' )
      mode = root[ ExtensionCliSaveMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'ExtensionCli.config' ]
      for k, v in sorted( extensionConfig.option.items(), key=lambda kv: kv[ 0 ] ):
         cmd = '%s %s' % ( k, v )
         cmds.addCommand( cmd.strip() )

      if extensionConfig.enabled:
         cmds.addCommand( 'no disabled' )
   # pylint: enable-msg=unused-variable

def _createDaemonCliSavePlugin( daemonName, ExtensionCliSaveMode ):
   # pylint: disable-msg=unused-variable
   @CliSave.saver( 'Tac::Dir', 'daemon/agent/config' )
   def saveExtensionCliConfig( entity, root, sysdbRoot, options ):
      extensionConfig = entity.get( daemonName )
      if extensionConfig is None:
         return

      if not extensionConfig.option and not extensionConfig.enabled:
         # if the mode is empty don't add any commands
         return

      CliSave.GlobalConfigMode.addChildMode( ExtensionCliSaveMode )
      ExtensionCliSaveMode.addCommandSequence( 'ExtensionCli.config' )
      mode = root[ ExtensionCliSaveMode ].getOrCreateModeInstance( None )
      cmds = mode[ 'ExtensionCli.config' ]
      for k, v in sorted( extensionConfig.option.items(), key=lambda kv: kv[ 0 ] ):
         cmd = '%s %s' % ( k, v )
         cmds.addCommand( cmd.strip() )

      if extensionConfig.enabled:
         cmds.addCommand( 'no disabled' )
   # pylint: enable-msg=unused-variable

def _createCliSavePlugin( modeName, modeInfo, ExtensionCliSaveMode ):
   daemonName = modeInfo.get( 'daemon' )
   if daemonName is not None:
      _createDaemonCliSavePlugin( daemonName, ExtensionCliSaveMode )
   else:
      _createModeCliSavePlugin( modeName, ExtensionCliSaveMode )

def _createEosSdkDaemonLauncherConfig( daemonName ):
   ''' Programs into LauncherConfig '''
   assert daemonName not in gv.launcherConfifgDir
   daemonInfo = loadedDaemons[ daemonName ]
   daemonConfig = gv.launcherConfifgDir.newAgent( daemonName )
   daemonConfig.userDaemon = True
   daemonConfig.heartbeatPeriod = daemonInfo.get( 'heartbeatPeriod',
         DEFAULT_EOSSDK_DAEMON_HEARTBEAT )
   daemonConfig.useEnvvarForSockId = True # True for EosSdk
   daemonConfig.oomScoreAdj = LauncherDaemonConstants.DEFAULT_OOM_SCORE_ADJ
   daemonConfig.exe = daemonInfo[ 'exe' ]

   # Set up the runnability criteria, so the agent only runs when
   # the genericAgentCfg.enabled is True on the application. Meaning only if the
   # application is enabled do we run the daemon
   # (even if we try to start it anyway)
   daemonConfig.runnability = ( 'runnability', )
   daemonConfig.runnability.qualPath = ( 'daemon/agent/runnability/%s' % daemonName )

   for redProto in LauncherUtil.allRedProtoSet:
      daemonConfig.criteria[ redProto ] = LauncherUtil.activeSupervisorRoleName
   daemonConfig.stable = True

def _maybeCreateDaemonConfig( mode, daemonName ):
   genericAgentCfg = gv.eossdkConfigDir.get( daemonName )
   if genericAgentCfg is None:
      genericAgentCfg = gv.eossdkConfigDir.newEntity(
            'GenericAgent::Config', daemonName )
      genericAgentCfg.agentType = AGENT_TYPE.cliExtensionAgent
      gv.aclConfigDir.newEntity( 'Acl::ServiceAclTypeVrfMap', daemonName )
      _createEosSdkDaemonLauncherConfig( daemonName )
   else:
      if genericAgentCfg.agentType != AGENT_TYPE.cliExtensionAgent:
         mode.addError( 'Unable to create daemon \'%s\' because CLI daemon '
                        '\'%s\' already exists. Please delete the daemon'
                        % ( daemonName, daemonName ) )

def _createGotoModeCommandClass( modeName, modeInfo, ExtensionCliMode ):
   cmdInfo = modeInfo[ 'command' ]
   daemonName = modeInfo.get( 'daemon' )

   class GotoExtensionModeCmd( CliCommand.CliCommandClass ):
      syntax = cmdInfo[ 'syntax' ]
      noOrDefaultSyntax = cmdInfo[ 'noSyntax' ]
      data = _generateData( cmdInfo[ 'syntax' ], cmdInfo.get( 'data', {} ) )

      @staticmethod
      def handler( mode, args ):
         if daemonName is not None:
            _maybeCreateDaemonConfig( mode, daemonName )
         else:
            if modeName not in gv.cliExtensionConfig.config:
               gv.cliExtensionConfig.config.newMember( modeName )
         childMode = mode.childMode( ExtensionCliMode )
         mode.session_.gotoChildMode( childMode )

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         if daemonName is not None:
            genericAgentCfg = gv.eossdkConfigDir.get( daemonName )
            if ( genericAgentCfg is None or
                 genericAgentCfg.agentType != AGENT_TYPE.cliExtensionAgent ):
               return
            gv.eossdkConfigDir.deleteEntity( daemonName )
            gv.aclConfigDir.deleteEntity( daemonName )
            del gv.launcherConfifgDir[ daemonName ]
         else:
            del gv.cliExtensionConfig.config[ modeName ]

   BasicCli.GlobalConfigMode.addCommandClass( GotoExtensionModeCmd )

def _createCliMode( modeName, modeInfo ):
   modeKey = modeInfo[ 'modeKey' ]
   longModeKey = modeInfo.get( 'longModeKey', modeKey )

   class ExtensionBaseMode( CliMode.ConfigMode ):
      def __init__( self, param ):
         self.modeKey = modeKey
         self.longModeKey = longModeKey
         CliMode.ConfigMode.__init__( self, param )

      def enterCmd( self ):
         return modeInfo[ 'command' ][ 'syntax' ]

   class ExtensionCliMode( ExtensionBaseMode, BasicCli.ConfigModeBase ):
      name = longModeKey
      modeParseTree = CliParser.ModeParseTree()

      def __init__( self, parent, session ):
         ExtensionBaseMode.__init__( self, None )
         BasicCli.ConfigModeBase.__init__( self, parent, session )

      def getDaemonName( self ):
         return modeInfo.get( 'daemon' )

      def getModeName( self ):
         return modeName

   class ExtensionCliSaveMode( ExtensionBaseMode, CliSave.Mode ):
      def __init__( self, param ):
         ExtensionBaseMode.__init__( self, param )
         CliSave.Mode.__init__( self, param )

   CliSave.GlobalConfigMode.addChildMode( ExtensionCliSaveMode )
   ExtensionCliSaveMode.addCommandSequence( 'ExtensionCli.config' )

   _createCliSavePlugin( modeName, modeInfo, ExtensionCliSaveMode )
   _createGotoModeCommandClass( modeName, modeInfo, ExtensionCliMode )

   return ExtensionCliMode

def loadSyntaxYaml( yamlContents, checkVendor=False ):
   extContentsJson = yaml.load( yamlContents )

   if checkVendor:
      if 'vendor' not in extContentsJson:
         raise CliExtensionLoadError( 'Vendor information is missing' )
      CliExtensionValidator.valdiateVendorInfo( extContentsJson[ 'vendor' ] )

   for daemonName, daemonInfo in extContentsJson.get( 'eosSdkDaemons', {} ).items():
      if daemonName in loadedDaemons:
         raise CliExtensionLoadError(
               'Daemon Definition %s: Duplicate daemon name' %
               daemonName )
      CliExtensionValidator.validateEosSdkDaemon( daemonName, daemonInfo )
      loadedDaemons[ daemonName ] = daemonInfo

   loadedModes[ 'Privileged' ] = BasicCli.EnableMode
   loadedModes[ 'Unprivileged' ] = BasicCli.UnprivMode
   for modeName, modeInfo in extContentsJson.get( 'modes', {} ).items():
      if modeName in loadedModes:
         raise CliExtensionLoadError(
               'Mode Definition %s: Mode name has already been seen' % modeName )
      CliExtensionValidator.validateMode( modeName, modeInfo )
      loadedModes[ modeName ] = _createCliMode( modeName, modeInfo )

   for cmdName, cmdInfo in extContentsJson.get( 'commands', {} ).items():
      if cmdName in loadedCommands:
         raise CliExtensionLoadError(
               'Command Definition %s: Command name has already been seen' %
               cmdName )
      if cmdName not in registeredCommands:
         raise CliExtensionLoadError(
               'Command Definition %s: Command is missing a command handler' %
               cmdName )

      CliExtensionValidator.validateCmd( cmdName, cmdInfo )
      loadedCommands[ cmdName ] = cmdInfo # To keep track of duplicates

      # register the command with the parser!
      cmdClass = registeredCommands[ cmdName ]
      if issubclass( cmdClass, ShowCommandClass ):
         _registerShowCommand( cmdName, cmdInfo, cmdClass )
      else:
         _registerNonShowCommand( cmdName, cmdInfo, cmdClass )

def _loadSyntaxFile( path ):
   try:
      with open( path ) as f:
         contents = f.read()
   except EnvironmentError as e:
      print( 'Error: Unable to load syntax file %s due to: %s' % ( path, e ) )
      return

   try:
      loadSyntaxYaml( contents, checkVendor=True )
   except ( yaml.YAMLError,
            CliExtensionValidator.CliExtensionValidationError,
            CliExtensionLoadError ) as e:
      print( 'Error: Unable to parse syntax file %s due to: %s' % ( path, e ) )
      traceback.print_exc()

def initCliExtensions( entityManager ):
   if gv.extensionsLoaded:
      return
   gv.extensionsLoaded = True

   if not gv.useLocalMounts:
      gv.cliExtensionConfig = ConfigMount.mount( entityManager,
            'cli/extension', 'CliExtension::Config', 'wi' )
      gv.eossdkConfigDir = ConfigMount.mount( entityManager,
            'daemon/agent/config', 'Tac::Dir', 'wi' )
      gv.eossdkStatusDir = LazyMount.mount( entityManager,
            'daemon/agent/status', 'Tac::Dir', 'ri' )
      gv.aclConfigDir = ConfigMount.mount( entityManager,
            'daemon/acl/config', 'Tac::Dir', 'wi' )
      gv.launcherConfifgDir = ConfigMount.mount( entityManager,
            LauncherLib.agentConfigCliDirPath, 'Launcher::AgentConfigDir', 'wi' )
   else:
      gv.cliExtensionConfig = entityManager.root()[ 'cli' ][ 'extension' ]
      gv.eossdkConfigDir = entityManager.root()[ 'daemon' ][ 'agent' ][ 'config' ]
      gv.eossdkStatusDir = entityManager.root()[ 'daemon' ][ 'agent' ][ 'status' ]
      gv.aclConfigDir = entityManager.root()[ 'daemon' ][ 'acl' ][ 'config' ]
      gv.launcherConfifgDir = LauncherLib.agentConfigCliDir( entityManager.root() )

   _loadCliExtensions()

def _loadCliExtensions():
   cliExtensionDir = os.environ.get( 'CLI_EXTENSION_DIR',
         CliExtensionConsts.CLI_EXTENSION_DIR )
   try:
      if not os.path.isdir( cliExtensionDir ):
         # no extensions: nothing to do
         return

      for f in os.listdir( cliExtensionDir ):
         if not f.endswith( '.yaml' ):
            continue
         path = os.path.join( cliExtensionDir, f )
         _loadSyntaxFile( path )
   except Exception as e: # pylint: disable-msg=broad-except
      # ideally we should never hit here, but alas we can't always get what we want
      print( 'Exception', e )
      traceback.print_exc()

def registerCommand( name, cmdClass ):
   if not isinstance( name, str ):
      raise CliExtensionLoadError(
            'Unable to add Command Class %r'
            'due to invalid name %r' % ( cmdClass, name ) )

   if name in registeredCommands:
      raise CliExtensionLoadError(
            'Unable to register command "%s" because Command Class "%s" already '
            'registered with the same name' % ( name, registeredCommands[ name ] ) )

   if ( not inspect.isclass( cmdClass ) or
        ( not issubclass( cmdClass, ShowCommandClass ) and
          not issubclass( cmdClass, CliCommandClass ) ) ):
      raise CliExtensionLoadError(
            'Unable to register command "%s" because the Command Class "%s" does '
            'not inherit from "CliExtension.ShowCommandClass" or '
            '"CliExtension.CliCommandClass"' % ( name, cmdClass ) )

   registeredCommands[ name ] = cmdClass
