# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import copy
import inspect

import CliMatcher
import CliParserCommon
from CliParserStructs import ParserNode
import EbnfParser
import Tracing

th = Tracing.defaultTraceHandle()
t0 = th.trace0
t1 = th.trace1
t2 = th.trace2

# pylint: disable-msg=unsupported-assignment-operation

NAME_NO = '__no__'
NAME_DEFAULT = '__default__'

def isDefaultCmd( args ):
   return NAME_DEFAULT in args

def isNoCmd( args ):
   return NAME_NO in args

def isNoOrDefaultCmd( args ):
   return NAME_DEFAULT in args or NAME_NO in args

class Node( object ):
   '''User defined part of a parser ParserNode. This allows a user to add more
   control over how the internal ParserNode is created.
   guard: A function to dynamically not a command
   hidden: Will disable auto-complete for this node
   sensitive: Will replace the token with * in the aaaTokens
              (used for things like passwords)
   maxMatches: Defines how many times a particular node can be matched
   sharedMatchObj: if specified, can be shared by multiple nodes to shared the
                   match count.
   alias: name that is passed into the value function
   deprecatedByCmd: If this node matches a GuardError will be raised
                    (unless in the startup-config) and with the new command that
                    should be used instead.
   noResult: If the result of this node is not added to the result list
   storeSharedResult: Whether we store match result in sharedData of the context
   overrideCanMerge: Generically there are a bunch of types of syntaxes that
                     can't merge. For example 'a ( b | c ) d' and 'a b d e'
                     can't merge because we would introduce the syntax
                     a c d e. Maybe in the future we can improve our simple
                     merge algorithm, however for now if we want to merge something
                     like 'a ( b | c ) d' and 'a ( b | c ) e' we would have to mark
                     b and c as overrideCanMerge. However care must be taken for
                     a syntax like 'a b d e' not to be added, so please use with
                     caution. Typically should be use if all of the commands share
                     a common root.
   canMerge: Defaults to true, can be set to false to prevent tokens with same name
             from merging (instead of naming keyword tokens in the syntax differently
             than the keyword itself, like done sometimes to prevent 2 keywords
             with a different guard from merging).
             Note: the overrideCanMerge above is a missnomer, should be
             overrideCannotMerge (sone syntax nodes will be marked as unmergable by
             the EBNF parser, but we can override that determination and allow them
             to merge nonetheless in special cases where merging is still ok).
   '''
   __slots__ = ( 'matcher_', 'guard_', 'hidden_', 'sensitive_', 'maxMatches_',
                 'sharedMatchObj_', 'alias_', 'deprecatedByCmd_', 'noResult_',
                 'storeSharedResult_', 'overrideCanMerge_', 'canMerge_' )

   def __init__( self, matcher, guard=None, hidden=False, sensitive=False,
                 maxMatches=0, sharedMatchObj=None, alias=None,
                 deprecatedByCmd=None, noResult=False, storeSharedResult=False,
                 overrideCanMerge=False, canMerge=True ):
      assert isinstance( matcher, CliMatcher.Matcher )
      self.matcher_ = matcher
      if guard is not None:
         assert callable( guard ), 'Guard function must be callable'
      self.guard_ = guard
      self.hidden_ = hidden
      self.sensitive_ = sensitive
      self.maxMatches_ = maxMatches
      self.sharedMatchObj_ = sharedMatchObj
      self.alias_ = alias
      self.deprecatedByCmd_ = deprecatedByCmd
      self.noResult_ = noResult
      self.storeSharedResult_ = storeSharedResult
      self.overrideCanMerge_ = overrideCanMerge
      self.canMerge_ = canMerge

def getBoundedMethod( handler ):
   if ( hasattr( handler, 'im_self' ) and handler.im_self is None and
        issubclass( handler.im_class, _CliCommandCommonBase ) ):
      # This can happen if someone does the following in a command
      # class:
      #
      # handler = <some-global-function>
      #
      # This makes handler unbound, but we can just retrieve the underlying
      # function.
      handler = handler.im_func
   assert callable( handler ), "Function must be callable"
   return handler

class _CommandHandlerClass( object ):
   __slots__ = ( 'authz', 'syncAcct', 'acct', 'allowCache', 'authzFunc', 'acctFunc',
                 'autoConfigSessionAllowed', 'handler', 'cmdFilter', 'cmdSyntax',
                 'originalCls' )

   def __init__( self, authz, syncAcct, acct, allowCache, authzFunc, acctFunc,
                 autoConfigSessionAllowed, cmdFilter, handler, cmdSyntax,
                 originalCls ):
      self.authz = authz
      self.syncAcct = syncAcct
      self.acct = acct
      self.allowCache = allowCache
      self.authzFunc = authzFunc
      self.acctFunc = acctFunc
      self.autoConfigSessionAllowed = autoConfigSessionAllowed
      self.handler = getBoundedMethod( handler )
      self.cmdFilter = cmdFilter
      self.cmdSyntax = cmdSyntax
      self.originalCls = originalCls

   def __call__( self, *args, **kwargs ):
      return self.handler( args, kwargs )

class _CliCommandCommonBase( object ):
   ALLOWED_FIELDS = ()
   REQUIRED_FIELDS = ()
   data = None

   @classmethod
   def _assert( cls, condition, msg ):
      assert condition, "%s: %s" % ( cls.__name__, msg )

   @classmethod
   def _validateFields( cls ):
      # check to make sure that unknown fields aren't also being defined
      for i in cls.__dict__:
         if not i.startswith( '_' ) and not i.endswith( '_' ):
            cls._assert( i in cls.ALLOWED_FIELDS,
                         "Unknown field '%s' found; non-standard field must "
                         "start or end with '_'" % i )
      for field in cls.REQUIRED_FIELDS:
         assert getattr( cls, field, None ) is not None, \
            'Derived class must define % s' % field

   @classmethod
   def _checkData( cls ):
      for name, value in cls.data.iteritems():
         try:
            cls._assert( ( isinstance( value, ( CliMatcher.Matcher, Node, str,
                                                CliExpressionFactory ) ) or
                           inspect.isclass( value ) and
                           issubclass( value, CliExpression ) ),
                         "data '%s' must be an instance of Matcher, Node, string, "
                         "CliExpressionFactory or a subclass of CliExpression"
                         % name )
         except TypeError:
            # In `data`, if a value is a Rule, rather than a Matcher,
            # we get an unclear error from `issubclass` that `matcher`
            # is not a class or type.
            # This `except` will be unnecessary once the old parser is gone.
            cls = name.__class__.__name__
            assert False, 'Token %r is not a Matcher, but %s.' % ( name, cls )

   @classmethod
   def _updateSyntax( cls, field ):
      # pylint: disable=no-value-for-parameter
      setattr( cls, field, cls._expandSyntax( getattr( cls, field ) ) )
      # pylint: enable=no-value-for-parameter

   @classmethod
   def _getExpression( cls, key, value ):
      if inspect.isclass( value ):
         if issubclass( value, CliExpression ):
            return value
      elif isinstance( value, CliExpressionFactory ):
         return value.get( key )
      return None

   @classmethod
   def _expandSyntax( cls, syntax ):
      # Given a syntax it will return an expanded form of that syntax. Meaning
      # if that syntax has subexpressions in it, the syntax here will be replaced
      # with the expression given. This needs to be done before the data field is
      # is also expanded, because once that is done we lose information on the
      # subexpression.
      # pylint: disable-msg=protected-access
      for key, value in cls.data.iteritems():
         value = cls._getExpression( key, value )
         if value is not None:
            newSyntaxTokens = []
            for token in EbnfParser.tokenize( syntax ):
               if token in EbnfParser.EBNF_TOKENS:
                  newSyntaxTokens.append( token )
               elif token != key:
                  newSyntaxTokens.append( token )
               else:
                  newSyntaxTokens.append( '( %s )' % value._expression() )
            syntax = ' '.join( newSyntaxTokens )
      # pylint:enable-msg=protected-access
      # special case the trailing elipsis
      syntax = syntax.replace( '...', '[ ... ]' )
      return syntax

   @classmethod
   def _expandData( cls ):
      # given data it will expand out any data that is a subexpression
      data = {}
      for key, value in cls.data.iteritems():
         expr = cls._getExpression( key, value )
         if expr is not None:
            expressionData = expr._data()  # pylint: disable-msg=protected-access
            for subKey in expressionData.iterkeys():
               if subKey in data:
                  assert expressionData[ subKey ] == data[ subKey ], \
                     "duplicate usage of token-name '%s' in syntax (note that " \
                     "expressions are like macros and get expanded)." % subKey
            data.update( expressionData )
         else:
            if key in data:
               assert data[ key ] == value
            else:
               data[ key ] = value
      return data

   @classmethod
   def _expandAdapters( cls ):
      adapters = []
      for key, value in cls.data.iteritems():
         value = cls._getExpression( key, value )
         if value is not None:
            # pylint: disable-msg=protected-access
            adapters.extend( value._adapters() )
            # pylint: enable-msg=protected-access
      adapter = getattr( cls, 'adapter', None )
      if adapter is not None:
         adapters.append( getBoundedMethod( adapter ) )
      return adapters

class _CliCommandBase( _CliCommandCommonBase ):
   authz = True
   syncAcct = False
   acct = True # no way to turn off accounting
   allowCache = True
   authzFunc = None
   acctFunc = None
   autoConfigSessionAllowed = True
   cmdFilter = False

   @classmethod
   def _createParseTreeHelper( cls, ebnfNode, handler, visitedEbnfNodes, data,
                               hidden, adapters ):
      if ebnfNode in visitedEbnfNodes:
         return visitedEbnfNodes[ ebnfNode ]

      # add a cmdHandler IFF the node is terminal
      if ebnfNode.isTerminal():
         cmdHandler = handler
         adaptersList = adapters
      else:
         cmdHandler = None
         adaptersList = None

      # make sure that matcher info
      name = ebnfNode.getName()
      if name == '...':
         dataElipsis = data.get( '...' )
         assert dataElipsis is None or dataElipsis is TRAILING_GARBAGE
         data[ '...' ] = TRAILING_GARBAGE
      cls._assert( name in data,
                   'Didn\'t find MetaData for %s in data!' % name )
      if isinstance( data[ name ], str ):
         # we are special casing a string type of matcher to be the help
         # description of a keyword rule, since this is the most common
         # case
         matcher = CliMatcher.KeywordMatcher( keyword=name,
                                              helpdesc=data[ name ] )
         parseNodeMetaData = Node( matcher=matcher )
      elif isinstance( data[ name ], CliMatcher.Matcher ):
         parseNodeMetaData = Node( matcher=data[ name ] )
      else:
         assert isinstance( data[ name ], Node )
         # TODO:write test case for this, to ensure that if we have 2 nodes and one
         # is hidden and one is not that both don't end up hidden
         parseNodeMetaData = copy.copy( data[ name ] )

      # <internal_ is a special keyword that should only be used in the case of the
      # internal Cli infra. This should not inherit from the command if it is hidden
      # or not, it should decide for itself.
      if not name.startswith( '<internal_' ):
         parseNodeMetaData.hidden_ |= hidden

      # create the node
      canMerge = True if parseNodeMetaData.overrideCanMerge_ else ebnfNode.canMerge()
      if not parseNodeMetaData.canMerge_:
         canMerge = False

      node = ParserNode( name, parseNodeMetaData, cmdHandler,
                         ebnfNode.isRepeatable(), adaptersList,
                         canMerge )
      visitedEbnfNodes[ ebnfNode ] = node

      for nextEbnfNode in ebnfNode.getNextNodes():
         createTreeHelper = cls._createParseTreeHelper
         nextNode = createTreeHelper( nextEbnfNode, handler, visitedEbnfNodes,
                                      data, hidden, adapters )
         node.addNextNode( nextNode )
      return node

   @classmethod
   def _getRootNodes( cls, syntaxEbnfTree, data, cmdHandler, hidden, adapters ):
      '''
      return a list of root nodes given an ebnf syntax and the meta data associated
      with each node.
      '''
      if not syntaxEbnfTree:
         return None

      nodes = []
      visitedEbnfNodes = {}

      # pylint: disable-msg=no-member
      for ebnfNode in syntaxEbnfTree.getNextNodes():
         node = cls._createParseTreeHelper( ebnfNode, cmdHandler, visitedEbnfNodes,
                                            data, hidden, adapters )
         nodes.append( node )

      return nodes

   @classmethod
   def _createCmdHandler( cls, handler, cmdSyntax ):
      if not handler:
         return None
      return _CommandHandlerClass(
         authz=cls.authz,
         syncAcct=cls.syncAcct,
         acct=True, # no way to turn off accounting
         allowCache=cls.allowCache,
         authzFunc=cls.authzFunc,
         acctFunc=cls.acctFunc,
         autoConfigSessionAllowed=cls.autoConfigSessionAllowed,
         cmdFilter=cls.cmdFilter,
         handler=handler,
         cmdSyntax=cmdSyntax,
         originalCls=cls )

   @classmethod
   def _getSyntaxAndHandler( cls, syntax, handler ):
      if getattr( cls, syntax, None ) is None:
         return ( None, None )
      return ( EbnfParser.parseEbnf( getattr( cls, syntax ) ),
               cls._createCmdHandler( getattr( cls, handler ),
                                      getattr( cls, syntax ) ) )

   @classmethod
   def _generateSyntaxAndData( cls ):
      # Implement this when dynamicSyntax_ is True
      assert False, "Subclass must implement this if dynamicSyntax_ is True"

   @classmethod
   def prettySyntax( cls, truncate=200 ):
      '''Extract pretty syntax from a command class.
      We assume that the no/def syntax is the same as the normal syntax,
      even though, in reality, some arguments may be optional in the no/def form.
      So, the results are not 100% accurate, but good enough.'''
      # We use `getattr`( cls, 'noOrDefaultSyntax', None )` and such,
      # because childeren of `ShowCommandClass` don't have these attributes.
      noDefSyntax = getattr( cls, 'noOrDefaultSyntax', None )
      if noDefSyntax:
         noSyntax = None
         defSyntax = None
      else:
         noSyntax = getattr( cls, 'noSyntax', None ) or ''
         noSyntax = noSyntax.replace( NAME_NO, '' ).strip()
         defSyntax = getattr( cls, 'defaultSyntax', None ) or ''
         defSyntax = defSyntax.replace( NAME_DEFAULT, '' ).strip()
         # Some commands are written with no and def separately. Simplify.
         if noSyntax and defSyntax:
            noDefSyntax = noSyntax

      result = getattr( cls, 'syntax', None )
      if result:
         if noDefSyntax:
            result = '[no|default] ' + result
         elif noSyntax:
            result = '[no] ' + result
         elif defSyntax:
            result = '[default] ' + result
      elif noDefSyntax:
         result = '(no|default) ' + noDefSyntax
      elif noSyntax:
         result = 'no ' + noSyntax
      elif defSyntax:
         result = 'default ' + defSyntax
      else:
         assert False, 'Is it possible that we found no syntaxes?'

      # Normalize whitespace with split, join.
      result = ' '.join( result.split() )
      if truncate and len( result ) > truncate:
         return result[ : truncate ] + ' ...'
      return result

   @classmethod
   def verbatimSyntax( cls ):
      '''Return the syntax verbatim.'''
      syntaxes = []
      syntax = getattr( cls, 'syntax', None )
      if syntax:
         syntaxes.append( syntax )

      noDef = getattr( cls, 'noOrDefaultSyntax', None )
      if noDef:
         syntaxes.append( '(no|default) ' + noDef )
      else:
         no = getattr( cls, 'noSyntax', None )
         if no:
            syntaxes.append( no.replace( NAME_NO, 'no' ) )
         default = getattr( cls, 'defaultSyntax', None )
         if default:
            syntaxes.append( default.replace( NAME_DEFAULT, 'default' ) )

      for syntax in syntaxes:
         yield ' '.join( syntax.split() )

class CliCommandClass( _CliCommandBase ):
   '''
   syntax: EBNF Syntax for this command
   noOrDefaultSyntax: EBNF No or Default syntax for this command
   noSyntax: EBNF syntax for this command
   defaultSyntax: EBNF Default syntax for this command
   handler: Command handler
   noOrDefaultHandler: No or Default Handler
   noHandler: No Handler
   defaultHandler: Default Handler
   data:
   authz: If authorization is enabled if this command gets authorized (default: true)
   syncAcct: Wait for accounting before handler is invoked (default: false)
   allowCache: Allow the parser to cache results of this command (default: true)
   authzFunc: Authorization function to authorize this command
   acctFunc: Accounting function to account this command
   hidden: Enable autocomplete and help completion for this command (default: false)
   autoConfigSessionAllowed:
   adapter:
   '''
   ALLOWED_FIELDS = ( 'syntax', 'noOrDefaultSyntax', 'noSyntax', 'defaultSyntax',
                      'handler', 'noOrDefaultHandler', 'noHandler', 'defaultHandler',
                      'data', 'hidden', 'authz', 'syncAcct', 'allowCache',
                      'authzFunc', 'acctFunc', 'hidden', 'autoConfigSessionAllowed',
                      'rootNodesCacheEnabled',
                      'adapter', 'initialize' )
   REQUIRED_FIELDS = ()

   syntax = None
   noOrDefaultSyntax = None
   noSyntax = None
   defaultSyntax = None
   hidden = False
   rootNodes = None

   dynamicSyntax_ = False
   initialized_ = False

   @classmethod
   def initialize( cls ):
      if cls.initialized_:
         return
      t1( "initialize", cls.__name__ )

      if cls.dynamicSyntax_:
         cls._generateSyntaxAndData()

      cls._validateFields()

      # when we initialize the class we take the fields that the class with created
      # and then we get to work. This is generally what we are tying to do:
      # 1) Check that the data is valid
      # 2) Expand noOrDefault fields into the noSyntax and defaultSyntax
      # 3) get all of the relevent adapters (must be done before expanding data)
      # 4) Update the syntaxes (basically expanding the syntax which are expressions)
      # 5) We don't need the Expressions in the data anymore, so also replace those
      # 6) Get the ENBF tree and cmdHandler and create function callbacks of them
      cls._checkData()
      cls._expandNoOrDefaultSyntax()
      adapters = cls._expandAdapters()
      for syntax in ( 'syntax', 'noSyntax', 'defaultSyntax' ):
         if getattr( cls, syntax, None ):
            cls._updateSyntax( syntax )
      if getattr( cls, 'syntax', None ) is not None:
         cls._assert( not cls.syntax.startswith( 'no ' ),
                      'Please use noSyntax instead' )
         cls._assert( not cls.syntax.startswith( 'default ' ),
                      'Please use defaultSyntax instead' )
      data = cls._expandData()

      syntaxInfo = cls._getSyntaxAndHandler( 'syntax', 'handler' )
      noSyntaxInfo = cls._getSyntaxAndHandler( 'noSyntax', 'noHandler' )
      defaultSyntaxInfo = cls._getSyntaxAndHandler( 'defaultSyntax',
                                                    'defaultHandler' )
      assert syntaxInfo[ 0 ] or noSyntaxInfo[ 0 ] or defaultSyntaxInfo[ 0 ], \
             'syntax, noSyntax, defaultSyntax, and noOrDefaultSyntax missing'

      def rootNodesGenerator():
         results = []
         for i in ( syntaxInfo, noSyntaxInfo, defaultSyntaxInfo ):
            if not i[ 0 ]:
               continue
            result = cls._getRootNodes( i[ 0 ], data, i[ 1 ],
                                        cls.hidden, adapters )
            results.extend( result )
         return results

      def _cachedRootNodes():
         if cls.cachedRootNodes_ is None:
            cls.cachedRootNodes_ = rootNodesGenerator()
         return cls.cachedRootNodes_

      cls.rootNodes = staticmethod( rootNodesGenerator )
      if getattr( cls, 'rootNodesCacheEnabled', False ):
         cls.cachedRootNodes_ = None
         cls.cachedRootNodes = staticmethod( _cachedRootNodes )
      cls.initialized_ = True

   @classmethod
   def _expandNoOrDefaultSyntax( cls ):
      if getattr( cls, 'noOrDefaultSyntax', None ) is not None:
         cls._assert( getattr( cls, 'noSyntax', None ) is None,
                      'noSyntax not expected' )
         cls._assert( getattr( cls, 'defaultSyntax', None ) is None,
                      'defaultSyntax not expected' )
         cls.noSyntax = cls.noOrDefaultSyntax
         cls.defaultSyntax = cls.noOrDefaultSyntax

      noOrDefaultHandler = getattr( cls, 'noOrDefaultHandler', None )
      if noOrDefaultHandler is not None:
         cls._assert( getattr( cls, 'noHandler', None ) is None,
                      'noHandler not expected' )
         cls._assert( getattr( cls, 'defaultHandler', None ) is None,
                      'defaultHandler not expected' )
         noOrDefaultHandler = staticmethod( noOrDefaultHandler )
         setattr( cls, 'noHandler', noOrDefaultHandler )
         setattr( cls, 'defaultHandler', noOrDefaultHandler )

      # Make sure we have reasonable values for syntax, noSyntax,
      # and defaultSyntax.
      for k, h in ( ( 'syntax', 'handler' ),
                    ( 'noSyntax', 'noHandler' ),
                    ( 'defaultSyntax', 'defaultHandler' ) ):
         if getattr( cls, k, None ) is not None:
            cls._assert( getattr( cls, h, None ) is not None,
                         'If %s is present so must %s' % ( k, h ) )
            assert isinstance( getattr( cls, k ), str )
         else:
            cls._assert( getattr( cls, h, None ) is None,
                         'If %s is present so must %s' % ( h, k ) )

      # if we have no or default syntax we should prepend it to the syntax
      for triple in ( ( 'noSyntax', NAME_NO, NO_PARSER_NODE ),
                      ( 'defaultSyntax', NAME_DEFAULT, DEFAULT_PARSER_NODE ) ):
         syntaxName, token, node = triple
         syntax = getattr( cls, syntaxName, None )
         if syntax:
            cls._assert( token not in cls.data, # pylint: disable-msg=E1135
                         '%r is not user definable' % token )
            setattr( cls, syntaxName, token + ' ' + syntax )
            cls.data[ token ] = node

class CliExpression( _CliCommandCommonBase ):
   ALLOWED_FIELDS = ( 'data', 'expression', 'adapter', 'initialize' )
   REQUIRED_FIELDS = ( 'data', 'expression' )

   expression = None
   data = None
   adapters = None

   initialized_ = False

   @classmethod
   def _expression( cls ):
      cls.initialize()
      return cls.expression

   @classmethod
   def _data( cls ):
      cls.initialize()
      return cls.data

   @classmethod
   def _adapters( cls ):
      cls.initialize()
      return cls.adapters

   @classmethod
   def initialize( cls ):
      if cls.initialized_:
         return
      t1( "initialize", cls.__name__ )
      cls._validateFields()
      cls._checkData()
      adapters = cls._expandAdapters()
      expression = cls.expression
      EbnfParser.parseEbnf( expression ) # ensure that is it a correct EBNF sequence
      cls._updateSyntax( 'expression' )
      cls.data = cls._expandData()
      cls.adapters = adapters
      cls.initialized_ = True

class CliExpressionFactory( object ):

   def __init__( self ):
      self.expressions_ = {}

   def get( self, name ):
      expr = self.expressions_.get( name )
      if not expr:
         expr = self.generate( name )
         assert inspect.isclass( expr ) and issubclass( expr, CliExpression ), \
            "CliExpressionFactory.generate() must return a CliExpression class " \
            "but got %s" % ( type( expr ) )
         self.expressions_[ name ] = expr
      return expr

   def generate( self, name ):
      # Subclass must implement this
      raise NotImplementedError

class OrExpressionFactory( CliExpressionFactory ):
   def __init__( self ):
      self.subExprs_ = dict() # name -> expr
      CliExpressionFactory.__init__( self )

   def __ior__( self, ( exprName, expr ) ):
      t0( "OrExpression add", exprName )
      assert exprName not in self.subExprs_, \
         "%s: expression %s already present" % ( self, exprName )
      self.subExprs_[ exprName ] = expr
      return self

   def __str__( self ):
      return '( ' + ' | '.join( str( x ) for x in self.subExprs_ ) + ' )'

   def generate( self, name ):
      class OrExpression( CliExpression ):
         argName_ = name

         data = { name + '_' + k: v for k, v in self.subExprs_.iteritems() }
         expression = ' | '.join( data.iterkeys() )

         # note data can be altered by expression expansion, so we save
         # the original interested names
         orNames_ = data.keys()

         @classmethod
         def adapter( cls, mode, args, argsList ):
            for n in cls.orNames_:
               if n in args:
                  value = args.pop( n )
                  args[ cls.argName_ ] = value
                  break

      return OrExpression

class _SetEnumMatcher( CliMatcher.EnumMatcher ):
   def __init__( self, name, **kargs ):
      super( _SetEnumMatcher, self ).__init__( **kargs )
      self.name_ = name

   def getValidKeywords( self, context ):
      if not context.sharedResult:
         return self.keywords_
      seenKeywords = context.sharedResult.get( self.name_ )
      if not seenKeywords:
         return self.keywords_
      return [ i for i in self.keywords_ if i not in seenKeywords ]

   def getValidCompletions( self, context ):
      if not context.sharedResult:
         return self.completions_
      seenKeywords = context.sharedResult.get( self.name_ )
      if not seenKeywords:
         return self.completions_
      return [ c for c in self.completions_ if c.name not in seenKeywords ]

class SetEnumMatcher( CliExpressionFactory ):
   '''
   This matcher is a type of EnumMatcher, but matches matches an Enum multiple times.
   So for example if I have an Enum of Red, Blue, Green, I could create this
   matcher to accept these enums in any order, but only accepts each one once.
   For example Red Blue would be valid as well as Green Red, but Red Red would
   not be valid. It takes a data dict which is a mapping from the keyword to its
   help description.
   '''
   def __init__( self, data, **kargs ):
      CliExpressionFactory.__init__( self )
      self.data_ = data
      self.kargs_ = kargs

   def generate( self, name ):
      matcher = _SetEnumMatcher( name=name, keywordsHelpDescMapping=self.data_ )
      node = Node( matcher, storeSharedResult=True, **self.kargs_ )

      class _SetEnumExpr( CliExpression ):
         expression = '{ %s }' % name
         data = { name: node }
      return _SetEnumExpr

def setCliExpression( data, hiddenData=None, optional=False, name=None ):
   if hiddenData:
      allData = data.copy()
      allData.update( hiddenData )
   else:
      allData = data
   setExpression = '{ ' + ' | '.join( allData.keys() ) + ' }'
   if optional:
      setExpression = '[ %s ]' % setExpression

   def defaultAdapter( mode, args, argsList ):
      results = []
      for arg in argsList:
         if arg[ 0 ] not in allData:
            continue
         results.append( args[ arg[ 0 ] ] )
         args.pop( arg[ 0 ] )
      args[ name ] = results

   setData = {}
   for k, v in allData.iteritems():
      if hiddenData:
         hidden = True if k in hiddenData else False
      else:
         hidden = False
      if isinstance( v, str ):
         matcher = CliMatcher.KeywordMatcher( keyword=k,
                                              helpdesc=v )
         parseNodeMetaData = Node( matcher=matcher, hidden=hidden )
      elif isinstance( v, CliMatcher.Matcher ):
         parseNodeMetaData = Node( matcher=v, hidden=hidden )
      else:
         parseNodeMetaData = v
      parseNodeMetaData.maxMatches_ = 1
      setData[ k ] = parseNodeMetaData

   class SetExpression( CliExpression ):
      expression = setExpression
      data = setData
      adapter = staticmethod( defaultAdapter ) if name else None

   return SetExpression

# We define some default nodes/matchers below that are useful to be defined
# within the parser
NO_MATCHER = CliMatcher.KeywordMatcher( keyword='no',
                                 helpdesc='Disable the command that follows',
                                 common=True )
NO_PARSER_NODE = Node( matcher=NO_MATCHER )
DEFAULT_MATCHER = CliMatcher.KeywordMatcher( keyword='default',
                                 helpdesc='Set a command to its defaults',
                                 common=True )
DEFAULT_PARSER_NODE = Node( DEFAULT_MATCHER )
TRAILING_GARBAGE = Node( CliMatcher.StringMatcher( helpname='LINE',
                                                   helpdesc='LINE' ),
                         hidden=True )

# Some convenience functions

def singleNode( matcher, **kargs ):
   # creating a node that matches only once
   kargs[ 'maxMatches' ] = 1
   return Node( matcher=matcher, **kargs )

def singleKeyword( keyword, helpdesc, priority=CliParserCommon.PRIO_KEYWORD,
                   **kargs ):
   # creating a keyword node that matches only once
   return singleNode( CliMatcher.KeywordMatcher( keyword,
                                                 helpdesc,
                                                 priority=priority ),
                      **kargs )

def guardedKeyword( keyword, helpdesc, guard,
                    priority=CliParserCommon.PRIO_KEYWORD,
                    **kargs ):
   # creating a guarded keyword node
   return Node( matcher=CliMatcher.KeywordMatcher( keyword, helpdesc=helpdesc,
                                                   priority=priority ),
                guard=guard, **kargs )

def hiddenKeyword( keyword, helpdesc,
                    priority=CliParserCommon.PRIO_KEYWORD,
                    **kargs ):
   # creating a hidden keyword node
   return Node( matcher=CliMatcher.KeywordMatcher( keyword, helpdesc=helpdesc,
                                                   priority=priority ),
                hidden=True, **kargs )
