# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import os
import simplejson
import sys
import tempfile
import threading
import time
import traceback
from cStringIO import StringIO

import BasicCli
import BasicCliSession
import CapiCliCommon
import CliCommon
import CliModel
import CliParser
import ConfigMount
import Tac
import Tracing

ERR_408_TEXT = "Command exceeded its allowed time limit"

__defaultTraceHandle__ = Tracing.Handle( 'CliApi' )
th = Tracing.defaultTraceHandle()
t1 = th.trace1
t2 = th.trace2
t3 = th.trace3

class CapiStatus( object ):
   """ Status codes for a CliCommandResponse. """
   SUCCESS = 200 # httplib.OK
   ERROR = 400 # httplib.BAD_REQUEST
   UNAUTHORIZED = 401 # httplib.UNAUTHORIZED
   FORBIDDEN = 403 # httplib.FORBIDDEN
   NOT_FOUND = 404 # httplib.NOT_FOUND
   REQUEST_TIMEOUT = 408 # httplib.REQUEST_TIMEOUT
   NOT_EXECUTED = 412 # httplib.PRECONDITION_FAILED
   CONFIG_LOCKED = 423 # httplib.LOCKED
   CLOSED_NO_RESP = 444 # "Connection Closed Without Response"; not in httplib
   INTERNAL_ERROR = 500 # httplib.INTERNAL_SERVER_ERROR
   NOT_CAPI_READY = 501 # httplib.NOT_IMPLEMENTED


# All the following types of exceptions are translated into NOT_FOUND:
NOT_FOUND_EXCEPTIONS = ( CliCommon.InvalidInputError,
                         CliCommon.AmbiguousCommandError,
                         CliCommon.IncompleteCommandError,
                         CliCommon.IncompleteTokenError,
                         CliCommon.GuardError )

class TimedThread( threading.Thread ):
   """ Thread class to execute a function and return the results or HTTP 408 """
   def __init__( self, func, args, timeout ):
      self.func = func
      self.args = args
      self.timeout = timeout
      self.envCopy = dict( os.environ )
      self.stop = []
      threading.Thread.__init__( self, target=self.wrapper )

   def wrapper( self ):
      os.environ = self.envCopy
      self.func( *self.args, stop=self.stop )

   def start( self ):
      threading.Thread.start( self )
      self.join( self.timeout )
      self.stop.append( 'You shall not pass!' )

class OutputBuffer( object ):
   """Something akin to a StringIO that correctly handles fileno().

   Use this if you want to capture stdout or stderr while running
   subprocesses or doing other similar things that may directly use
   fd 1 or 2.

   This uses a cStringIO under the hood and falls back to a temporary file
   when fileno() is called.  Only write operations are supported.  Seeking
   is not supported.  Unsupported methods are left unimplemented to trigger
   errors as early as possible (and give a chance to pylint to find out too).
   """

   def __init__( self ):
      self.buf_ = StringIO()
      self.tmpfile_ = None
      self.pipeFile_ = None

   def usePipe( self, pipeFile ):
      self.pipeFile_ = pipeFile

   def fileno( self ):
      if self.pipeFile_:
         return self.pipeFile_
      elif self.tmpfile_ is None:
         self.tmpfile_ = tempfile.TemporaryFile( prefix="CapiBuffer" )
      # Before handing out the file descriptor, let's make sure we flush
      # anything that may have been written from Python to the tempfile,
      # that could still be buffered.
      self.tmpfile_.flush()
      return self.tmpfile_.fileno()

   def flush( self ):
      if self.tmpfile_:
         self.tmpfile_.flush()
      self.buf_.flush()

   def close( self ):
      if self.tmpfile_:
         self.tmpfile_.close()
         self.tmpfile_ = None
      self.buf_.close()

   def isatty( self ):
      return False

   def tell( self ):
      pos = self.buf_.tell()
      if self.tmpfile_:
         pos += self.tmpfile_.tell()
      return pos

   def truncate( self, size=None ):
      bufsize = self.buf_.tell()
      filesize = self.tmpfile_.tell() if self.tmpfile_ else 0
      if size is None:
         size = bufsize + filesize
      if self.tmpfile_:
         if size <= bufsize:
            self.tmpfile_.close()
            self.tmpfile_ = None
         else:
            self.tmpfile_.truncate( size - bufsize )
            return
      self.buf_.truncate( size )

   def write( self, s ):
      if self.pipeFile_:
         os.write( self.pipeFile_, s )
      elif self.tmpfile_:
         self.tmpfile_.write( s )
      else:
         self.buf_.write( s )

   def writelines( self, iterable ):
      if self.pipeFile_:
         for i in iterable:
            os.write( self.pipeFile_, i )
            os.write( "\n" )
      elif self.tmpfile_:
         self.tmpfile_.writelines( iterable )
      else:
         self.buf_.writelines( iterable )

   def getvalue( self ):
      assert not self.pipeFile_, "Can't get value with pipe installed"
      buf = self.buf_.getvalue()
      if self.tmpfile_:
         self.tmpfile_.seek( 0 )
         buf += self.tmpfile_.read()
         self.tmpfile_.close()
         self.tmpfile_ = None
      return buf

class IOManager( object ):
   """Context manager that redirects stdout/stderr to a StringIO, and
   overrides stdin when provided with an input string."""

   def __init__( self ):
      self.output_ = OutputBuffer()
      self.input_ = None
      self.inputStr_ = None
      self.savedStdinFd_ = None
      self.savedStdoutFd_ = None
      self.savedStderrFd_ = None

   def usePipe( self, pipeFile ):
      self.output_.usePipe( pipeFile )

   def replaceFd( self, new, original ):
      saved = None
      if new != original:
         saved = os.dup( original )
         os.dup2( new, original )
      return saved

   def restoreFd( self, saved, original ):
      if saved:
         os.dup2( saved, original )
         os.close( saved )

   def __enter__( self ):
      self.output_.truncate( 0 )
      sys.stdout.flush()
      self.savedStdoutFd_ = self.replaceFd( self.output_.fileno(),
                                            sys.stdout.fileno() )
      self.savedStderrFd_ = self.replaceFd( self.output_.fileno(),
                                            sys.stderr.fileno() )

      if self.inputStr_:
         # Use a temporary file containing the contents of the input as stdin
         inIO = tempfile.TemporaryFile()
         inIO.write( self.inputStr_ )
         inIO.flush()
         inIO.seek( 0 )
         self.input_ = inIO
         self.savedStdinFd_ = self.replaceFd( inIO.fileno(), sys.stdin.fileno() )

      return self

   def __exit__( self, excType, value, excTraceback ):
      sys.stdout.flush()
      self.restoreFd( self.savedStdoutFd_, sys.stdout.fileno() )
      self.restoreFd( self.savedStderrFd_, sys.stderr.fileno() )
      self.restoreFd( self.savedStdinFd_, sys.stdin.fileno() )

      self.savedStdinFd_ = None
      self.savedStdoutFd_ = self.savedStderrFd_ = None
      # Clear the input so it isn't reused multiple times
      self.input_ = self.inputStr_ = None

   def inputIs( self, inputStr ):
      """Sets the input string which will be fed as stdin next time
      this class is used. The input will be cleared after one use."""
      self.inputStr_ = inputStr

   def get( self ):
      """Returns a string with the captured output."""
      return self.output_.getvalue()

   def close( self ):
      self.output_.close()

class CliCommand( object ):

   # Let us use 'input' as a valid parameter name:
   # pylint: disable-msg=W0622
   def __init__( self, cmd, revision=None, input=None ):
      """ A structure representing a CLI command along with any relevant options.
      - cmdStr: the CLI command
      - revision: the desired Model revision
      - inputStr: a string which will be fed as stdin during the command's execution
      """
      self.cmdStr = cmd
      self.revision = revision
      self.input = input

   def __str__( self ):
      """ Provide a nice view of the command, i.e.
      "'show aliases' (revision='1' input='Hello World!')"
      """
      ret = "'%s'" % self.cmdStr
      if self.revision or self.input:
         annotation = ''
         if self.revision:
            annotation += 'revision=%r' % self.revision
         if self.input:
            annotation += 'input=%r' % self.input
         ret += '(%s)' % annotation
      return ret

class TextResponse( CliModel.Model ):
   """ A simple Model which represents the output of a command in
   'text' mode. """
   # No need to implement a render() method, as it should never be
   # called for this model.
   output = CliModel.Str( "ASCII output from the CLI command" )

class CapiExecutor( object ):
   """ Main class responsible for executing CLI commands and returning
   CliCommandResponses. External programs that wish to interact with a
   programatic CLI should instantiate an instance of this class, and
   use the executeCommand[s] methods to run commands. """

   def __init__( self, cli, session, stateless=True ):
      self.cli_ = cli
      self.session_ = session
      self.stateless_ = stateless
      t2( "CapiExecutor initialized" )

   def warmupCache( self ):
      cliSessionName = "capi-%s-warmup" % os.getpid()
      commands = [ CliCommand( "configure session %s" % cliSessionName ),
                   CliCommand( "rollback clean-config" ),
                   CliCommand( "copy startup-config session-config" ),
                   CliCommand( "exit" ),
                   CliCommand( "no configure session %s" % cliSessionName ) ]
      self.executeCommands( CliCommon.MAX_PRIV_LVL, commands )

   def _maybeRunPreCmdsFn( self, preCommandsFn, textOutput, timestamps ):
      if preCommandsFn:
         assert callable( preCommandsFn ), "preCommandsFn must be callable"
         t2( "We have a preCommandsFn" )
         for command in preCommandsFn():
            t2( "Running preCommand", command )
            response = self._execute( command, textOutput, timestamps )
            assert response.status == CapiStatus.SUCCESS, (
               "Command '%s' did not succeed, received %s" % ( command, response ) )

   def _maybeRunPostCmdFn( self, postCommandsFn, errorSeen,
                           textOutput, timestamps ):
      if postCommandsFn:
         assert callable( postCommandsFn ), "postCommandsFn must be callable"
         t2( "We have a postCommand fnc and errorSeen was:", errorSeen )
         for command in postCommandsFn( errorSeen ):
            t2( "Running postCommand", command )
            response = self._execute( command, textOutput, timestamps )
            assert response.status == CapiStatus.SUCCESS, (
               "Command '%s' did not succeed, received %s" % ( command, response ) )

   def _makeTimeoutResponse( self ):
      errors = [ ERR_408_TEXT ]
      rt = CapiStatus.REQUEST_TIMEOUT
      return CapiCliCommon.CliCommandResponse( rt, errors=errors )

   def _executeCommands( self, responses, commands, stopOnError, textOutput,
                         timestamps, preCommandsFn, postCommandsFn,
                         globalVersion, expandAliases, streamFd, stop=None ):
      """Here, the `stop` parameter exists only when the request is done with
         a timeout. In that case, `stop` will e an empty list, but it will
         become non-empty when the parent decides to gracefully stop the child"""

      if responses:
         timedOutResponse = self._makeTimeoutResponse()
      try:
         self._maybeRunPreCmdsFn( preCommandsFn, textOutput, timestamps )
         try:
            errorSeen = False
            for i, command in enumerate( commands ):
               responses[ i ] = timedOutResponse
               t2( "Running command", command )
               if streamFd is not None and i > 0:
                  os.write( streamFd, "," )

               try:
                  resp = self._execute( command, textOutput, timestamps,
                                        globalVersion, expandAliases, streamFd )
                  if stop:
                     return

                  if streamFd is not None and not textOutput:
                     # TODO implement this with polymorphism
                     if type( resp.model ) is not CliModel.DeferredModel:
                        encoder = simplejson.JSONEncoder()
                        for chunk in encoder.iterencode( resp.result ):
                           os.write( streamFd, chunk )
               finally:
                  BasicCliSession.doCommandHandlerCleanupCallback()

               responses[ i ] = resp
               errorSeen |= resp.status != CapiStatus.SUCCESS
               if stopOnError and errorSeen:
                  break
         finally:
            self._maybeRunPostCmdFn( postCommandsFn, errorSeen, textOutput,
                                     timestamps )
      finally:
         finalRes = responses[ -1 ] if responses else None
         self._terminateSession( finalRes )

   def executeCommands( self, commands, stopOnError=True,
                        textOutput=False, timestamps=False,
                        preCommandsFn=None, postCommandsFn=None,
                        globalVersion=None, autoComplete=False, requestTimeout=None,
                        expandAliases=False, streamFd=None ):
      """Executes the given Cli commands at the given privilege level.

      Args:
        - commands: An iterable of `CliCommand`s to parse and execute.
        - stopOnError: Whether or not to stop executing commands on the first
          command that fails.  True is the safest, otherwise a sequence of
          commands such as [ "interface et1", "interface et99", "shutdown" ]
          will shut down et1 instead of another typo'ed interface name.
        - textOutput: If True, return the unstructured text output printed by
          the Cli, even if the command returned an Model.
        - timestamps: True if timestamps should be included in the response. This
          will include how long it took to execute and when the command started to
          execute
        - preCommandsFn: Function that return a list of commands that will run before
          the commands are run.
        - postCommandsFn: Function that return a list of commands that will run
          before the commands are run.
        - globalVersion: The global version of the API to use if the command
          does not override it with a specific revision.
      Returns:
        A list of CliCommandResponse objects

      Example:
        >> capi.executeCommands( 15, [ CliCommand( "configure" ),
                                       CliCommand( "interface Et1" ),
                                       CliCommand( "shutdown" ) ] )
      """
      self._updateSession( textOutput=textOutput, autoComplete=autoComplete )

      cmdNotExecuted = CapiCliCommon.CliCommandResponse( CapiStatus.NOT_EXECUTED,
                                           errors=[ "An earlier command failed" ] )
      responses = [ cmdNotExecuted ] * len( commands )
      args = ( responses, commands, stopOnError, textOutput, timestamps,
               preCommandsFn, postCommandsFn,
               globalVersion, expandAliases, streamFd )
      if requestTimeout is None:
         self._executeCommands( *args )
      else:
         TimedThread( self._executeCommands, args, requestTimeout ).start()
      return responses

   def executeCommand( self, command, **kwargs ):
      """Executes the given CliCommand at the given privilege level.
      Returns a CliCommandResponse. """
      return self.executeCommands( [ command ], **kwargs )[ 0 ]

   def getHelpInfo( self, command, isText=False ):
      """ Get the help informations for each possible completions of
      the given CliCommand."""
      self._updateSession( textOutput=isText )

      response = None
      try:
         tokens, partialToken, completions = \
            self._getCompletionsHelper( command, startWithPartialToken=False )

         helpInfos = { c.name: c.help for c in completions }
         t2( 'Tokens: ', tokens )
         t2( 'Partial Token: ', partialToken )
         t2( 'Completions: ', completions )
         t2( 'HelpInfos: ', helpInfos )
         t2( 'Mode: ', self.session_.mode_ )

         return helpInfos

      finally:
         self._terminateSession( None )

      return response

   def _execute( self, command, wantText, timestamps=False,
                 globalVersion=None, expandAliases=False, streamFd=None ):
      """Executes one CliCommand in the given mode."""
      ioManager = IOManager()
      self.session_.shouldPrintIs( wantText )

      # Pass output format to c code printers via env
      if wantText:
         self.session_.outputFormat_ = "text"
      else:
         self.session_.outputFormat_ = "json"

      # handlers that print json via CliPrint will need to know the revision before
      # even starting to respond, so put it where they can pick it up. Translation
      # of version to revision is done in BasicCli once we know the model.
      self.session_.requestedModelRevision_ = command.revision
      self.session_.requestedModelVersion_ = globalVersion
      ioManager.inputIs( command.input )

      ts = time.time()
      execStartTime = ts if timestamps else None

      excType = excTraceback = None
      if streamFd is not None:
         ioManager.usePipe( streamFd )
      result = None
      isInternalError = False
      with ioManager:
         try:
            tokens = CliParser.textToTokens( command.cmdStr,
                                             mode=self.session_.mode_ )
            expandedTokens = None
            if expandAliases:
               tokenized = BasicCliSession.expandAlias( self.session_.mode_,
                                                        self.session_.cliConfig,
                                                        command, tokens, True )
               if len( tokenized ) > 1:
                  raise CliCommon.CommandIncompatibleError( "Multi-line aliases "
                                                     "are currently not supported" )
               else:
                  expandedTokens = tokenized[ 0 ]
            else:
               expandedTokens = tokens

            if expandedTokens:
               aaa = ( self.session_.authenticationEnabled() or
                       self.session_.authorizationEnabled() )
               result = self.session_.runTokenizedCmd( expandedTokens,
                                                       aaa=aaa,
                                                       fromCapi=True )
         except Exception:  # pylint: disable-msg=W0703
            # do this before getting the output as it might add errors
            excType, result, excTraceback = sys.exc_info()
            isInternalError = self._handleException( excType, result, excTraceback,
                                                     command )

      output = ioManager.get() if streamFd is None else ""
      t2( "Output:", output )

      ioManager.usePipe( None )
      ioManager.close()
      te = time.time()
      execDuration = ( te - ts ) if timestamps else None

      t = ( te - ts ) * 1000
      t2( "Executed", command, "in %.1fms" % t )

      deferredModel = False
      validateModel = CliModel.shouldValidateModel( result )
      if validateModel is False:
         result = type( result )
      try:
         deferredModel = issubclass( result, CliModel.DeferredModel )
      except TypeError as e:
         pass
      if ( result and hasattr( result, '__dict__' ) and
           result.__dict__.get( "__cliPrinted__" ) ):
         deferredModel = True
         result = type( result )

      if isinstance( result, Exception ):
         if isInternalError:
            # We have to do this after closing the ioManager or the output
            # will go into streaming file instead of agent logs.
            self._printInternalError( excType, result, excTraceback )
         ret = self._onException( result, wantText, output,
                                  execStartTime=execStartTime,
                                  execDuration=execDuration )
      elif streamFd is not None and deferredModel:
         # no need to return data, the result is already being written to the pipe
         return CapiCliCommon.CliCommandResponse( CapiStatus.SUCCESS,
                                                  model=CliModel.DeferredModel(),
                                                  execStartTime=execStartTime,
                                                  execDuration=execDuration )
      elif deferredModel:
         # value function may write to stream directly for speed, in this
         # case there is no python model that needs to be toDict-ed, the
         # captured stream already is json. Convert it back to py for now.
         if not wantText:
            # fix empty outputs (if people dont even call start/end)
            if output == "":
               output = "{}"
            # convert json to py, an exception will appear like that:
            # "error": {"message": "No JSON object could be decoded"
            try:
               j = simplejson.loads( output )
            except Exception: # pylint: disable=broad-except
               print( output )
               print( ' '.join( hex( ord( c ) ) for c in output[ : 10 ] ) )
               print( ' '.join( hex( ord( c ) ) for c in output[ -10 : ] ) )
               raise
            # check conformance to declared model
            try:
               if self.session_.requestedModelRevisionIsLatest_:
                  if self.cli_.cliConfig_.validateOutput and validateModel:
                     CliModel.unmarshalModel( result, j, degraded=False )
            except CliModel.ModelUnknownAttribute as e:
               t2( "bad deferred model instance", str( e ) )
               warnMsg = CliCommon.SHOW_OUTPUT_UNEXPECTED_ATTRIBUTE_WARNING
               self.session_.addWarning( "%s %s" % ( warnMsg, str( e ) ) )
            t2( "simplejson load:", j )
            ret = CapiCliCommon.CliCommandResponse( CapiStatus.SUCCESS, j,
                                                    isDict=True )
         else:
            result = TextResponse( output=output )
            return CapiCliCommon.CliCommandResponse( CapiStatus.SUCCESS,
                                                     model=result )
      elif not wantText and isinstance( result, CliModel.Model ):
         ret = self._onModelReturn( result,
                                    globalVersion, command.revision,
                                    execStartTime=execStartTime,
                                    execDuration=execDuration )
      else:
         # Everything else. In this case we will ignore the return value of the
         # value funciton, and just return whatever was printed to the screen.
         # This means that we will ignore non-model return values for json output.
         # however this is probably ok behavior
         ret = self._onEmptyResult( wantText, output,
                                    execStartTime=execStartTime,
                                    execDuration=execDuration )

      origErrorCount = len( self.session_.errors_ )
      self._setSessionOutputs( ret )
      # call toDict and cache the result for later, cannot delay it
      # further because of the per command cleanup callbacks (RibCapi).
      ret.computeResultDict( streaming=streamFd is not None )
      # Errors can be added late, during toDict, when using generators: detect that
      # condition and if so rebuild the response. Note that if warning/messages
      # are added during generation, those will not show in the response, and if
      # the generation returned an extra error and data, the data is discarded.
      if origErrorCount != len( self.session_.errors_ ):
         ret = CapiCliCommon.CliCommandResponse( CapiStatus.ERROR,
                             execStartTime=execStartTime, execDuration=execDuration )
         self._setSessionOutputs( ret )
         ret.computeResultDict( streaming=streamFd is not None )
      return ret

   def _getCompletionsHelper( self, cmd, startWithPartialToken ):
      completions = []
      if CliParser.ignoredComment( self.session_.mode_, cmd.strip() ):
         # We don't need to print any help for a comment we ignore.
         return completions

      allTokens = CliParser.textToTokens( cmd, mode=self.session_.mode_ )

      if not cmd or cmd != cmd.rstrip():
         tokens = allTokens
         partialToken = ''
      else:
         tokens = allTokens[ : -1 ]
         partialToken = allTokens[ -1 ]

      # partialToken is the empty string if the line was empty or there was
      # whitespace at the end of the line, or the last token from the line
      # otherwise.  tokens is a list of the rest of the tokens from the line, if
      # any.
      try:
         completions = self.session_.getCompletions(
            tokens, partialToken, startWithPartialToken=startWithPartialToken )
         unguardedCompletions = [ c for c in completions if c.guardCode is None ]
         if unguardedCompletions:
            completions = unguardedCompletions
         else:
            for c in completions:
               c.help = c.guardCode # BUG277628 help should be 'not available ...'
      except CliParser.ParserError, e:
         t1( 'There was an error getting completions:', e )

      return ( tokens, partialToken, completions )

   def getCompletions( self, cmd, preamble=None, startWithPartialToken=False ):
      self._updateSession()
      try:
         for preambleCmd in preamble or []:
            cliCmd = CliCommand( preambleCmd )
            resp = self._execute( cliCmd, False )
            if 'errors' in resp.result:
               return resp.result[ 'errors' ], {}
         _, _, completions = self._getCompletionsHelper( cmd, startWithPartialToken )
         return {}, completions
      finally:
         self._terminateSession( None )

   def _updateSession( self, textOutput=False, autoComplete=False ):
      assert self.session_ is not None
      self.session_.clearMessages()
      self.session_.shouldPrintIs( textOutput )
      self.session_.autoCompleteIs( autoComplete )

   def gotoUnprivMode( self, response=None ):
      while type( self.session_.mode ) is not BasicCli.UnprivMode:
         try:
            prev = self.session_.mode
            self.session_.gotoParentMode()
            t3( "Went back from mode", prev, "to", self.session_.mode )
         except Exception:  # pylint: disable-msg=W0703
            excType, exception, excTraceback = sys.exc_info()
            msg = "Exception while leaving mode %r: %s" % ( prev.name, exception )
            t1( msg )
            print( msg )  # Print to the agent log.
            traceback.print_exception( excType, exception, excTraceback )
            if response:
               response.status = CapiStatus.INTERNAL_ERROR
               if response.result.get( "errors" ):
                  response.result[ "errors" ].append( msg )
               else:
                  response.result[ "errors" ] = [ msg ]
            # brute force going to our parent
            self.session_.mode_ = self.session_.mode_.parent_

   def _terminateSession( self, response ):
      """Returns to UnprivMode and kills the current Session.

      It's important that we leave all the modes and modelets we may have
      entered as some modes trigger side-effects upon being exited, such
      as changes made to an ACL not being truly recorded in Sysdb until the
      submode is left.
      If a `response` CliCommandResponse is provided, we'll update it
      with any errors, warnings or messages that we get while leaving
      modes.
      """
      self.session_.clearMessages()
      if self.stateless_:
         # this means that we expect that the session should be restored to
         # pristine conditions
         self.gotoUnprivMode( response )

      if response:
         # Collect any new errors, warnings or messages that could have been
         # emitted while the session was torn down and add them to the response.
         if self.session_.errors_:
            if response.status == CapiStatus.SUCCESS:
               # We thought we succeeded, but exiting caused an error,
               # so overwrite the response status
               response.status = CapiStatus.ERROR
            if response.result.get( "errors" ):
               response.result[ "errors" ] += self.session_.errors_
            else:
               response.result[ "errors" ] = self.session_.errors_
         if self.session_.warnings_:
            if response.result.get( "warnings" ):
               response.result[ "warnings" ] += self.session_.warnings_
            else:
               response.result[ "warnings" ] = self.session_.warnings_
         if self.session_.messages_:
            if response.result.get( "messages" ):
               response.result[ "messages" ] += self.session_.messages_
            else:
               response.result[ "messages" ] = self.session_.messages_
      self.session_.clearMessages()

   def _setSessionOutputs( self, response ):
      """ Updates the `response` CliCommandResponse with any errors,
      warnings, or messages logged in the session. If the
      corresponding array already exists, we append the new outputs to
      the end of that array, otherwise we create a new member"""
      assert isinstance( response, CapiCliCommon.CliCommandResponse )
      response.errors.extend( self.session_.errors_ )
      response.warnings.extend( self.session_.warnings_ )
      response.messages.extend( self.session_.messages_ )

   def _handleException( self, excType, exception, excTraceback, command ):
      t1( "Handling exception", excType or type( exception ), ":", exception )

      exc_info = ( excType, exception, excTraceback )
      try:
         self.session_._handleCliException( exc_info, # pylint: disable-msg=W0212
                                            command,
                                            handleKeyboardInterrupt=False )
         return False
      except: # pylint: disable=bare-except
         self.session_.addError( str( exception ) )
         return True

   def _printInternalError( self, excType, exception, excTraceback ):
      # TODO(tsuna): Syslog something here.
      msg = "Internal error %s: %s" % ( excType, exception )
      # Print to the agent log.
      print( "\n".join( "CLI Exception: " + line
                        for line in msg.split( "\n" ) ) )
      f = StringIO()
      traceback.print_exception( excType, exception, excTraceback, file=f )
      print( "\n".join( "CLI Exception: " + line for line in
                        f.getvalue().split( "\n" ) ) )

   def _onException( self, exception, wantText, output,
                     execStartTime=None, execDuration=None ):
      # translate error code
      if isinstance( exception, NOT_FOUND_EXCEPTIONS ):
         status = CapiStatus.NOT_FOUND
      elif isinstance( exception, CliCommon.ApiError ):
         status = exception.ERR_CODE
      elif isinstance( exception, CliCommon.AuthzDeniedError ):
         status = CapiStatus.UNAUTHORIZED
      elif isinstance( exception, ConfigMount.ConfigMountProhibitedError ):
         status = CapiStatus.CONFIG_LOCKED
      elif isinstance( exception, CliParser.AlreadyHandledError ):
         if self.session_.errors_:
            status = CapiStatus.ERROR
         else:
            status = CapiStatus.SUCCESS
      else:
         status = CapiStatus.INTERNAL_ERROR

      response = CapiCliCommon.CliCommandResponse( status,
                                                   execStartTime=execStartTime,
                                                   execDuration=execDuration )
      if wantText:
         response.model = TextResponse( output=output )
      return response

   def _onModelReturn( self, model,
                       globalVersion, revision,
                       execStartTime=None, execDuration=None ):
      """Takes the dict result of a successful command and transforms it into
      a CliCommandResponse with the appropriate status"""
      status = CapiStatus.ERROR if self.session_.errors_ else CapiStatus.SUCCESS
      response = CapiCliCommon.CliCommandResponse( status, model,
                                               execStartTime=execStartTime,
                                               execDuration=execDuration,
                                     globalVersion=globalVersion, revision=revision )
      if not model.__public__:
         response.warnings.append( "Model '%s' is not a public model and is subject "
                                   "to change!" % type( model ).__name__ )

      return response

   def _onConfigCmdReturn( self, output, execStartTime=None, execDuration=None ):
      """ Returns a CliCommandResponse, populated with any
      errors/warnings/messages generated. If output was produced, it
      is inserted into the messages array """
      status = CapiStatus.ERROR if self.session_.errors_ else CapiStatus.SUCCESS
      response = CapiCliCommon.CliCommandResponse( status,
                    execStartTime=execStartTime, execDuration=execDuration )
      if output:
         response.messages = [ output ]
      return response

   def _onEmptyResult( self, wantText, output,
                       execStartTime=None, execDuration=None ):
      result = TextResponse( output=output )
      if not wantText:
         # We didn't get a Model back, but this was not a show
         # commands, so we weren't expecting a model back. Just go
         # and return any errors/warnings/messages that occurred.
         return self._onConfigCmdReturn( output,
                                         execStartTime=execStartTime,
                                         execDuration=execDuration )
      elif not self.session_.errors_:
         # They explicitly asked for text.
         status = CapiStatus.SUCCESS
      else:
         status = CapiStatus.ERROR
      return CapiCliCommon.CliCommandResponse( status, model=result,
                     execStartTime=execStartTime, execDuration=execDuration )

   def shutdown( self ):
      """Must be call to safely dispose of this object."""
      pass
