#!/usr/bin/env python

# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import print_function

import os
import textwrap
import platform
import rpm
import Tac
import socket

import CEosHelper

def genKickStartConfig( disabledAgents=None, filePath=None, enableCapi=False,
                        enableHostSyslog=False,
                        enableRoutingModelMultiAgent=False,
                        disableSpanningTree=False,
                        enableRouting=False,
                        enableRoutedMode=False,
                        configHostname=False ):
   if disabledAgents is None:
      disabledAgents = []

   if not filePath:
      filePath = os.path.join( "/", "mnt", "flash", "kickstart-config" )

   with open( filePath, "w" ) as f:
      for agent in disabledAgents:
         f.write( "agent %s shutdown\n" % agent )

      if enableCapi:
         # Enable http-server on localhost
         f.write( textwrap.dedent( """ 
         management api http-commands
            no shutdown
         !
         management http-server
            protocol http localhost
         !
         """ ) )

      if enableHostSyslog:
         # Enable sync warning events to SONiC
         f.write( textwrap.dedent( """ 
         logging on
         logging trap Warning
         logging host 127.0.0.1
         match-list input string cEosEvents
            match regex Thermostat
         !
         logging policy match invert-result match-list cEosEvents discard
         """ ) )

      if enableRoutingModelMultiAgent:
         # Enable multi-agent routing protocols model
         f.write( textwrap.dedent( """
         service routing protocols model multi-agent
         """ ) )

      if disableSpanningTree:
         # Enable multi-agent routing protocols model
         f.write( textwrap.dedent( """
         spanning-tree mode none
         """ ) )

      if enableRouting:
         f.write( textwrap.dedent( """
         ip routing
         """ ) )

      if enableRoutedMode:
         f.write( textwrap.dedent( """
         switchport default mode routed
         """ ) )

      if configHostname:
         f.write( textwrap.dedent( """
         hostname %s
         """ % socket.gethostname() ) )

## Handle config passed in by container orchestrator. Append the configuration to
## kickstart file
def handleExtKickstart( kickstartConfigFile=None, extConfigFile=None ):
   if not extConfigFile:
      extConfigFile = os.path.join( "/", "mnt", "flash", "config",
                                    "kickstart-config" )
   if not kickstartConfigFile:
      kickstartConfigFile = os.path.join( "/", "mnt", "flash",
                                          "kickstart-config" )
   with open( kickstartConfigFile, "a" ) as kickstartFd:
      with open( extConfigFile, "r" ) as extConfigFd:
         kickstartFd.write( extConfigFd.read() )

def setLinuxKernelForwarding( veosConfigFile=None ):
   if not veosConfigFile:
      veosConfigFile = os.path.join( "/", "mnt", "flash", "veos-config" )
   with open( veosConfigFile, "w+" ) as f:
      f.write( "MODE=linux" )

def runningKernelVersion():
   return platform.uname()[ 2 ]

def installedKernelVersion():
   ts = rpm.TransactionSet()
   # pylint: disable-msg=E1101
   rpms = ts.dbMatch( 'name', 'EosKernel' )
   # pylint: enable-msg=E1101
   installedVersionStr = ''
   if len( rpms ) == 1:
      for r in rpms:
         installedVersionStr = "%s-%s" % ( r[ 'version' ], r[ 'release' ] )
   return installedVersionStr

# We need to verify that cEOS is running on the same
# kernel version as the binary of the kernel that cEos
# is bundled with.
def checkKernelVersion():
   # Check the running Kernel with the binary
   versionMatch = False
   try:
      versionMatch = runningKernelVersion() == installedKernelVersion()
   finally:
      # Ignore exception due to missing rpms, etc
      pass
   return versionMatch

def applyCeosMode( mode ) :
   if mode.checkKernelVersion:
      assert checkKernelVersion(), "Kernel version check failed for cEOS"

   if mode.generateKickStart:
      blacklistedAgents = os.getenv( 'EOS_AGENT_BLACKLIST', "" )
      blacklistedAgents = [ agent for agent in blacklistedAgents.split( ',' )
                            if agent != "" ]
      disabledAgents = mode.disabledAgents + blacklistedAgents

      genKickStartConfig(
         disabledAgents=disabledAgents,
         enableCapi=mode.enableCapi,
         enableHostSyslog=mode.enableHostSyslog,
         enableRoutingModelMultiAgent=mode.enableRoutingModelMultiAgent,
         disableSpanningTree=mode.disableSpanningTree,
         enableRouting=mode.enableRouting,
         enableRoutedMode=mode.enableRoutedMode,
         configHostname=mode.configHostname,
      )

   if mode.extKickstart:
      handleExtKickstart()

   if mode.linuxKernelForwarding:
      setLinuxKernelForwarding()

def createSaiDefaultIntfs():
   for intf in [ 'cpu', 'vxlan', 'fabric', 'internal1_1', 'internal1_2' ]:
      Tac.run( [ 'ip', 'link', 'add', intf, 'type', 'dummy' ],
               stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
      Tac.run( [ 'ip', 'link', 'set', intf, 'up' ],
               stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )

   # When dummy fabric netdevice is created, mtu will be set to 1500.
   # Since fabric mtu is 1500, we will not able to configure mtu >1500
   # for SVIs. To fix this, we set mtu of dummy fabric netdevice to
   # EOS fabric mtu.
   Tac.run( [ 'ip', 'link', 'set', 'fabric', 'mtu', '9216' ],
            stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )

def CEosInit():
   if not CEosHelper.isCeos():
      return

   applyCeosMode( CEosHelper.getCeosMode() )
   if os.environ.get( "CEOS_NAMESPACE" ):
      createSaiDefaultIntfs()
