# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import re
import simplejson as json
import textwrap

import EosCloudInitLib as cloudInit
import Fru
from SfFruHelper import SfFruHelper
from SfFruHelper import INTFTYPE_CONTAINER, PORT, DEVICE
from SfFruHelper import PORT_INVALID, PORT_VIRT_START, PORT_VIRT_END
from SfFruHelper import PORT_MGMT
import Tac
import Tracing

__defaultTraceHandle__ = Tracing.Handle( "Fru.SfFruHelper" )
t0 = Tracing.trace0

class CEOSRFruHelper( SfFruHelper ):
   def __init__( self, driver ):
      SfFruHelper.__init__( self, driver )
      self._contPortId = PORT_INVALID

   def getEventFifoPath( self ):
      return "/ceosr/UdevToSfaHotPlugPipe"

   def hasManagementIntf( self ):
      return False

   def useKernelDevNames( self ):
      return True

   def isContainerInterface( self, devName ):
      if ( self.isDeviceUserForceMapped( devName ) and
           not self.isPhysicalDevice( devName ) ):
         # Force mapped virtual interface
         return True

      # Different CNI provider create container interface with
      # different prefix.
      # Calico create interfaces with prefix: cali
      # Flannel create interfaces with prefix: veth
      # Cilium create interfaces with prefix: lxc
      if( devName.startswith( "cali" ) or
          devName.startswith( "veth" ) or
          devName.startswith( "lxc" ) ):
         return True
      return False

   def sortDevNames( self, devNames ):
      # In cEOSR we manage three differeent type of interfaces
      # 1. Physical interfaces
      # 2. User force mapped interfaces from the configuration
      #    as part of FORCE_MAP_INTERFACE_REGEX_LIST config
      # 3. Application POD interfaces
      phyDevs = []
      conDevs = []
      forceMappedDevs = []

      t0( "Unsorted interfaces: %s" % str( devNames ) )
      for d in devNames:
         if self.isContainerInterface( d ):
            if self.isDeviceUserForceMapped( d ):
               forceMappedDevs.append( d )
            else:
               conDevs.append( d )
         else:
            phyDevs.append( d )

      phyDevs = super( CEOSRFruHelper, self ).sortDevNames( phyDevs )
      conDevs = sorted( conDevs )
      forceMappedDevs = sorted( forceMappedDevs )

      # User forced mapped interfaces need to be before the application
      # container interfaces.
      # It needs to be sorted in the order mentioned in the configuration
      # and not in alphabetical order.
      forceMappedDevsTmp = []
      for d1 in self._forceMappedIntfRegxList:
         for d2 in forceMappedDevs:
            if re.match( d1, d2 ) and d2 not in forceMappedDevsTmp:
               forceMappedDevsTmp.append( d2 )
      forceMappedDevs = forceMappedDevsTmp

      sortedDevs = phyDevs + forceMappedDevs + conDevs
      t0( "Final sorted interfaces: %s" % str( sortedDevs ) )
      return sortedDevs

   def handleInterfaceEvent( self, intfName ):
      # CEOSR CNI plugin sends us events in this JSON format
      # { "namespace":   "default",
      #   "pod":         "busybox1-5d9f79cb7b-87sw5",
      #   "containerID": "09c32fc5d6f5197f221ce408e1d6d502e92f9a15a15f6429ca6b3ce6",
      #   "command":     "ADD",
      #   "interface":   "calif920e908dbc",
      #   "ipAddress":   "10.244.0.1/24" }
      t0( "HotPlug called for CEOSR: %s" % intfName )
      try:
         podConf = json.loads( intfName )
      except ValueError:
         t0( "Invalid interface event: %s" % intfName )
         return

      intf = podConf.get( "interface", "" )
      command = podConf.get( "command", "" )
      if not self.isContainerInterface( intf ):
         t0( "Invalid interface: %s" % intf )
         return

      if command == "ADD":
         self.handleInterfaceEventAdd( intf, **podConf )
      elif command == "DEL":
         self.handleInterfaceEventDel( intf )
      else:
         t0( "Invalid CNI command" )

   def getNewPort( self, devName, **extraArgs ):
      if not self.isContainerInterface( devName ):
         return super( CEOSRFruHelper, self ).getNewPort( devName )

      t0( "getNewPort for container interface: %s" % devName )

      # BUG403483: namespace and pod name comes from hotplugin handler,
      # but in case of cEOSR restart and with pre-existing container interfaces,
      # we need to figure out how to find a pod name.
      namespace = extraArgs.pop( "namespace", "unknown" )
      pod = extraArgs.pop( "pod", "unknown" )

      portId = self.deviceCache[ PORT ][ devName ]
      port = self._driver.ethPortDir.newPort( portId )
      port.description = "Container:%d:%s:%s" % ( portId, namespace, pod )
      port.role = "Switched"
      port.macAddr = self.getMac( devName )
      port.label = portId
      return port, INTFTYPE_CONTAINER

   def resetPortId( self ):
      super( CEOSRFruHelper, self ).resetPortId()

      self._contPortId = PORT_INVALID
      # Reset the current contPortId pointer
      for portId in self.deviceCache[ PORT ].values():
         if portId > self._contPortId and portId >= PORT_VIRT_START and \
            portId <= PORT_VIRT_END:
            self._contPortId = portId

   def getNextPortId( self, devName ):
      if not self.isContainerInterface( devName ):
         return super( CEOSRFruHelper, self ).getNextPortId( devName )

      t0( "getNextPortId for container interface: %s" % devName )
      newPortId = self.getNextPortIdFromPool(
            self._contPortId, PORT_VIRT_START, PORT_VIRT_END )
      if newPortId:
         self._contPortId = newPortId
         self._portIdPool[ newPortId ] = False # Mark as not available
         t0( "getNextPortId: portId %d allocated for %s" % ( newPortId, devName ) )
         return newPortId

      t0( "Ran out of portIds( %d - %d ) for %s" %
          ( PORT_VIRT_START, PORT_VIRT_END, devName ) )
      return 0

   def addInterface( self, devName, **extraArgs ):
      if not self.isContainerInterface( devName ):
         super( CEOSRFruHelper, self ).addInterface( devName, **extraArgs )
         return

      port, _ = self.getNewPort( devName, **extraArgs )
      if port is None:
         t0( "Ignoring unrecognized interface %s" % devName )
         return

      invEntities = [ port ]
      phyName = self.getPhyName( devName, port.id )
      ethName = "Ethernet%d" % port.label

      # Create an Inventory::PhyEthtoolPhy or SfaPhyDir
      phy = self._driver.phyEthtoolPhyDir.newPhy( phyName )
      invEntities.append( phy )
      t0( "%s added to phy/ethtool/config" % devName )
      phy.port = port

      phy = self._driver.sfaPhyDir.newPhy( phyName )
      invEntities.append( phy )
      phy.port = port

      # Create a DependentSet for each of the entity
      # under inventory/genericPc/component/...
      # All dependents can be cleaned up when the entity is deleted.
      for e in invEntities:
         Fru.newDependentSet( e )

      # Cleanup stale containerIntf status
      containerIntfStatus = self._driver.containerIntfStatusDir.intfStatus
      if ethName in containerIntfStatus:
         del containerIntfStatus[ ethName ]
         t0( "Deleted intf from containerIntfStatus %s" % ethName )

      # Create Interface::ContainerIntfStatus
      podNs = extraArgs.pop( "namespace", "" )
      podName = extraArgs.pop( "pod", "" )
      podId = extraArgs.pop( "containerID", "" )
      address = extraArgs.pop( "ipAddress", "0.0.0.0/0" ).split( "/" )
      ipAddrWithMask = Tac.newInstance( 'Arnet::IpAddrWithMask', address[ 0 ],
                                      int( address[ 1 ] ) )
      t0( "Creating ContainerIntfStatus: devName=%s, ethName=%s, " %
          ( devName, ethName ) +
          "ipAddrWithMask=%s, podNs=%s, podName=%s, podId=%s" %
          ( ipAddrWithMask, podNs, podName, podId ) )
      Fru.Dep( containerIntfStatus, port ).newMember(
               ethName, ipAddrWithMask, podNs, podName, podId )
      t0( "%s added to ContainerIntfStatusDir" % devName )

      t0( "Added device %s, mac %s, pci %s" %
          ( devName, self.getMac( devName ), self.getPci( devName ) ) )

   def generateIntfConfigStr( self, devName ):
      if self.deviceCache[ PORT ][ devName ] == PORT_MGMT:
         ifname = "Management1"
      else:
         ifname = "Ethernet%d" % self.deviceCache[ PORT ][ devName ]
      intfConfig = ""
      ipaddr = cloudInit.getIP( devName )
      if ipaddr:
         intfConfig = textwrap.dedent( """
         interface %s
         ip address %s
         """ % ( ifname, ipaddr ) )
         gw = cloudInit.getDefaultGw( devName )
         if gw:
            intfConfig += textwrap.dedent( """
            ip route 0.0.0.0/0 %s
            """ % gw )
      return intfConfig

   def updateIntfConfigAll( self ):
      intfConfigs = ""
      for devName in self.deviceCache[ DEVICE ]:
         intfConfigs += self.generateIntfConfigStr( devName )
      self.updateRunningConfig( intfConfigs )

   def updateIntfConfig( self, devName ):
      intfConfig = self.generateIntfConfigStr( devName )
      self.updateRunningConfig( intfConfig )
