#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""BGP Monitoring Protocol."""

import dpkt
import BgpDpkt
import struct, socket

# BGP Monitoring Protocol - RFC 7854

# Protocol Version
BMP_VERSION = 3

# Message Types
ROUTE_MONITORING = 0
STATS_REPORT = 1
PEER_DOWN = 2
PEER_UP = 3
INITIATION = 4
TERMINATION = 5
ROUTE_MIRROR = 6

# Information TLV Types
INFO_STRING = 0
INFO_SYSDESCR = 1
INFO_SYSNAME = 2

# Termination TLV Types
TERM_STRING = 0
TERM_REASON = 1

# Termination Reason Codes
TERM_ADMINCLOSE = 0
TERM_UNSPEC = 1
TERM_RESOURCE = 2
TERM_REDUNDANT = 3
TERM_PERMANENT = 4

# Peer Type
PEER_GLOBAL = 0
PEER_RD = 1
PEER_LOCAL = 2

# Peer Flags
PEERFLAG_IPV6 = 0x80
PEERFLAG_POST_POLICY = 0x40
PEERFLAG_AS2FORMAT = 0x20

# Fixed Lengths
BMP_PEERHDR_LEN = 42

# Peer Down Reason Codes
PEERDOWN_LOCAL_NOTIFICATION = 1
PEERDOWN_FSM = 2
PEERDOWN_REMOTE_NOTIFICATION = 3
PEERDOWN_REMOTE_UNKNOWN = 4
PEERDOWN_CONFIG = 5

class BMPPeerHeader( dpkt.Packet ):
   __hdr__ = (
      ( 'peer_type', 'B', PEER_GLOBAL ),
      ( 'peer_flags', 'B', 0 ),
      ( 'peer_distinguisher', '8s', 0 ),
      ( 'peer_address', '16s', 0 ),
      ( 'peer_as', 'L', 0 ),
      ( 'peer_bgp_id', 'L', 0 ),
      ( 'timestamp', 'L', 0 ),
      ( 'timestamp_us', 'L', 0 )
   )

   def unpack( self, buf ):
      dpkt.Packet.unpack( self, buf )
      self.data = ''

   def __len__( self ):
      return self.__hdr_len__

   def __str__( self ):
      return self.pack_hdr()


class BMP( dpkt.Packet ):
   __hdr__ = (
      ( 'version', 'B', 3 ),
      ( 'length', 'L', 6 ),
      ( 'type', 'B', INITIATION )
   )

   def unpack( self, buf ):
      dpkt.Packet.unpack( self, buf )
      self.data = self.data[ :self.length - self.__hdr_len__ ]
      if self.type == ROUTE_MONITORING:
         self.data = self.routemonitoring = self.RouteMonitoring( self.data )
      elif self.type == PEER_DOWN:
         self.data = self.peerdown = self.PeerDown( self.data )
      elif self.type == PEER_UP:
         self.data = self.peerup = self.PeerUp( self.data )
      elif self.type == INITIATION:
         self.data = self.initiation = self.Initiation( self.data )
      elif self.type == TERMINATION:
         self.data = self.termination = self.Termination( self.data )

   class RouteMonitoring( dpkt.Packet ):
      __hdr__ = ()

      def unpack( self, buf ):
         dpkt.Packet.unpack( self, buf )
         self.peer_hdr = BMPPeerHeader( self.data )
         self.data = self.data[ BMP_PEERHDR_LEN: ]
         self.update = BgpDpkt.BGP( self.data )


   class PeerDown( dpkt.Packet ):
      __hdr__ = ()

      def unpack( self, buf ):
         dpkt.Packet.unpack( self, buf )
         self.peer_hdr = BMPPeerHeader( self.data )
         self.data = self.data[ BMP_PEERHDR_LEN: ]
         self.reason = self.REASON( self.data )

      class REASON( dpkt.Packet ):
         __hdr__ = (
            ( 'reason', 'B', 0 ),
         )

         def unpack( self, buf ):
            dpkt.Packet.unpack( self, buf )

            if ( self.reason == PEERDOWN_LOCAL_NOTIFICATION ) or \
               ( self.reason == PEERDOWN_REMOTE_NOTIFICATION ):
               self.notification = BgpDpkt.BGP( self.data )
            elif self.reason == PEERDOWN_FSM:
               self.fsm_code = struct.unpack( '!H', self.data )[ 0 ]
               self.data = ''


   class PeerUp( dpkt.Packet ):
      __hdr__ = ()

      def unpack( self, buf ):
         dpkt.Packet.unpack( self, buf )
         self.peer_hdr = BMPPeerHeader( self.data )
         self.data = self.data[ BMP_PEERHDR_LEN: ]
         self.peer_up = self.PEERUP( self.data )

      class PEERUP( dpkt.Packet ):
         __hdr__ = (
            ( 'local_addr', '16s', 0 ),
            ( 'local_port', 'H', 0 ),
            ( 'remote_port', 'H', 0 )
         )

         def unpack( self, buf ):
            dpkt.Packet.unpack( self, buf )
            self.sent_open = BgpDpkt.BGP( self.data )
            self.data = self.data[ len( self.sent_open ): ]
            self.rcvd_open = BgpDpkt.BGP( self.data )
            self.data = self.data[ len( self.rcvd_open ): ]
            self.information = self.data

   class Initiation( dpkt.Packet ):
      __hdr__ = ()

      def unpack( self, buf ):
         dpkt.Packet.unpack( self, buf )
         tlvList = []
         tlvLen = len( self.data )
         while tlvLen > 0:
            tlv = self.TLV( self.data )
            self.data = self.data[ len( tlv ): ]
            tlvLen -= len( tlv )
            tlvList.append( tlv )
         self.data = self.tlvs_ = tlvList

      def __len__( self ):
         return self.__hdr_len__ + \
            sum( map( len, self.tlvs_ ) )

      def __str__( self ):
         tlvs = ''.join( map( str, self.tlvs_ ) )
         self.tlv_len = len( tlvs )
         return self.pack_hdr() + tlvs

      class TLV( dpkt.Packet ):
         __hdr__ = (
            ( 'type', 'H', 0 ),
            ( 'length', 'H', 0 )
         )

         def unpack( self, buf ):
            dpkt.Packet.unpack( self, buf )
            self.data = self.data[ :self.length ]

         def __len__( self ):
            return self.__hdr_len__ + self.length

         def __str__( self ):
            return self.pack_hdr() + self.data


   class Termination( dpkt.Packet ):
      __hdr__ = ()

      def unpack( self, buf ):
         dpkt.Packet.unpack( self, buf )
         tlvList = []
         tlvLen = len( self.data )
         while tlvLen > 0:
            tlv = self.TLV( self.data )
            self.data = self.data[len( tlv ): ]
            tlvLen -= len( tlv )
            tlvList.append( tlv )
         self.data = self.tlvs_ = tlvList

      def __len__( self ):
         return self.__hdr_len__ + \
            sum( map( len, self.tlvs_ ) )

      def __str__( self ):
         tlvs = ''.join( map( str, self.tlvs_ ) )
         self.tlv_len = len( tlvs )
         return self.pack_hdr() + tlvs

      class TLV( dpkt.Packet ):
         __hdr__ = (
            ( 'type', 'H', 0 ),
            ( 'length', 'H', 0 )
         )

         def unpack( self, buf ):
            dpkt.Packet.unpack( self, buf )
            self.data = self.data[ :self.length ]
            if self.type == TERM_REASON:
               self.reason_code = struct.unpack( '!H', self.data )[ 0 ]
               self.data = ''

         def __len__( self ):
            return self.__hdr_len__ + self.length

         def __str__( self ):
            return self.pack_hdr() + self.data
