#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from enum import Enum
import Tac
import Arnet
from IpLibConsts import DEFAULT_VRF
from socket import AF_INET, AF_INET6
from collections import namedtuple, OrderedDict
from RouteMapLib import ( getExtCommValue, commValueToPrint,
      CommunityType, getNumberFromCommValue )
import re
import UtmpDump
import Logging
import BgpLogMsgs

# pylint: disable-msg=C0321

Afi = Tac.Type( "Routing::Bgp::Afi" )
Safi = Tac.Type( "Routing::Bgp::Safi" )

Ip6AddrType = Tac.Type( 'Arnet::Ip6Addr' )
PeerConfigKeyType = Tac.Type( 'Routing::Bgp::PeerConfigKey' )

# Enum for Neighbor Cli No-or-default mode, and associated code.
class NoOrDefault( Enum ):
   NO = 1
   DEFAULT = 2

def vpnAfToAfiSafi( af ):
   """Helper method to convert vpnAf to its corresponding AfiSafi"""
   return Tac.Type( 'Routing::Bgp::AfiSafi' ).vpnAfToAfiSafi( af )

vpnAfTypeCliKeywords = sorted( [ 'evpn', 'vpn-ipv4', 'vpn-ipv6' ] )
vpnTypeAttrs = sorted( [ 'EtVpn', 'VpnIpv4', 'VpnIpv6' ] )
# Get all the vpnAf types discounting the unknown type
vpnAfs = sorted( set( Tac.Type( 'Routing::Bgp::VpnAf' ).attributes ) -
                 set( [ Tac.Type( 'Routing::Bgp::VpnAf' ).vpnAfUnknown ] ) )
vpnAfTypes = [ vpnAfToAfiSafi( vpnAf ) for vpnAf in vpnAfs ]

# Maps used to translate the CLI vpnType token to the suffix used in the
# routeTarget{Import|Export}" attribute within Routing::Bgp::Config and vice-versa
vpnTypeAttrMap = {}
vpnTypeAttrMapInv = {}

# Maps used to translate the CLI vpnType token to the vpnAf key used in
# the routeMap{Import|Export}" collections within Routing::Bgp::Config and vice-versa
vpnAfTypeMap = {}
vpnAfTypeMapInv = {}

vpnTypeAttrMap = dict( zip( vpnTypeAttrs, vpnAfTypeCliKeywords ) )
vpnTypeAttrMapInv = { v: k for k, v in vpnTypeAttrMap.iteritems() }
vpnAfTypeMap = dict( zip( vpnAfTypes, vpnAfTypeCliKeywords ) )
vpnAfTypeMapInv = { v: k for k, v in vpnAfTypeMap.iteritems() }

v4Uni = Tac.const( Tac.Value( "Routing::Bgp::AfiSafi", 'afiIpv4', 'safiUnicast' ) )
v6Uni = Tac.const( Tac.Value( "Routing::Bgp::AfiSafi", 'afiIpv6', 'safiUnicast' ) )
l2vpnEvpn = Tac.const( Tac.Value( "Routing::Bgp::AfiSafi", 'afiL2vpn', 'safiEvpn' ) )
v4Lu = Tac.const( Tac.Value( "Routing::Bgp::AfiSafi", 'afiIpv4', 'safiMplsLabels' ) )
v6Lu = Tac.const( Tac.Value( "Routing::Bgp::AfiSafi", 'afiIpv6', 'safiMplsLabels' ) )
v4Multicast = Tac.const( Tac.Value(
                           "Routing::Bgp::AfiSafi", 'afiIpv4', 'safiMulticast' ) )
v6Multicast = Tac.const( Tac.Value(
                           "Routing::Bgp::AfiSafi", 'afiIpv6', 'safiMulticast' ) )
v4MplsVpn = Tac.const( Tac.Value(
                           "Routing::Bgp::AfiSafi", 'afiIpv4', 'safiMplsVpn' ) )
v6MplsVpn = Tac.const( Tac.Value(
                           "Routing::Bgp::AfiSafi", 'afiIpv6', 'safiMplsVpn' ) )
evpn = Tac.const( Tac.Value( "Routing::Bgp::AfiSafi", 'afiL2vpn', 'safiEvpn' ) )
esAfiSafi = Tac.const( Tac.Value(
                           "Routing::Bgp::AfiSafi", 'afi48BitMac', 'safiPriv244' ) )
unknownAfi = Tac.const( Tac.Value(
                           "Routing::Bgp::AfiSafi", 'afiUnknown', 'safiNone' ) )
cliAfToAf = {
      'ipv4': v4Uni,
      'ipv4-labeled-unicast': v4Lu,
      'ipv4-multicast': v4Multicast,
      'ipv6': v6Uni,
      'ipv6-labeled-unicast': v6Lu,
      'ipv6-multicast': v6Multicast,
      'evpn': evpn,
      'vpn-ipv4': v4MplsVpn,
      'vpn-ipv6': v6MplsVpn,
}
afToCliAf = { v: k for k, v in cliAfToAf.iteritems() }

def PeerConfigKey( s ):
   """
   This routine is used by Bgp CliPlugin code to construct PeerConfigKey objects from
   strings as in PeerConfigKey( '1.1.1.1' ). To create a "null" PeerConfigKey, use
   PeerConfigKeyType().
   """
   if type( s ) is PeerConfigKeyType:
      return s
   return Tac.ValueConst( "Routing::Bgp::PeerConfigKey", stringValue=str( s ) )

def stripLinkLocalIfName( peerKey ):
   """
   Certain test infrastructure expects link-local next-hops to be represented only
   using the link-local address (and not qualified by '%<ifname>'). This helper
   routine robustly strips the ifname, if necessary, returning the unaltered
   next-hop, if not.
   """
   key = PeerConfigKey( peerKey )
   return key.v6Addr.stringValue if key.type == 'peerIpv6Ll' else key.stringValue

def isValidV6PeerAddr( addr ):
   try:
      if type( addr ) is Ip6AddrType:
         return True

      # if we match an ipv4 addr ignore
      if Arnet.IpAddrFullCompiledRe.match( addr ):
         return False

      m = Arnet.Ip6AddrCompiledRe.match( addr )
      if m is None or m.group( 0 ) != addr:
         return False

      Arnet.IpAddress( addr, addrFamily=AF_INET6 )
   except ValueError:
      return False
   else:
      return True

def isValidLinkLocalPeer( addr ):
   m = re.match( r'[0-9a-fA-F:]+%[0-9a-zA-Z]+', addr )
   if m and m.group() == addr:
      return True
   else:
      return False

def isValidPeerAddr( addr ):
   """
   Function to verify whether the addr string in PeerConfig is a valid neighbor
   IP address or the name for a peer-group
   """
   return isValidV6PeerAddr( addr ) or Arnet.IpAddrFullCompiledRe.match( addr )

def createKey( addrOrName ):
   return PeerConfigKey( addrOrName )

# The TACC models defined in /src/Bgp/Bgp.tac contains AF specific attributes
# for configuration under address-family mode, where the AF is added explicitly
# to the attribute name string.
#
# The following is used generate a mapping used to lookup Bgp config and
# Peer config attributes by address-family. The generated dict has the
# following structure:
#
# {'af': {'evpn': 'afEvpn',
#         'ipv4': 'afV4Uni',
#         'ipv4 labeled-unicast': 'afV4UniLab',
#         'ipv4 sr-te': 'afV4SrTe',
#         'ipv6': 'afV6Uni',
#         'ipv6 labeled-unicast': 'afV6UniLab',
#         'ipv6 sr-te': 'afV6SrTe',
#         'link-state': 'afLinkState',
#         'path-selection': 'afDps',
#         'vpn-ipv4': 'afVpnV4',
#         'vpn-ipv6': 'afVpnV6',
#         'flow-spec ipv4': 'afV4Flowspec',
#         'flow-spec ipv6': 'afV6Flowspec'},
#  'apRecv': {'all': 'apRecv',
#             'evpn': 'apRecvEvpn',
#             'ipv4': 'apRecvIPv4Uni',
#             'ipv4 labeled-unicast': 'apRecvIPv4LabeledUni',
#             'ipv6 labeled-unicast': 'apRecvIPv6LabeledUni',
#             'ipv6': 'apRecvIPv6Uni',
#             'vpn-ipv4': 'apRecvVpnV4',
#             'vpn-ipv6': 'apRecvVpnV6'},
#  'apSend': {'all': 'apSend', 'ipv4': 'apSendIPv4Uni', 'ipv6': 'apSendIPv6Uni'},
#  'routeMapIn': {'all': 'routeMapIn',
#                 'evpn': 'routeMapEvpnIn',
#                 'ipv4': 'routeMapV4UniIn',
#                 'ipv6': 'routeMapV6UniIn',
#                 'vpn-ipv4': 'routeMapVpnV4In',
#                 'vpn-ipv6': 'routeMapVpnV6In'},
#  'routeMapOut': {'all': 'routeMapOut',
#                 'evpn': 'routeMapEvpnOut',
#                 'ipv4': 'routeMapV4UniOut',
#                 'ipv6': 'routeMapV6UniOut',
#                 'vpn-ipv4': 'routeMapVpnV4Out',
#                 'vpn-ipv6': 'routeMapVpnV6Out'}}
#
# For a given attribute type and AF, we can retrieve the attribute name
# corresponding to that AF. Similarly, we can also determine all the attributes
# corresponding to a given AF.

addressFamilies = { 'ipv4': ( 'afiIpv4', 'safiUnicast' ),
                    'ipv6': ( 'afiIpv6', 'safiUnicast' ),
                    'evpn': ( 'afiL2vpn', 'safiEvpn' ),
                    'ipv4 multicast': ( 'afiIpv4', 'safiMulticast' ),
                    'ipv6 multicast': ( 'afiIpv6', 'safiMulticast' ),
                    'ipv4 labeled-unicast': ( 'afiIpv4', 'safiMplsLabels' ),
                    'ipv6 labeled-unicast': ( 'afiIpv6', 'safiMplsLabels' ),
                    'flow-spec ipv4': ( 'afiIpv4', 'safiFlowspec' ),
                    'flow-spec ipv6': ( 'afiIpv6', 'safiFlowspec' ),
                    'ipv4 sr-te': ( 'afiIpv4', 'safiSrTe' ),
                    'ipv6 sr-te': ( 'afiIpv6', 'safiSrTe' ),
                    'vpn-ipv4': ( 'afiIpv4', 'safiMplsVpn' ),
                    'vpn-ipv6': ( 'afiIpv6', 'safiMplsVpn' ),
                    'rt-membership': ( 'afiIpv4', 'safiRtMembership' ),
                    'link-state': ( 'afiLinkState', 'safiLinkState' ),
                    'path-selection': ( 'afiIpv4', 'safiDps' ),
                    }

# The following maps a given AF against substrings that are appended in
# attribute names. Unfortunately, our naming pattern for AF specific attributes
# are not consistent, so we need to maintain a list of all possible substrings
# for each AF.
AfDesc = namedtuple( 'AfDesc', 'afStrs vrfSupported' )
AfDesc.__new__.__defaults__ = ( [], False )

afDescMap = OrderedDict()
afDescMap[ 'ipv4' ] = AfDesc( [ 'V4', 'AfV4', 'V4Uni', 'IPv4Uni' ], True )
afDescMap[ 'ipv6' ] = AfDesc( [ 'V6', 'AfV6', 'V6Uni', 'IPv6Uni' ], True )
afDescMap[ 'ipv4 labeled-unicast' ] = AfDesc( [ 'V4UniLab', 'IPv4LabeledUni',
                                                'AfV4LabeledUni' ] )
afDescMap[ 'ipv6 labeled-unicast' ] = AfDesc( [ 'V6UniLab', 'IPv6LabeledUni',
                                                'AfV6LabeledUni' ] )
afDescMap[ 'ipv4 multicast' ] = AfDesc( [ 'V4Multicast', 'AfV4Multicast' ], True )
afDescMap[ 'ipv6 multicast' ] = AfDesc( [ 'V6Multicast', 'AfV6Multicast' ], True )
afDescMap[ 'evpn' ] = AfDesc( [ 'Evpn', 'AfEvpn' ] )
afDescMap[ 'ipv4 sr-te' ] = AfDesc( [ 'V4UniSrTe', 'V4SrTe', 'AfV4SrTe' ] )
afDescMap[ 'ipv6 sr-te' ] = AfDesc( [ 'V6UniSrTe', 'V6SrTe', 'AfV6SrTe' ] )
afDescMap[ 'link-state' ] = AfDesc( [ 'LinkState', 'AfLinkState' ] )
afDescMap[ 'path-selection' ] = AfDesc( [ 'Dps', 'AfDps' ] )

# List of AFs supported by VRF
def bgpConfigAttrAfList( vrfName=DEFAULT_VRF ):
   return [ af for af, afDesc in afDescMap.iteritems() \
               if vrfName == DEFAULT_VRF or afDesc.vrfSupported ]

def initConfigAttrsAfMap( configType, configAttrsAfMap ):
   # Clear configAttrsAfMap and rebuild it so that there is no duplication.
   configAttrsAfMap.clear()
   attrNames = Tac.Type( configType ).attributes
   for attrName in attrNames:
      afMatch = None
      afMatchStr = None
      for af, afDesc in afDescMap.iteritems():
         for afStr in afDesc.afStrs:
            # Skip the Default/Present attributes. Also skip the
            # defaultV4/defaultV6  since those are not configured
            # under address-family mode.
            if attrName.endswith( 'Default' ) or \
                  attrName.endswith( 'Present' ) or \
                  attrName.startswith( 'defaultV4' ) or \
                  attrName.startswith( 'defaultV6' ):
               continue
            # Otherwise, check whether the afStr is present in the attrName. We
            # save the longest matching afStr so that we can match the most
            # specific af. For example, afV4UniLab should match against
            # V4UniLab instead of V4Uni.
            if afStr in attrName and \
                  ( afMatchStr is None or len( afMatchStr ) < len( afStr ) ):
               afMatch = af
               afMatchStr = afStr

      if afMatch:
         attrNameAfBase = attrName.replace( afMatchStr, '' )
         afMap = configAttrsAfMap.get( attrNameAfBase )
         if not afMap:
            afMap = {}
            configAttrsAfMap[ attrNameAfBase ] = afMap
            if attrNameAfBase in attrNames:
               afMap[ 'all' ] = attrNameAfBase
         afMap[ afMatch ] = attrName

# Initializes the map to lookup Bgp config attributes by AF
bgpConfigAttrsAfMap = {}
initConfigAttrsAfMap( "Routing::Bgp::Config", bgpConfigAttrsAfMap )

# Initializes the map to lookup Peer config attributes by AF
peerConfigAttrsAfMap = {}
initConfigAttrsAfMap( "Routing::Bgp::PeerConfig", peerConfigAttrsAfMap )

# @afMap : Dict with AF that needs to be appended to afDescMap as key
# and list of possible suffixes in the attribute names for the AF as value
# @vrfSupported: bool suggesting if true, af configuration supported in
# non-default vrfs, else not.
def updateConfigAttrsAfMap( afMap, vrfSupported=False ):
   # Update afDescMap to also contain the new address families.
   # It is used to build address family lists by the CliSave plugin.
   for af, afStrs in afMap.iteritems():
      afDescMap[ af ] = AfDesc( afStrs, vrfSupported )
   # Rebuild bgpConfigAttrsAfMap and peerConfigAttrsAfMap according
   # to the updated afDescMap
   initConfigAttrsAfMap( "Routing::Bgp::Config", bgpConfigAttrsAfMap )
   initConfigAttrsAfMap( "Routing::Bgp::PeerConfig", peerConfigAttrsAfMap )

attributesNotNeedingCleanup = set( ( 'parent', 'asNumber', 'defaultPeerConfig' ) )
collAttrsToSkip = set( (
   # ignore this common entity attribute that happens to be a collection
   'entity',
   # NOTE: must not clear default, pre-populated collections since we need to
   # use these to restore the active collection with default values
   # its unfortunate we can't just use the collAttrsDelAll() function :(
   'nhResolutionRibProfileConfigDefault',
   ) )

def cleanupAsnConfig( config ):
   config.asnNotation = config.asnNotationDefault

def cleanupBgpConfig( config ):
   # clear all the collections
   # NOTE: not using collAttrsDelAll() because this PyClient proxy doesn't provide
   # access to methods of the entity - so instead just clear() all the collections
   collAttrs = [ i.name for i in config.tacType.attributeQ if i.isCollection ]
   collAttrs = set( collAttrs ) - collAttrsToSkip
   for attr in collAttrs:
      getattr( config, attr ).clear()

   # set remaining, writable, non-collection attributes to defaults values
   entAttrs = [ i.name for i in config.tacType.attributeQ if i.writable and
                                                          not i.isCollection ]
   cleanupAttrs = ( set( entAttrs ) - attributesNotNeedingCleanup )
   # TODO: Try a deepcopy from defaultBgpConfig
   defaultBgpConfig = Tac.root.newEntity( 'Routing::Bgp::Config', 'defaultBgp' )
   for attr in cleanupAttrs:
      setattr( config, attr, getattr( defaultBgpConfig, attr ) )
   config.maxEcmp = config.maxEcmpInvalid
   config.confedId = 0

   # NOTE: lacking access to collAttrsDelAll() function means we have to manually
   # restore active configs from the default config values
   for key in getattr( config, 'nhResolutionRibProfileConfigDefault' ).keys():
      getattr( config, 'nhResolutionRibProfileConfig' )[ key ] = \
            getattr( config, 'nhResolutionRibProfileConfigDefault' )[ key ]

def cleanupUcmpConfig( config ):
   # set config to all defaults
   defaultUcmpConfig = Tac.root.newEntity( 'RoutingLib::Ucmp::UcmpConfig',
                                           'defaultUcmp' )
   config.ucmpEnabled = defaultUcmpConfig.ucmpEnabled
   config.maxUcmp = defaultUcmpConfig.maxUcmpInvalid
   config.ucmpDeviation = defaultUcmpConfig.ucmpDeviationInvalid
   config.ucmpLinkbwDelay = defaultUcmpConfig.ucmpLinkbwDelayInvalid
   config.ucmpLinkbwWeighted = defaultUcmpConfig.ucmpLinkbwWeighted
   config.ucmpLinkbwRecursive = defaultUcmpConfig.ucmpLinkbwRecursive

   config.ucmpTriggerFecThreshold = defaultUcmpConfig.ucmpTriggerFecThreshold
   config.ucmpClearFecThreshold = defaultUcmpConfig.ucmpClearFecThreshold

def bgpGRConfigSetting( vrfBgpConfig, addrFamily ):
   # Use the fact that GR cannot be configured both globally and in address family
   # specific way. 
   bgpGRConfigValues = []
   if addrFamily == AF_INET:
      bgpGRConfigValues.append( vrfBgpConfig.gracefulRestartAfV4 )
   else:
      bgpGRConfigValues.append( vrfBgpConfig.gracefulRestartAfV6 )

   bgpGRConfigValues = [ vrfBgpConfig.gracefulRestart ]

   configured = False
   configValue = None
   for bgpGRConfigValue in bgpGRConfigValues:
      if bgpGRConfigValue != "isInvalid" :
         configured = True
         configValue = True if bgpGRConfigValue == "isTrue" else False
         break
   
   return ( configured, configValue )

   
def nbrGRConfigSetting(  nbrConfig, addrFamily ):
   nbrGRConfigPresentFlags = []
   nbrGRConfigValues = []
   if addrFamily == AF_INET:
      nbrGRConfigPresentFlags.append( nbrConfig.gracefulRestartAfV4Present )
      nbrGRConfigValues.append( nbrConfig.gracefulRestartAfV4 )
   else:
      nbrGRConfigPresentFlags.append( nbrConfig.gracefulRestartAfV6Present )
      nbrGRConfigValues.append( nbrConfig.gracefulRestartAfV6 )
      
   nbrGRConfigPresentFlags.append( nbrConfig.gracefulRestartPresent )
   nbrGRConfigValues.append( nbrConfig.gracefulRestart )

   configured = False
   configValue = None
   for nbrGRConfigPresentFlag, nbrGRConfigValue in zip(
      nbrGRConfigPresentFlags, nbrGRConfigValues ):
      if nbrGRConfigPresentFlag:
         configured = True
         configValue = nbrGRConfigValue
         break

   return ( configured, configValue )

def isBgpGRNbrImpacted( vrfName, neighbor,
                        vrfBgpConfig, defaultVrfBgpConfig ):
   
   bgpConfigs = [ vrfBgpConfig ]
   nbrConfigs = [ vrfBgpConfig.neighborConfig[ neighbor ] ]
   if vrfName != DEFAULT_VRF:
      bgpConfigs.append( defaultVrfBgpConfig )

   if ( nbrConfigs[0].isPeerGroupPeer ):
      pg = nbrConfigs[0].peerGroupKey
      if pg in defaultVrfBgpConfig.neighborConfig:
         nbrConfigs.append( defaultVrfBgpConfig.neighborConfig[ pg ] )

   # GR is disabled by default unless explicitly configured.
   grEnabled = { AF_INET  : False,
                 AF_INET6 : False }

   for addrFamily in ( AF_INET, AF_INET6 ):
      for config in nbrConfigs + bgpConfigs:
         if type( config ) == Tac.Type( "Routing::Bgp::Config" ):
            grConfigSetting = bgpGRConfigSetting
         else:
            assert type( config ) == Tac.Type( "Routing::Bgp::PeerConfig" )
            grConfigSetting = nbrGRConfigSetting
            
         configured, configValue = grConfigSetting( config, addrFamily )
         if configured:
            grEnabled[ addrFamily ] = configValue
            break
            
   return False in grEnabled.values()
      
def bgpNsfGRWarn( bgpConfig, bgpVrfConfigDir ):
   # This function returns a list of GrWarnInfo tuples indicating
   # neighbors that are impacted if user proceeds with asu when bgp is configured 
   # but not GR. We may warning in some false positive case as we blindly look at
   # whether GR is enabled or not in neighbor's configare and do not know if
   # neighbor is actually GR capable or not.
   # For each vrf including default vrf
   #   -  We do not throw warning if bgp is not configured( no as number )
   #         or is configured to be down( shutdown )
   #   -  We iterate through neighbors to see if GR is disabled or unconfigured.
   #         and through warning if we come across one or more neighbors with GR
   #         disabled or unconfigured.

   GrWarnInfo = namedtuple( "GrWarnInfo", [ "vrfName", "neighborsImpacted" ] )
   def vrfBgpGRWarn( vrfName, vrfBgpConfig, defaultVrfBgpConfig ):
      if( vrfConfig.asNumber == 0 or
          vrfConfig.shutdown ):
         # Bgp is not configured or it is configured to be down.
         return None

      nbrsImpacted = []
      for nbr in vrfBgpConfig.neighborConfig:
         if( isBgpGRNbrImpacted( vrfName, nbr, vrfBgpConfig,
                                 defaultVrfBgpConfig ) ):
            nbrsImpacted.append( nbr )

      if nbrsImpacted:
         return GrWarnInfo( vrfName, nbrsImpacted )

      return None

   warnList = []
   vrfNames = [ DEFAULT_VRF ] + bgpVrfConfigDir.vrfConfig.keys()
   for vrfName in vrfNames:
      vrfConfig = bgpConfig if vrfName == DEFAULT_VRF else \
                  bgpVrfConfigDir.vrfConfig[ vrfName ]
      grWarnInfo = vrfBgpGRWarn( vrfName, vrfConfig, bgpConfig )
      if grWarnInfo:
         warnList.append( grWarnInfo )

   return warnList

# Returns a tuple of two booleans indicating if graceful-restart is configured
# at the global level or at any address-familiy levels
def isGracefulRestartGlobalOrAfConfigured( config ):
   isPeerConfig = ( type( config ) == Tac.Type( "Routing::Bgp::PeerConfig" ) )
   grAttrs = peerConfigAttrsAfMap[ 'gracefulRestart' ]
   anyAfConfigured = False
   allAfConfigured = False
   for af, attr in grAttrs.iteritems():
      if ( ( isPeerConfig and getattr( config, attr + 'Present' ) ) or \
            ( not isPeerConfig and getattr( config, attr ) != 'isInvalid' ) ):
         if af == 'all':
            allAfConfigured = True
         else:
            anyAfConfigured = True
   return ( allAfConfigured, anyAfConfigured )


def haveConflictingGracefulRestart( noOrDefault, addrFamily,
                                    anyAfConfigured, allAfConfigured ):
   # Global and Af GR commands may pass "default", while Neighbor commands use the
   # enum.  Eventually all commands will use the enum.
   return ( noOrDefault not in [ "default", NoOrDefault.DEFAULT ] and
            ( ( not addrFamily and anyAfConfigured ) or
              ( addrFamily and allAfConfigured ) ) )

def routeTargetToExtCommU64Value( rt ):
   extComm = getExtCommValue( 'rt ' + rt )
   return getNumberFromCommValue( extComm )

def routeTargetToPrint( extCommValue, asdotConfigured=False ):
   value = getExtCommValue( extCommValue )
   return commValueToPrint( value, commType=CommunityType.communityTypeExtended,
                            asdotConfigured=asdotConfigured, withTypeStr=False )

def createPeerSpecifier( prefix, value, useAsn=True, includeRouterId=False ):
   peerSpec = Tac.Value( 'Routing::Bgp::PeerSpecifier', prefix )
   asnOrPeerFilter = Tac.Value( 'Routing::Bgp::AsnOrPeerFilter' )
   asnOrPeerFilter.hasAsn = useAsn
   if useAsn:
      asnOrPeerFilter.asn = int( value )
   else:
      asnOrPeerFilter.peerFilter = str( value )
   peerSpec.asnOrPeerFilter = asnOrPeerFilter
   peerSpec.includeRouterId = includeRouterId
   return peerSpec

def getPeerSpecifierValue( peerSpec ):
   if peerSpec.hasAsn:
      return peerSpec.asn
   else:
      return peerSpec.peerFilter

def cmpPeerSpecifier( peerSpec, hasAsn, value ):
   return peerSpec.hasAsn == hasAsn and \
       getPeerSpecifierValue( peerSpec ) == value

def doLogClearIpBgp( mode, transportAddrFamily=None, peer=None, soft=False,
                     direction=None, vrfName=None, errors=False,
                     counters=False, resetMsg='' ):
   if peer is None:
      if transportAddrFamily is None:
         mode.addWarning( "Clearing all IPv4 and IPv6 peering sessions" )
      elif transportAddrFamily == 'ip':
         mode.addWarning( "Clearing all IPv4 peering sessions" )
      elif transportAddrFamily == 'ipv6':
         mode.addWarning( "Clearing all IPv6 peering sessions" )

   peeringStr = " peering%s " % ( 's' if peer is None else '' )
   objectStr = " counters " if counters else " errors " if errors else peeringStr

   vrfStr = "(vrf %s)" % vrfName

   if peer is None:
      if transportAddrFamily is None:
         targetStr = " all neighbors "
      elif transportAddrFamily == 'ip':
         targetStr = " all IPv4 neighbors "
      elif transportAddrFamily == 'ipv6':
         targetStr = " all IPv6 neighbors "
      else:
         raise Exception( "Invalid address family" )
   else:
      if peer.type == 'peerGroup':
         targetStr = " peer-group %s " % ( peer.stringValue )
      else:
         targetStr = " neighbor %s " % ( peer.stringValue )

   directionStr = '(in)' if direction == 'in' else \
                  '(out)' if direction == 'out' else \
                  '(in and out)'

   actionStr = ' reset ' if errors or counters else \
               ' hard reset ' if not ( soft or direction ) else \
               ' soft %s reset ' % ( directionStr )

   wasStr = ' were' if errors or counters or peer is None else ' was'

   reasonStr = ' reason: %s' % resetMsg if resetMsg else ''
   logStr = objectStr + 'for' + targetStr + vrfStr + wasStr + actionStr

   uInfo = UtmpDump.getUserInfo()

   # pylint: disable-msg=E1101
   Logging.log( BgpLogMsgs.BGP_PEER_CLEAR, logStr.strip(),
                uInfo[ 'user' ], uInfo[ 'tty' ], uInfo[ 'ipAddr' ],
                reasonStr )

def getVpwsName( vpwsEviName ):
   return "vpws." + vpwsEviName

def getVpwsEviName( vpwsName ):
   return vpwsName[ 5 : ] if vpwsName.startswith( "vpws." ) else ""
