#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import time
from socket import AF_INET, AF_INET6, inet_pton
import Tac
IntervalType = Tac.Type( 'Bfd::BfdInterval' )
MultiplierType = Tac.Type( 'Bfd::BfdMultiplier' )
operSessionType = Tac.Type( "Bfd::OperSessionType" )
perlinkMode = Tac.Type( "Bfd::PerlinkMode" )

class VxlanPeerConfig( object ):
   def __init__( self, ip, flags, vni, innerSrcIp, innerDstMac, innerSrcMac,
                 outerDstIp, outerSrcIp, outerDstPort, outerSrcPort, 
                 minTx=IntervalType.defval, minRx=IntervalType.defval,
                 multiplier=MultiplierType.defval ):
      ip = Tac.Value( 'Arnet::IpGenAddr', ip )
      self.peer = Tac.Value( 'Bfd::Peer', ip, 'default' )
      self.peer.type = 'vxlanTunnel'
      self.vxlanTunnel = Tac.Value( 'Bfd::VxlanTunnel' )
      self.vxlanTunnel.flags = flags
      vniTacValue = Tac.Value( 'Arnet::VxlanVniReserved' )
      vniTacValue.vni = vni
      self.vxlanTunnel.vni = vniTacValue
      self.vxlanTunnel.innerSrcIp = Tac.Value( 'Arnet::IpGenAddr', innerSrcIp )
      self.vxlanTunnel.innerDstMac = innerDstMac
      self.vxlanTunnel.innerSrcMac = innerSrcMac
      self.vxlanTunnel.outerDstIp = Tac.Value( 'Arnet::IpGenAddr', outerDstIp )
      self.vxlanTunnel.outerSrcIp = Tac.Value( 'Arnet::IpGenAddr', outerSrcIp )
      self.vxlanTunnel.outerDstPort = outerDstPort
      self.vxlanTunnel.outerSrcPort = outerSrcPort
      self.intervalParams = Tac.Value( 'Bfd::BfdIntervalConfig', minTx, minRx, 
                                       multiplier )
      self.intf = ''

def getVxlanValidFlags():
   flags = Tac.Value( 'Arnet::VxlanFlags', 0 )
   flags.vniValid = 1
   return flags

def compare_peer( x, y ):
   if x[ 0 ].startswith( "Port-Channel" ):
      return -1
   elif y[ 0 ].startswith( "Port-Channel" ):
      return 1
   else:
      if x[ 0 ] < y[ 0 ]:
         return -1
      elif x[ 0 ] > y[ 0 ]:
         return 1
      else:
         return 0

def compareIp( x, y ):
   ip1, ip2 = x[ 0 ], y[ 0 ]
   addrFamily = AF_INET if '.' in str( ip1 ) else AF_INET6
   ip1 = inet_pton( addrFamily, str( ip1 ) )
   ip2 = inet_pton( addrFamily, str( ip2 ) )
   if ip1 < ip2:
      return -1
   elif ip1 > ip2:
      return 1
   return 0

def compareHwSession( x, y ):
   ipX = x.ip if hasattr( x, 'ip' ) else x[ 'ip' ]
   ipY = y.ip if hasattr( y, 'ip' ) else y[ 'ip' ]
   addrFamilyX = AF_INET if '.' in str( ipX ) else AF_INET6
   addrFamilyY = AF_INET if '.' in str( ipY ) else AF_INET6
   if addrFamilyX != addrFamilyY:
      return -1 if addrFamilyX == AF_INET else 1
   return compareIp( [ ipX ], [ ipY ] )

def dispTime( timeSecs, shortDisp=False ):
   if timeSecs == 0:
      return "NA"

   if shortDisp is False:
      disp = "%s" % ( time.strftime( "%b %d %H:%M:%S %Y",
                                     time.localtime( timeSecs ) ) )
   else:
      disp = "%s" % ( time.strftime( "%m/%d/%y %H:%M",
                                     time.localtime( timeSecs ) ) )
   return disp

diagEnumToReason = { 'diagNone' :               'No Diagnostic',
                     'diagCtrlTimeout' :        'Detect Time Exp',
                     'diagEchoFail' :           'Echo Function Failed',
                     'diagNeighDown' :          'Nbr Signaled Down',
                     'diagForwardingReset' :    'Forwarding Plane Reset',
                     'diagPathDown' :           'Path Down',
                     'diagConcatPathDown' :     'Concatenated Path Down',
                     'diagAdminDown' :          'Administratively Down',
                     'diagRevConcatPathDown' :  'Reverse Concatenated Path Down',
                   }

authType = Tac.Type( "Bfd::BfdAuthType" )
authEnumTypeToText = { authType.authNone           : "None",
                       authType.authPassword       : "Password",
                       authType.authKeyedMd5       : "Keyed Md5",
                       authType.authMeticulousMd5  : "Meticulous Md5",
                       authType.authKeyedSha1      : "Keyed Sha1",
                       authType.authMeticulousSha1 : "Meticulous Sha1",
                       authType.authInherit        : "None",
                       authType.authDebugSeqNum    : "None", }

authEnumNumToText = { 0    : "None",
                      1    : "Password",
                      2    : "Keyed Md5",
                      3    : "Meticulous Md5",
                      4    : "Keyed Sha1",
                      5    : "Meticulous Sha1",
                      254  : "None",
                      255  : "None", }


authTypeDict = { authType.authNone : 0,
                 authType.authPassword : 1,
                 authType.authKeyedMd5 : 2,
                 authType.authMeticulousMd5 : 3,
                 authType.authKeyedSha1 : 4,
                 authType.authMeticulousSha1 : 5,
                 authType.authInherit : 254,
                 authType.authDebugSeqNum : 255 }

operSessionEnumToType = { operSessionType.sessionTypeNormal :        'normal',
                          operSessionType.sessionTypeLagLegacy :     'per-link',
                          operSessionType.sessionTypeLagRfc7130 :    'RFC7130',
                          operSessionType.sessionTypeMicroLegacy :   'per-link',
                          operSessionType.sessionTypeMicroRfc7130 :  'RFC7130',
                          operSessionType.sessionTypeVxlanTunnel :   'VXLAN',
                          operSessionType.sessionTypeMultihop :      'multihop',
                          operSessionType.sessionTypeSbfdInitiator : 'initiator',
                          operSessionType.sessionTypeSbfdReflector : 'reflector',
                        }

def operSessionTypeForPerlink( perLink ):
   if perLink == perlinkMode.rfc7130 or perLink == perlinkMode.rfc7130Interop :
      return operSessionEnumToType[ operSessionType.sessionTypeLagRfc7130 ]
   if perLink == perlinkMode.legacy:
      return operSessionEnumToType[ operSessionType.sessionTypeLagLegacy ]
   return operSessionEnumToType[ operSessionType.sessionTypeNormal ]
 
diagEnumToValue = {  'diagNone' :               0,
                     'diagCtrlTimeout' :        1,
                     'diagEchoFail' :           2,
                     'diagNeighDown' :          3,
                     'diagForwardingReset' :    4,
                     'diagPathDown' :           5,
                     'diagConcatPathDown' :     6,
                     'diagAdminDown' :          7,
                     'diagRevConcatPathDown' :  8,
                   }

def sbfdLocalIntfDefault():
   return Tac.Value( 'Arnet::IntfId', '' )

def initiatorConfigDefault():
   intervalDefault = Tac.Type( 'Bfd::BfdInterval' ).defval
   multDefault = Tac.Type( 'Bfd::BfdMultiplier' ).defval
   return Tac.Value( 'Bfd::InitiatorTimerConfig', intervalDefault, multDefault )

def reflectorMinRxDefault():
   return Tac.Type( 'Bfd::BfdInterval' ).defval

def reflectorLocalDiscIpDefault():
   return Tac.Value( 'Arnet::IpAddr', 0 )
