#!/usr/bin/python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import collections
import errno
import os
import pprint
import re
import select
import socket
import sys
import textwrap
import threading
import time
from functools import partial

import Ark
import BasicCliModes
import CliAaa
import CliCommon
import ConfigureLock
import CliInputInterface
import CliParser
import CliParserCommon
import CliSession
import CliSessionDataStore
import CliSessionOnCommit
import ConfigMount
import LazyMount
import Logging
import MemoryStats
import ShowCommand
from Syscall import gettid
import Tac
import TacSigint
import TerminalUtil
import Tracing
import UtmpDump

th = Tracing.defaultTraceHandle()
t0 = th.trace0
t2 = th.trace2
t3 = th.trace3
t7 = th.trace7
t8 = th.trace8

_compiledRegexAliases = None
_regexAliasLock = threading.Lock()
aliasParamRe = re.compile( r'%([1-9]\d*)$' )
reAliasParamRe = re.compile( aliasParamRe.pattern[ : -1 ] )

CONFIG_LOCK = ConfigureLock.ConfigureLock()
cmdHistoryPos = 0
DEFAULT_CMD_HISTORY_SIZE = 100
CMD_HISTORY_SIZE = DEFAULT_CMD_HISTORY_SIZE
CMD_HISTORY = [ None ] * CMD_HISTORY_SIZE
THREAD_ID_CMD_MAPPING = collections.defaultdict( list )

Logging.logD( id='SYS_DEPRECATED_CMD',
   severity=Logging.logNotice,
   format=( 'Deprecated command \'%s\' was executed outside of startup-config by '
            '%s on %s (%s)' ),
   explanation=( 'Deprecated commands will not be allowed to '
                  'execute outside of startup-config.' ),
   recommendedAction=Logging.NO_ACTION_REQUIRED )

class CmdHistory( object ):
   def __init__( self ):
      self.threadId = gettid()
      self.cmd = None
      self.startTime = time.time()
      self.endTime = None

   def duration( self ):
      # duration that the command ran. Will return the amount of time run
      # if we don't have an endTime yet
      if self.endTime:
         return self.endTime - self.startTime
      else:
         return time.time() - self.startTime

   def __str__( self ):
      cmd = self.cmd if self.cmd is not None else '***Parsing***'
      return ( 'TID: %s, Duration: %s%s, Start: %s, End: %s, Cmd: %s' %
               ( self.threadId, self.duration(), '' if self.endTime else '*',
                 self.startTime, self.endTime, cmd ) )

def resetRegexAliases( mode, onSessionCommit=True ):
   global _compiledRegexAliases
   _compiledRegexAliases = None

def getCompiledRegexAliases( sysdbRegexAliases ):
   global _compiledRegexAliases
   if ( _compiledRegexAliases is None or
        len( _compiledRegexAliases ) != len( sysdbRegexAliases ) ):
      # Config agent restarted; reload aliases.
      with _regexAliasLock:
         newAliases = collections.OrderedDict()
         for index, alias in sysdbRegexAliases.items():
            newAliases[ re.compile( alias.cmdAlias ) ] = index
         _compiledRegexAliases = newAliases

   return _compiledRegexAliases

@Ark.synchronized( _regexAliasLock )
def addCompiledRegex( compiledRegex, sysdbRegexAliases ):
   # Make a copy so we don't disturb potential readers.
   global _compiledRegexAliases
   newAliases = collections.OrderedDict( _compiledRegexAliases or {} )

   key = sysdbRegexAliases.keys()[ -1 ] + 1 if sysdbRegexAliases else 1
   newAliases[ compiledRegex ] = key
   _compiledRegexAliases = newAliases
   return key

def getRegexAliasIndex( pattern, sysdbRegexAliases ):
   if _compiledRegexAliases:
      for regex, index in _compiledRegexAliases.items():
         if regex.pattern == pattern:
            return index
   return 0

@Ark.synchronized( _regexAliasLock )
def delRegexAlias( pattern, sysdbRegexAliases ):
   global _compiledRegexAliases
   if not _compiledRegexAliases:
      return

   # Make a copy so we don't disturb potential readers.
   newAliases = collections.OrderedDict()
   for regex, index in _compiledRegexAliases.items():
      if regex.pattern == pattern:
         del sysdbRegexAliases[ index ]
      else:
         newAliases[ regex ] = index

   _compiledRegexAliases = newAliases

def getNextCmdHistoryPos():
   global cmdHistoryPos
   currCmdHistoryPos = cmdHistoryPos
   cmdHistoryPos += 1
   return currCmdHistoryPos % CMD_HISTORY_SIZE

# Support for additional format specifiers to be registered by CliPlugins
promptFormatSpecifiers = {}

def addFormatSpecifier( spec, func ):
   promptFormatSpecifiers[ spec ] = func

# Register a command to be invoked (with no arguments) by
# the readline thread at the beginning of its next iteration
# through the readline loop.
readlineWorkQueue = []

def registerReadlineThreadWork( cmd ):
   readlineWorkQueue.append( cmd )

def drainReadlineWorkQueue():
   while readlineWorkQueue:
      cmd = readlineWorkQueue.pop( 0 )
      t3( "Processing pending readline command", cmd )
      cmd()

class ProfilerRecord( object ):
   """ A function decorator initalized with a profiler. """

   def __init__( self, profilerFuncName ):
      self.profilerFuncName = profilerFuncName

   def __call__( self, func ):
      def wrapper( *args, **kwargs ):
         session = args[ 0 ] # Expecting this to be of type Session
         profilerEnabled = session.sessionData( 'BasicCliCommands.profilerEnabled' )
         if not profilerEnabled:
            return func( *args, **kwargs )

         profiler = session.sessionData( self.profilerFuncName )
         profiler.enable()
         try:
            return func( *args, **kwargs )
         finally:
            profiler.disable()

      return wrapper

CLI_COMMAND_CACHE = {}
commandHandlerCleanupLockThreadLocal = threading.local()

AaaUser = collections.namedtuple( 'AaaUser', 'user sessionId uid gid' )
AliasHelp = collections.namedtuple( 'AliasHelp',
                                    [ 'aliasCmd', 'origCmd', 'helpStr' ],
                                    verbose=False )

hostnameHandler_ = None

def registerHostnameHandler( func ):
   global hostnameHandler_
   hostnameHandler_ = func

######################################################################
#
#  A CLI session.
#
MODE_STACK_OPEN = 0
MODE_STACK_COMMITTED = 1
MODE_STACK_RESTORED = 2

class ModeStack( object ):
   def __init__( self, mode ):
      self.mode = mode
      self.stack = []
      self.state = MODE_STACK_OPEN

def copyResults( result ):
   if type( result ) == dict:
      copiedResults = {}
      for k, v in result.iteritems():
         copiedResults[ k ] = copyResults( v )
      return copiedResults
   elif type( result ) == list:
      copiedResults = []
      for v in result:
         copiedResults.append( copyResults( v ) )
      return copiedResults
   elif type( result ) == tuple:
      copiedResults = []
      for v in result:
         copiedResults.append( copyResults( v ) )
      return tuple( copiedResults )

   return result

class Session( object ):
   """
   A Session object stores the current CLI mode and provides the ability to run
   CLI commands in the context of that mode, and to move between parent and
   child modes.
   """

   def __init__( self, initialModeClass, entityManager, configFile=None,
                 privLevel=1, disableAaa=False, disableAutoMore=False,
                 disableGuards=False,
                 startupConfig=False, secureMonitor=False,
                 standalone=False, interactive=False,
                 shouldPrint=True, isEapiClient=False,
                 skipConfigCheck=False, autoComplete=True, cliInput=None,
                 cli=None, aaaUser=None ):
      """Initialize a session starting at the given mode with the given
      privilege level."""

      self.entityManager_ = entityManager
      self.configFile_ = configFile
      self.disableAaa_ = disableAaa
      self.standalone_ = standalone
      self.privLevel_ = privLevel
      self.filterCommandDepth_ = 0 # how many filter commands are running
      self.showCommandDepth_ = 0 # how many levels of show commands are running
      self.disableAutoMore_ = disableAutoMore
      self.disableGuards_ = disableGuards
      # This flag is set when we are loading startup-config.
      self.startupConfig_ = startupConfig
      # Only support forcing secureMonitor to True. None means getting it from AAA.
      self.secureMonitor_ = secureMonitor or None
      # The 'interactive' flag is set to True when a user is interacting
      # with the CLI (not loading config from a file, for example). This can
      # be used to decide whether we allow user interaction (e.g., prompting
      # for confirmation).
      self.interactive_ = interactive
      # The 'skipConfigCheck' flag helps skipping config check.
      #
      # Certain command handlers have expensive config validation (e.g.,
      # overlapping ranges). In certain cases, when we load a config from
      # file we can skip this check if we believe the config file contains
      # validated configuration. Particularly, this flag is set to True in
      # the following two cases:
      # 1) loading startup-config
      # 2) configure replace
      #
      # It's not set to True when the user does "copy <file> running-config".
      #
      # Note it's better we avoid config check in all cases and just let agents
      # deal with invalid configs. The agents have to deal with invalid
      # intermediate configs anyway, due to how config replace is performed.
      self.skipConfigCheck_ = skipConfigCheck
      self.shouldPrint_ = shouldPrint # Whether we should print to stdout or not.
      self.isEapiClient_ = isEapiClient
      self.displayJson_ = False # Used for show running-config | json
      self.messages_ = []
      self.warnings_ = []
      self.errors_ = []
      self.sessionData_ = {}  # Plugins can use this dict to store per-session data.
      self.commandConfirmation_ = True # whether to prompt for confirmation
      # used by "cliprint" C printing library.
      self.outputFormat_ = "text" # whether to print in cli/text or json format
      self.requestedModelVersion_ = 1
      self.requestedModelRevision_ = 1
      self.requestedModelRevisionIsLatest_ = True
      self.aaaUser_ = aaaUser
      self.terminalCtx_ = TerminalUtil.TerminalContext( self )
      self.terminalCtx_.saveTerminalLength()
      self.terminalCtx_.saveTerminalWidth()
      # used when debugging memory usage, enabled by a cli command
      self.memoryStatsContext = MemoryStats.MemoryStatsContext( self )

      self.autoComplete_ = autoComplete
      self.parserDebug_ = False

      # None or a string name. If string name, wrap a session around *every*
      # cli config command.
      self.autoConfigSession_ = None
      self.autoConfigSessionNumCmds_ = 0

      # Tricky things happen when the command line is around 80-char limit
      # of the terminal window (scrolling, linewrap, etc). Our CliTest's
      # VT100 emulator isn't perfect. It's much easier to just avoid long
      # prompt by limiting how long the hostname can be.
      hostLen = int( os.environ.get( 'CLI_PROMPT_HOSTNAME_LIMIT', '999' ) )
      self.promptHostnameLimit_ = hostLen

      # Instantiate the initial mode last so the Session state is available
      self.mode_ = initialModeClass( None, self )
      self.mode_.onInitialMode()
      self.modeStack_ = []
      # remember which mode parsed the command for history saving purpose
      self.parsingMode_ = None
      self.origMode_ = None
      self.cliConfig = None
      self.configHistory = None
      self.cliSessionStatus = None
      if self.entityManager_:
         try:
            self.cliConfig = self.sysdbRoot[ 'cli' ][ 'config' ]
         except KeyError:
            pass
         try:
            self.cliSessionStatus = self.sysdbRoot[ 'cli' ][ 'session' ][ 'status' ]
         except KeyError:
            pass

      self.cliInput = cliInput if cliInput else CliInputInterface
      self.cli_ = cli
      if cli:
         self.cliConfig = cli.cliConfig_
         self.configHistory = cli.cliConfigHistory_

   # printing in c code, valueFunction needs to know output format, which is an
   # enum in c code (converted here from json-rpc string). TODO: remove magic value
   # or use some tac conversion magic (which would make cli depend on agent-cmd-req)
   def outputFormat( self ):
      if self.outputFormat_ == "text":
         return 1 # ofText enum Value from CommandAgentRequest.tac
      else:
         return 2 # ofJson enum Value from CommandAgentRequest.tac

   def outputFormatIsJson( self ):
      if self.outputFormat_ == "json":
         return True
      return False

   def outputFd( self ):
      return int( sys.stdout.fileno() )

   def memoryDebugCfgIs( self, heapcheck, mallinfo, rss, pyObjects, gc, log ):
      global CMD_HISTORY_SIZE, CMD_HISTORY, DEFAULT_CMD_HISTORY_SIZE, cmdHistoryPos
      args = [ heapcheck, mallinfo, rss, pyObjects, gc, log ]
      if all( e is None for e in [ heapcheck, mallinfo, rss, pyObjects, gc, log ] ):
         self.memoryStatsContext.cfgIs( None )
         CMD_HISTORY_SIZE = DEFAULT_CMD_HISTORY_SIZE
         cmdHistoryPos = 2 # leaves a trace that history was squashed
      else:
         MemoryDebugCfg = collections.namedtuple( 'MemoryDebugCfg', [ 'heapcheck',
                            'mallinfo', 'rss', 'pyObjects', 'gc', 'log' ] )
         self.memoryStatsContext.cfgIs( MemoryDebugCfg( *args ) )
         # don't want to wait 100 cmd to see memory stabilize... or confuse anybody
         # looking at the history after we squashed index 0 for a while.
         CMD_HISTORY = [ None ] * DEFAULT_CMD_HISTORY_SIZE
         cmdHistory = CmdHistory()
         cmdHistory.cmd = "reset history for memory debugs"
         CMD_HISTORY[ 1 ] = cmdHistory
         CMD_HISTORY_SIZE = 1

   # the requested revision of the model that shall be returned
   def requestedModelRevision( self ):
      return self.requestedModelRevision_

   def gotoParentMode( self, temporaryModeChange=False ):
      """Switch the current mode to the current mode's parent."""
      assert self.mode_.session_ is self
      assert self.mode_.parent_
      if not temporaryModeChange:
         # call onExit for all saved temporary mode changes
         if self.modeStack_:
            self.commitMode( )
         # Let the current mode know that we're leaving it.
         if self.mode_ != self.mode_.parent_:
            self.mode_.onExit()
      else:
         # Save the current mode so we may call its onExit() later.
         if self.modeStack_:
            self.modeStack_[ -1 ].stack.append( self.mode_ )
      self.mode_ = self.mode_.parent_
      assert self.mode_.session_ is self

   def gotoChildMode( self, mode ):
      """Switch the current mode to a child of the current mode."""

      assert self.mode_.session_ is self
      assert mode.parent_ is self.mode_, ( "Tried to enter mode %s whose parent"
            " is %s, but we're in mode %s" % ( mode, mode.parent_, self.mode_ ) )
      self.mode_ = mode
      assert self.mode_.session_ is self

   # Mode change management.
   #
   # During runCmd() parsing, the mode could change temporarily until we find
   # a parent that accepts the command. ModeStack keeps track of those changes
   # so that:
   # 1. it remembers the original mode so it can revert to the original mode.
   # 2. it keeps track of the temporary mode changes so it can call onExit()
   # for those modes in case the changes turn out to be permanent.
   #
   # Since command handlers can call runCmd() to run sub commands (e.g., show tech),
   # each level tracks its own data.

   def _saveMode( self ):
      t2( "save the current mode", self.mode_ )
      self.modeStack_.append( ModeStack( self.mode_ ) )

   def _restoreMode( self ):
      modeStack = self.modeStack_[ -1 ]
      assert modeStack.state != MODE_STACK_COMMITTED
      modeStack.state = MODE_STACK_RESTORED
      self.mode_ = modeStack.mode
      t2( "restore mode to %s" % self.mode_ )
      modeStack = self.modeStack_[ -1 ]
      del modeStack.stack[ : ]

   def commitMode( self ):
      # now we've changed mode, call onExit() on all walked modes
      modeStack = self.modeStack_[ -1 ]
      assert modeStack.state != MODE_STACK_RESTORED
      modeStack.state = MODE_STACK_COMMITTED
      stack = modeStack.stack
      t2( "commit mode to %s" % self.mode_ )
      for m in stack:
         assert m != self.mode_
         t2( "%s: onExit()" % m.name )
         m.onExit( )
      del stack[ : ]

   def updateParsingMode( self, mode ):
      # We add the command to the history of the parsing mode
      # (may be a parent of the current mode). In case of aliases,
      # there might be more modes, so we only use the first one
      if self.parsingMode_ is None:
         self.parsingMode_ = mode

   def addToHistoryEventTable( self, commandSource, configSource,
                               configDest, cmdSourceURLScheme="",
                               cmdDestURLScheme="", runningConfigChanged=False ):
      '''Add an element the historyEventTable. Further, we could generate traps
      as per CONFIG-MAN-MIB for the events added to historyEventTable.
      Currently, we add four kinds of events to the historyEventTable:
      1. Entering config mode.
      2. Exiting config mode.
      3. Copy command affecting startup/running config.
      4. write erase.
      '''
      if not self.configHistory:
         return
      historyEventTable = self.configHistory.historyEventTable
      eventEntry = Tac.newInstance( "Cli::HistoryEventTableEntry",
                                    self.configHistory.historyEventCount,
                                    Tac.now(), commandSource,
                                    configSource, configDest,
                                    cmdSourceURLScheme,
                                    cmdDestURLScheme )

      while len( historyEventTable ) >= CliCommon.HISTORY_TABLE_LEN:
         historyEventTable.deq()

      historyEventTable.enq( eventEntry )
      t2( 'Added entry to history table' )
      self.configHistory.historyEventCount += 1

      if runningConfigChanged:
         self.configHistory.runningConfigLastChanged = Tac.now()

   def modeOfLastPrompt( self ):
      # Certain modes with onExit() handler need to know the last prompt mode
      # so pass this in through the session. Here is one example for config-acl:
      #
      # config# ip access-list foo
      # config-acl-foo# <configure acl foo>
      # config-acl-foo# no ip access-list foo
      # config#
      #
      # The current CLI behavior is that it tries to execute the no command in
      # the parent mode (since the current mode does not parse it), then calls
      # the onExit() handle of the config-acl mode and stays at the global config
      # mode. The ACL is then committed in onExit() after already being deleted!
      # Certainly not the expected behavior.
      #
      # By passing in the modeOfLastPrompt information, the no command handler
      # could set a flag in that mode similarly as the following logic:
      #
      # def noAclConfig( mode, aclName ):
      #     lastMode = mode.session_.modeOfLastPrompot()
      #     if isinstance( lastMode, AclConfigMode ) and \
      #        lastMode.aclName == aclName:
      #        # deleting self while inside config-acl mode
      #        # do not commit the acl upon exit
      #        lastMode.commitOnExit = False
      #     <...>
      #
      # This is probably going to be revisited when we have the real GroupChange
      # mode, but for now it's a quick fix.
      if not self.modeStack_:
         return None
      return self.modeStack_[ -1 ].mode

   @ProfilerRecord( 'BasicCliCommands.parserProfiler' )
   def _parseTokens( self, tokens, aaa ):
      '''Parses tokens in the current mode.

      This method may be overridden in a subclass to parse tokens differently
      (e.g. with autocompletion disabled).
      '''
      if self.mode_.__class__.isConfigMode():
         return self.mode_.parse( tokens, autoComplete=self.autoComplete_,
                                  authz=aaa, acct=aaa )
      else:
         with ConfigMount.ConfigMountDisabler():
            return self.mode_.parse( tokens, autoComplete=self.autoComplete_,
                                     authz=aaa, acct=aaa )

   def shouldPrintIs( self, shouldPrint ):
      """ Set the shouldPrint attribute. If True, Cli will print
      output as usual. Otherwise, we'll (try) not to print anything by
      not calling the model's render function and will abstain from
      printing warnings and errors. Old commands that have not yet
      converted will still print """
      self.shouldPrint_ = shouldPrint

   def shouldPrint( self ):
      """ Get the shouldPrint attribute """
      return self.shouldPrint_

   def isEapiClient( self ):
      """ Get the isEapiClient attribute """
      return self.isEapiClient_

   def displayJsonIs( self, displayJson ):
      """ Set the displayJson value, used by "show running-config | json"
      and show session-config | json so that they will return the correct
      output to the CLI """
      self.displayJson_ = displayJson

   def displayJson( self ):
      """ Get the displayJson value """
      return self.displayJson_

   def clearMessages( self ):
      """ Empty the message, warning, and error arrays. """
      self.messages_ = []
      self.warnings_ = []
      self.errors_ = []

   def addMessage( self, msg ):
      """ Store a message in our session's array, and prints
      it if the shouldPrint flag is set. """
      if self.shouldPrint():
         CliCommon.printMessage( msg )
      self.messages_.append( str( msg ) )

   def checkLineInfo( self, cmd, lineNo ):
      """ Fill in missing cmd and line info from the config file. """
      configFile = self.configFile_
      # It is possible that we have been given a config file but we
      # haven't started reading yet, e.g., the file could not be
      # opened. Also, sometimes configFile is given as a string.
      if getattr( configFile, 'lineNo', 0 ) > 0:
         if cmd is None:
            cmd = configFile.currentLine
         if lineNo is None:
            lineNo = configFile.lineNo
      return cmd, lineNo

   def addWarning( self, msg, cmd=None, lineNo=None ):
      """ Store a warning message in our session's array, and prints
      the warning message if the shouldPrint flag is set. """
      cmd, lineNo = self.checkLineInfo( cmd, lineNo )
      if self.shouldPrint():
         CliCommon.printWarningMessage( msg, cmd, lineNo )
      self.warnings_.append( str( msg ) )

   def addError( self, msg, cmd=None, lineNo=None ):
      """ Store an error message in our session's array, and prints
      the error message if the shouldPrint flag is set. """
      cmd, lineNo = self.checkLineInfo( cmd, lineNo )

      if self.shouldPrint():
         CliCommon.printErrorMessage( msg, cmd, lineNo )
      self.errors_.append( str( msg ) )

   def hasError( self ):
      """Returns True if at least one error was logged via addError()."""
      return bool( self.errors_ )

   def aaaUser( self ):
      """ Returns the AaaUser for the session """
      t0( 'aaaUser', self.aaaUser_ )
      return self.aaaUser_

   def secureMonitor( self ):
      """Returns whether the current user is a secure-monitor operator"""
      if self.secureMonitor_ is None:
         # Delay the evaluation until someone asks for it since it requires mounts
         data = CliAaa.authenSessionData( self.mode_ ) or {}
         # the value is a boolean string in session data
         self.secureMonitor_ = ( data.get( CliCommon.SECURE_MONITOR_ATTR_NAME ) ==
                                 "True" )
         t0( 'secureMonitor', self.secureMonitor_ )
      return self.secureMonitor_

   def sessionDataIs( self, name, value ):
      """Stores some arbitrary session-specific data in this Session.

      This allows plugins to store their own per-session data in a map
      keyed by the given name.

      Args:
        name: A name to be able to retrieve this value later via sessionData().
        value: An arbitrary value to store in this session.
      """
      self.sessionData_[ name ] = value

   def sessionData( self, name, defaultValue=None ):
      """Returns the session-specific data of the given name, or the defaultValue."""
      return self.sessionData_.get( name, defaultValue )

   def close( self ):
      """ Sessions might take a little while to be garbage collected. So when we
      are done with a session we can close whatever resources we are using and
      let the GC collect the rest when it gets a chance. """
      self.sessionData_ = {}

   def promptFormat( self ):
      assert self.cliConfig
      return self.cliConfig.prompt

   def getParentHistoryKeys( self ):
      result = []
      mode = self.parsingMode_ or self.origMode_
      modes = []
      while mode.parent_:
         modes.append( mode.parent_ )
         mode = mode.parent_
      for m in reversed( modes ):
         result.append( self.getHistoryKeys( m ) )
      return result

   def getOrigModeHistoryKeys( self ):
      mode = self.parsingMode_ or self.origMode_
      return self.getHistoryKeys( mode )

   def getHistoryKeys( self, mode=None ):
      if not mode:
         mode = self.mode_
      return ( mode.historyKey(),
               mode.parent_.historyKey() if mode.parent_ else "" )

   def tabComplete( self, cmd ):
      sys.stdout.write( "\r" )
      lists = self.getCompletionLists( cmd )
      completions, unguardedCompletions, partialToken, aliasCompletions = lists
      if not completions and not aliasCompletions:
         return ''
      if not unguardedCompletions and not aliasCompletions:
         # Help needs to be printed on a new line (unlike tab-completion). Also
         # print tab (like we do with ?), so a session log makes sense to reader.
         print( "" )
         self.printHelp( cmd )
         return ''
      return self.getCommandSuffix( unguardedCompletions, partialToken,
                                    aliasCompletions )

   def printHelp( self, cmd ):
      ( _, partialToken, completions, aliasCompletions, status ) = \
                                          self.getCompletions_( cmd )
      if status == 'invalid':
         print( '% Unrecognized command' )
      elif status == 'ambiguous':
         print( '% Ambiguous command' )
      elif status == 'guarded':
         print( '% Unavailable command' )
      elif status == 'error':
         pass
      else:
         assert status == 'ok'
         unguardedCompletions = [ c for c in completions if c.guardCode is None ]
         if unguardedCompletions:
            # If we have a mix of guard-blocked things and guard-allowed things,
            # then focus on the guard-allowed things.
            completions = unguardedCompletions
         else:
            # All we have is guard-blocked things, so expose them.
            pass
         completions = sorted( completions )

         # If the user is mid-way through a token, just print a concise list of
         # the completions.  Otherwise, print a neatly formatted name and help
         # text for each completion.  Also, if all completions are guard-blocked,
         # then show the verbose help, so the user can see that they're guard-
         # blocked (and see why).
         aliasHelp = [ i.helpStr for i in aliasCompletions ]

         def showCmdFunc():
            return self._printCompletions( self.mode_, completions, partialToken,
                                           aliasHelp )
         ShowCommand.runShowCommand( self.mode_, showCmdFunc )

   def _printCompletions( self, mode, completions, partialToken, aliasHelp ):
      if partialToken == '' or ( completions and completions[ 0 ].guardCode ):
         maxNameLen = max( [ len( c.name ) for c in completions ] )
         lineNotPrinted = True
         maxRowLen = TerminalUtil.ioctl_GWINSZ()[ 1 ]
         if maxRowLen == 0:
            maxRowLen = TerminalUtil.defaultTerminalWidth
         maxHelpLen = maxRowLen - ( maxNameLen + 4 )
         for common in False, True:
            for c in completions:
               commonCompletion = mode.commonHelpSupported and c.common
               if commonCompletion == common:
                  # Prints the line once if there are common commands to print
                  if commonCompletion and lineNotPrinted:
                     print( ' ', '-' * 40 )
                     lineNotPrinted = False
                  helpMsg = c.help if c.help is not None else ''
                  assert ( helpMsg.lower() not in
                           CliParserCommon.DEPRECATED_HELP_STRINGS )
                  assert ( c.name.lower() not in
                           CliParserCommon.DEPRECATED_HELP_STRINGS )
                  if not c.guardCode and len( helpMsg ) > maxHelpLen:
                     # Fits the helpMsg within the terminal width
                     # First tries to separate by words, but if a word is longer
                     # than the max allowed length, it will split the word
                     helpLines = textwrap.wrap( helpMsg, maxHelpLen )
                     helpMsg = helpLines[ 0 ] + '\n'
                     for line in helpLines[ 1 : ]:
                        helpMsg = helpMsg + ' ' * ( maxNameLen + 4 ) + line + '\n'
                     helpMsg = helpMsg[ : -1 ]
                  print( '  %-*s  %s' % ( maxNameLen, c.name,
                     c.guardCode or helpMsg ) )
      elif len( completions ) == 1 and not aliasHelp and completions[ 0 ].guardCode:
         # Special case: the user has just one choice, and it is blocked
         # by a guard.  Let them know why it's blocked.
         c = completions[ 0 ]
         print( '  %s  %s' % ( c.name, c.guardCode ) )
      else:
         names = [ c.name for c in completions ]
         helpStrs = aliasHelp + names
         lines = self._formatListForTermWidth( helpStrs )
         for l in lines:
            print( l )
      print()

   def _formatListForTermWidth( self, strings ):
      """Formats a list of strings into rows and columns (not necessarily of
      equal width), increasing left to right then top to bottom, such that the
      length of each line is less than or equal to termWidth.  This is very
      similar to the output of 'ls -x'."""

      termWidth = TerminalUtil.ioctl_GWINSZ()[ 1 ]
      if termWidth <= 0:
         termWidth = 80 # if query unsuccessful, default to 80

      # The algorithm we use is optimized for simplicity, not speed.  We first
      # try fitting the strings into n columns, and if that fails, we try fiting
      # them into n-1 columns, and so on.  To get an initial value of n we look
      # at the first few items in the list.
      n = 0
      while ( n < len( strings ) and
              len( '  '.join( strings[ : n + 1 ] ) ) <= termWidth ):
         n += 1

      # n is the maximum number of strings that fit on the first line.  If no
      # strings fit, just use one column.
      if n == 0:
         n = 1

      while True:
         # Organize the strings into n columns.
         groups = []
         tmpStrings = strings[ : ]
         while tmpStrings:
            group = tmpStrings[ : n ]
            while len( group ) < n:
               group.append( '' )
            groups.append( group )
            tmpStrings = tmpStrings[ n : ]

         # pylint: disable-msg=W0631
         columns = [ map( lambda g : g[ i ], groups ) for i in range( n ) ]
         columnWidths = [ max( map( len, c ) ) for c in columns ]

         # If the strings fit into n columns (or if n is 1), we're done.
         # Otherwise, try again with a smaller value of n.
         if sum( columnWidths ) + ( 2 * ( n - 1 ) ) <= termWidth:
            break
         elif n == 1:
            break
         else:
            n -= 1

      # Format the list of groups of strings into a list of lines.
      fmt = '  '.join( [ '%%-%ds' % columnWidths[ i ] for i in range( n ) ] )
      lines = [ fmt % tuple( g ) for g in groups ]

      return lines

   def initTerminal( self ):
      assert self.cliConfig
      if self.cliConfig.terminalLengthOverride >= 0:
         self.terminalCtx_.terminalLengthOverrideIs(
            self.cliConfig.terminalLengthOverride )
      if self.cliConfig.terminalWidthOverride >= 0:
         self.terminalCtx_.terminalWidthOverrideIs(
            self.cliConfig.terminalWidthOverride )

   def _printParserDebug( self, cmdHistory, parseResults ):
      if not self.parserDebug_:
         return

      now = time.time()
      print( '---------Start Parser Debug Info---------' )
      try:
         print( 'Command:', cmdHistory.cmd )
         if parseResults:
            cliModel = None
            if 'fnWithArgs' in parseResults[ 'kargs' ]:
               cmdHandler = parseResults[ 'kargs' ][ 'fnWithArgs' ][ 0 ]
               cliModel = parseResults[ 'kargs' ][ 'fnWithArgs' ][ 3 ]
            else:
               cmdHandler = parseResults[ 'valueFunc' ]
               if isinstance( cmdHandler, ShowCommand.ShowCommandHandler ):
                  cliModel = cmdHandler.cliModel_
                  cmdHandler = cmdHandler.handler_
            print( 'Command Handler:', CliParserCommon.debugFuncName( cmdHandler ) )
            cmdPkg = 'Unknown'
            if hasattr( cmdHandler, 'func_code' ):
               cmdFile = cmdHandler.func_code.co_filename
               try:
                  rpm = Tac.run( [ 'rpm', '-qf', cmdFile, '--queryformat=%{NAME}' ],
                                 stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
                  # handles something like 'a-b-c' by just taking the 'a-b'
                  if '-' in rpm:
                     cmdPkg = '-'.join( rpm.split( '-' )[ : -1 ] )
                  else:
                     cmdPkg = rpm
               except Tac.SystemCommandError:
                  pass
            pp = pprint.PrettyPrinter()
            print( 'Command Args:' )
            pp.pprint( parseResults[ 'kargs' ] )
            print( 'Command Package:', cmdPkg )
            print( 'Command CLI Model:', cliModel )
            aaaInfo = parseResults[ 'aaa' ]
            print( 'Command AAA authzTokens:', aaaInfo[ 'authzTokens' ] )
            if aaaInfo[ 'authzFunc' ]:
               print( 'Command AAA authzFunc:', aaaInfo[ 'authzFunc' ] )
            if not aaaInfo[ 'requireAuthz' ]:
               print( 'Command AAA requireAuthz: False' )
            print( 'Command AAA acctTokens:', aaaInfo[ 'acctTokens' ] )
            if aaaInfo[ 'acctFunc' ]:
               print( 'Command AAA acctFunc:', aaaInfo[ 'acctFunc' ] )
            if not aaaInfo[ 'requireAcct' ]:
               print( 'Command AAA requireAcct: False' )
         else:
            print( 'Parser Results: No parser results' )
         print( 'Thread ID:', cmdHistory.threadId )
         print( 'Start Time:', cmdHistory.startTime )
         print( 'Parse Time (ms):', ( now - cmdHistory.startTime ) * 1000 )
         if parseResults:
            autoConfigSessionAllowed = parseResults[ 'autoConfigSessionAllowed' ]
            print( 'Auto Config Session Allowed:', autoConfigSessionAllowed )
            print( 'Allow Parser Cache:', parseResults[ 'allowCache' ] )
         print( 'Parser Cache Enabled:', self.cachingEnabled() )
      except Exception as e: # pylint: disable-msg=broad-except
         print( 'Error occurred', e )
      print( '----------End Parser Debug Info----------' )
      print()

   @ProfilerRecord( 'BasicCliCommands.valueFuncProfiler' )
   def _invokeValueFunc( self, mode, parseResults, authz, acct ):
      if not parseResults:
         return None

      if ( parseResults[ "cmdDeprecatedBy" ] and parseResults[ "aaa" ] and
            self.cliConfig and self.cliConfig.deprecatedCmdsSyslogMsg ):
         cmd = ' '.join( parseResults[ "aaa" ][ "acctTokens" ] )
         info = UtmpDump.getUserInfo()
         # pylint: disable-msg=undefined-variable
         SYS_DEPRECATED_CMD( cmd, info[ 'user' ], info[ 'tty' ], info[ 'ipAddr' ] )
         # pylint: enable-msg=undefined-variable

      self.updateParsingMode( mode )
      self.performAaa( authz, acct, mode, parseResults )
      valueFunc = parseResults[ "valueFunc" ]
      kargs = parseResults[ "kargs" ]
      modelet = CliParser.getModelet( mode, valueFunc )
      if not isinstance( mode, BasicCliModes.ConfigModeBase ):
         # Not a config command
         with ConfigMount.ConfigMountDisabler():
            t3( "calling final function", valueFunc,
                "with ConfigMounts disabled, with kargs", kargs )
            return valueFunc( modelet, **kargs )
      else:
         configMntProhibited = not CONFIG_LOCK.canRunConfigureCmds()
         with ConfigMount.GlobalConfigDisabler( configMntProhibited ):
            return self._runConfigCmd( mode, modelet, parseResults )

   def _runConfigCmd( self, mode, modelet, parseResults ):
      autoConfigSessionAllowed = parseResults[ "autoConfigSessionAllowed" ]
      valueFunc = parseResults[ "valueFunc" ]
      kargs = parseResults[ "kargs" ]
      if ( autoConfigSessionAllowed and
            mode.autoConfigSessionAllowed and
            # If onExit method has been defined in mode's class (and not
            # inherited from parent class), then it indicates that this
            # mode has grouped commands. Disable auto config session in
            # this case.
            ( isinstance( mode, BasicCliModes.GlobalConfigMode ) or
              ( 'onExit' not in vars( type( mode ) ) ) ) and
            not mode.session_.inConfigSession() and
            mode.session_.autoConfigSessionEnabled() ):
         assert isinstance(
                     mode.session_.autoConfigSession_, str )
         t3( "calling final function", valueFunc,
             "within a ConfigSession, with kargs", kargs )
         with ConfigSessionWrapper( mode ):
            return valueFunc( modelet, **kargs )
      else:
         if ( mode.session_ and
              mode.session_.inConfigSession() ):
            em = mode.session_.entityManager_
            sessionName = CliSession.currentSession( em )
            sStatusDir = CliSession.sessionStatus.pendingChangeStatusDir
            pcs = sStatusDir.status.get( sessionName )
            if pcs and pcs.completed:
               raise CliSession.SessionAlreadyCompletedError( sessionName )

         if ( not mode.inhibitImplicitModeChange and
              mode.session_.modeStack_ ):
            # We are going to change mode, so we call onExit() handlers
            # for the mode stack before calling the command handler.
            t2( "%s: commitMode" % mode.name )
            mode.session_.commitMode()
         t3( "calling final function", valueFunc, "with kargs",
             kargs )
         return valueFunc( modelet, **kargs )

   def cachingEnabled( self ):
      # We must be in an config session, not have cache disabled AND either an
      # abuild or over eapi
      enabled = ( self.inConfigSession() and not os.getenv( 'DISABLE_CLI_CACHE' ) and
                  ( self.isEapiClient() or os.getenv( 'ABUILD' ) ) )
      t7( "Caching enabled", enabled )
      return enabled

   def getCliValueFuncFromCache( self, mode, commandStr ):
      if not ( mode.allowCache and self.cachingEnabled() ):
         return None
      modeKey = self.getKeyFromMode( mode )
      result = CLI_COMMAND_CACHE.get( modeKey, {} ).get( commandStr )
      t0( "Found cached handler for", commandStr, "in mode", modeKey,
          "with results", result )
      return result

   def getKeyFromMode( self, mode ):
      if hasattr( mode, "instanceRuleKey" ) and mode.instanceRuleKey():
         return "%s-%s" % ( type( mode ), mode.instanceRuleKey() )
      elif hasattr( mode, "enterCmd" ) and mode.enterCmd():
         return "%s-%s" % ( type( mode ), mode.enterCmd() )
      else:
         return type( mode )

   def putCliValueFuncInCache( self, mode, commandStr, parseResults ):
      if not ( mode.allowCache and parseResults[ 'allowCache' ] ):
         return

      modeKey = self.getKeyFromMode( mode )
      t0( "Adding to CLI cache '", commandStr, "' in mode '", modeKey,
          "' with results", parseResults )

      cache = CLI_COMMAND_CACHE.setdefault( modeKey, {} )
      cache[ commandStr ] = parseResults

   def getParserResult( self, modeAcceptingCommand, commandStr, tokens, aaa ):
      if not self.cachingEnabled():
         return self._parseTokens( tokens, aaa )

      cachedResults = self.getCliValueFuncFromCache( modeAcceptingCommand,
                                                     commandStr )
      if cachedResults:
         return cachedResults
      else:
         return self._parseTokens( tokens, aaa )

   def performAaa( self, authz, acct, mode, parseResults ):
      if not parseResults[ "aaa" ]:
         return
      aaaInfo = parseResults[ "aaa" ]
      if aaaInfo[ "requireAuthz" ] and authz and self.authorizationEnabled():
         authzFunc = aaaInfo[ "authzFunc" ]
         if authzFunc is None:
            authzFunc = CliAaa.authorizeCommand
         authorized, message = authzFunc( mode,
                                          self.privLevel_,
                                          aaaInfo[ "authzTokens" ] )
         if not authorized:
            cmd = " ".join( aaaInfo[ "authzTokens" ] )
            raise CliParser.AuthzDeniedError( cmd, message )

      if aaaInfo[ "requireAcct" ] and acct and self.acctEnabled():
         acctFunc = aaaInfo[ "acctFunc" ]
         if acctFunc is None:
            acctFunc = CliAaa.sendCommandAcct
         acctFunc( mode, self.privLevel_,
                   aaaInfo[ "acctTokens" ] )
         # For synchrous accounting, we will wait for few seconds
         # ( currently 15 seconds ) for command accounting to be
         # completed before executing the command.
         if aaaInfo[ "requireSyncAcct" ]:
            waitTime = int( os.environ.get( 'AAA_SYNC_ACCT_WAIT_TIME', '15' ) )
            CliAaa.flushAcctQueue( mode, waitTime=waitTime )

   def runTokenizedCmd( self, tokens, aaa, fromCapi=False ):
      startTime = time.time()
      cmdHistory = CmdHistory()
      CMD_HISTORY[ getNextCmdHistoryPos() ] = cmdHistory
      drainReadlineWorkQueue()
      self.clearMessages()
      self._saveMode()
      savedMode = self.modeStack_[ -1 ]
      commandStr = " ".join( tokens )

      firstParseError = None
      try:
         # If the command failed in the current mode, see if it
         # is accepted in any parent configuration mode.  This
         # is user-friendly, and in particular it means that the
         # startup-config file doesn't need to contain any
         # 'exit' commands.
         while True:
            try:
               # Try parsing the tokens in the current mode. Error is
               # passed on to, and handled by the calling function. If the
               # command succeeds, and does not itself cause a mode change,
               # and the mode's inhibitImplicitModeChange is True, then we
               # revert the mode to modeOfLastPrompt. e.g., config-if#show
               # interfaces. After running this, we would like to stay back
               # in config-if mode. Hence EnableMode.inhibitImplicitModeChange
               # is True. However, when running "prompt xyz" in config-if mode,
               # where "prompt xyz" is a config mode command, we want to remain
               # back in config mode itself after the command execution. This is
               # the industry standard, and we can have commands following this
               # one which assume that we are in config mode. Thus
               # configMode.inhibitImplicitModeChange is False. Also
               # config-if#vlan 1, although accepted in Exec mode, changes the
               # mode itself to config-vlan-1# . Here, we will not try to revert
               # back to modeoflastPrompt.
               modeAcceptingCommand = self.mode_
               # NB getParserResult might change the 'tokens' object
               parseResults = self.getParserResult( modeAcceptingCommand,
                                                    commandStr, tokens, aaa )
               if parseResults and parseResults[ "aaa" ]:
                  # if we have results and we have AAA tokens we should replace the
                  # cmd with that so that we don't by accidently spill sensitive
                  # tokens, ie passwords, into the log file in plain text
                  commandStr = ' '.join( parseResults[ "aaa" ][ "acctTokens" ] )
               cmdHistory.cmd = commandStr
               t0( "Results for command '", commandStr, "'", parseResults )
               self._printParserDebug( cmdHistory, parseResults )
               parseResCopy = copyResults( parseResults )
               THREAD_ID_CMD_MAPPING[ gettid() ].append( cmdHistory )
               try:
                  with self.memoryStatsContext:
                     result = self._invokeValueFunc( modeAcceptingCommand,
                                                     parseResCopy,
                                                     authz=aaa, acct=aaa )
               finally:
                  THREAD_ID_CMD_MAPPING[ gettid() ].pop()
                  if len( THREAD_ID_CMD_MAPPING[ gettid() ] ) == 0:
                     del THREAD_ID_CMD_MAPPING[ gettid() ]
               t0( "Final Results for command '", commandStr, "'", result )
               if parseResults and self.cachingEnabled():
                  self.putCliValueFuncInCache( modeAcceptingCommand, commandStr,
                                               parseResults )
               if ( self.mode_ == modeAcceptingCommand and
                    self.mode_.inhibitImplicitModeChange ):
                  self._restoreMode( )
               else:
                  self.commitMode( )

               return result

            except ( CliCommon.InvalidInputError,
                     CliCommon.IncompleteTokenError ), e:
               t0( "InvalidInputError", e )
               # Invalid command.  Try the next parent
               # configuration mode, if any.
               #
               # In unpriv mode, try parsing in EnableMode for friendlier
               # error messages, except if we already came down from a
               # higher mode (in which case firstParseError is already set)
               if ( not firstParseError
                    and isinstance( self.mode_, BasicCliModes.UnprivMode )
                    and self.mode_.childMode(
                           BasicCliModes.EnableMode ).checkCommandExists( tokens ) ):
                  msg = CliCommon.PRIV_CMD_IN_UNPRIV_MODE_ERROR_MSG
                  raise CliCommon.InvalidInputError( "Invalid input" + " " + msg )
               if firstParseError is None:
                  # Save the ParseError of the first parse.
                  firstParseError = e

            if ( self.mode_.parent_ and self.mode_.parent_ != self.mode_ and
                 savedMode.state == MODE_STACK_OPEN ):
               # self.mode_.parent_ may be equal to self.mode_
               # when loading a config file, so the check is
               # necessary to prevent an infinite loop. For the
               # mode where we start, the mode change may be
               # temporary if we don't find a mode to parse the
               # current command.  If it's not temporary, we'll
               # finish the mode change processing in our
               # 'finally' clause below.
               #
               # The check for savedMode.state is to detect the case
               # where this runCmd() is called inside a command handler,
               # and gives invalid input error. We don't want to go up
               # to the parent in this case since the command is technically
               # accepted and the mode is actually already committed.
               # See BUG99838.
               self.gotoParentMode( True )

               # since we are going to call this function again in the for loop
               # this means that we need to cleanup anything that we have
               # outstanding
               doCommandHandlerCleanupCallback()
            else:
               # The command wasn't accepted in any parent
               # configuration mode.  Restore the original mode
               # and raise the original exception
               raise firstParseError # pylint: disable-msg=E0702

      finally:
         t0( "Finally done running cmd", commandStr )
         # If we're leaving oldMode, call its exit hook in case
         # it has one.  Having this check here should avoid
         # needing it anywhere in the nest of conditionals
         # above.
         sys.stdout.flush()
         # clear the mode
         if self.modeStack_[ -1 ].state == MODE_STACK_OPEN:
            self._restoreMode( )
         self.modeStack_.pop()
         # some commands open sockets that need to be closed, doing it
         # here makes sure cleanup happens even on control-C or broken
         if not fromCapi: # Capi does the cleanup in CliApi.py
            doCommandHandlerCleanupCallback()

         endTime = time.time()
         cmdHistory.endTime = endTime
         if self.sessionData( 'BasicCliCommands.timingEnabled' ):
            deltaTime = endTime - startTime
            timingInfo = self.sessionData( 'BasicCliCommands.timingInfo' )
            if commandStr not in timingInfo:
               timingInfo[ commandStr ] = { "totalTime": 0, "details": [] }
            timingInfo[ commandStr ][ "totalTime" ] += deltaTime
            timingInfo[ commandStr ][ "details" ].append( deltaTime )

   def runCmd( self, cmd, aaa=True ):
      """Attempts to parse and run the given command in the current mode, or in
      a parent configuration mode if parsing fails in the current mode.  If
      aaa=False no authorization check or accounting will be performed for
      the command, which is useful when a command's value function invokes
      other commands, e.g. 'show tech-support'."""
      self.parsingMode_ = None
      self.origMode_ = self.mode_
      # Erase traces of previous ^C (the control-c might hit after the command
      # it was intended to hit finished, and thus could not clean it up)
      TacSigint.clear()

      tokens = CliParser.textToTokens( cmd, mode=self.mode_ )
      if tokens:
         # Check first token against existing aliases, expand the
         # alias to full command(s), and check whether the alias is
         # provided with enough parameters. If so, then run each
         # command in this alias.
         # skipConfigCheck is set on 'config replace' and
         # loading startup-config. No aliases. BUG225798
         if self.skipConfigCheck():
            tokenized = [ tokens ]
         else:
            tokenized = expandAlias( self.mode_, self.cliConfig, cmd, tokens, True )

         for t in tokenized:
            self.runTokenizedCmd( t, aaa )

   def getCompletions( self, tokens, partialToken, startWithPartialToken=False ):
      error = None
      currModeCompletions = None

      if self.cliConfig and len( tokens ) > 0:
         # Check first token against existing aliases, expand as much
         # as possible. If this alias contains multi-line commands, we
         # get completions for the right positional parameter.
         cmd = ' '.join( tokens )
         tokens = expandAlias( self.mode_, self.cliConfig, cmd, tokens, False )[ -1 ]

      try:
         currModeCompletions = self.mode_.getCompletions(
            tokens, partialToken, startWithPartialToken=startWithPartialToken )
         if currModeCompletions:
            return currModeCompletions
      except CliCommon.InvalidInputError:
         error = sys.exc_info()

      # Find completions from all parent modes.
      parentModeCompletions = set()
      thisMode = self.mode_.parent_
      while thisMode:
         try:
            thisModeCompletions = thisMode.getCompletions(
               tokens, partialToken, startWithPartialToken=startWithPartialToken )
            if thisModeCompletions:
               parentModeCompletions.update( thisModeCompletions )
         except CliCommon.ParseError:
            # Although it may have been instructive to only handle InvalidInputError
            # and let Ambiguous Grammer exception be passed up, that may hinder
            # the possibility to show completions from other modes which may have
            # valid completions. This differs from the current mode completion code
            # above, which can raise Ambiguous Grammer error. However I hope, we
            # can cope up with this model.
            pass
         thisMode = thisMode.parent_

      if not parentModeCompletions and error is not None:
         raise error[ 0 ], error[ 1 ], error[ 2 ] # pylint: disable-msg=E0702
      return parentModeCompletions

   def exitConfigMode( self ):
      """Exit out of Config Mode."""
      while isinstance( self.mode_, BasicCliModes.ConfigModeBase ):
         self.gotoParentMode()

   def parserDebugIs( self, parserDebug ):
      self.parserDebug_ = parserDebug

   def autoCompleteIs( self, autoComplete ):
      self.autoComplete_ = autoComplete

   def authenticationEnabled( self ):
      return not self.disableAaa_

   def authorizationEnabled( self ):
      return not self.disableAaa_

   def disableAaaIs( self, disableAaa ):
      self.disableAaa_ = disableAaa

   def acctEnabled( self ):
      return self.authorizationEnabled()

   def changePrivLevel( self, newPrivLevel ):
      self.privLevel_ = newPrivLevel

   def guardsEnabledIs( self, guardsEnabled ):
      self.disableGuards_ = not guardsEnabled

   def guardsEnabled( self ):
      return not self.disableGuards_

   def skipConfigCheck( self ):
      """ True if the session has explictly set to skip config check
      or if it is startup config"""
      return self.skipConfigCheck_ or self.startupConfig()

   def isStandalone( self ):
      """ True if the session is in standalone mode """
      return self.standalone_

   def startupConfig( self ):
      """ True if this session is being run as a startup config """
      return self.startupConfig_

   def commandConfirmation( self ):
      """ True if a command should ask for confirmation. If we are not interactive
      then we should also not require command confirmation."""
      if not self.interactive_:
         return False

      # there are 2 variables that control this. One is the variables is in the
      # session, and the other one is in Sysdb.
      if self.cliConfig:
         return self.commandConfirmation_ and self.cliConfig.commandConfirmation
      else:
         return self.commandConfirmation_

   def isInteractive( self ):
      """ True if this is an "interactive" session. The session is NOT
      interactive while executing startup-config, 'load config', and when
      stdin is not a terminal, e.g. 'Cli -c' and 'echo command | Cli' """
      return self.configFile_ is None and self.interactive_

   def inConfigSession( self ):
      """ True if this Cli process is inside a Config Session. """
      return CliSession.currentSession( self.entityManager_ ) is not None

   def maybeCallConfigSessionOnCommitHandler( self, key, handler ):
      """ Invokes the supplied handler closure, immediately if not in a config
      session, else after session commit. """

      if self.inConfigSession():
         CliSession.registerSessionOnCommitHandler( self.entityManager_,
                                                    key, handler )
      else:
         handler( self.mode_, onSessionCommit=False )

   def exitFromConfigSession( self ):
      """Keep exiting to parent mode until no longer in configSession"""
      while self.inConfigSession() and self.mode_.parent_:
         self.gotoParentMode()

   def autoConfigSessionIs( self, value ):
      if not bool( value ):
         self.autoConfigSession_ = None
         self.autoConfigSessionNumCmds_ = 0
      else:
         autoSessionPrefix = 'autosess_%d_' % os.getpid()
         value = CliSession.uniqueSessionName( self.entityManager_,
                                               prefix=autoSessionPrefix,
                                               okToReuse=True )
         self.autoConfigSession_ = value
         self.autoConfigSessionNumCmds_ = CliCommon.AUTO_CONFIG_SESSION_NUM_CMDS

   def autoConfigSessionEnabled( self ):
      return ( self.autoConfigSession_ is not None and
               self.autoConfigSessionNumCmds_ > 0 )

   sysname = property( lambda self: self.entityManager_.sysname() )
   sysdbRoot = property( lambda self: self.entityManager_.root() )
   entityManager = property( lambda self: self.entityManager_ )
   mode = property( lambda self: self.mode_ )
   configFile = property( lambda self: self.configFile_ )
   cli = property( lambda self: self.cli_ )

   def loadDynamicAliases( self ):
      if self.cli_:
         self.cli_.loadDynamicAliases()

   def getCommandSuffix( self, completions, partialToken, aliasCompletions ):
      t3( "Getting completions for", completions, "with a partial token of'",
          partialToken, "' and alias completions of", aliasCompletions )
      partialToken = partialToken.lower()
      cn = []

      def addCompletions( allCompletions, nameFn ):
         for c in allCompletions:
            # normally, partial token should be a prefix of the completions
            # but if not, we add '\b' to offset those characters that are not.
            name = nameFn( c )
            comm = os.path.commonprefix( ( partialToken, name.lower() ) )
            numb = len( partialToken ) - len( comm )
            cn.append( '\b' * numb + name[ len( comm ) : ] )

      addCompletions( completions, lambda x: x.name )
      addCompletions( aliasCompletions, lambda x: x.aliasCmd )

      compl = os.path.commonprefix( cn )
      if ( len( cn ) == 1 and ( ( completions and not completions[ 0 ].partial )
                                or ( aliasCompletions ) ) ):
         # Append a space if there was a unique completion, and the completion
         # gives the complete token (rather than a directory name, for example).
         compl += ' '

      t3( "Returning an alias completion of", compl )
      return compl

   def getCompletionLists( self, cmd ):
      ( _, partialToken, completions, aliasCompletions, _ ) = \
                     self.getCompletions_( cmd, startWithPartialToken=True )
      completions = list( completions )

      # It's only OK to try to extend the token to a partial or full
      # literal completion, plus we do not tab-complete guard-blocked tokens.
      # If you want to know why it's not completing, press "?", and you'll
      # see the guard code.
      completions = [ c for c in completions if c.literal ]
      unguardedCompletions = [ c for c in completions if c.guardCode is None ]
      return ( completions, unguardedCompletions, partialToken, aliasCompletions )

   def getCompletions_( self, cmd, startWithPartialToken=False ):
      if CliParser.ignoredComment( self.mode_, cmd.strip() ):
         # We don't need to print any help for a comment we ignore.
         return ( [], '', [], [], 'error' )

      allTokens = CliParser.textToTokens( cmd, mode=self.mode_ )

      if cmd == '' or cmd != cmd.rstrip():
         tokens = allTokens
         partialToken = ''
      else:
         tokens = allTokens[ : -1 ]
         partialToken = allTokens[ -1 ]

      # partialToken is the empty string if the line was empty or there was
      # whitespace at the end of the line, or the last token from the line
      # otherwise.  tokens is a list of the rest of the tokens from the line, if
      # any.

      try:
         completions = self.getCompletions(
            tokens, partialToken, startWithPartialToken=startWithPartialToken )
         if not completions:
            status = 'invalid'
         else:
            status = 'ok'
      except CliCommon.InvalidInputError:
         completions = set()
         status = 'invalid'
      except CliCommon.AmbiguousCommandError:
         completions = set()
         status = 'ambiguous'
      except CliCommon.GuardError:
         completions = set()
         status = 'guarded'
      except CliCommon.ParserError:
         # other generic parse error
         completions = set()
         status = 'error'

      aliasCompletions = self._getAliasCompletions( tokens, partialToken )
      if aliasCompletions:
         status = 'ok'

      return ( tokens, partialToken, completions, aliasCompletions, status )

   def _getAliasCompletions( self, tokens, partialToken ):
      if not partialToken:
         return []

      # we have to be the first token unless the first one is no/default
      if ( tokens and ( len( tokens ) > 1 or
                        tokens[ 0 ].lower() not in ( "no", "default" ) ) ):
         return []

      aliasCompletions = {}
      partialTokenLower = partialToken.lower()
      # The order of lists in the below addition makes sure we give precedence
      # to the regular alias over a dynamic alias with the same name.
      aliases = self.cliConfig.dynamicAlias.items() + self.cliConfig.alias.items()
      for cmdAliasLower, alias in aliases:
         if not cmdAliasLower.startswith( partialTokenLower ):
            continue

         origCmd = self._getAliasOrigCmd( alias )
         helpStr = '*%s="%s"' % ( alias.cmdAlias, origCmd )
         aliasCompletions[ alias.cmdAlias ] = \
            AliasHelp( alias.cmdAlias, origCmd, helpStr )

      t3( "Matched aliases", tokens )
      return sorted( aliasCompletions.values(),
                     key=lambda aliasInfo: aliasInfo.aliasCmd )

   def _getAliasOrigCmd( self, alias ):
      if len( alias.originalCmd ) > 1:
         # Multi-line command alias.
         firstSeq = alias.originalCmd.keys()[ 0 ]
         return "%s ..." % alias.originalCmd[ firstSeq ]
      elif len( alias.originalCmd ) == 1:
         return alias.originalCmd.values()[ 0 ]
      else: # Empty alias.
         return ""

   def prompt( self, promptPrefix ):
      promptFormat = self.promptFormat()
      drainReadlineWorkQueue()
      promptString = ""

      i = 0
      while i < len( promptFormat ):
         if ( promptFormat[ i ] == '%' ) and ( i < ( len( promptFormat ) - 1 ) ):
            i += 1
            c = promptFormat[ i ]
            if c == 'h' or c == 'H':
               # Find the hostname.
               hostname = None
               if hostnameHandler_:
                  hostname = hostnameHandler_()
               if not hostname:
                  hostname = socket.gethostname()
               if c == 'h':
                  # only up to the first dot.
                  j = hostname.find( '.' )
                  if j != -1:
                     hostname = hostname[ : j ]
               promptString += hostname[ : self.promptHostnameLimit_ ]
            elif c == 'p':
               promptString += self.mode_.shortPrompt()
            elif c == 'P':
               promptString += self.mode_.longPrompt()
            elif c == 's':
               promptString += ' '
            elif c == 't':
               promptString += '\t'
            elif c == '%':
               promptString += '%'
            elif c == 'D':
               more = promptFormat[ i + 1 : ]
               m = re.match( "{([^}]*)}", more )
               if not m:
                  timeFormat = "%c"
               else:
                  timeFormat = m.group( 1 )
                  i += len( timeFormat ) + 2
               promptString += time.strftime( timeFormat )
            elif c in promptFormatSpecifiers:
               promptString += promptFormatSpecifiers[ c ]( self )
         else:
            promptString += promptFormat[ i ]
         i += 1

      # add secure-monitor indication
      if self.secureMonitor():
         promptPrefix += "(secure)"
      # add memory stats into the prompt if so configured
      msc = self.memoryStatsContext.cfg
      if msc and not msc.log:
         promptPrefix += self.memoryStatsContext.state.msg
      return promptPrefix + promptString

   def _printCommandError( self, f, cmd, exc ):
      print( 'Command:', cmd, file=f )
      print( '', file=f )
      ( ty, val, tb ) = exc
      import Excepthook
      Excepthook.printException( ty, val, tb, dest=f )

   def _tryToGetErrorLogfileNumberAndCleanup( self, errorLogLimit=100 ):
      def alnum_key( s ):
         """ Key to sort a list alphanumeric or the "natural" way. """
         def convert( text ):
            return int( text ) if text.isdigit() else text
         return [ convert( c ) for c in re.split( '([0-9]+)', s ) ]

      oldLogFilenames = sorted( os.listdir( CliCommon.getCliLogDir() ),
                                key=alnum_key )
      logFileNum = 0
      if oldLogFilenames:
         if len( oldLogFilenames ) >= errorLogLimit:
            # too many error logs, cleanup the oldest half
            errorLogLowMark = len( oldLogFilenames ) // 2
            for x in oldLogFilenames[ : errorLogLowMark ]:
               try:
                  os.remove( "%s/%s" % ( CliCommon.getCliLogDir(), x ) )
               except OSError, e:
                  if e[ 1 ] == errno.ENOENT:
                     # the file just disappeared - maybe another
                     # Cli session deleted it.
                     pass
                  raise
            oldLogFilenames = oldLogFilenames[ errorLogLowMark : ]
         for lastLogFilename in reversed( oldLogFilenames ):
            m = re.match( r'err-(\d+)\.log', lastLogFilename )
            if m:
               logFileNum = int( m.group( 1 ) ) + 1
               break
      return logFileNum

   def _handleCliInternalError( self, cmd, lineNo=None ):
      if self.shouldPrint():
         print()
      self.addError( 'Internal error', cmd=cmd, lineNo=lineNo )
      logFileNum = self.writeInternalError( cmd )
      if logFileNum is not None:
         self.addError( "To see the details of this error, run the command "
                        "'show error %d'" % logFileNum )

   def writeInternalError( self, cmd ):
      # Return the error log file number, or None if we couldn't
      # create an error log file.
      exc = sys.exc_info()
      try:
         # Write information about the exception to a new file named
         # err-NNNNNN.log in the /var/log/cli directory.
         while True:
            try:
               logFileNum = self._tryToGetErrorLogfileNumberAndCleanup()
               logFilename = os.path.join( CliCommon.getCliLogDir(),
                                           'err-%06d.log' % logFileNum )
               fd = os.open( logFilename, os.O_WRONLY | os.O_CREAT | os.O_EXCL,
                             0666 )
               break
            except OSError, e:
               if e[ 0 ] != errno.EEXIST:
                  print( "Error saving logs:", e.strerror )
                  raise

               # The file already existed.  Go around the loop again and try
               # to get another filename.

         f = os.fdopen( fd, 'w' )
         self._printCommandError( f, cmd, exc )
         f.close()
         return logFileNum
      except EnvironmentError:
         # We can't write to /var/log/cli.  Print the exception information to
         # stdout instead.
         print( "--------------------------------" )
         print( "Exception calling value function" )
         print( "--------------------------------" )
         self._printCommandError( sys.stdout, cmd, exc )
         return None

   def handleAlreadyHandledError( self, e ):
      if e.msgType == e.TYPE_ERROR:
         self.addError( e.message )
      elif e.msgType == e.TYPE_WARNING:
         self.addWarning( e.message )
      elif e.msgType == e.TYPE_INFO:
         self.addMessage( e.message )

   def _handleCliException( self, exc_info, cmd, lineNo=None,
                            handleKeyboardInterrupt=True ):
      """Handles a exception raised by the CLI by printing an error message."""
      def _printCommandAborted():
         self.addMessage( '' )
         self.addMessage( 'Command aborted' )

      try:
         # BUG 9003: Due to the way we take over Python interrupt handling,
         # we may have come here with a different exception (e.g., EINTR)
         # which was actually caused by SIGINT. So we check it here.
         TacSigint.check()
         raise exc_info[ 0 ], exc_info[ 1 ], exc_info[ 2 ]
      except ( IOError, select.error ), e:
         if handleKeyboardInterrupt and e.args[ 0 ] == errno.EINTR:
            _printCommandAborted()
         else:
            raise
      except KeyboardInterrupt:
         if handleKeyboardInterrupt:
            _printCommandAborted()
         else:
            raise
      except ( CliCommon.AmbiguousCommandError,
               CliCommon.IncompleteCommandError,
               CliCommon.IncompleteTokenError,
               CliCommon.InvalidInputError,
               CliCommon.GuardError,
               CliCommon.AuthzDeniedError,
               CliCommon.ApiError ), e:
         # FIXME: this is just to keep existing behavior. Not sure if it's
         # intended.
         #
         # CAPI: "Invalid input (at token 0: 'this-is-a-bad-command')"
         # CLI:  "Invalid input"
         error = str( e ) if self.isEapiClient() else e.message
         if error:
            self.addError( error, cmd, lineNo )
      except CliCommon.AlreadyHandledError, e:
         self.handleAlreadyHandledError( e )
      except ( LazyMount.LocallyReadOnlyException,
               ConfigMount.ConfigMountProhibitedError ), e:
         self.addError( str( e ) )
      except CliSession.SessionAlreadyCompletedError, e:
         self.addError( str( e ) )
         if self.isInteractive():
            # when loading a file, we start in configMode, not enableMode, so we
            # never exit config session
            self.exitFromConfigSession()

   def handleCliException( self, exc_info, cmd, lineNo=None,
                           handleKeyboardInterrupt=True ):
      try:
         self._handleCliException( exc_info, cmd, lineNo=lineNo,
                                   handleKeyboardInterrupt=handleKeyboardInterrupt )
      except:
         # any unhandled exception is handled here.
         self._handleCliInternalError( cmd, lineNo )

def parseNoOrDefault( tokens ):
   if tokens[ 0 ].lower() in ( "no", "default" ):
      noOrDefault = tokens[ 0 ]
      rest = tokens[ 1 : ]
      return ( [ noOrDefault ], rest )
   else:
      return ( [], tokens )

def expandAlias( mode, cliConfig, cmd, tokens, raiseOnError ):
   cmd = getattr( cmd, 'cmdStr', cmd ) # BUG506172.
   tokenize = partial( CliParser.textToTokens, mode=mode )
   noMatchResult = [ tokens ]
   noOrDefault, tokens = parseNoOrDefault( tokens )
   if not tokens or cliConfig is None:
      return noMatchResult
   else:
      regexMatch = None
      aliasMaxParams = len( tokens )

      # Look for "dumb" aliases first.
      firstToken = tokens[ 0 ].lower()
      aliasCfg = ( cliConfig.alias.get( firstToken ) or
                   cliConfig.dynamicAlias.get( firstToken ) )

      # Nothing found; look for regex aliases.
      if aliasCfg is None:
         if cliConfig.regexAlias:
            compiledRegexAliases = getCompiledRegexAliases( cliConfig.regexAlias )
            for regex, index in compiledRegexAliases.items():
               t7( 'trying', regex.pattern, 'against', cmd )
               regexMatch = regex.match( cmd )
               if regexMatch:
                  t7( 'match!' )
                  aliasCfg = cliConfig.regexAlias.get( index )
                  if aliasCfg is None:
                     return noMatchResult
                  retTokens = []
                  # `format` uses 0-based params, but alias templates are 1-based.
                  groups = [ None ]
                  # Optional groups that didn't match are `None`, so use ''.
                  groups += [ g or '' for g in regexMatch.groups() ]
                  # This needs some deciphering and TODO: cleanup
                  for seq in aliasCfg.originalCmd:
                     originalCmd = aliasCfg.originalCmd[ seq ]
                     # Alias param string to `format`-compatible one.
                     # e.g., 'foo %1 %2 bar %3' -> 'foo {1} {2} bar {3}'.
                     template = reAliasParamRe.sub( r'{\1}', originalCmd )
                     try:
                        substituted = template.format( *groups )
                     except IndexError:
                        if raiseOnError:
                           raise CliCommon.IncompleteCommandError
                        return retTokens
                     retTokens.append( tokenize( substituted ) )
                  # Append any leftovers past the match. e.g., pipes.
                  retTokens[ -1 ] += tokenize( cmd[ regexMatch.end() : ] )
                  return retTokens

      # Found anything?
      if aliasCfg is None:
         return noMatchResult

      # Expand aliased commands
      numParam = 0
      maxParam = 0
      lastSeq = aliasCfg.originalCmd.keys()[ -1 ]
      retTokens = []
      for seq in aliasCfg.originalCmd:
         expandedTokens = []
         originalCmdTokens = tokenize( aliasCfg.originalCmd[ seq ] )
         for token in originalCmdTokens:
            m = aliasParamRe.match( token )
            if m:
               numParam = int( m.group( 1 ) )
               # Store maximum positional parameter seen so far
               if maxParam < numParam:
                  maxParam = numParam
               if numParam >= aliasMaxParams:
                  # Not all of the arguments were provided
                  if raiseOnError:
                     raise CliCommon.IncompleteCommandError

                  retTokens.append( noOrDefault + expandedTokens )
                  return retTokens
               else:
                  expandedTokens.append( tokens[ numParam ] )
            else:
               expandedTokens.append( token )
         if seq == lastSeq:
            #  Extra tokens are only appended to the last command in
            #  the alias.
            expandedTokens = expandedTokens + tokens[ maxParam + 1 : ]
         retTokens.append( noOrDefault + expandedTokens )
      return retTokens

class ConfigSessionWrapper( object ):
   """Context manager that wraps a session around a single function (usually
   the value function of a rule, in a mode)"""
   def __init__( self, mode ):
      self.active_ = False
      self.mode_ = mode
      assert( mode.session_ and
              mode.session_.autoConfigSession_ and
              isinstance( mode.session_.autoConfigSession_, str ) )
      self.sessionName_ = mode.session_.autoConfigSession_

      if CliSession.isSessionPresent( self.sessionName_ ):
         # Get a different session name.
         autoSessionPrefix = 'autosess_%d_' % os.getpid()
         self.sessionName_ = CliSession.uniqueSessionName(
                                 self.mode_.session_.entityManager_,
                                 prefix=autoSessionPrefix,
                                 okToReuse=True )

   def __enter__( self ):
      if self.mode_.session_ and not self.mode_.session_.inConfigSession():
         response = CliSession.enterSession( self.sessionName_,
                                             self.mode_.session_.entityManager_ )
         if response:
            self.mode_.addError( response )
         else:
            self.active_ = True
            if self.mode_.session_.autoConfigSessionNumCmds_ > 0:
               self.mode_.session_.autoConfigSessionNumCmds_ -= 1

   def __exit__( self, _exceptionType, _value, _traceback ):
      if self.active_:
         try:
            if _exceptionType:
               CliSession.abortSession( self.mode_.session_.entityManager_ )
            else:
               response = CliSession.commitSession(
                     self.mode_.session_.entityManager_ )
               if response:
                  self.mode_.addError( response )
               else:
                  name = self.sessionName_
                  CliSessionDataStore.commit( self.mode_, name )
                  CliSessionOnCommit.invokeSessionOnCommitHandlers( self.mode_,
                                                                    name )
         except KeyboardInterrupt:
            pass

         CliSession.exitSession( self.mode_.session_.entityManager_ )
         if not _exceptionType:
            # Delete session here (so we can re-use next __enter__ )
            response = CliSession.deleteSession( self.mode_.session_.entityManager_,
                                                 self.sessionName_ )
            if response:
               self.mode_.addWarning( response )

# Can be set by a cli command's handler, in case some cleanup is required when
# the command is over, as the handler might not have had a chance do it in
# cases like a control-c (KeyboardInterrupt) or broken pipe (IOError) interrupts.
# The handler would have to find its state via a global variable.
# In case of recursion ("watch" or "show tech" commands): if those need a callback
# too, then commandHandlerCleanupCallback should become a list, and cleanup needs
# to go in steps, unless the top level calls all callbacks in case its unwinding uses
# special exceptions that don't "stop" at runCmd().
def setCommandHandlerCleanupCallback( func ):
   t3( gettid(), "setCommandHandlerCleanupCallback with", func )
   if not func:
      return

   if getattr( commandHandlerCleanupLockThreadLocal, 'func', None ) is not None:
      # this function can get multiple times within a thread, but
      # the cleanup will only be called once. So if we already have
      # a function that means that we should already have this lock
      t3( gettid(),
          "setCommandHandlerCleanupCallback func is already set for thread" )
      return

   commandHandlerCleanupLockThreadLocal.func = func

# invoke the cleanup handler registered by the command handler (if any)
def doCommandHandlerCleanupCallback():
   t3( gettid(), "doCommandHandlerCleanupCallback" )
   # lets check if this thread installed a commandHandlerCleanupCallback or not
   # if it did not, we should not release any sorts of locks
   if getattr( commandHandlerCleanupLockThreadLocal, 'func', None ) is None:
      t3( gettid(), "doCommandHandlerCleanupCallback this thread doesn't have a "
          "commandHandlerCleanup" )
      return
   commandHandlerCleanupLockThreadLocal.func()
   commandHandlerCleanupLockThreadLocal.func = None
