# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import glob
import os

from EosCloudInitLib import writeFile
from SfFruHelper import SfFruHelper
from SfFruHelper import DPS_INTF, SYS_CLASS_NET, PORT_MGMT
import Tac
import Tracing

__defaultTraceHandle__ = Tracing.Handle( "Fru.SfFruHelper" )
t0 = Tracing.trace0

class AzureFruHelper( SfFruHelper ):
   def __init__( self, driver ):
      SfFruHelper.__init__( self, driver )
      self._ncpus = os.sysconf( "SC_NPROCESSORS_ONLN" )
      self._startcpu = 2
      if self._ncpus <= 4:
         self._startcpu = 1

   def hasManagementIntf( self ):
      return False

   def getRealDevName( self, devName ):
      loPath = os.path.join( SYS_CLASS_NET, devName, "lower_*" )
      if not glob.glob( loPath ):
         return None
      return glob.glob( loPath )[ 0 ].split( "lower_" )[ 1 ]

   def isMellanoxDriver( self, devName ):
      # Read driver from kernel and not from deviceCache
      return self.doGetDriver( devName ) in [ "mlx4_en", "mlx5_core" ]

   def isIgnoredInterface( self, devName ):
      skipIntf = False

      if devName == DPS_INTF:
         skipIntf = False
      elif not os.path.exists( os.path.join( SYS_CLASS_NET, devName, "device" ) ):
         # Skip interfaces like tunnelTap0, dummy0, etc.
         t0( "Skip virtual interface: %s" % devName )
         skipIntf = True
      elif self.isMellanoxDriver( devName ):
         t0( "Skip Mellanox interface: %s" % devName )
         return True

      return skipIntf

   def sortDevNames( self, devNames ):
      return sorted( devNames, key=lambda intf: filter( str.isdigit, intf ) )

   def getNewDevName( self, oldDevName, portId ):
      # Ebra agent crashes on Azure if the interface are renamed to et<x>_hv/mlx,
      # these devices are hidden device and Ebra agent must not react.
      # Fix is to rename the hidden hv_netvsc and mlx4_en/mlx5_core
      # interfaces to hv_et<x> and mlx_et<x> to avoid Ebra agent
      # reacting to these interfaces.

      if oldDevName == DPS_INTF:
         return oldDevName

      if portId == PORT_MGMT:
         newDevName = "ma1"
      else:
         newDevName = "%s%d" % ( self._frontPanelPrefix, portId )

      if self.isMellanoxDriver( oldDevName ):
         newDevName = "mlx_" + newDevName
      elif self.veosConfig( "MODE" ) == "sfe" and portId != PORT_MGMT:
         newDevName = "hv_" + newDevName

      return newDevName

   def doGetPci( self, devName ):
      # On Azure platform synthetic interface ( hv_netvsc ) does not have
      # pci address, instead uuid of the interface is used to bind to DPDK
      if self.getDriver( devName ) == "hv_netvsc":
         sysPath = os.path.join( '/', 'sys', 'class', 'net', devName, 'device' )
         if not os.path.exists( sysPath ):
            return ""

         realPath = Tac.run( [ "realpath", sysPath ], stdout=Tac.CAPTURE ).strip()
         return os.path.basename( realPath )
      else:
         return super( AzureFruHelper, self ).doGetPci( devName )

   def renameInterface( self, oldDevName, portId ):
      # First rename hv_netvsc interface
      newDevName = \
            super( AzureFruHelper, self ).renameInterface( oldDevName, portId )

      # Now rename corresponding lower_ethX mlx interface
      realDev = self.getRealDevName( newDevName )
      if realDev:
         super( AzureFruHelper, self ).renameInterface( realDev, portId )

      return newDevName

   def getIrqlist( self, devName ):
      irqs = None
      sysPath = os.path.join( SYS_CLASS_NET, devName, "device/msi_irqs" )
      if os.path.exists( sysPath ):
         output = Tac.run( [ "ls", sysPath ], stdout=Tac.CAPTURE )
         irqs = output.strip().split( "\n" )
      return irqs

   def setIrqAffinity( self, devName ):
      irqs = self.getIrqlist( devName )
      if irqs is None:
         return
      cpu = self._startcpu
      for irq in irqs:
         if cpu >= self._ncpus:
            cpu = self._startcpu
         irqPath = os.path.join( "/proc/irq/", irq, "smp_affinity" )
         cpuStr = str( hex( 2 ** cpu ).lstrip( "0x" ) )
         if os.path.exists( irqPath ):
            writeFile( irqPath, cpuStr )
            t0( "setIrqAffinity %s to %s" % ( cpuStr, irqPath ) )
            cpu += 1

   def addInterface( self, devName, **extraArgs ):
      super( AzureFruHelper, self ).addInterface( devName, **extraArgs )

      # Only the irq affinity if more than 2 cores on Sfa
      if self.veosConfig( "MODE" ) == "linux" and self._ncpus > 2:
         self.setIrqAffinity( devName )
