# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This deals with all the AWS cloud side. This currently uses boto2 AWS SDK. 
# It should be upgraded to boto3 in future.
#
import ArPyUtils.Decorators
import boto.vpc
import Tracing
import threading
import socket
import Tac
import CloudHaBackend
import exceptions
from CloudException import BackendException

t0Saved = Tracing.t0
t1Saved = Tracing.t1
t2Saved = Tracing.t2

def debugMsg( *s ):
   msg = '** %s **: ' % ( threading.current_thread().name ) 
   for i in s:
      msg += str( i )
   return msg

def t0( *msg ):
   t0Saved( debugMsg( msg ) )

def t1( *msg ):
   t1Saved( debugMsg( msg ) )

def t2( *msg ):
   t2Saved( debugMsg( msg ) )

class AwsAccess( object ):
   def __init__( self, region, key, access, port=None, proxy=None,
         proxy_port=None, proxy_user=None, proxy_pass=None):
      self.region = region
      self.key = key
      self.access_key = access
      self.is_secure = True
      self.port = port
      self.proxy = proxy
      self.proxy_port = proxy_port
      self.proxy_user = proxy_user
      self.proxy_pass = proxy_pass
   
   def __str__( self ):
      return "Access Object:Region:" + self.region + "Port:" + self.port \
         + "Proxy:" + self.proxy + "Proxy Port:" + self.proxy_port \
         + "Proxy User:" + self.proxy_user


class AwsParsedConfig( object ):
   def __init__( self, access, localRoutes, peerRoutes ):
      assert access and isinstance( access, AwsAccess)
      self.access = access 
      self.localRoutes = localRoutes
      self.peerRoutes = peerRoutes

# This class should never be used directly as this interface with AWS SDK 
# which uses blocking calls and may create issues with procMgr during boot 
# as the networking may not be up. Use wrapper class instead. You need to 
# be careful adding any tacc related code here due to thread lock issues.
class AwsBackend( CloudHaBackend.BackendBase ):

   def __init__( self, parsedConfig ):
      super( AwsBackend, self ).__init__()
      assert parsedConfig and isinstance( parsedConfig, AwsParsedConfig )
      self.parsedConfig = parsedConfig
      self.conn = None

   def getVpcConnection( self ):
      try:
         access = self.parsedConfig.access
         region = access.region
         key = access.key
         access_key = access.access_key
         self.conn = boto.vpc.connect_to_region( region, aws_access_key_id=key, 
            aws_secret_access_key=access_key, port=access.port, 
            proxy=access.proxy, proxy_port=access.proxy_port,
            proxy_user=access.proxy_user, 
            proxy_pass=access.proxy_pass ) 
      except exceptions.Exception, e:
         t0( 'Failed to get vpc connection %s' % e )
         self.updateResultMessage( "Network connectivity issue" )
         self.updateResultFailureType( 'TRANSIENT' )
         raise BackendException( 'Error getting AWS vpc connection' )

      assert self.conn
      return self.conn

   def getInstance( self, Id=None ):
      assert self.conn
      if not Id:
         Id = Tac.run( [ 'wget', '-q', '-O', '-', \
            'http://169.254.169.254/latest/meta-data/instance-id' ], \
            stdout=Tac.CAPTURE )
      instance = self.conn.get_only_instances( [ Id ] ) 
      if not len( instance ) == 1:
         t0(' instance not found for id %s ' % Id )
         return None
      return instance[0]

   def getRouteTables( self, ids ):
      if not self.conn:
         self.getVpcConnection()
      fetchedTable = self.conn.get_all_route_tables( route_table_ids=ids )
      if len( fetchedTable ) != len( ids ):
         t0( 'Failed to get all the routeTable IDs. Ignoring it', \
               fetchedTable, ids )
         # boto SDK gets confused if there is route propagation set in the 
         # route table and gives us list with bunch of 'None' along with the 
         # expected route table IDs. We ignore the len of the list but do validate 
         # that the route table indeed exists correctly as done below.
      fetchedRtTableId = [ i.id for i  in fetchedTable ]
      for i in ids:
         if i not in fetchedRtTableId :
            t0( 'Error: Unable to locate routeTable ID %s' % i )
            raise BackendException( \
               'Unable to retrieve route table id %s' % i )
      return fetchedTable

   # Extract the default route from the tableId
   @staticmethod
   def getRouteInRouteTable( table, dest ):
      for i in table.routes:
         if i.destination_cidr_block == dest:
            return i

    # Extract the default route from the tableId
   @staticmethod
   def getDefaultRouteForRouteTable( table ):
      return AwsBackend.getRouteInRouteTable( table, r'0.0.0.0/0' )
   
   def replaceRoute( self, routeTable, dest, intf, dryRun=False ):
      t1( 'replaceRoute called for %s , dest %s, intf %s ' % \
         ( routeTable, dest, intf ) )
      assert self.conn
      conn = self.conn
      route = AwsBackend.getRouteInRouteTable( routeTable, dest )
      msg = 'Failed to replaced route for %s , dest %s, intf %s' % \
               ( routeTable, dest, intf )
      try:
         ret = conn.replace_route( routeTable.id, route.destination_cidr_block, \
            interface_id=intf, dry_run=dryRun )
      except boto.exception.EC2ResponseError , e:
         t2( 'replace_route: Exception response', e )
         if dryRun and e.error_code == 'DryRunOperation' and \
            not e.message.find('Request would have succeeded, but'\
            ' DryRun flag is set.'):
            return True
         else:
            t0( msg, 'exception: %s '% e )
            raise BackendException( msg )

      t0(' Replaced route for %s, dest %s, intf %s' % \
            ( routeTable, dest, intf ) )
      return ret

   def updatePeerRouteTables( self ):
      return self.updateRouteTable( peer=True )

   def updateLocalRouteTables( self ):
      return self.updateRouteTable( peer=False )
      
   def updateRouteTable( self, peer ):
      t1( "UpdateRouteTable called for peer: %s" % peer )
      try:
         routes = self.parsedConfig.peerRoutes if peer else \
                     self.parsedConfig.localRoutes
         for rts, intf in routes.iteritems():
            rtId, dest = rts
            rt = self.getRouteTables( [ rtId ] )[ 0 ]
            self.replaceRoute( rt, dest, intf, dryRun=False )
      except( socket.error, socket.herror, socket.gaierror ) as e:
         t0( 'Failed to update routing tables: ', e )
         self.updateResultMessage( "Network connectivity issue" )
         self.updateResultFailureType( 'TRANSIENT' )
         return False
      except( boto.exception.BotoServerError, \
            boto.exception.BotoClientError ) as e:
         t0( e )
         self.updateResultMessage( e.message )
         return False
      except exceptions.Exception, e:
         t0( 'Failed to update %s routing tables: %s ' % ( \
            'peer' if peer else 'local', e ) )
         self.updateResultMessage( str( e ) )
         return False
      return True

   @ArPyUtils.Decorators.retry( retryCheckEmbeddedMethodName='transientFailure',
                                attempts=4, retryInterval=30 )
   def configValidator( self ):
      t1( 'Invoked AWS Backend configValidator' )
      try:
         self.getVpcConnection()
         # Now run through validations for local and peer routes
         self.configRouteValidator()
         self.configRouteValidator( doPeerRoutes=False )
      except ( socket.gaierror, socket.error, socket.herror ) as e:
         # Protect against all networking issues.
         t0( 'Received socket error from boto: %s ' % e )
         self.updateResultMessage( 'Network connectivity issue' )
         self.updateResultFailureType( 'TRANSIENT' )
         return False
      except exceptions.Exception, e:
         t0( str( e ) )
         if isinstance( e, boto.exception.BotoServerError ):
            self.updateResultMessage( e.message )
         else:   
            self.updateResultMessage( str( e ) )
         return False
      else:
         t0( 'Successful in parsing AWS HA config' )
         return True

   def configRouteValidator( self, doPeerRoutes=True ):
      t1( 'Invoked AWS Backend configRouteValidator, doPeerRoutes %s' % \
            ( doPeerRoutes ) )
      assert( self.conn )
      conn = self.conn
      if doPeerRoutes:
         cfgRoutes = self.parsedConfig.peerRoutes
      else:
         cfgRoutes = self.parsedConfig.localRoutes
      t2('Routes to check %s ' % cfgRoutes )
      #
      # Make sure the config is sane
      # These are the various checks done:
      # -Check the route table IDs are valid.
      # -Check the dest prefix are valid
      # -Corresponding Network interface IDs are valid and 
      #  assigned to self.
      # -Indirectly check our credentials are fine and we can control the routes.
      #
      cfgRouteTableIds = { i[ 0 ] for i in cfgRoutes.keys() }
      fetchedTable = self.getRouteTables( list( cfgRouteTableIds ) )
      fetchedRtTableId = [ i.id for i  in fetchedTable ]

      # Unfortunately boto2 lib doesn't seem to give us interface Id for  
      # route...This works fine with management console/aws-cli etc.
      #  Check that the interface do belong to us.
      cfgNifs = [ i for i in cfgRoutes.values() ]
      Filter = {}
      Filter[ 'attachment.instance-id' ] = self.getInstance().id 
      gotNifs = conn.get_all_network_interfaces( filters=Filter )
      gotNifsId = [ i.id for i in gotNifs ]
      for i in cfgNifs:
         if i not in gotNifsId:
            t0( 'configured network interface %s not found as assigned to me' \
                  %  i )
            raise BackendException( \
               'next-hop interface %s does not belong to me' % i )
      for routeDest, intf in cfgRoutes.items():
         rt, dest = routeDest
         fetchedRoute = fetchedTable[ fetchedRtTableId.index( rt ) ]
         route = AwsBackend.getRouteInRouteTable( fetchedRoute, dest ) 
         if not route:
            msg = 'No route found for %s in rt %s '% ( dest, rt )
            t0( msg )
            raise BackendException( msg )
         # This can be self or peer instance ID.
         instance_id = route.instance_id
         # Make sure this is valid for self or peer eos instance
         peer_or_self = self.getInstance( instance_id )
         if not peer_or_self:
            msg ='Bad veos instance %s for routing next hop ' \
                  % ( instance_id )
            t0( msg )
            raise BackendException( msg )

         # make sure we can actually change the route as needed during HA.
         if not self.replaceRoute( fetchedRoute, dest, intf, dryRun=True ) :
            msg = 'Unable to validate route replacement for %s '\
                  %  route
            t0( msg )
            raise BackendException( msg )
      return True
