#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tracing
import AsuPStore
from CliPlugin.AsuPStoreModel import ReloadHitlessBlockingReason
from CliPlugin.AsuPStoreModel import ReloadHitlessWarningReason
import Tac

__defaultTraceHandle__ = Tracing.Handle( "StpAsu" )
t0 = __defaultTraceHandle__.trace0

class StpAsuPStoreEventHandler( AsuPStore.PStoreEventHandler ):
   '''Stp mode other than NoSTP or MSTP blocks ASU2 hitless reload.
      AsuPStorePlugin's hitlessReloadSupported is used for this
      purpose. Other apis do not apply, and should not be registered.
   '''
   def __init__( self, stpConfig, stpPortMode, errDisabled ):
      self.stpConfig_ = stpConfig
      self.errDisabled_ = errDisabled
      self.stpPortMode_ = stpPortMode
      self.mlagConfigured_ = False
      forceForwarding = \
         Tac.Type( 'EthIntf::PortModeConfig::PortMode' ).forceForwarding
      if self.stpPortMode_ and self.stpPortMode_.entity.get( 'mlag' ) :
         mlagStpPortModes = self.stpPortMode_.entity[ 'mlag' ].portMode.values()
         # We only care about presence of atleast one interface in stp/portMode/mlag
         # with portMode as 'forceForwarding'. In reality only the Mlag PeerLink
         # interface is present within stp/portMode/mlag.portMode collection anyways.
         if forceForwarding in mlagStpPortModes:
            self.mlagConfigured_ = True
         else:
            self.mlagConfigured_ = False
      AsuPStore.PStoreEventHandler.__init__( self )

   def hitlessReloadSupported( self ):
      warningList, blockingList = [], []
      # only MSTP and NoSTP are supported
      if self.stpConfig_ and ( self.stpConfig_.version != "none" ) and \
                             ( self.stpConfig_.version != "mstp" ):
         t0( 'hitlessReloadSupport: stpConfig.version = ',
             self.stpConfig_.version )
         # Stp does not support Asu2, for modes other than NoSTP, Mstp
         blockingList.append( ReloadHitlessBlockingReason( reason='stpMode' ) )

      # if STP mode is MSTP, check:
      #   - all bridge ports are edge ports
      #   - bpduguard is enabled for all bridge ports
      if self.stpConfig_ and ( self.stpConfig_.version == "mstp" ):
         portfastBpduguard = self.stpConfig_.portfastBpduguard
         mstiConfigs = self.stpConfig_.stpiConfig[ 'Mst' ].mstiConfig.values()
         if not self.mlagConfigured_:
            for intf, config in ( it for mstiConfig in mstiConfigs
                                     for it in mstiConfig.mstiPortConfig.items() ):
               if( ( config.enabled == True ) and
                   ( self.stpConfig_.portConfig[ intf ].adminEdgePort == False ) ):
                  t0( 'hitlessReloadSupport: admingEdgePort is not set for ',
                      intf )
                  blockingList.append( ReloadHitlessBlockingReason(
                     reason='edgePort' ) )
                  break

            for intf, config in ( it for mstiConfig in mstiConfigs
                                     for it in mstiConfig.mstiPortConfig.items() ):
               portBpduguard = self.stpConfig_.portConfig[ intf ].bpduguard
               if( ( config.enabled == True ) and
                   ( ( portBpduguard == 'bpduguardDisabled' ) or
                     ( ( portBpduguard == 'bpduguardDefault' ) and
                       ( portfastBpduguard == False ) ) ) ):
                  t0( 'hitlessReloadSupport: Bpduguard is not enabled for ',
                      intf )
                  blockingList.append( ReloadHitlessBlockingReason(
                     reason='stpBpduguard' ) )
                  break

         if self.errDisabled_:
            for intf in ( it for mstiConfig in mstiConfigs
                             for it in mstiConfig.mstiPortConfig ):
               if self.errDisabled_.intfStatus.has_key( intf ):
                  t0( 'hitlessReloadSupport: port ', intf, ' is errDisabled' )
                  warningList.append( ReloadHitlessWarningReason(
                                          reason='stpErrDisabled' ) )
                  break

      return ( warningList, blockingList )

def Plugin( ctx ):
   featureName = 'Stp'
   if ctx.opcode() == 'CheckHitlessReloadSupported':
      entityManager = ctx.entityManager()
      mg = entityManager.mountGroup()
      stpConfig = mg.mount( 'stp/config', 'Stp::Config', 'r' )
      stpPortMode = mg.mount( 'stp/portMode', 'Tac::Dir', 'ri' )
      errDisabled = mg.mount( 'interface/errdisable/cause/bpduguard',
                              'Errdisable::CauseStatus', 'r' )
      mg.close( blocking=True )
      ctx.registerAsuPStoreEventHandler( featureName,
                                         StpAsuPStoreEventHandler( stpConfig,
                                                                   stpPortMode,
                                                                   errDisabled ) )
   else:
      ctx.registerAsuPStoreEventHandler( featureName,
                                         StpAsuPStoreEventHandler( None, None,
                                                                   None ) )
