# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from McastCommonCliLib import AddressFamily
import AsuPStore
from CliPlugin.AsuPStoreModel import ReloadHitlessWarningReason
import Tac
import PimLib

class PimAsuPStoreEventHandler( AsuPStore.PStoreEventHandler ):
   '''We should warn ASU2 hitless reload if PIM join/prune holdtime is less
   than 420 seconds on an interface. AsuPStorePlugin's hitlessReloadSupported is
   used for this purpose. Other apis do not apply, and should not be registered.
   '''
   def __init__( self, pimCtx ):
      if pimCtx:
         self.pimGlobalStatus = pimCtx[ 'pimGlobalStatus' ]
         self.pimAllStatusColl = pimCtx[ 'pimAllStatusColl' ]
      else:
         self.pimGlobalStatus = None
         self.pimAllStatusColl = None

      AsuPStore.PStoreEventHandler.__init__( self )

   def hitlessReloadSupported( self ):
      warningList, blockingList = None, None

      if self.pimGlobalStatus is None or self.pimAllStatusColl is None:
         return ( warningList, blockingList )

      bidirWarning = True if self.pimGlobalStatus.pimEnabledBidirVrf else False

      jpIntfWarning = False
      for vrfName in self.pimGlobalStatus.pimEnabledSparseModeVrf:
         status = self.pimAllStatusColl.vrfStatus.get( vrfName )
         if not status:
            continue

         for intfId in status.pimIntf:
            pimIntf = status.pimIntf.get( intfId )
            if not pimIntf:
               continue

            # Look at every PIM interface.  Check the holdtime is greater than
            # 2 * 210 seconds.  210 is the default holdtime.
            if 2 * pimIntf.joinPruneHoldTimeDefault > pimIntf.joinPruneHoldTime:
               jpIntfWarning = True
               break

      if jpIntfWarning or bidirWarning:
         warningList = []

         if jpIntfWarning:
            warningList.append( ReloadHitlessWarningReason( reason='pimHoldTime' ) )

         if bidirWarning:
            warningList.append( ReloadHitlessWarningReason(
               reason='pimBidirEnabled' ) )

      return ( warningList, blockingList )

def Plugin( ctx ):
   featureName = 'Pim'
   pimCtx = {}

   if ctx.opcode() == 'CheckHitlessReloadSupported':
      entityManager = ctx.entityManager()

      pimCtx = {}
      pimCtx[ 'pimGlobalStatus' ] = PimLib.getPimGlobalStatus(
            entityManager, AddressFamily.ipv4, blocking=True )
      pimCtx[ 'pimAllStatusColl' ] = PimLib.getPimAllStatusColl( AddressFamily.ipv4 )

      ctx.registerAsuPStoreEventHandler( featureName,
                                         PimAsuPStoreEventHandler( pimCtx ) )
   else:
      ctx.registerAsuPStoreEventHandler( featureName,
                                         PimAsuPStoreEventHandler( None ) )
