#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import AsuPStore
import Tracing
import MlagMountHelper
from CliPlugin.AsuPStoreModel import ReloadHitlessBlockingReason, \
   ReloadHitlessWarningReason, modifyValueOfWarningReasonsDict
from MlagShared import NEG_STATUS_CONNECTED
from CliPlugin.MlagShowCli import portCategorySummary, showMlagInt
from CliPlugin.MlagWarningCli import mlagInterfaceArgs

__defaultTraceHandle__ = Tracing.Handle( 'MlagAsuPStore' )
t0 = Tracing.trace0

class MlagAsuPStoreEventHandler( AsuPStore.PStoreEventHandler ):
   '''Mlag configuration blocks ASU2 hitless
   reload. AsuPStorePlugin's hitlessReloadSupported is used for this
   purpose. Other apis do not apply, and should not be registered.
   '''
   def __init__( self, mlagAsuContext ):
      self.funcs = { 'lacpLagIdConfig' : self.storeLacpLagIdConfig,
                     'stpInputConfigMlag' : self.storeStpInputConfigMlag,
                   }
      if mlagAsuContext:
         self.mlagConfig_ = mlagAsuContext[ 'mlagConfig' ]
         self.mlagStatus_ = mlagAsuContext[ 'mlagStatus' ]
         self.lacpOverrideDir_ = mlagAsuContext[ 'lacpOverrideDir' ]
      super( MlagAsuPStoreEventHandler, self ).__init__()

   def isPrimaryOrSecondary( self ):
      return self.mlagStatus_.mlagState in ( 'primary', 'secondary' )

   def hitlessReloadSupported( self ):
      '''
      If Mlag is not configured, don't show any warning.
      Warnings:
      1. Show warning for image upgrade for backwards-compatibility.
      2. Show warning if mlag/non-mlag reload delay is configured,
         it may result in some loss.
      3. Show warning if any of the mlag ports are active-partial.
      Blockers:
      1. Block reload hitless if mlag is not established.
      2. Block reload hitless if mlag state is unstable.
      '''

      warningList = []
      blockingList = []

      # This is used to currently block the feature as a toggle. This check will be
      # removed once all the muts are merged for mlag-asu2.
      if not self.mlagConfig_.configured():
         return ( [], [] )

      warningList.append( ReloadHitlessWarningReason(
         reason="mlagVersionCompatibility" ) )

      activePartialLocalPorts = portCategorySummary( self.mlagStatus_ )[ 3 ]
      if activePartialLocalPorts > 0:
         nonActiveModeWarning = "The following MLAGs are not in Active mode. " \
                                "Traffic to or from these ports will be lost " \
                                "during the upgrade process:\n"
         activePartialPorts = showMlagInt( None, mlagInterfaceArgs ).renderStr()
         modifyValueOfWarningReasonsDict( 'mlagPortsActivePartial',
                                           nonActiveModeWarning +
                                           activePartialPorts )
         warningList.append( ReloadHitlessWarningReason(
            reason='mlagPortsActivePartial' ) )

      if( self.mlagConfig_.reloadDelay.delay > 0 or
          self.mlagConfig_.reloadDelayNonMlag.delay > 0 ):
         warningList.append( ReloadHitlessWarningReason(
            reason='reloadDelayConfigured' ) )

      if self.mlagStatus_.negotiationStatus != NEG_STATUS_CONNECTED:
         blockingList.append( ReloadHitlessBlockingReason(
            reason='mlagNotEstablished' ) )

      if not self.isPrimaryOrSecondary():
         blockingList.append( ReloadHitlessBlockingReason(
            reason='mlagStateUnstable' ) )

      return ( warningList, blockingList )

   def getSupportedKeys( self ):
      return [ 'lacpLagIdConfig', 'stpInputConfigMlag' ]

   def getKeys( self ):
      return [ 'lacpLagIdConfig', 'stpInputConfigMlag' ]

   def storeLacpLagIdConfig( self ):
      '''LacpLagIdConfig is stored as  - 
      { lagIntfId : { systemId, portIdOffset, portIdMax, lagKey } }
      '''
      t0( self.__class__.__name__, __name__, 'store LacpLagIdConfig.' )
      lacpOverrideDir = self.lacpOverrideDir_
      pyCol = {}
      for lagIntfId, systemId in lacpOverrideDir.systemId.iteritems():
         info = pyCol.setdefault( lagIntfId, {} )
         info[ 'systemId' ] = systemId
      for lagIntfId, portIdRange in lacpOverrideDir.portIdRange.iteritems():
         info = pyCol.setdefault( lagIntfId, {} )
         info[ 'portIdOffset' ] = portIdRange.portIdMin
         info[ 'portIdMax' ] = portIdRange.portIdMax
      for lagIntfId, lagKey in lacpOverrideDir.lagKey.iteritems():
         info = pyCol.setdefault( lagIntfId, {} )
         # lagKey in lacpOverrideDir is not used
         # If lagKey is set, Lag agent will assert
         info[ 'lagKey' ] = lagKey
      return pyCol

   def storeStpInputConfigMlag( self ):
      '''stpInputConfigMlag is stored as - ( valid, bridgeAddr )
      '''
      t0( self.__class__.__name__, __name__, 'store stpInputConfigMlag' )
      valid = False
      bridgeAddr = None
      if self.mlagStatus_.systemId != '00:00:00:00:00:00':
         valid = True
         bridgeAddr = self.mlagStatus_.systemId
      return ( valid, bridgeAddr )

   def save( self, pStoreIO ):
      keys = self.getKeys()
      for k in keys:
         t0( 'saving', k )
         pStoreItem = self.funcs[k]()
         pStoreIO.set( k, pStoreItem )

def Plugin( ctx ):
   featureName = 'Mlag'
   mlagConfig = None
   mlagStatus = None
   mlagAsuContext = {}

   if ctx.opcode() == 'GetSupportedKeys':
      ctx.registerAsuPStoreEventHandler( featureName,
                                         MlagAsuPStoreEventHandler( None ) )
      return

   entityManager = ctx.entityManager()
   mg = entityManager.mountGroup()
   # Mount mlag/config, Mlag::Config and its dependent paths
   mlagConfig = MlagMountHelper.mountMlagConfig( mg )
   # Mount mlag/status, Mlag::Status and its dependent paths
   mlagStatus = MlagMountHelper.mountMlagStatus( mg )
   lacpOverrideDir = mg.mount( 'lag/input/lacpoverride/mlag',
                               'Lag::Input::LacpOverrideDir',
                               'r' )
   mg.close( blocking=True )

   mlagAsuContext[ 'mlagConfig' ] = mlagConfig
   mlagAsuContext[ 'mlagStatus' ] = mlagStatus
   mlagAsuContext[ 'lacpOverrideDir' ] = lacpOverrideDir

   ctx.registerAsuPStoreEventHandler(
      featureName, MlagAsuPStoreEventHandler( mlagAsuContext ) )
