#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AsuPStore
from CliPlugin.AsuPStoreModel import ReloadHitlessBlockingReason

class MaintenanceModeAsuPStoreEventHandler( AsuPStore.PStoreEventHandler ):
   '''We should block ASU2 hitless reload if either of following is true:
   1. Atleast one maintenance unit is quiesced
   2. on-boot is enabled in default unit profile
   3. on-boot is configured in user-configured unit profile associated with unit
   AsuPStorePlugin's hitlessReloadSupported is used for this purpose.
   Other apis do not apply, and should not be registered.
   '''
   def __init__( self, maintenanceInputDir, unitDefaultProfile, unitConfigDir,
                 unitProfileStatusDir ):
      self.maintenanceInputDir = maintenanceInputDir
      self.unitDefaultProfile = unitDefaultProfile
      self.unitConfigDir = unitConfigDir
      self.unitProfileStatusDir = unitProfileStatusDir
      AsuPStore.PStoreEventHandler.__init__( self )

   def hitlessReloadSupported( self ):
      warningList, blockingList = None, None
      # 1 : maintenance unit is quiesced
      quiescedUnits = len( self.maintenanceInputDir.unit )
      defaultUnitProfileName = self.unitDefaultProfile.profileName
      defaultUnitProfile = self.unitProfileStatusDir.status.get(
         defaultUnitProfileName )
      onBootEnabledInDefaultProfile = False
      # 2 : on-boot enabled in default unit profile
      if defaultUnitProfile:
         onBootEnabledInDefaultProfile = defaultUnitProfile.onBootEnabled
      atleastOneOnBootUnit = False
      # 3 : on-boot in user configured profile assciate with unit
      for unit in self.unitConfigDir.config:
         unitConfig = self.unitConfigDir.config.get( unit )
         unitProfile = None
         if unitConfig.profileName:
            unitProfile = self.unitProfileStatusDir.status.get(
               unitConfig.profileName )
         else:
            unitProfile = defaultUnitProfile
         if unitProfile and unitProfile.onBootEnabled:
            atleastOneOnBootUnit = True
            break

      if quiescedUnits or onBootEnabledInDefaultProfile or atleastOneOnBootUnit:
         blockingList = [ ReloadHitlessBlockingReason(
               reason='maintenanceModeConfigured' ) ]
      return ( warningList, blockingList )

def Plugin( ctx ):
   featureName = 'MaintenanceMode'
   if ctx.opcode() == 'CheckHitlessReloadSupported':
      entityManager = ctx.entityManager()
      mg = entityManager.mountGroup()
      maintenanceInputDir = mg.mount( 'maintenance/unit/input/cli',
                                      'Maintenance::Unit::Input', 'r' )
      unitDefaultProfile = mg.mount( 'maintenance/profile/config/default/unit',
                                     'Maintenance::Profile::DefaultProfile', 'r' )
      unitConfigDir = mg.mount( 'maintenance/unit/config',
                                'Maintenance::Unit::ConfigDir', 'r' )
      unitProfileStatusDir = mg.mount( 'maintenance/profile/status/unit',
                                       'MaintenanceUnitProfile::StatusDir', 'r' )
      mg.close( blocking=True )
      ctx.registerAsuPStoreEventHandler( featureName,
                                         MaintenanceModeAsuPStoreEventHandler(
            maintenanceInputDir, unitDefaultProfile, unitConfigDir,
            unitProfileStatusDir ) )
   else:
      ctx.registerAsuPStoreEventHandler( featureName,
                                         MaintenanceModeAsuPStoreEventHandler(
            None, None, None, None ) )
