#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AsuPStore
from collections import defaultdict

class IraPStoreEventHandler( AsuPStore.PStoreEventHandler ):
   def __init__( self, routingHardwareConfig, routing6HardwareConfig, ipStatus,
                 ip6Status ):
      AsuPStore.PStoreEventHandler.__init__( self )
      self.routingHardwareConfig_ = routingHardwareConfig
      self.routing6HardwareConfig_ = routing6HardwareConfig
      self.ipStatus_ = ipStatus
      self.ip6Status_ = ip6Status

   def getInterfaceInfo( self ):
      def _createInterface():
         rv = {}
         rv[ 'ipAddress' ] = ''
         rv[ 'secondaryIpAddress' ] = []
         rv[ 'ip6Address' ] = []
         rv[ 'vrf' ] = ''
         return rv

      def _addIpAddress( intfName, ipAddress ):
         interface[ intfName ][ 'ipAddress' ] = ipAddress

      def _addSecondaryIpAddress( intfName, secondaryIpAddress ):
         interface[ intfName ][ 'secondaryIpAddress' ].append( secondaryIpAddress )

      def _addIp6Address( intfName, ip6Address ):
         interface[ intfName ][ 'ip6Address' ].append( ip6Address )

      def _addVrf( intfName, vrfName ):
         interface[ intfName ][ 'vrf' ] = vrfName

      interface = defaultdict( _createInterface )

      # IPv4 Addresses
      for intfName, ipIntfStatus in self.ipStatus_.ipIntfStatus.iteritems():
         if intfName.startswith( 'Management' ):
            continue
         _addIpAddress( intfName, ipIntfStatus.activeAddrWithMask.stringValue )
         _addVrf( intfName, ipIntfStatus.vrf )
         # Store secondary IP Addresses
         for ipAddrWithMask in ipIntfStatus.activeSecondaryWithMask:
            _addSecondaryIpAddress( intfName, ipAddrWithMask.stringValue )

      # IPv6 Addresses
      for intfName, ip6IntfStatus in self.ip6Status_.intf.iteritems():
         if intfName.startswith( 'Management' ):
            continue
         _addVrf( intfName, ip6IntfStatus.vrf )
         for addrWithMask in ip6IntfStatus.addr:
            _addIp6Address( intfName, addrWithMask.stringValue )

      return interface

   def save( self, pStoreIO ):
      pStoreIO.set( 'interfaces', self.getInterfaceInfo() )

   def getSupportedKeys( self ):
      return [ 'interfaces' ]

   def getKeys( self ):
      return [ 'interfaces' ]


def Plugin( ctx ):
   featureName = 'Ira'

   if ctx.opcode() == 'GetSupportedKeys':
      routingHardwareConfig = routing6HardwareConfig = ipStatus = ip6Status = None
   else:
      entityManager = ctx.entityManager()
      mg = entityManager.mountGroup()
      if ctx.opcode() == 'CheckHitlessReloadSupported':
         routingHardwareConfig = mg.mount( 'routing/hardware/config',
               'Routing::Hardware::Config', 'r' )
         routing6HardwareConfig = mg.mount( 'routing6/hardware/config',
               'Routing6::Hardware::Config', 'r' )
         ipStatus = ip6Status = None
      else:
         ipStatus = mg.mount( 'ip/status', 'Ip::Status', 'r' )
         ip6Status = mg.mount( 'ip6/status', 'Ip6::Status', 'r' )
         routingHardwareConfig = routing6HardwareConfig = None
      mg.close( blocking=True )

   ctx.registerAsuPStoreEventHandler(
         featureName,
         IraPStoreEventHandler( routingHardwareConfig,
                                routing6HardwareConfig,
                                ipStatus, ip6Status ) )
