#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
The EthIntf ASU PStore plugin is used for preserving the names of runtime-errdisabled
interfaces accross the reboot. Runtime-errdisable interfaces are those interfaces 
that are errdisabled due to the runtime conditions (such as link flaps) as opposed
to user config (such as portgroup-disabled).

Also used for storing all MTUs so interfaces that are brought up by AFPR have the
correct MTU.
"""

import Tac, Tracing
import AsuPStore
from CliPlugin.AsuPStoreModel import ReloadHitlessWarningReason

traceHandle = Tracing.defaultTraceHandle()
t0 = traceHandle.trace0

# Errdisable cause groups for interfaces that are errdisabled due to the runtime
# conditions, as opposed to due to user config, such as port-group-shutdown.
# Interfaces errdisabled with bpduguard cause group are taken care of with the
# stpErrDisabled warning.
# When adding new cause groups here, also update test/EthIntfAsuPStoreTest.py.
warnErrdisableCauseGroups = [
   'acl',
   'arp-inspection',
   'dot1x',
   'license-enforce',
   'link-flap',
   'loopprotectguard',
   'mlagissu',
   'no-internal-vlan',
   'portchannelguard',
   'portsec',
   'speed-misconfigured',
   'stuck-queue',
   'tapagg',
   'tpid',
   'uplink-failure-detection',
   'xcvr-misconfigured',
   'xcvr-overheat',
   'xcvr-unsupported',
   ]

class EthIntfPStoreEventHandler( AsuPStore.PStoreEventHandler ):

   def __init__( self, errdisableCauses, intfStatuses ):
      self.errdisableCauses = errdisableCauses
      self.intfStatuses = intfStatuses
      super( EthIntfPStoreEventHandler, self ).__init__()

   def save( self, pStoreIO ):
      reloadDownErrdisableIntfs = set()
      for cause in self.errdisableCauses:
         for intf in self.errdisableCauses[ cause ].intfStatus:
            t0( 'Saving errdisable intf %s (%s).' % ( intf, cause ) )
            reloadDownErrdisableIntfs.add( intf )
      pStoreIO.set( 'reloadDownErrdisableIntfs', list( reloadDownErrdisableIntfs ) )

      intfMtus = {}
      for name, intfStatus in self.intfStatuses.intfStatus.iteritems():
         if name.startswith( 'Unconnected' ) or \
            intfStatus.operStatus != 'intfOperUp':
            continue
         intfMtus[ name ] = { 'l2Mtu' : intfStatus.l2Mtu, 'mtu' : intfStatus.mtu }
      pStoreIO.set( 'intfMtus', intfMtus )

   def getKeys( self ):
      return [ 'reloadDownErrdisableIntfs' ]

   def getSupportedKeys( self ):
      return [ 'reloadDownErrdisableIntfs' ]
   
   def hitlessReloadSupported( self ):
      warningList, blockingList = [], []
      for cause in self.errdisableCauses:
         if self.errdisableCauses[ cause ].causeGroup in warnErrdisableCauseGroups:
            if self.errdisableCauses[ cause ].intfStatus:
               t0( "Runtime errdisabled interfaces present for cause %s. "
                   "Adding warning." % cause )
               warningList.append( 
                  ReloadHitlessWarningReason( reason='runtimeErrdisabledIntfs' ) )
               break
      return warningList, blockingList 

def Plugin( ctx ):
   featureName = 'EthIntf'
   
   if ctx.opcode() == 'Restore':
      return
   if ctx.opcode() == 'GetSupportedKeys':
      ctx.registerAsuPStoreEventHandler( 
         featureName, EthIntfPStoreEventHandler( None, None ) )
      return

   entityManager = ctx.entityManager()
   mg = entityManager.mountGroup()
   
   errdisableCauses = mg.mount( 'interface/errdisable/cause', 'Tac::Dir', 'ri' )

   intfStatuses = mg.mount( 'interface/status/eth/intf',
                            'Interface::EthIntfStatusDir', 'r' )

   mg.close( blocking=True )
   ctx.registerAsuPStoreEventHandler( 
      featureName, EthIntfPStoreEventHandler( errdisableCauses, intfStatuses ) )
   
