#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, AsuPStore
import Tracing
from collections import defaultdict
from Ethernet import convertMacAddrCanonicalToDisplay

__defaultTraceHandle__ = Tracing.Handle( "AleVxlanMacAddrAsu" )
t0 = Tracing.trace0

VxlanEntryType = Tac.Type( 'Vxlan::EntryType' )

class AleVxlanMacAddrPStoreEventHandler( AsuPStore.PStoreEventHandler ):
   ''' VXLAN remote macs are saved to pstore.
   '''
   def __init__( self, vxlanHwStatusDir, bridgingStatus ):
      AsuPStore.PStoreEventHandler.__init__( self )
      self.vxlanHwStatusDir_ = vxlanHwStatusDir
      self.bridgingStatus_ = bridgingStatus

   def hitlessReloadSupported( self ):
      return ( [], [] )

   def save( self, pStoreIO ):
      t0( 'Saving vxlanHwStatusDir asu state' )
      pStoreIO.set( 'vxlanFdbStatus', self.getVxlanFdbStatusInfo() )

   def getSupportedKeys( self ):
      return [ 'vxlanFdbStatus' ]

   def getKeys( self ):
      return [ 'vxlanFdbStatus' ]

   def getVxlanFdbStatusInfo( self ):
      vxlanFdbStatus = defaultdict( list )
      learnedHost = self.vxlanHwStatusDir_.fdbStatus.learnedHost
      for lh in learnedHost.itervalues():
         if lh.entryType == VxlanEntryType.learnedMac:
            vlan = lh.macVlanPair.vlanId
            macaddr = lh.macVlanPair.macAddr
            try:
               fdbStatus = self.bridgingStatus_.fdbStatus[ vlan ]
               bridgingLh = fdbStatus.learnedHost[ macaddr ]
            except KeyError:
               continue
            vxlanFdbStatus[ vlan ].append(
               ( convertMacAddrCanonicalToDisplay( macaddr ),
                 str( lh.remoteVtepAddr ), bridgingLh.intf ) )
      return vxlanFdbStatus
   
def Plugin( ctx ):
   featureName = 'AleVxlanMacAddrSm'
   if ctx.opcode() == 'Store':
      entityManager = ctx.entityManager()
      mg = entityManager.mountGroup() 
      vxlanHwStatusDir = mg.mount( 'vxlan/hardware/status',
            'Vxlan::VxlanHwStatusDir', 'r' )
      bridgingStatus = mg.mount( 'bridging/status', 'Bridging::Status', 'r' )
      mg.close( blocking=True )
   else:
      vxlanHwStatusDir = None
      bridgingStatus = None

   ctx.registerAsuPStoreEventHandler(
         featureName,
         AleVxlanMacAddrPStoreEventHandler( vxlanHwStatusDir, bridgingStatus ) )
