#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""Assert: A module to export the helpful unittest.TestCase assertions.

As the unittest.TestCase assertions are contained within the TestCase
class, we need to create a singleton instance of a TestCase (using the
TestCase.id method as the method to initially run, to avoid the
regular side effect of starting the test case.

For information about the available assertion methods, see the Python
unittest documentation at:

   https://docs.python.org/2/library/unittest.html

In particular, the majority of the assertions are listed here:

   https://docs.python.org/2/library/unittest.html#assert-methods

These assertions can then be used in the following way:

---snip---

import Assert

# Start up test framework and populate the 'longString' variable

expected = '''We expect that there will be some long
text appearing in the longString output.'''

Assert.assertMultiLineEqual( expected, longString )
Assert.assertIn( 3, [ 1, 2, 3 ] )

---snip---

These methods are far more useful than Python's built in assert(),
because they print meaningful output (based on the assertion type)
without the programmer otherwise having to provide an assertion
message. This means that objects not immediately on the stack can
easily be tested and useful results appear in test failures.

"""

from __future__ import absolute_import, division, print_function
import six
import unittest

# Binds a test case to the module for access to the assertion methods
# (in Python, this is effectively a singleton, since module scope code
# is only executed once per program no matter how many times the
# module is imported)
__tc = unittest.TestCase( methodName='id' )

def assertAlmostEqual( *a, **kw ):
   __tc.assertAlmostEqual( *a, **kw )

def assertAlmostEquals( *a, **kw ):
   __tc.assertAlmostEquals( *a, **kw )

def assertDictContainsSubset( *a, **kw ):
   __tc.assertDictContainsSubset( *a, **kw )

def assertDictEqual( *a, **kw ):
   __tc.assertDictEqual( *a, **kw )

def assertDictEqualUnordered( a, b, maxDepth=15 ):
   def _sortDictLists( node, depth ):
      assertLessEqual( depth, maxDepth )
      if not node:
         return
      for k, v in node.items():
         if isinstance( v, dict ):
            _sortDictLists( v, depth+1 )
         elif isinstance( v, list ):
            node[ k ] = sorted( v )
            for listItem in node[ k ]:
               _sortDictLists( listItem, depth+1 )
   _sortDictLists( a, 0 )
   _sortDictLists( b, 0 )
   assertDictEqual( a, b )

def assertEqual( *a, **kw ):
   __tc.assertEqual( *a, **kw )

def assertEquals( *a, **kw ):
   __tc.assertEquals( *a, **kw )

def assertFalse( *a, **kw ):
   __tc.assertFalse( *a, **kw )

def assertGreater( *a, **kw ):
   __tc.assertGreater( *a, **kw )

def assertGreaterEqual( *a, **kw ):
   __tc.assertGreaterEqual( *a, **kw )

def assertIn( *a, **kw ):
   __tc.assertIn( *a, **kw )

def assertIs( *a, **kw ):
   __tc.assertIs( *a, **kw )

def assertIsInstance( *a, **kw ):
   __tc.assertIsInstance( *a, **kw )

def assertIsNone( *a, **kw ):
   __tc.assertIsNone( *a, **kw )

def assertIsNot( *a, **kw ):
   __tc.assertIsNot( *a, **kw )

def assertIsNotNone( *a, **kw ):
   __tc.assertIsNotNone( *a, **kw )

def assertCountEqual( *a, **kw ):
   six.assertCountEqual( __tc, *a, **kw )

# assertItemsEqual renamed to assertCountEqual in Python 3
assertItemsEqual = assertCountEqual

def assertLess( *a, **kw ):
   __tc.assertLess( *a, **kw )

def assertLessEqual( *a, **kw ):
   __tc.assertLessEqual( *a, **kw )

def assertListEqual( *a, **kw ):
   __tc.assertListEqual( *a, **kw )

def assertMultiLineEqual( *a, **kw ):
   __tc.assertMultiLineEqual( *a, **kw )

def assertNotAlmostEqual( *a, **kw ):
   __tc.assertNotAlmostEqual( *a, **kw )

def assertNotAlmostEquals( *a, **kw ):
   __tc.assertNotAlmostEquals( *a, **kw )

def assertNotEqual( *a, **kw ):
   __tc.assertNotEqual( *a, **kw )

def assertNotEquals( *a, **kw ):
   __tc.assertNotEquals( *a, **kw )

def assertNotIn( *a, **kw ):
   __tc.assertNotIn( *a, **kw )

def assertNotIsInstance( *a, **kw ):
   __tc.assertNotIsInstance( *a, **kw )

def assertNotRegexpMatches( *a, **kw ):
   __tc.assertNotRegexpMatches( *a, **kw )

def assertRaises( *a, **kw ):
   return __tc.assertRaises( *a, **kw )

def assertRaisesRegexp( *a, **kw ):
   return __tc.assertRaisesRegexp( *a, **kw )

def assertRegexpMatches( *a, **kw ):
   __tc.assertRegexpMatches( *a, **kw )

def assertSequenceEqual( *a, **kw ):
   __tc.assertSequenceEqual( *a, **kw )

def assertSetEqual( *a, **kw ):
   __tc.assertSetEqual( *a, **kw )

def assertTrue( *a, **kw ):
   __tc.assertTrue( *a, **kw )

def assertTupleEqual( *a, **kw ):
   __tc.assertTupleEqual( *a, **kw )

def assert_( *a, **kw ):
   __tc.assert_( *a, **kw )

def maxDiffIs( maxDiff ):
   __tc.maxDiff = maxDiff
