#!/bin/python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# connect uses the environmental variables to attach to a server:
# if ATPPortNumber is defined, then it contains a string of the following form:
#       host:port
# host is the ip address of a server or "localhost" or empty.  If host is
# empty, the : should be absent as well.  port is the port number on which
# a connection is to be made.  In this case, a TCP connection is made.
#
# if ATP_SOCKNAME is defined, then it contains a string for a unix domain
# sock and unix domain sockets are used.

# if both are defined, then ATPSockName is used.

# if neither is defined, then a unix domain socket is used as defined
# in ClientLibConstants

import os
import re
import threading
import time
import socket
import sys

# Tests in Arrow should import Marco before importing this module.
# We do not import Marco here as other EOS code may wish to use the TacMarco version
# of MarcoDebug.
import MarcoDebug as Debug

from Arrow.tableImpl import TableImpl
import Arrow.ClientLibConstants as clc
import Arrow.ArrowProtocolBindings as apb
import Arrow.messageHandler as mhi
from Arrow.messageHandler import HandshakeError, LostConnectionError
from baseRow import BaseRow
from Arrow.arMetadata.arClass import arClass
from Arrow.arMetadata.arTable import arTable
from Arrow.arMetadata.SystemDescription import SystemDescription
from Arrow.External.Publication import Publication
from Arrow.External.Subscription import Subscription
import Arrow.monotonic_time as mTime
import signal

threadLock = threading.Lock()
threadLocalData = threading.local()

mType = clc.ArrowMessageType()
dh = Debug.Handle( handleName="ArrowExternalClient" )

connectionUid = 1

# The time, in seconds, that a socket operation should block before timing out.
defaultSocketTimeoutTime = 0.1
def socketTimeoutTime():
   if ( not hasattr( threadLocalData, "socketTimeoutTime" )):
      threadLocalData.socketTimeoutTime = defaultSocketTimeoutTime
   return threadLocalData.socketTimeoutTime

# set to 5 minutes by default
defaultMsgTimeoutTime = 300.0
def msgTimeoutTime():
   if ( not hasattr( threadLocalData, "msgTimeoutTime" )):
      threadLocalData.msgTimeoutTime = defaultMsgTimeoutTime
   return threadLocalData.msgTimeoutTime

# set to 20 seconds by default
defaultInitTimeoutTime = 20.0
def initTimeoutTime():
   if ( not hasattr( threadLocalData, "initTimeoutTime" )):
      threadLocalData.initTimeoutTime = defaultInitTimeoutTime
   return threadLocalData.initTimeoutTime

def getNewConnectionId():
   global connectionUid
   threadLock.acquire()
   connId = connectionUid
   connectionUid += 1
   threadLock.release()
   return "%d-%d" % ( os.getpid(), connId )

class ClientConnect( object ):

   def __init__( self, kind, address ):
      self.sock_ = self.connect( kind, address )

   def socket( self ):
      return self.sock_

   def shutdown( self ):
      self.sock_.shutdown( socket.SHUT_RDWR )
      self.sock_.close()

   def connect( self, kind, address ):
      if not ( kind == socket.AF_INET or kind == socket.AF_UNIX ):
         raise ValueError, "invalid socket type: %s" % kind

      sock = socket.socket( kind, socket.SOCK_STREAM )
      sock.setsockopt( socket.SOL_SOCKET, socket.SO_REUSEADDR, 1 )

      dh.DEBUG8( "connecting: kind = %d address = %s" % ( kind, address ) )

      if kind == socket.AF_INET:
         l = address.split( ':', 2 )
         servAddr = (( "localhost", int( l[0] ) ) if len( l ) == 1
                     else ( l[0], int( l[1] ) ))
         sock.connect( servAddr )
      else:
         allTries = 50
         tries = 0
         startTime = time.time()
         while True:
            try:
               sock.connect( address )
               break
            except socket.error, e:
               tries += 1
               if ( tries > allTries or
                    ( time.time() > ( startTime + initTimeoutTime() ) )):
                  fmt = "failed to connect: kind = %d, address = %s, error = %s"
                  dh.DEBUG5( fmt % ( kind, address, e ) )
                  raise HandshakeError, fmt % ( kind, address, e )
               time.sleep( 0.25 )

      dh.DEBUG5( "connected: kind = %d address = %s" % ( kind, address ) )
      return sock

class Dispatcher( object ):

   def __init__( self, messageHandler, agentName, clientWorker ):
      self.mh_ = messageHandler
      self.dispatchTable = dict()
      self.pathDispatchTable = dict()
      self.arTHdlr = HandlerArTable( self )
      self.addTableHandler( clc.atTablePath, self.arTHdlr, tableId=clc.atTableId )
      self.arCHdlr = HandlerArClass( self )
      self.addTableHandler( clc.acTablePath, self.arCHdlr, tableId=clc.acTableId )
      self.sdHdlr = HandlerSystemDescription( self, clientWorker )
      self.addTableHandler( clc.sdTablePath, self.sdHdlr )
      # the following can only be filled in after we get arClass entries
      # for them
      self.sbHdlr = None
      self.puHdlr = None
      self.sysname_ = clientWorker.sysname_
      self.agentName_ = agentName

   def dbname( self ):
      return self.sysname_

   def agentName( self ):
      return self.agentName_

   # add a handler for a table
   # entries here are maintained by the arTable handler.  When a row
   # added to arTable, if there is a handler for the table, but either
   # the tableId or output path is not registered, then the unregistered
   # one is registered.
   # When an arTable row is deleted, the corresponding handler entries
   # are removed.
   def processTableIdForAddTableHandler( self, tablePath, handler, tableId ):
      if tableId != None:
         self.dispatchTable[tableId] = handler
      else:
         path = arTable()
         path.tablePathIs( tablePath )
         atRow = self.arTHdlr[path]
         if atRow != None:
            self.dispatchTable[atRow.oid()] = handler

   def addTableHandler( self, tablePath, handler, tableId=None ):
      if handler is None:
         dh.DEBUG1( "%s:%s attempting to add null handler for %s"
                    % ( self.dbname(), self.agentName(), tablePath ) )
         assert False
      try:
         tHdlr = self.pathDispatchTable[tablePath]
         if tHdlr is not None:
            if tHdlr.transient():
               tblDescript = "for %s%s" % ( tablePath, 
                                             ( "" if tableId is None
                                               else
                                               " [" + str( tableId ) + "]" ))
               msg = "%s:%s:%s replacing transient handler %s with correct handler"
               dh.DEBUG5( msg % ( self.dbname(), self.agentName(),
                                  self.sdHdlr.sysDescOid(), tblDescript ) )
               self.pathDispatchTable[tablePath] = handler
               self.processTableIdForAddTableHandler( tablePath, handler, tableId )
               tHdlr.transferOut( handler )
            else:
               # Don't we want to allow multiple handlers for the same table?
               # In this implementation, after there's one handler for a table,
               # trying to add any other handlers will be a no-op, except for
               # the error message.
               msg = ( "%s:%s:%s already have %%s non-transient handler for %s" %
                       ( self.dbname(), self.agentName(),
                         self.sdHdlr.sysDescOid(), tablePath ))
               if ( not (( handler == tHdlr ) or
                         ( handler.cw_ == tHdlr.cw_ and
                           handler.request_ == tHdlr.request_ ))):
                  msg2 = msg % "different"
                  # assert tHdlr.transient(), msg2
                  dh.DEBUG1( msg2 )
                  dh.DEBUG1( "Ignoring new, conflicting, handler" )
               else:
                  msg2 = msg % "identical"
                  dh.DEBUG1( msg2 )
                  dh.DEBUG1( "Ignoring new, redundant, handler" )
         else:
            dh.DEBUG1( "%s:%s has path '%s' with a null handler.  "
                       "Implementing correct handler" %
                       ( self.dbname(), self.agentName(), tablePath ) )
         return
      except KeyError:
         pass
      self.pathDispatchTable[tablePath] = handler
      self.processTableIdForAddTableHandler( tablePath, handler, tableId )

   # removeTableHandler is very forgiving as the handler may be
   # removed as a consequence of any one of several causes:
   #  Publication.rowDel
   #  arTable.rowDel
   def removeTableHandler( self, tablePath, tableId=None ):
      dh.DEBUG5( "%s removing table handler for tablePath: %s"
                 % ( self.sysname_, tablePath ) )
      if tableId is not None:
         try:
            del self.dispatchTable[tableId]
            del self.pathDispatchTable[tablePath]
         except KeyError:
            pass
      else:
         art = arTable()
         art.tablePathIs( tablePath )
         arr = self.arTHdlr[art]
         if not arr is None:
            del self.dispatchTable[arr.oid()]
            del self.pathDispatchTable[tablePath]
         else:
            # we have a tablePath but no tableId.  Try to
            # delete by finding the handler in the pathDispatchTable
            # set of values
            try:
               hdlr = self.pathDispatchTable[tablePath]
               tbd = []
               for i in self.dispatchTable.items():
                  if i[1] == hdlr:
                     tbd.append(i[0])
               for i in tbd:
                  del self.dispatchTable[i]
               del self.pathDispatchTable[tablePath]
            except KeyError:
               pass        # don't care since it is gone
            except Exception, msg:
               dh.DEBUG1( "failed attempt to remove handler for %s with: %s"
                          % ( tablePath, msg ) )

   def handlerDefined( self, tableId ):
      return tableId in self.dispatchTable

   def bindHandlerToTableId( self, tableId, tablePath ):
      if self.handlerDefined( tableId ):
         return         # maybe we should say something here
      hdlr = self.pathDispatchTable.get( tablePath )
      if hdlr is None:
         return         # maybe we should say something here, as well
      self.dispatchTable[tableId] = hdlr


   # When a zero tableId is delivered, it may be because the
   # table can not yet been defined.
   #
   # When this occurs, if pathname is not null (not == "") then
   # it is looked up and if, found, the registered handler is invoked
   def analyzeZeroTableId( self, msgType, pathname, row1, row2 ):
      if pathname is None or pathname == "":
         return True
      hdlr = self.pathDispatchTable.get( pathname )
      if hdlr is None:
         return False
                  
      if msgType == mType.rowIs:
         getattr( hdlr, "onRow" )( row1, pathname )
      elif msgType == mType.rowDel:
         getattr( hdlr, "onRowDel" )( row1, pathname )
      elif msgType == mType.rowDelRange:
         getattr( hdlr, "onRowDelRange" )( row1, row2, pathname )
      else:
         assert False, "Unknown protocol message: %d" % msgType

      return True

   # next -- handle all incoming messages from the server.
   #@return -- true, successful input handling
   #           false, not successful input handling
   def next( self ):
      ( msgType, tableId, pathname, row1, row2 ) = self.mh_.readAndDecodeMsg()
      if msgType == 0:
         dh.DEBUG5( "connection dropped" )
         if self.mh_.running_:
            raise LostConnectionError, "connection dropped"
         else:
            return 0

      dh.DEBUG9( "%s:%s received msgType = %d tableId = %d path = %s"
                 % ( self.sysname_, self.agentName_, msgType, tableId, pathname ) )
      
      if tableId == 0:
         return self.analyzeZeroTableId( msgType, pathname, row1, row2 )
      
      try:
         handler = self.dispatchTable[tableId]
      
         if msgType == mType.rowIs:
            handler.onRow( row1, pathname )
         elif msgType == mType.rowDel:
            handler.onDelete( row1, pathname )
         elif msgType == mType.rowDelRange:
            handler.onDeleteRange( row1, row2, pathname )
         else:
            assert False, "Unknown protocol message: %d" % msgType
            
      except KeyError:
         return self.analyzeZeroTableId( msgType, pathname, row1, row2 )
      return True
      

   def fetchArTableId( self, tablePath ):
      art = arTable()
      art.tablePathIs( tablePath )
      arc = self.arTHdlr[art]
      if arc is None:
         return 0
      else:
         return arc.oid()


class HandlerBase( object ):
   def __init__( self, **kw ):
      if "rowType" in kw:
         rowType = kw["rowType"]
      elif "name" in kw and "arCHdlr" in kw:
         ac = arClass()
         name = kw["name"]
         ac.typeNameIs ( name )
         ac = ( kw["arCHdlr"] )[ac]
         if not ac:
            raise KeyError, "no class row for %s" % name
         rowType = ac.oid()
      else:
         raise NameError, "Missing rowType or name, and arChdlr"

      self.rowType = rowType
      # don't unilaterally generate oids -- do it on a row by row basis
      self.table_ = TableImpl( rowType, genOid=False )

   # implement a[key] returns value at key else None
   def __getitem__( self, key ):
      return self.table_.row( key )

   def transient( self ):
      return False

   # implement a[key] = value
   def __setitem__( self, key, row ):
      assert row.rowType() == 0 or row.rowType() == self.rowType, \
         "Attempted to set bad row type"
      row.rowTypeIs( self.rowType )
      self.table_.rowIs( row )

   def __delitem__( self, key ):
      self.table_.rowDel( key )

   def rows( self ):
      return self.table_.rows()

   def onDeleteRange( self, start, end, pathName ):
      # we expect that there will never be very many rows
      # in the table and that deletions will occur only after the
      # desired action is complete.  So, we will use a full table scan
      # to delete the appropriate rows from the table.
      # We deliver the set of keys whose rows are to be deleted.
      dh.DEBUG9( "onDeleteRange: start = %s, end = %s"
                 % ( start.toString() if start != None else "None",
                     end.toString() if end != None else "None" ) )
      toBeDeleted = []
      for k in self.table_.iterkeys():
         # FIXME: have to check for null here!!!
         if (( not start or start < k ) and 
             ( not end or k < end )):
            toBeDeleted.append( k )
      return toBeDeleted


# HandlerTransient is instantiated when a wrapped row arrives for which we have no
# handler.  When finally a handler is created for the class, we transfer ownership
# to the provided handler and deliver the rows to the new handler.
class HandlerTransient( HandlerBase ):
   def __init__( self, pathName ):
      HandlerBase.__init__( self, rowType = 0 )
      self.pathName_ = pathName
      self.tStore_ = []

   def transient( self ):
      return True

   def onRow( self, row, pathName=None ):
      dh.DEBUG9( "received transient row for %s" % self.pathName_ )
      self.tStore_.append( ( row, pathName, True ) )

   def onDelete( self, row, pathName ):
      dh.DEBUG9( "received transient rowDel for %s" % self.pathName_ )
      # No coalescing -- we don't have the type of the row, so can't compare
      # keys.
      self.tStore_.append( ( row, pathName, False ) )
      return True

   def transferOut( self, handler ):
      while len( self.tStore_ ) > 0:
         row, path, isRow = self.tStore_[ 0 ]
         if isRow:
            handler.onRow( row, path )
         else:
            handler.onDelete( row, path )
         self.tStore_ = self.tStore_[ 1: ]

class HandlerArClass( HandlerBase ):
   def __init__( self, dsp ):
      self.dsp_ = dsp
      super( HandlerArClass, self ).__init__( rowType = clc.acClassId )

   def onRow( self, row, pathName=None ):
      aRow = arClass( row=row )
      self[aRow] = aRow
      dh.DEBUG8( "%s:%s installing arClass for %s [%s]"
                 % ( self.dsp_.dbname(), self.dsp_.agentName(), aRow.typeName(),
                     aRow.oid() ) )
      return True

   def onDelete( self, row, pathName ):
      del self[row]
      return True

class HandlerArTable( HandlerBase ):
   def __init__( self, dsp ):
      self.dsp_ = dsp
      super( HandlerArTable, self ).__init__( rowType = clc.atClassId )

   def onRow( self, row, pathName=None ):
      aRow = arTable( row=row )
      dh.DEBUG8( "rcvd table row (%s,%s): %s"
                 % ( self.dsp_.dbname(), self.dsp_.agentName(), aRow.toString() ) )
      try:
         oldARow = self[aRow]
         if oldARow is not None and oldARow.oid() != aRow.oid():
            self.onDelete( row, aRow.tablePath() )
      except KeyError:
         pass
      except Exception:
         pass

      self[aRow] = aRow

      dh.DEBUG9( "arTable: oid = %d, tablePath = %s"
                 % ( aRow.oid(), aRow.tablePath() ) )

      if not self.dsp_.handlerDefined( aRow.oid() ):
         hdlr = self.dsp_.pathDispatchTable.get( aRow.tablePath() )
         if hdlr is None:
            dh.DEBUG9( "arTable no handler for tableId %d path '%s'"
                       "  Using transient handler"
                       % ( aRow.oid(), aRow.tablePath() ) )
            self.dsp_.addTableHandler( aRow.tablePath(),
                                       HandlerTransient( aRow.tablePath() ),
                                       tableId=aRow.oid() )
            return True
         self.dsp_.bindHandlerToTableId( aRow.oid(), aRow.tablePath() )
      return True

   def onDelete( self, row, pathName ):
      art = arTable( row=row )
      dh.DEBUG9( " delete artable row: %s" % str( art ) )
      # invoke removeTableHandler before removing the arTable row
      self.dsp_.removeTableHandler( art.tablePath(), tableId = art.oid() )

      del self[row]
      return True

class HandlerSystemDescription( HandlerBase ):
   def __init__( self, dsp, cw ):
      self.dsp_ = dsp
      self.cw_ = cw
      super( HandlerSystemDescription, self ).__init__( rowType = clc.sdClassId )

   def sysDescOid( self ):
      return self.cw_.sysDescOid_

   def getByOid(self, oid ):
      #search for row with oid
      for row in self.table_.values():
         if row.oid() == oid:
            return row
      else:
         return None
   
   def onRow( self, iRow, pathName=None ):
      if iRow is None or iRow == "":
         return
      row = SystemDescription( row=iRow )
      self[row] = row
      dh.DEBUG9( "SysDesc %d out of %s" % ( row.oid(), self.rows() ))

      # if this is an echo of our SystemDescription, simply accept it
      dh.DEBUG5( "%s,%s,%s rcvd SysDesc: oid = %d, connectId = %s for %s, %s"
                 % ( self.cw_.sysname_, self.cw_.agentName_, self.cw_.connectionId(),
                     row.oid(), row.connection_id(), row.sysName(),
                     row.agentName() ) )
      if (( self.cw_.sysname_ == row.sysName() ) and
          ( self.cw_.agentName_ == row.agentName() ) and
          ( row.connection_id() == self.cw_.connectionId() ) and
          ( row.oid() != 0 ) ):
         dh.DEBUG5( " setting sysdesc oid to %d for %s::%s:%s"
                    % ( row.oid(), self.cw_.sysname_, self.cw_.agentName_,
                        self.cw_.connectionId() ) )
         self.cw_.setSysDescOid( row.oid() )
         return True

      # other systemdescriptions.  We don't care.
      return True

   def onDelete( self, row, pathName ):
      row = SystemDescription( row=row )
      dh.DEBUG8( "SysDesc del: sysname = %s, agentname = %s, connectionId = %s"
                 % ( row.sysName(), row.agentName(), row.connection_id() ) )
      if row.connection_id() == self.cw_.connectionId():
         self.cw_.connected = False
      del self[row]
      return True

   def onDeleteRange( self, start, end, pathName ):
      sdStart = SystemDescription( row=start )
      sdEnd = SystemDescription( row=end )
      toBeDeleted = HandlerBase.onDeleteRange( self, sdStart, sdEnd, pathName )
      for k in toBeDeleted:
         self.onDelete( k.buf_, pathName )
      return True

class HandlerSubscription( HandlerBase ):
   def __init__( self, arCHdlr, cw ):
      dh.DEBUG5( "[%s:%s:%s] Subscription handler"
                 % ( cw.dbname(), cw.agentName(), cw.sysDescOid_ ) )
      self.cw_ = cw
      super( HandlerSubscription, self ).__init__(
         name="Arrow::External::Subscription",
         arCHdlr=arCHdlr )

   def getByOid(self, oid ):
      #search for row with oid
      for row in self.table_.values():
         if row.oid() == oid:
            return row
      else:
         return None
      
   def onRow( self, iRow, pathName=None ):
      row = Subscription( row=iRow )
      dh.DEBUG8( "received Subscription oid = %d" % row.oid() )
      self[row] = row

      dh.DEBUG8( "%s:%s:%s rcvd Subscription: sys_desc = %d, outputPath = %s"
                 % ( self.cw_.sysname_, self.cw_.agentName_, self.cw_.sysDescOid_,
                     row.system_description(), row.outputPath() ) )
      return True

   def onDelete( self, row, pathName ):
      row = Subscription( row=row )
      # the following is here to permit forward progress.  It should
      # be removed when the server properly delivers a deleted Publication
      # in response to a deleted subscription
      del self[row]
      return True

   def onDeleteRange( self, start, end, pathName ):
      sStart = Subscription( row=start )
      sEnd = Subscription( row=end )
      toBeDeleted = HandlerBase.onDeleteRange( self, sStart, sEnd, pathName )
      dh.DEBUG9( " Ex::Sub len %d to be deleted (from [%s:%s])" %
                 ( len( toBeDeleted ), 
                   sStart.outputPath() if ( sStart ) else ".", 
                   sEnd.outputPath() if ( sEnd ) else "." ) )
      for k in toBeDeleted:
         dh.DEBUG9( "Sub del: %s" % k.outputPath() )
         self.onDelete( k.buf_, pathName )
      return True

class HandlerPublication( HandlerBase ):
   def __init__( self, arCHdlr, cw ):
      dh.DEBUG5( "[%s:%s:%s] Publication handler"
                 % ( cw.dbname(), cw.agentName(), cw.sysDescOid_ ) )
      super( HandlerPublication, self ).__init__(
         name="Arrow::External::Publication",
         arCHdlr=arCHdlr )
      self.cw_ = cw
      self.sidToOutputPath = {}

   def signalFailureInPub( self, msg, sb, pub ):
      # subscription failed.
      self.cw_.dsp_.sbHdlr.onDelete( sb, sb.outputPath() )
      self.cw_.signalFailure( msg, pub.error() )
      return True

   def onRow( self, iRow, pathName=None ):
      pub = Publication( row=iRow )
      dh.DEBUG9( "%s:%s:%s rcvd Publication: subId = %d, tableId = %d, status = %d"
                 % ( self.cw_.sysname_, self.cw_.agentName_, self.cw_.sysDescOid_,
                     pub.subscription_id(), pub.tableId(), pub.status() ) )
      dh.DEBUG9( "                  err = '%s'" % pub.error() )

      # validate that we have a corresponding subscription.
      sb = self.cw_.dsp_.sbHdlr.getByOid( pub.subscription_id() )

      if sb is None:
         # we can't handle this, as we have no way to understand this
         # publication row.
         # publish a log message and give up (we are in the wrong thread to do
         # anything else)
         dh.DEBUG1( "%s:%s:%s no subscription %d for received publication with "
                    "status %s and message %s"
                    % ( self.cw_.sysname_, self.cw_.agentName_, self.cw_.sysDescOid_,
                        pub.subscription_id(),
                        "success" if pub.status() else "failure",
                        pub.error() ) )
         # Ignore this msg, and move on to the next
         return True
      outputPath = sb.outputPath()
      if ( self.cw_.sysDescOid_ and
           ( not sb.system_description() == self.cw_.sysDescOid_ )):
         dh.DEBUG5( "%s:%s ignoring Publication for %s with sysdesc oid=%d, "
                    "expecting %d"
                    % ( self.cw_.sysname_, self.cw_.agentName_, outputPath,
                        sb.system_description(), self.cw_.sysDescOid_ ) )
         sdrow = self.cw_.dsp_.sdHdlr.getByOid( sb.system_description )
         if( sdrow ):
            dh.DEBUG9( "  incoming sysdesc: %s:%s" % ( sdrow.sysName(),
                                                       sdrow.agentName() ) )
         else:
            dh.DEBUG5( "  no matching incoming sysdesc" )
            return True
         # sdrow must exist
         if ( self.cw_.sysname_ != sdrow.sysName() ):
            # We know this is totally irrelevant to us. Return and process
            # next msg
            dh.DEBUG8( " sysname in row (%s) doesn't match our sysname (%s)"
                       % ( sdrow.sysName(), self.cw_.sysname_ ) )
            return True
         if ( self.cw_.agentName_ != sdrow.agentName() ):
            # We know this is totally irrelevant to us. Return and process
            # next msg
            dh.DEBUG8( " agentName in row (%s) doesn't match our name (%s)"
                       % ( sdrow.agentName(), self.cw_.agentName_ ) )
            return True
         dh.DEBUG1( " For %s, in %s:%s:%s, sysdesc: %s, matches sysname and agent"
                    % ( outputPath, self.cw_.sysname_, self.cw_.agentName_,
                        self.cw_.sysDescOid_, sdrow ) )
         assert False, " spurious sysdesc match"

      self.sidToOutputPath[pub.subscription_id()] = outputPath
      msg = self.cw_.runningQueries.get( outputPath )
      if msg is None:
         dh.DEBUG1( "%s:%s:%s no request message for output path %s"
                    % ( self.cw_.sysname_, self.cw_.agentName_, self.cw_.sysDescOid_,
                        outputPath ) )
         # Not one of ours
         return True

      # If we have received this before, check if the status
      # has changed from ok to not ok.
      oldPub = self[pub]
      if oldPub is not None:
         # new version of existing query
         if not pub.status():
            return self.signalFailureInPub( msg, sb, pub )
         else:
            return True

      dh.DEBUG9( "%s:%s:%s entered publication for %s"
                 % ( self.cw_.sysname_, self.cw_.agentName_, self.cw_.sysDescOid_,
                     outputPath ) )
      self[pub] = pub
      outputPath = sb.outputPath()
      msg = self.cw_.runningQueries[outputPath]
      if not pub.status():
         return self.signalFailureInPub( msg, sb, pub )

      # either assert that msg.modifier is None, or else return
      # (with error message) if msg.modifier is not None...
      dh.DEBUG5( "%s:%s:%s wrapping handler for path %s tableId = %d"
                 % ( self.cw_.sysname_, self.cw_.agentName_, self.cw_.sysDescOid_,
                     outputPath, pub.tableId() ) )
      iqWH = IQHandlerWrapper( msg, self.cw_ )
      self.cw_.dsp_.addTableHandler( outputPath, iqWH, tableId=pub.tableId() )
      if msg.nature == clc.qmbWriter:
         # have a writer.  create handle and invoke callback
         tablePath = msg.parameters.get( "writer._output_" )
         wtr = Writer( pub.tableId(), tablePath, pub.typeId(), self.cw_ )
         msg.modifier = wtr
         msg.callbacks.onReady( wtr )

      return True

   def onDelete( self, row, pathName ):
      pu = Publication( row=row )
      outputPath = self.sidToOutputPath[pu.subscription_id()]
      assert outputPath is not None
      self.cw_.dsp_.removeTableHandler( outputPath, tableId = pu.tableId() )
      self.cw_.tearDownSubscriptionHandling( outputPath )
      del self[pu]
      return True

   def onDeleteRange( self, start, end, pathName ):
      pStart = Publication( row=start )
      pEnd = Publication( row=end )
      toBeDeleted = HandlerBase.onDeleteRange( self, pStart, pEnd, pathName )

      dh.DEBUG9( " Ex::Pub onDeleteRange [%s:%s] (%d to del)" % 
                 ( "." if not pStart else pStart.publication_id(), 
                   "." if not pEnd else pEnd.publication_id(), 
                   len( toBeDeleted ) ) )
      for k in toBeDeleted:
         dh.DEBUG9( "Pub del: %d" % k.subscription_id() )
         self.onDelete( k.buf_, pathName )
      return True
#
# frontend/backend communication objects
#
class QMsgBase( object ):
   def __init__( self, query, parameters, oneShot, signalChannel, callbacks,
                 nature=clc.qmbQuery, toTime=msgTimeoutTime() ):
      monoTime = mTime.monotonic_time()
      if toTime is None:
         timeoutTime = sys.float_info.max
      else:
         timeoutTime = monoTime + toTime
      self.query = query
      self.parameters = parameters
      self.signalChannel = signalChannel
      self.nature = nature
      self.oneShot = oneShot
      self.timeout = timeoutTime
      self.success = True
      self.errorMsg = None
      self.callbacks = callbacks
      dh.DEBUG9( "queued up msg [%s] %s with params %s at %f with timeout %s"
                 % ( nature, query, parameters, monoTime, str( timeoutTime ) ) )
   def queryText( self ):
      return self.query
   def extractTypeName( self, query, kind ):
      qualifiedName = r"((?:\w+\s*::)+)\s*(\w+)"
      ms = r"^.*%s\s+%s(?:\s+NAMED\s+%s\s*)?;\s*$" % ( kind,
                                                       qualifiedName,
                                                       qualifiedName )
      m = re.match( ms, query, re.IGNORECASE | re.DOTALL )
      if m is None:
         raise SyntaxError, "query missing proper %s clause: %s" % ( kind, query )
      # remove the trailing "::"
      nestedNamespaces = m.group( 1 )[:-2]
      namespaces = [ name.strip() for name in nestedNamespaces.split( "::" ) ]
      prefix = "::".join( namespaces )
      name = prefix + "::" + m.group( 2 )
      return name
   def parametersIs( self, parameters ):
      self.parameters = parameters

class IqMsg( QMsgBase ):
   def __init__( self, query, parameters, callbacks,
                 oneShot, signalChannel ):
      super( IqMsg, self ).__init__( query, parameters,
                                     oneShot, signalChannel, callbacks,
                                     nature=clc.qmbQuery )
   def typeName( self ):
      return self.extractTypeName( self.queryText(), "INTO" )

class InqMsg( QMsgBase ):
   def __init__( self, qualifiedQueryName, typename,
                 parameters, callbacks,
                 oneShot, signalChannel ):
      # skip adding the ";" at end of the qualifiedQueryName
      super( InqMsg, self ).__init__( qualifiedQueryName, parameters,
                                      oneShot, signalChannel, callbacks,
                                      nature=clc.qmbQuery )
      self.typename = typename
   
   def typeName( self ):
      return self.typename

class IwMsg( QMsgBase ):
   def __init__( self, name, parameters, callbacks, signalChannel,
                 oneShot=False ):
      super( IwMsg, self ).__init__( "SELECT * INTO %s;" % name,
                                     parameters, oneShot, signalChannel,
                                     callbacks, 
                                     nature=clc.qmbWriter )
      self.modifier = None
   def typeName( self ):
      return self.extractTypeName( self.queryText(), "INTO" )

class IrMsg( QMsgBase ):
   def __init__( self, name, parameters, callbacks, signalChannel,
                 oneShot=False ):
      super( IrMsg, self ).__init__( "SELECT * FROM %s;" % name,
                                     parameters, oneShot, signalChannel,
                                     callbacks, 
                                     nature=clc.qmbReader )
   def typeName( self ):
      return self.extractTypeName( self.queryText(), "FROM" )

class CqMsg( QMsgBase ):
   def __init__( self, query, parameters, signalChannel ):
      super( CqMsg, self ).__init__( query, parameters,
                                     False, signalChannel, None )

class CancelMsg( QMsgBase ):
   def __init__( self, signalChannel ):
      super( CancelMsg, self ).__init__( None, None, False,
                                         signalChannel, None )

class WrMsg( QMsgBase ):
   def __init__( self, tableId, row ):
      QMsgBase.__init__( self, None, None, False, None, None )
      self.tableId = tableId
      self.row = row

class Writer( object ):
   def __init__( self, tableId, tablePath, typeId, cw ):
      self.tableId_ = tableId
      self.tablePath_ = tablePath
      self.typeId_ = typeId 
      self.cw_ = cw
      self.table_ = TableImpl( typeId, genOid=True )

   def rowIs( self, row ):
      if not row:
         dh.DEBUG1( "Null row passed" )
         raise ValueError( "Null row passed" )

      if row.rowType() != self.typeId_:
         dh.DEBUG1( "expected type: %d; got: %d" % ( self.typeId_, row.rowType() ) )
         raise TypeError( "expected type: %d; got: %d" %
                          ( self.typeId_, row.rowType() ) )
      r = self.table_.rowIs( row )
      cl = self.cw_.clientLib_
      cl.appendRequests( [ WrMsg( self.tableId_, r ) ] )
      return r

   # TODO: Should support def rowDel( row ).... also need to define a 
   #       msg/request type (analagous to WrMsg .. )
   
   def close( self ):
      self.cw_.tearDownSubscriptionHandling( self.tablePath_ )

   def table( self ):
      return self.table_

#
# ExternalClientLib front end
#
class ExternalClientLib( object ):

   # Even though the below class won't be useful in real DUTs because of the
   # CliServer non-main-thread signal handler issue, keep it around because
   # it'll still be useful in namespace DUTs where we don't run CliServer
   class sigintHandler( object ):
      def __init__( self ):
         self.interrupted_ = None
         self.oldHandler_ = None

      def __enter__( self ):
         # set up signal handler to catch Ctrl-C
         self.interrupted_ = False
         try:
            self.oldHandler_ = signal.signal( signal.SIGINT, self.handleSignal  )
         except: # pylint: disable-msg=W0702
            # This can happen in real DUTs, where this arrow code runs in the
            # non-main thread of CliServer and python doesn't allow signals
            # handlers to be registered in the non-main threads.
            pass
         return self

      def __exit__( self, exc_type, exc_value, _traceback ):
         if self.oldHandler_ is not None:
            try:
               signal.signal( signal.SIGINT, self.oldHandler_ )
            except: # pylint: disable-msg=W0702
               pass

      def handleSignal( self, signum, stack ):
         self.interrupted_ = True

      def interrupted( self ):
         return self.interrupted_

   def __init__( self, sysname, agentName,
                 msgTimeout=msgTimeoutTime(), initTimeout=initTimeoutTime(),
                 clientInterruptedFn=None ):
      threadLocalData.msgTimeoutTime = msgTimeout
      threadLocalData.initTimeoutTime = initTimeout
      self.clientInterruptedFn = clientInterruptedFn
      self.requests_ = []
      self.running_ = True
      self.runner_ = ClientWorker( self, sysname, agentName )
      self.runner_.start()

   def dbname( self ):
      return self.runner_.sysname_

   def agentName( self ):
      return self.runner_.agentName_

   def checkForClientErrors( self ):
      if self.runner_.err() is not None:
         dh.DEBUG9( "raising client error: %s" % self.runner_.err() )
         err = self.runner_.err()
         self.runner_.errIs( None )
         self.runner_.terminate()
         self.running_ = False
         raise err # pylint: disable-msg=E0702
      return True

   def clientInterrupted( self ):
      if self.clientInterruptedFn:
         return self.clientInterruptedFn()
      else:
         return False

   def handleSync( self, sync, signalChannel ):
      if not self.running_:
         dh.DEBUG1( "ExternalClientLib is no longer running" )
         signalChannel.set()
      if sync:
         with self.sigintHandler() as sh:
            while self.checkForClientErrors() and not signalChannel.wait( 0.1 ):
               if sh.interrupted() or self.clientInterrupted():
                  dh.DEBUG1( "received SIGINT, exiting..." )
                  raise KeyboardInterrupt
      else:
         return signalChannel

   def handleSignalFailure( self, msg, sync ):
      self.runner_.signalFailure( msg, "Connection already closed" )
      if sync:
         return
      else:
         return msg.signalChannel

   def prepareQueryParameters( self, parameters, callbacks, oneShot ):
      assert hasattr( callbacks, "onRow" )
      if( not oneShot ):
         assert hasattr( callbacks, "onDelete" )
      else:
         p2 = dict( parameters )
         for k, v in p2.iteritems():
            if clc.qpOutputSuffix in k:
               if self.runner_:
                  p2[k] = self.runner_.uniquifyOutputPath( v )
         parameters = p2
      return parameters

   def invokeNamedQuery( self, qualifiedQueryName, typename,
                         parameters, callbacks, sync=True, oneShot=True ):
      dh.DEBUG5( "%s:%s:%s invokeNamedQuery called with name = %s parameters = %s"
                 % ( self.dbname(), self.agentName(), self.runner_.sysDescOid_,
                     qualifiedQueryName, parameters ) )
      signalChannel = threading.Event()
      msg = InqMsg( qualifiedQueryName, typename, None,
                    callbacks, oneShot, signalChannel )
      if not self.runner_.running_:
         return self.handleSignalFailure( msg, sync )
      self.checkForClientErrors()
      parameters = self.prepareQueryParameters( parameters, callbacks, oneShot )
      msg.parametersIs( parameters )
      self.requests_.append( msg )
      return self.handleSync( sync, signalChannel )

   # invoke the indicated query in the ClientWorker
   # callbacks should implement:
   #   onRow( row ) (required)
   #   onDelete( row )  [ only if oneShot=False ]
   #   onError( msg ) [optional]
   # row is a derivative of BaseRow.  Row must be of the correct type
   # and not a null row
   # where msg is a instance which has a boolean attribute
   #   success and a string attribute errorMsg
   def invokeQuery( self, query, parameters, callbacks,
                    sync=True, oneShot=True ):
      dh.DEBUG5( "%s:%s:%s invokeQuery called with query = %s parameters = %s"
                 % ( self.dbname(), self.agentName(), self.runner_.sysDescOid_,
                     query, parameters ) )
      signalChannel = threading.Event()
      msg = IqMsg( query, None, callbacks, oneShot, signalChannel )
      if not self.runner_.running_:
         return self.handleSignalFailure( msg, sync )
      self.checkForClientErrors()
      parameters = self.prepareQueryParameters( parameters, callbacks, oneShot )
      msg.parametersIs( parameters )
      self.requests_.append( msg )
      return self.handleSync( sync, signalChannel )

    # name indicates the type of the table to read
    # tablePath is the unique path of the table to be read
    # callbacks should implement:
    #   onRow( row ) (required)
    #   onDelete( row ) (required)
    #   onError( msg ) [optional]
    # row is a derivative of BaseRow.  Row must be of the correct type
    # and not a null row
    # where msg is a instance which has a boolean attribute
    #   success and errorMsg
   def invokeReader( self, name, tablePath, callbacks, sync=False ):
      dh.DEBUG5( "%s:%s:%s invokeReader called with name = %s parameters = %s"
                 % ( self.dbname(), self.agentName(), self.runner_.sysDescOid_,
                     name, tablePath ) )
      assert hasattr( callbacks, "onRow" )
      assert hasattr( callbacks, "onDelete" )
      signalChannel = threading.Event()
      params = { "reader._output_" : tablePath, }
      msg = IrMsg( name, params, callbacks, signalChannel )
      if not self.runner_.running_:
         return self.handleSignalFailure( msg, sync )
      self.checkForClientErrors()
      self.requests_.append( msg )
      return self.handleSync( sync, signalChannel )

    # name indicate the type of the table to write.
    # tablePath is the unique path of the table to be written.
    # callbacks implement:
    #   onReady( handle )
    #   onError( msg )
    # where handle is a write accessor with services:
    #   rowIs( row )
    #   close()
    # [TODO: should support rowDel( row ) in the future ]
    # row is a derivative of BaseRow.  Row must be of the correct type
    # and not a null row
    # where msg is a instance which has a boolean attribute
    #   success and errorMsg
   def invokeWriter( self, name, tablePath, callbacks, sync=False ):
      dh.DEBUG5( "%s:%s:%s invokeWriter called with name = %s parameters = %s"
                 % ( self.dbname(), self.agentName(), self.runner_.sysDescOid_,
                     name, tablePath ) )
      assert hasattr( callbacks, "onReady" )
      signalChannel = threading.Event()
      params = { "writer._output_" : tablePath, }
      msg = IwMsg( name, params, callbacks, signalChannel )
      if not self.runner_.running_:
         return self.handleSignalFailure( msg, sync )
      self.checkForClientErrors()
      self.requests_.append( msg )
      return self.handleSync( sync, signalChannel )

   # cancel the indicated query from the Clientworker
   def cancelQuery( self, query, parameters, sync=True ):
      signalChannel = threading.Event()
      msg = CqMsg( query, parameters, signalChannel )
      if not self.runner_.running_:
         self.running_ = False
         return self.handleSignalFailure( msg, sync )
      self.checkForClientErrors()
      self.requests_.append( msg )
      return self.handleSync( sync, signalChannel )

   # terminate all client processing and terminate the ClientWorker
   def cancelAll( self, sync=True ):
      signalChannel = threading.Event()
      msg = CancelMsg( signalChannel )
      if not self.runner_.running_:
         return self.handleSignalFailure( msg, sync )
      self.checkForClientErrors()
      self.requests_.append( msg )
      self.handleSync( sync, signalChannel )
      self.runner_.terminate()
      self.running_ = False

   def appendRequests( self, requests ):
      self.requests_ += requests

   def getRequests( self ):
      return list( self.requests_ )

   def removeRequests( self, requestsToRemove ):
      for r in requestsToRemove:
         # remove works from the start of the list, so if requests_ and
         # requestsToRemove share a common prefix this will be O(n)
         self.requests_.remove( r )

   def typeId( self, typeName ):
      if( self.runner_ ):
         return self.runner_.typeId( typeName )
      else:
         return 0

   def registered( self, outputPath ):
      return( self.runner_ and self.runner_.registered( outputPath ))

   def running( self ):
      return self.running_

class ClientWorker( threading.Thread ):
   def __init__( self, cl, sysname, agentName ):
      threading.Thread.__init__( self )
      self.clientLib_ = cl
      self.agentName_ = agentName
      self.sysname_ = sysname
      self.cc_ = None
      self.mh_ = None
      self.dsp_ = None
      self.runningQueries = dict()
      self.sbwPath_ = None
      self.sysDescOid_ = None
      self.connectionId_ = None
      self.connected = False
      self.running_ = True
      self.err_ = None

   def running( self ):
      return self.running_

   def dbname( self ):
      return self.sysname_

   def agentName( self ):
      return self.agentName_

   def run( self ):
      # FIXME this initialization must take nature of connection into account
      #       and deal with both TCP as well as Unix domain sockets
      self.connectionId_ = getNewConnectionId()
      try:
         self.cc_ = ClientConnect( socket.AF_UNIX,
                                   apb.getArrowDomainSocket( self.dbname() ) )

         self.mh_ = mhi.MessageHandler( self.cc_.socket(),
                                        timeoutHandler = self.mnTimeoutHandler,
                                        timeoutTime = socketTimeoutTime() )
         self.dsp_ = Dispatcher( self.mh_, self.agentName_, self )
         self.sbwPath_ = clc.sbwTablePath % self.agentName_
         self.initialProtocol()
         self.connected = True

         # process all input and output
         while self.dsp_.next() and self.running_:
            pass
         dh.DEBUG9( "%s:%s:%s completed run" % ( self.sysname_, self.agentName_,
                                              self.sysDescOid_ ))
      except BaseException as e:  # pylint: disable-msg=W0703
         print e
         dh.DEBUG5( "%s:%s:%s got exception (%s) in ClientWorker.run [%r]"
                    % ( self.sysname_, self.agentName_, self.sysDescOid_, str( e ),
                        repr( e ) ) )
         self.errIs( e )
      finally:
         self.terminate()

   def terminate( self ):
      if self.running_:
         dh.DEBUG5( "%s:%s:%s terminating ClientWorker"
                    % ( self.sysname_, self.agentName_, self.sysDescOid_ ) )
         tbd = []
         for k in self.runningQueries.iterkeys():
            tbd.append( k )
         for k in tbd:
            self.tearDownSubscriptionHandling( k )
         if self.mh_ is not None:
            dh.DEBUG5( "sysname: " + self.sysname_ + " stats: "
                       + self.mh_.statsToString() )
            self.mh_.terminate()
         if self.cc_ is not None:
            self.cc_.shutdown()
         self.running_ = False

   def connectionId( self ):
      return self.connectionId_

   def err( self ):
      return self.err_

   def errIs( self, err ):
      self.err_ = err

   def setSysDescOid( self, oid ):
      self.sysDescOid_ = oid

   def signalFailure( self, msg, errorString ):
      msg.success = False
      msg.errorMsg = errorString
      msg.signalChannel.set()
      if(  msg.callbacks and 
           hasattr( msg, "onError" ) and
           msg.onError ):
         msg.callbacks.onError( msg )

   def signalTimeout( self, msg ):
      dh.DEBUG1( "request %s timed out" % msg )
      self.signalFailure( msg, "connection timed out. Likely dropped." )

   def tearDownSubscriptionHandling( self, outputPath ):
      msg = self.runningQueries.get( outputPath, None )
      if( msg is None ):
         dh.DEBUG9( "%s:%s teardown for %s, but already deleted." %
                    ( self.sysname_, self.agentName_, outputPath ) )
         return
      s = msg.subscription
      dh.DEBUG9( "%s:%s tearDown... k = %s sub = %s"
                 % ( self.sysname_, self.agentName_, outputPath, s ) )
      if s is not None:
         try:
            del self.runningQueries[outputPath]
         except KeyError:
            pass
         msg.signalChannel.set()

   def removeSubscription( self, qMsg ):
      s = qMsg.subscription
      if s is not None:
         self.mh_.sendRowDel( 0, s, self.sbwPath_ )

   def normalizeOutputPath( self, outputPath ):
      return outputPath
      #return "external/%d/%s" % ( self.sysDescOid_, outputPath )

   def uniquifyOutputPath( self, outputPath ):
      return ( outputPath + "." + str( os.getpid()) + "." + 
               # need to add ip addr into this, too.
               str( mTime.monotonic_time() ))

   # registerQuery is invoked when an IqMsg message is
   # queued up by a client by calling invokeQuery.  registerQuery
   # checks for timeliness and if we are beyond boot strap,
   # then the query parameters are extracted, the output path
   # is discovered, and the output type is learned and looked up.
   # If the system is not ready or the output table is not yet
   # registered, then registeryQuery takes no action and returns
   # true indicating that the request should be requeued.
   #
   # If registerQuery should proceed, it then constructs a
   # Subscription row including the required information.
   # The Subscription row is stored in the runningQueries
   # dictionary indexed by its output path.  The row is also
   # sent (rowIs) to the server to initiate the query.
   # Finally, the user callbacks are wrapped in an
   # IQHandlerWrapper and registered.
   def obtainSubPubTypeId( self, kind ):
      ac = arClass()
      ac.typeNameIs( "Arrow::External::" + kind )
      return self.dsp_.arCHdlr[ac]

   def registerSubPub( self ):
      sb = self.obtainSubPubTypeId( "Subscription" )
      if self.dsp_.sbHdlr is None:
         if sb != None:
            self.dsp_.sbHdlr = HandlerSubscription( self.dsp_.arCHdlr, self )
            self.dsp_.addTableHandler( clc.sbTablePath, self.dsp_.sbHdlr )
         else:
            dh.DEBUG9( "registerSubPub fails: no sbHdlr" )
            return None

      if self.dsp_.puHdlr is None:
         pu = self.obtainSubPubTypeId( "Publication" )
         if pu != None:
            self.dsp_.puHdlr = HandlerPublication( self.dsp_.arCHdlr, self )
            self.dsp_.addTableHandler( clc.puTablePath, self.dsp_.puHdlr )
         else:
            dh.DEBUG9( "registerSubPub fails: no pub typeId" )
            return None

      dh.DEBUG9( "registerSubPub completes with %s"
                 % ( "True" if sb != None else "False" ) )
      return sb.oid() if sb != None else None

   def validateIQ( self, msg, normalize ):
      reschedReturn = ( True, None, None, None )
      if not self.connected:
         # not yet connected to server.
         # reschedule this op for later
         dh.DEBUG9( "not processing invokeQuery: client not connected" )
         return reschedReturn

      acId = self.registerSubPub()
      if acId is None:
         dh.DEBUG9( "not processing invokeQuery: Sub/Pub not yet registered" )
         return reschedReturn

      keys = []
      values = []
      outputPath = None
      for k, v in msg.parameters.iteritems():
         keys.append( k )
         values.append( v )
         # the following could be a pattern match to ensure we are seeing what
         # we expect (^\w+._output_$).
         if clc.qpOutputSuffix in k:
            outputPath = self.normalizeOutputPath( v ) if normalize else v

      if outputPath is None:
         errorMsg = "Query parameters do not contain a table path.\n"
         errorMsg += "Please be sure that there is a parameter of the "
         errorMsg += 'form "xxx._output_":outputPath.'
         self.signalFailure( msg, errorMsg )
         dh.DEBUG9( "invokeQuery failed: no designated output table in %s"
                    % msg.parameters )
         return ( False, None, None, None )

      # everything checks out.  Return acquired values.
      return ( None, keys, values, outputPath )

   def processRegistration( self, msg, normalize ):
      subscriptionId = self.registerSubPub()
      if subscriptionId is None:
         dh.DEBUG9( "SubPub requeuing query = %s" % msg.queryText() )
         return True  # requeue the request

      if self.sysDescOid_ is None:
         dh.DEBUG9( "SysDesc requeuing query = %s" % msg.queryText() )
         return True

      ( requeue, keys, values, outputPath ) = self.validateIQ( msg, normalize )
      if ( outputPath and self.registered( outputPath ) ):
         errorMsg = "Previous subscription for %s still existent" 
         self.signalFailure( msg, errorMsg % outputPath )
         dh.DEBUG5( " %s:%s:%s Rejecting subscription for output path %s: %s"
                    % ( self.dbname(), self.agentName(), self.sysDescOid_,
                        outputPath, errorMsg ) )
         return False

      if requeue != None:
         dh.DEBUG9( "Validation requeuing query = %s" % msg.queryText() )
         return requeue

      # construct and deliver Subscription
      s = Subscription()
      s.rowTypeIs( subscriptionId )
      s.outputPathIs( outputPath )
      s.queryTextIs( msg.queryText() )
      s.system_descriptionIs( self.sysDescOid_ )
      name = msg.typeName()
      s.typeNameIs( name )
      s.type_signatureIs( "" )
      s.desiredFlagsIs( 0 )
      s.flagsIs( 0 )
      s.standingIs( not msg.oneShot )
      s.paramNameIs( keys )
      s.paramValueIs( values )

      self.dsp_.sbHdlr.onRow( s.buf_ )

      dh.DEBUG5( "%s:%s:%s issuing subscription with outputPath %s for query %s"
                 % ( self.dbname(), self.agentName(), self.sysDescOid_,
                     outputPath, msg.queryText() ) )
      msg.subscription = s

      #locally register the query's subscription
      self.runningQueries[outputPath] = msg

      # finally, deliver subscription request to server
      # return value from sendRow, in case we need to requeue
      return self.mh_.sendRowIs( 0, s, self.sbwPath_ )


   def registerQuery( self, msg ): # takes an IqMsg or an InqMsg
      dh.DEBUG9( "registerQuery query = %s" % msg.queryText() )
      return self.processRegistration( msg, True )

   def registerReader( self, msg ):  # takes an IrMsg # FIXME not fully implemented
      dh.DEBUG9( "registerReader query = %s" % msg.queryText() )
      return self.processRegistration( msg, False )

   def registerWriter( self, msg ): # takes an IwMsg # FIXME not fully implemented
      dh.DEBUG9( "registerWriter query = %s" % msg.queryText() )
      return self.processRegistration( msg, False )

   def registered( self, outputPath ):
      return bool( self.runningQueries.get( outputPath ))

   def writeRow( self, msg ):
      return self.mh_.sendRowIs( msg.tableId, msg.row, None )

   def unregisterQuery( self, msg ):
      outputPath = ""
      for k, v in msg.parameters.iteritems():
         if clc.qpOutputSuffix in k:
            outputPath = v
            break

      try:
         msg = self.runningQueries[outputPath]
      except KeyError:
         # signal completion of request even though it is a failure
         self.signalFailure( msg, "no query with output path %s" % outputPath )
         dh.DEBUG9( "no query for path = '%s'" % outputPath )
         return False

      self.mh_.sendRowDel( 0, msg.subscription, self.sbwPath_ )
      # deletion of the entry in runningQueries waits for onDelete
      # for the killed Publication as a result of the deletion of the
      # Subscription.
      msg.signalChannel.set()

   def mnTimeoutHandler( self ):
      completedRequests = []
      socketBufferFull = False
      requests = self.clientLib_.getRequests()
      monoTime = mTime.monotonic_time()
      for e in requests:
         if e.timeout <= monoTime:
            dh.DEBUG9( "timed out %s" % e )
            self.signalTimeout( e )
            completedRequests.append( e )
            continue

         dh.DEBUG9( "processing request = %s" % e )
         requeue = False
         if isinstance( e, IqMsg ):
            requeue = self.registerQuery( e )
         elif isinstance( e, InqMsg ):
            requeue = self.registerQuery( e )
         elif isinstance( e, IrMsg ):
            requeue = self.registerReader( e )
         elif isinstance( e, IwMsg ):
            requeue = self.registerWriter( e )
         elif isinstance( e, CqMsg ):
            requeue = self.unregisterQuery( e )
         elif isinstance( e, WrMsg ):
            if socketBufferFull:
               requeue = True
            else:
               requeue = self.writeRow( e )
               socketBufferFull = requeue
         elif isinstance( e, CancelMsg ):
            # need to cancel all outstanding operations here
            self.terminate()
            e.signalChannel.set()
         else:
            # unknown request
            dh.DEBUG9( "unknown request" )

         if not requeue:
            completedRequests.append( e )

      self.clientLib_.removeRequests( completedRequests )

   def typeId( self, typeName ):
      if( not self.dsp_ ):
         return 0
      # if dsp_ exists then dsp_.arCHdlr also exists
      arc = arClass()
      arc.typeNameIs( typeName )
      nArc = self.dsp_.arCHdlr[arc]
      if nArc is None:
         return 0
      return nArc.oid()

   def initialProtocol( self ):
      # send SystemDescription row containing:
      sd = SystemDescription()
      sd.rowTypeIs( clc.sdTableId )
      sd.sysNameIs( self.sysname_ )
      sd.agentNameIs( self.agentName_ )
      sd.connection_idIs( self.connectionId() )
      # indicate that we are an external client
      sd.propertiesIs( 0x1 )
      sd.oidIs( 1 ) # fake to at least have an oid
      # now, deliver the SystemDescription to the server
      dh.DEBUG9( "%s publishing SystemDescription" % self.sysname_ )
      self.mh_.sendRowIs( 0, sd, clc.sdTablePath )

# IQHandlerWrapper is used to intercept calls to user callbacks
# (onRow, onDelete, onDeleteRange) to perform any system services
# including:
#   1. periodic ptime delivery via SubscriptionRow
#   2. termination of the query output stream when a null row
#      is received
class IQHandlerWrapper( object ):
   def __init__( self, e, cw ):
      self.cw_ = cw
      self.dsp_ = cw.dsp_
      self.mh_ = cw.mh_
      self.request_ = e
      dh.DEBUG9( "request received for e = %s" % e )
      self.count_ = 0
      self.outputPath_ = None
      self.table_ = None
      for k, v in e.parameters.iteritems():
         if clc.qpOutputSuffix in k:
            self.outputPath_ = v
            break
      assert self.outputPath_ is not None

      self.tableId_ = self.dsp_.fetchArTableId( self.outputPath_ )

   def transient( self ):
      return False

   def updateSubscription( self ):
      self.count_ += 1
      if ( self.count_ % clc.PtimeNotificationInterval ) == 0:
         msg = self.cw_.runningQueries[self.outputPath_]
         s = msg.subscription
         self.mh_.sendRowIs( 0, s, self.cw_.sbwPath_ )

   def onRow( self, row, pathName=None ):
      callbacks = self.request_.callbacks
      if( getattr( callbacks, 'makeRow', None ) is None ):
         br = BaseRow( row = row )
      else:
         br = callbacks.makeRow( row=row )
         row = br
      dh.DEBUG9( "wrapped row length = %d" % len( br ) )

      if not br or len( br ) < 8:
         s = self.request_.subscription
         dh.DEBUG9( "%s %s:%s received null row for table %s" 
                    % ( self, self.cw_.dbname(), self.cw_.agentName(),
                        s.outputPath() ) )
         self.cw_.removeSubscription( self.request_ )
         return False

      # create table if it doesn't exist, and if appropriate
      if( self.table_ is None and 
          not self.request_.oneShot and
          getattr( callbacks, 'makeRow', None )):
         self.table_ = TableImpl( br.rowType(), genOid=False )

      # if table exists, store row
      if( self.table_ is not None ):
         self.table_.rowIs( br )
      self.updateSubscription()
      dh.DEBUG9( "%s %s:%s delivering row to callback for %s"
                 % ( self, self.cw_.dbname(), self.cw_.agentName(),
                     pathName or self.tableId_ ) )
      self.request_.callbacks.onRow( row, pathName )
      return True

   def onDelete( self, row, pathName ):
      callbacks = self.request_.callbacks
      if( getattr( callbacks, 'makeRow', None ) is not None ):
         row = callbacks.makeRow( row=row )
      self.updateSubscription()
      dh.DEBUG9( "%s %s:%s delivering deletion to callback for %s"
                 % ( self, self.cw_.dbname(), self.cw_.agentName(),
                     pathName or self.tableId_ ) )
      self.request_.callbacks.onDelete( row, pathName )
      if( self.table_ is not None ):
         self.table_.rowDel( row )
      return True

   def onDeleteRange( self, startRow, endRow, pathName ):
      callbacks = self.request_.callbacks
      convertedRows = False
      if( getattr( callbacks, 'makeRow', None ) is not None ):
         startRow = callbacks.makeRow( row=startRow )
         endRow = callbacks.makeRow( row=endRow )
         convertedRows = True
      if( getattr( callbacks, 'onDeleteRange', None ) is not None ):
         dh.DEBUG9( "%s %s:%s onDeleteRange callback for %s"
                    % ( self, self.cw_.dbname(), self.cw_.agentName(),
                        self.outputPath_ ) )
         self.updateSubscription()
         return callbacks.onDeleteRange( startRow, endRow, pathName )
      elif( self.table_ is not None and convertedRows ):
         toBeDeleted = []
         dh.DEBUG9( "onDeleteRange for %s converted to onDelete" % self.outputPath_ )
         dh.DEBUG9( " start: %s end: %s"
                    % ( startRow.toString(), endRow.toString() ) )
         for k in self.table_.iterkeys():
            if (( not startRow or startRow < k ) and 
                ( not endRow or k < endRow )):
               toBeDeleted.append( k )
         for k in toBeDeleted:
            self.onDelete( k, pathName )
         return True
      else:
         msg = "onDeleteRange not currently implemented for %s"
         assert False, msg % self.outputPath_
         # an onDeleteRange represents a single source change because
         # it represents a consolidation of row deletions.

