# -*- mode: python -*-
# Copyright (c) 2006-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""Arastra networking utilities.

This module contains utilities relating to networking, in particular, for
manipulating IP addresses.
"""

import collections
import functools
import random
import re
import socket

import Tac
from ArPyUtils import naturalOrderKey

# Moved some utilities that used to be here to Ark.IpUtils so that ArosTest
# doesn't depend on Arnet.  Not all of these utilities are used within this file,
# but that's fine: we're importing them for compatibility with old code that used
# to import them from here.
# pylint: disable-msg=W0611
from IpUtils import Mask

# This is required by nextAddress4() and nextAddress6(). These should
# ideally use the Arnet types and not the IpUtils types.
import IpUtils

Ip6AddrRe = '[A-Fa-f0-9:.]{0,46}'
IpAddrRe = '(\d+)\.(\d+)\.(\d+)\.(\d+)$'
IpAddrReFull = '^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}'\
               '(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$'
IpAddrCompiledRe = re.compile( IpAddrRe )
IpAddrFullCompiledRe = re.compile( IpAddrReFull )
Ip6AddrCompiledRe = re.compile( Ip6AddrRe )

IpAddrWithFullMaskCompiledRe = \
   re.compile( "(\d+\.\d+\.\d+\.\d+)(?:/(\d+\.\d+\.\d+\.\d+))?$" )
IpAddrWithMaskCompiledRe = re.compile( "(\d+\.\d+\.\d+\.\d+)(/\d+)?$" )
Ip6AddrWithMaskCompiledRe = re.compile( "^([A-Fa-f0-9:.]{0,46})(/\d+)?$" )
Ip6AddrWithFullMaskCompiledRe = \
      re.compile( "^([A-Fa-f0-9:.]{0,46})(/([A-Fa-f0-9:.]{0,46}))?$" )

mcastGroupCompiledRe = re.compile( '([0-9]+).([0-9]+).([0-9]+).([0-9]+)' )

def ip6MaskFromPrefix( prefixLength ):
   binaryString = "1" * prefixLength + "0" * ( 128 - prefixLength )
   hexString = format( int( binaryString, 2 ), '032x' )
   mask = ':'.join( re.findall( '.{4}', hexString ) )
   return mask

def intFromMask( sMask ):
   """
   Attempts to create a 32 bit integer from a string representing an ipv4 mask.

   s_mask will be converted to an int if it is in dotted-decimal or prefix format.
   """
   try:
      # mask is a prefix length, or a number that looks like one.
      mLen = int(sMask)
      return Mask._ipPrefixLenToMask[ mLen ]
   except ValueError:
      # mask is full, or some unsupported format.
      m = IpAddrCompiledRe.match( sMask )
      if m:
         return 0xffffffff  & \
             reduce( lambda a, b: a | b, 
                    [ int( m.groups()[i] ) << (24 - 8*i) for i in xrange(4) ] )
      return None

def intFromPrefix( prefix ):
   """
   Variation of intFromMask. Attempts to create a 64 bits integer from
   an ipv4 prefix.
   """
   address = intFromMask( prefix.address )
   return ((address << 8) + int(prefix.len)) if address is not None else None

def numFromAddr( addr ):
   """
   Convert an instance of Arnet::IpAddr or Arnet::Ip6Addr into a
   numeric type (int or long as necessary).

   As noted in BUG33844, using this conversion may be a sign that we
   need more functionality in the C++ layer for low-level address /
   mask hacking so that we can avoid implementing the same
   functionality in C++ and python.
   """
   if type( addr ) is Tac.Type( 'Arnet::IpAddr' ):
      return addr.value
   elif type( addr ) is Tac.Type( 'Arnet::Ip6Addr' ):
      return ( addr.word0 << 96 ) | ( addr.word1 << 64 ) \
          | ( addr.word2 << 32 ) | addr.word3
   else:
      raise ValueError()

def isContiguousMask( iMask ):
   """
   Determines whether or not the given 32 bit integer may represent 
   a contiguous ipv4 netmask.
   """
   return iMask is 0 or ~iMask is 0 or \
       not ( ( (~iMask >> 1) ^ ~iMask) >> 1) + iMask == 0
      

def AddrWithFullMask( sAddr, iMask=0 ):
   """
   Constructs a Tac.Value representing an Arnet::IpAddrWithFullMask.

   s_addr should be a string represetnign an ipv4 address in dotted decimal format.

   i_mask should be the integer form of the mask as it will be applied; 
   i.e., should not be a wildcard mask.
   
   If neither a mask nor a prefix length is specified, the address is taken
   to be a host address and the mask defaults to 0.0.0.0.
   """
   if type( sAddr ) is Tac.Type( 'Arnet::IpAddrWithFullMask' ) and iMask == 0:
      return sAddr
   sAddr = str( sAddr )

   m = IpAddrWithMaskCompiledRe.match( sAddr )
   if m:
      if iMask == 0:
         sAddr = m.group( 1 )
         if m.group( 2 ):
            iMask = Mask._ipPrefixLenToMask[ int( m.group( 2 )[ 1: ] ) ]
      return Tac.ValueConst( "Arnet::IpAddrWithFullMask", 
                             address=sAddr,
                             mask = (0xffffffff & iMask) )
   raise ValueError( 
      "Bad Address or Mask, addr=%s, mask=%s" % (sAddr, iMask))

def AddrWithMask( s ):
   """Constructs a Tac.Value representing an Arnet::AddrWithMask from
   a string (in the format '1.2.3.4' or '1.2.3.4/16')."""

   if type( s ) is Tac.Type( 'Arnet::IpAddrWithMask' ):
      return s
   s = str( s )

   m = IpAddrWithMaskCompiledRe.match( s )
   if m:
      if m.group( 2 ):
         maskLen = int( m.group( 2 )[ 1: ] )
      else:
         maskLen = 32
      return Tac.ValueConst(
         "Arnet::IpAddrWithMask", address=m.group( 1 ), len=maskLen )

   raise ValueError( "Expected prefix in addr/len format; got '" + s + "'" )

def IpAddr( s ):
   if type( s ) is Tac.Type( 'Arnet::IpAddr' ):
      return s
   return Tac.ValueConst( "Arnet::IpAddr", stringValue=str( s ) )

def Ip6Addr( s ):
   if type( s ) is Tac.Type( 'Arnet::Ip6Addr' ):
      return s
   return Tac.ValueConst( "Arnet::Ip6Addr", stringValue=str( s ) )

def IntfId( s ):
   if type( s ) is Tac.Type( 'Arnet::IntfId' ):
      return s
   return Tac.ValueConst( "Arnet::IntfId", stringValue=str( s ) )

def Ip6AddrWithMask( s, mask=128 ):
   """Constructs a Tac.Value representing an Arnet::Ip6AddrWithMask from
   a string (in the format 'A:B:C:D:E:F:G:H' or ''A:B:C:D:E:F:G:H/64')."""

   if type( s ) is Tac.Type( 'Arnet::Ip6AddrWithMask' ):
      return s
   s = str( s )
   
   m = Ip6AddrWithMaskCompiledRe.match( s )
   if m:
      if m.group( 2 ):
         maskLen = int( m.group( 2 )[ 1: ] )
      else:
         maskLen = mask
      return Tac.ValueConst( "Arnet::Ip6AddrWithMask",
                             address=Ip6Addr( m.group( 1 ) ),
                             len=maskLen )

   raise ValueError( "Expected prefix in addr/len format; got '" + s + "'" )

def Ip6AddrWithFullMask( s, mask="FFFF:FFFF:FFFF:FFFF:FFFF:FFFF:FFFF:FFFF" ):
   """Constructs a Tac.Value representing an Arnet::Ip6AddrWithFullMask from
   a string (with format 'A:B:C:D:E:F:G:H' or 'A:B:C:D:E:F:G:H/A:B:C:D:E:F:G:H')."""

   if type( s ) is Tac.Type( 'Arnet::Ip6AddrWithFullMask' ):
      return s
   s = str( s )
   m = Ip6AddrWithFullMaskCompiledRe.match( s )
   if m:
      if m.group( 2 ):
         mask = m.group( 2 )[ 1: ]
      else:
         mask = mask
      return Tac.ValueConst( "Arnet::Ip6AddrWithFullMask",
                              address=Ip6Addr( m.group( 1 ) ),
                              mask=Ip6Addr( mask ) )

   raise ValueError( "Expected mask in addr/mask format; got '" + s + "'" )

def Prefix( s ):
   if type( s ) is Tac.Type( 'Arnet::Prefix' ):
      return s

   addrWithMask = AddrWithMask( s )

   if not addrWithMask.validAsPrefix:
      raise ValueError( "'%s' is not a valid prefix" % s )

   return Tac.ValueConst( "Arnet::Prefix",
                          address=addrWithMask.address,
                          len=addrWithMask.len )

def Ip6Prefix( s ):
   if type( s ) is Tac.Type( 'Arnet::Ip6Prefix' ):
      return s

   addrWithMask = Ip6AddrWithMask( s )

   if not addrWithMask.validAsPrefix:
      raise ValueError( "'%s' is not a valid prefix" % s )

   return Tac.ValueConst( "Arnet::Ip6Prefix",
                          address=addrWithMask.address,
                          len=addrWithMask.len )

def EthAddr( ethAddrStr ):
   ethAddrObj = Tac.Value("Arnet::EthAddr" )
   ethAddrObj.stringValue = ethAddrStr
   return ethAddrObj

def uint64FromPrefix6( prefix ):
   """
   Attempts to create a 64 bits integer from a string representing an ipv6 prefix. 
   It returns None if the string is
   not in the correct format. 
   """
   try:
      ip = Ip6AddrWithMask( prefix )
   except ValueError:
      return None
   address = ( ip.address.word0 << 32 ) | ip.address.word1
   return (( address ) + int(ip.len))

def IpAddress( rep, addrFamily=socket.AF_INET ):
   '''Construct an IP address (v4 or v6) TAC object from one of
   several input formats.

   This can convert from string, int, or long. Passing in an object
   that is already an instance of Arnet::IpAddr or Arnet::Ip6Addr is
   also supported.
   '''

   if addrFamily == socket.AF_INET:
      typeName = "Arnet::IpAddr"
   elif addrFamily == socket.AF_INET6:
      typeName = "Arnet::Ip6Addr"
   else:
      raise ValueError()

   try:
      if type( rep ) == str:
         return Tac.ValueConst( typeName,
                                stringValue=rep )
      elif type( rep ) in ( int, long ):
         if addrFamily == socket.AF_INET:
            return Tac.ValueConst( typeName,
                                   value=rep )
         else:
            if type( rep ) is int:
               return Tac.ValueConst( typeName,
                                      word3=rep )
            else:
               word3 = rep & 0xffffffff
               word2 = ( rep >> 32 ) & 0xffffffff
               word1 = ( rep >> 64 ) & 0xffffffff
               word0 = ( rep >> 96 ) & 0xffffffff
               return Tac.ValueConst( typeName,
                                      word0=word0, word1=word1,
                                      word2=word2, word3=word3 )
      elif type( rep ) is Tac.Type( typeName ):
         # BUG31255 Allowing construction from an instance that is
         # already a TAC object is deprecated
         #
         # Round-tripping via stringValue is necessary to work around
         # a shortcoming in pylint; if we return rep itself, it
         # believes that we could be returning an integer and throws
         # bogus erros.
         return Tac.ValueConst( typeName, stringValue=rep.stringValue )
      else:
         raise TypeError()
   except IndexError:
      raise ValueError( "'%s' is not a valid IP address" % rep )

_intfNameKeyCache = collections.defaultdict( list )
def intfNameKey( item ):
   # We have a defaultdict.
   # If the key does not exist, it will create a list,
   # insert it in the cache, and return it.
   result = _intfNameKeyCache[ item ]
   if not result:
      # The list is empty, so we need to cache the result of `naturalOrderKey`.
      # NB: id( result ) == id( cache[ item ] ),
      # which means we're changing that same list.
      result += naturalOrderKey( item )
   return result

def compareIntfName( intf1, intf2 ):
   return cmp( intfNameKey( intf1 ), intfNameKey( intf2 ) )

# Convenience objects for the leftover usages of `fastSomethingSomething`.
fastIntfNameKey = intfNameKey

# Convenience API for sorting interfaces.
sortIntf = functools.partial( sorted, key=intfNameKey )

class Interface( object ):
   '''Some code would use the `intfNameKey` not for sorting,
   but to use the data in some other way.
   This no longer works, so we need another object with the appropriate attrs.
   '''
   __slots__ = ( 'basename', 'stack', 'mod', 'port', 'sub', 'intfNameRe' )
   intfNameRe = re.compile( r'(?P<basename>[^0-9/]+)'
                            r'(?:(?P<stack>\d+)/?)?'
                            r'(?:(?P<mod>\d+)/?)?'
                            r'(?:(?P<port>\d+))?'
                            r'(?:\.(?P<sub>\d+))?$' )

   def __init__( self, intfString ):
      def value( group ):
         return int( m.group( group ) or '0' )
      m = self.intfNameRe.match( intfString )
      self.basename = m.group( 'basename' )
      self.stack = value( 'stack' )
      self.mod = value( 'mod' )
      self.port = value( 'port' )
      self.sub = value( 'sub' )

def getMcastGroupAddresses( startAddr, numGroups ):
   groups = [ ]
   m = mcastGroupCompiledRe.match( startAddr )
   addr = ( int( m.group( 1 ) ) << 24 ) + \
          ( int( m.group( 2 ) ) << 16 ) + \
          ( int( m.group( 3 ) ) << 8 ) + \
          int( m.group( 4 ) )
   for i in range( numGroups ):
      groups.append( "%s.%s.%s.%s" % ( ( addr >> 24 ) & 0xFF,
                                       ( addr >> 16 ) & 0xFF,
                                       ( addr >> 8 ) & 0xFF,
                                       ( addr & 0xFF ) ) )
      addr = addr + 1
   return groups

def prefixStrToObject( prefixes, addrFamily ):
   '''Concerts a given prefix or a set of prefixes to list of AddrWithMask or
   Ip6AddrWithMask objects'''
   if type( prefixes ) is str:
      prefixes = [ prefixes ]
   if addrFamily == socket.AF_INET:
      return [ Tac.const( AddrWithMask(p) ) for p in prefixes ]
   else:
      return [ Tac.const( Ip6AddrWithMask(p) ) for p in prefixes ]

class Subnet( object ):
   """Represents an IP subnet (v4/v6), defined by an IP address and a subnet mask.  
   Subnet objects are immutable."""
   
   def __init__( self, a, b=None, addrFamily=socket.AF_INET ):
      """Constructs a Subnet object from a string 
      (in the format '1.2.3.4/16' for v4 and 1:2:3::4/64 for v6), a
      dict containing 'address' and 'len' items, an object with 'address' and 'len'
      attributes, or a pair of objects from which an IpAddress and Mask object,
      repsectively, can be constructed."""
      if addrFamily not in [ socket.AF_INET, socket.AF_INET6 ]:
         errMsg = "addrFamily must be either AF_INET or AF_INET6"
         raise ValueError( "%s: %s" % ( errMsg, addrFamily ) )
      
      self.addrFamily_ = addrFamily
      if b is None:
         # We got one parameter, which contains both an IP address and a mask
         if type( a ) == str:
            [ addr, mask ] = re.split( '/', a )
            self.address_ = IpAddress( addr, addrFamily=addrFamily )
            self.mask_ = Mask( int( mask ), addrFamily=addrFamily )
         elif type( a ) == dict:
            if type( a[ 'address' ] ) is str:
               self.address_ = IpAddress( a[ 'address' ], addrFamily=addrFamily )
            else:
               self.address_ = a[ 'address' ]
            self.mask_ = Mask( a[ 'len' ], addrFamily=addrFamily )
         else:
            self.address_ = IpAddress( a.address, addrFamily=addrFamily )
            self.mask_ = Mask( a.len, addrFamily=addrFamily )
      else:
         # We got two parameters, a separate IP address and mask.
         self.address_ = IpAddress( a, addrFamily=addrFamily )
         self.mask_ = Mask( b, addrFamily=addrFamily )

   def __str__( self ):
      """Returns the subnet as a string in the format '1.2.3.4/16' for v4 
      and 1:2:3::4/64 for v6) """
      return "%s/%d" % ( self.address_.stringValue, self.mask_.masklen_ )

   def _addrWithMask( self ):
      """This is a transitional helper method while refactoring the
      code to use TAC objects rather than Python. It generates an
      Arnet::AddrWithMask or Arnet::Ip6AddrWithMask. In the longer
      term, this Python class should become a thin wrapper around an
      instance of one of these types
      """

      if self.addrFamily_ == socket.AF_INET:
         return AddrWithMask( '%s/%s'
                              % ( self.address_, self.mask_.masklen_ ) )
      elif self.addrFamily_ == socket.AF_INET6:
         return Ip6AddrWithMask( '%s/%s'
                                 % ( self.address_, self.mask_.masklen_ ) )

   def toNum( self ):
      """Returns the subnet address as a non-negative int or long in host byte order.
      The subnet address of the subnet 1.2.3.4/16, for example, is 1.2.0.0."""
      if self.addrFamily_ == socket.AF_INET:
         return self.address_.value & self.mask_.toNum()
      elif self.addrFamily_ == socket.AF_INET6:
         return ( ( self.address_.word0 << 96 )
                  + ( self.address_.word1 << 64 )
                  + ( self.address_.word2 << 32 )
                  + self.address_.word3 ) & self.mask_.toNum()

   def toValue( self ):
      """Returns the subnet as a Tac.Value representing an Arnet::Prefix.
      The host bits in the network address are zeroed out by applying the mask. """
      if self.addrFamily_ == socket.AF_INET:
         return Prefix( self._addrWithMask().subnet.stringValue )
      elif self.addrFamily_ == socket.AF_INET6:
         return Ip6Prefix( self._addrWithMask().subnet.stringValue )
   
   def containsAddr( self, addr ):
      """Returns whether or not an IP address is contained within this subnet."""
      if type( addr ) == Tac.Type( 'Arnet::IpAddr' ) \
             and self.addrFamily_ != socket.AF_INET:
         return False
      elif type( addr ) == Tac.Type( 'Arnet::Ip6Addr' ) \
             and self.addrFamily_ != socket.AF_INET6:
         return False

      if type( addr ) in [ str, int, long ]:
         addr = IpAddress( addr, addrFamily=self.addrFamily_ )

      return self._addrWithMask().contains( addr )

   def numHostInSubnet( self ):
      """Returns the number of IP addresses that are contained within this subnet."""
      return self.mask_.numHostInSubnet()

   def overlapsWith( self, otherSubnet ):
      """Returns True if one subnet is a subset of the other."""
      if self.addrFamily_ == socket.AF_INET:
         smallerMask = Mask._ipPrefixLenToMask\
                    [ min( self.mask_.masklen_, otherSubnet.mask_.masklen_ ) ]
         return self.address_.value  & smallerMask == \
             otherSubnet.address_.value & smallerMask
      else:
         return self._addrWithMask().overlaps( otherSubnet._addrWithMask() )

   def isAllZerosAddress( self ):
      if self.addrFamily_ == socket.AF_INET:
         if self.mask_.masklen_ >= 31:
            #ignore special cases.
            return False
         mask = Mask._ipPrefixLenToMask[ self.mask_.masklen_ ]
         return ( ~mask & self.address_.value ) == 0
      else:
         mask = Mask._ip6PrefixLenToMask[ self.mask_.masklen_ ]
         return ( ~mask & self.address_.value ) == 0

   def isBroadcastAddress( self ):
      if self.addrFamily_ == socket.AF_INET:
         if self.mask_.masklen_ >= 31:
            # There is no broadcast address on a /31 or /32.
            return False
         addr = self.address_.value
         mask = Mask._ipPrefixLenToMask[ self.mask_.masklen_ ]
         return ( mask | addr ) == 0xffffffff
      else:
         return False

def IpGenPrefix( s ) :
   try:
      _addrWithMask = AddrWithMask( s )
      return Tac.ValueConst( "Arnet::IpGenPrefix", s )
   except AttributeError:
      pass
   except ValueError:
      pass
   try:
      _addrWithMask = Ip6AddrWithMask( s )
      return Tac.ValueConst( "Arnet::IpGenPrefix", s )
   except AttributeError:
      raise ValueError( "'%s' is not a valid prefix" % s )

def IpGenAddrWithMask( s ):
   try:
      _addrWithMask = AddrWithMask( s )
      return Tac.ValueConst( "Arnet::IpGenAddrWithMask", s )
   except AttributeError:
      pass
   except ValueError:
      pass
   try:
      _addrWithMask = Ip6AddrWithMask( s )
      return Tac.ValueConst( "Arnet::IpGenAddrWithMask", s )
   except AttributeError:
      raise ValueError( "'%s' is not a valid address" % s )

def IpGenAddrWithFullMask( s ):
   if s == '':
      return Tac.ValueConst( "Arnet::IpGenAddrWithFullMask", "" )
   try:
      m = IpAddrWithFullMaskCompiledRe.match( s )
      _addr = IpAddr( m.group( 1 ) )
      if m.group( 2 ):
         _mask = IpAddr( m.group( 2 ) )
      return Tac.ValueConst( "Arnet::IpGenAddrWithFullMask", s )
   except ( AttributeError, IndexError ):
      pass

   try:
      _addrWithMask = AddrWithMask( s )
      return Tac.ValueConst( "Arnet::IpGenAddrWithFullMask", s )
   except ( AttributeError, ValueError, IndexError ):
      pass

   try:
      _addrWithMask = Ip6AddrWithFullMask( s )
      return Tac.ValueConst( "Arnet::IpGenAddrWithFullMask", s )
   except ( AttributeError, IndexError ):
      pass

   try:
      _addrWithMask = Ip6AddrWithMask( s )
      return Tac.ValueConst( "Arnet::IpGenAddrWithFullMask", s )
   except ( AttributeError, IndexError, ValueError ):
      raise ValueError( "'%s' is not a valid address" % s )

def IpGenAddr( s ) :
   ipv6 = re.search( ":", s )
   if ipv6:
      _addr = Ip6Addr( s )
      return Tac.ValueConst( "Arnet::IpGenAddr", stringValue=s )
   else:
      try:
         _addr = IpAddress( s )
         return Tac.ValueConst( "Arnet::IpGenAddr", stringValue=s )
      except AttributeError:
         raise ValueError( "'%s' is not a valid address" % s )

def nextAddress4( addr, maskLen ):
   '''Get the next incremental IP addr given a prefix addr and mask len.
   For example, the next addr of 10.0.0.1/32 is 10.0.0.2/32, the next
   addr of 2.2.2.2/24 is 2.2.3.2/24. Exclude common broadcast addresses,
   but not other well-known special addresses (yet).'''
   step = 2 ** ( 32 - maskLen )
   if isinstance( addr, str ):
      addr = IpUtils.IpAddress( addr )
   addr = IpUtils.IpAddress( addr.toNum() + step )
   if maskLen == 32:
      if addr.toNum() % 256 == 0:
         addr = IpUtils.IpAddress( addr.toNum() + step )
      elif ( addr.toNum() + 1 ) % 256 == 0:
         addr = IpUtils.IpAddress( addr.toNum() + step + 1 )
   return addr

def nextAddress6( addr, maskLen=128 ):
   '''Get the next incremental IP addr given a prefix addr and mask len.
   For example, the next addr of 100::1/128 is 100::2/128, the next
   addr of 100::2/64  is 100:0:0:1::2/64'''
   if isinstance( addr, str ):
      addr = IpUtils.IpAddress( addr, addrFamily=socket.AF_INET6 )
   step = step = 2 ** ( 128 - maskLen )
   addr = IpUtils.IpAddress( addr.toNum() + step, addrFamily=socket.AF_INET6 )
   return addr

def nextAddrMask4( addrMask ):
   '''Get next addr/mask. For example 10.0.2.1/24 => 10.0.3.1/24'''
   if type( addrMask ) is str:
      addrMask = AddrWithMask( addrMask )
   addr = nextAddress4( addrMask.address, addrMask.len )
   return AddrWithMask( "%s/%s" % ( addr, addrMask.len ) )

def nextAddrMask6( addrMask ):
   if type( addrMask ) is str:
      addrMask = Ip6AddrWithMask( addrMask )
   assert addrMask.len == 64 # only do /64 for now
   addr = IpAddress( numFromAddr( addrMask.address ) + 2 ** 64,
                     addrFamily=socket.AF_INET6 )
   return Ip6AddrWithMask( addr, 64 )
      
def nextAddrInAddrMask4( addrMask ):
   '''
   Get next addr/mask with a subsequent /32 IP address. For example,
   10.0.2.1/24 => 10.0.2.2/24. Raise exception if the next address is out of
   current subnet.
   '''
   if type( addrMask ) is str:
      addrMask = AddrWithMask( addrMask )
   addr = nextAddress4( addrMask.address, 32 )
   if not addrMask.contains( str( addr ) ):
      raise Exception( "Next address out of subnet range %s" % addrMask )
   return AddrWithMask( "%s/%s" % ( addr, addrMask.len ) )

def nextAddrInAddrMask6( addrMask ):
   '''
   Return next address as a Ip6Addr value in /64 network of addrMask. No
   checking of boundary. Hopefully enough addresses.
   '''
   if type( addrMask ) is str:
      addrMask = Ip6AddrWithMask( addrMask )
   assert addrMask.len == 64
   addr = IpUtils.IpAddress( numFromAddr( addrMask.address ) + 1,
                             addrFamily=socket.AF_INET6 )
   return Ip6Addr( str( addr ) )

def randomEthAddr():
   '''Return a random unicast ethernet address.'''
   byte0 = 0
   byte1 = random.getrandbits( 8 )
   byte2 = random.getrandbits( 8 )
   byte3 = random.getrandbits( 8 )
   byte4 = random.getrandbits( 8 )
   byte5 = random.getrandbits( 8 )
   return '%02x:%02x:%02x:%02x:%02x:%02x' % (
      byte0, byte1, byte2, byte3, byte4, byte5 )
