# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys, os, re
import Tac, Cell
import TacSigint
from Arnet.NsLib import runMaybeInNetNs
from eunuchs.if_h import IFF_UP
import AgentDirectory

class VerifyException( Exception ):
   def __init__( self, fingerprint, data=None ):
      Exception.__init__( self )
      self.fingerprint = fingerprint
      self.data = data

class NotApplicableException( Exception ):
   pass

def runningAgents():
   return [ agent[ 'name' ] for agent in AgentDirectory.agentList() ]

class AdjustedModulePath():
   def __init__( self ):
      self.path = None
   def __enter__( self ):
      self.path = list( sys.path )
      sitepath = [ p for p in self.path if p.split('/')[-1] == 'site-packages' ]
      for p in sitepath:
         sys.path.insert( 0, p + '/VerifyPlugin' )
   def __exit__( self, *args ):
      sys.path[:] = self.path

def getModuleName( objType, actor ):
   return objType + 'Verify' + actor

def importVerifiers( objType, actors, entityManager ):
   module = None
   verifiers = {}
   for actor in actors:
      moduleName = getModuleName( objType, actor )
      try:
         module = __import__( moduleName )
      except: # pylint: disable-msg=W0702
         print 'TROUBLE VERIFYING, UNABLE TO IMPORT %s' % moduleName
         return verifiers

      try:
         verifiers[ moduleName ] = module.Verifier( entityManager )
      except NotApplicableException:
         pass
      except: # pylint: disable-msg=W0702
         print 'TROUBLE VERIFYING, UNABLE TO INSTANTIATE VERIFIER ' \
               'FOR ACTOR %s MODULE %s' % ( actor, moduleName )
   return verifiers

def verifyObject( verifiers, objType, obj, actors ):
   noFailure = True
   atleastOneSuccess = False

   for actor in actors:
      moduleName = getModuleName( objType, actor )
      if not verifiers.get( moduleName ) or not verifiers[ moduleName ]:
         continue
      try:
         verifiers[ moduleName ].verify( obj )
      except NotApplicableException:
         # obj not applicable to this actor (eg. Et1 to Ebra), ignore
         pass
      except VerifyException as e:
         noFailure = False
         print 'PROBLEM FINGERPRINT %s: %s' % ( actor, e.fingerprint )
         if e.data is not None:
            try:
               print e.fingerprint % e.data
            except: # pylint: disable-msg=W0702
               # ignore any exception that occurs during formatting
               print 'Unable to format data from actor %s' % actor
               print 'Data received is ', e.data
      except Exception as e1:
         noFailure = False
         print 'Unknown problem during verification in actor %s: ' % actor, e1
      else:
         # verification of 'obj' state managed by 'actor' was successful
         atleastOneSuccess = True

   return atleastOneSuccess and noFailure

def verify( objType, objects, actors, entityManager ):
   with AdjustedModulePath():
      verifiers = importVerifiers( objType, actors, entityManager )
      verifiedObjects = []
      for obj in objects:
         try:
            TacSigint.check()
         except: # pylint: disable-msg=W0702
            # keyboard interrupt - stop the verification
            print 'Verification interrupted'
            break
         if verifyObject( verifiers, objType, obj, actors ):
            verifiedObjects.append( obj )
      if len( verifiedObjects ):
         print 'Verification successful for ', verifiedObjects
      for verifier in verifiers.values():
         verifier.cleanup()

def deviceMacAddr( devName, ns ):
   filename = os.path.join( '/sys/class/net', devName, 'address' )
   sysfsAttribute = runMaybeInNetNs( ns, [ 'cat', filename ],
                                     asRoot=True, stdout=Tac.CAPTURE )
   return sysfsAttribute.strip()

def deviceEnabled( devName, ns ):
   try:
      filename = os.path.join( '/sys/class/net', devName, 'flags' )
      sysfsAttribute = runMaybeInNetNs( ns, [ 'cat', filename ],
                                        asRoot=True, stdout=Tac.CAPTURE )
      flags = int( sysfsAttribute, 16 )
      enabled = bool( flags & IFF_UP )
   except: # pylint: disable-msg=W0702
      return False
   else:
      return enabled

def devicePresent( devName, ns ):
   stmp = runMaybeInNetNs( ns, [ 'ls', '/sys/class/net' ],
                           asRoot=True, stdout=Tac.CAPTURE )
   return devName in set( stmp.split() )

def verifyKernelDevice( intfName, sysdb ):
   devName = intfName.lower().replace( '/', '_' )
   m = re.match( r'([a-z\-]+)(\d.*)', devName )
   devType = m.group( 1 ) if m.group( 1 ) in [ 'vlan', 'vxlan' ] \
                          else m.group( 1 )[ 0:2 ]
   devName = devType + m.group( 2 )

   allIntfConfigDir = sysdb.entity.get( 'interface/config/all' )
   allIntfStatusDir = sysdb.entity.get( 'interface/status/all' )
   allIntfStatusLocalDir = sysdb.entity.get( Cell.path( 'interface/status/local' ) )
   allIntfNamespaceConfigDir = sysdb.entity.get( Cell.path( 'interface/nsconfig' ) )
   bridgingConfig = sysdb.entity.get( 'bridging/config' )

   config = allIntfConfigDir.intfConfig.get( intfName )
   status = allIntfStatusDir.intfStatus.get( intfName )
   statusLocal = allIntfStatusLocalDir.intfStatusLocal.get( intfName )

   if not status or status.forwardingModel != 'intfForwardingModelRouted':
      print 'Skipping kernel device verification for non-routed interface %s' % \
            intfName
      return

   nsName = ''
   nsConfig = allIntfNamespaceConfigDir.intfNamespaceConfig.get( intfName )
   if nsConfig and nsConfig.netNsName != '' and \
                   nsConfig.netNsName != 'default':
      nsName = nsConfig.netNsName
      if nsConfig.netNsName != statusLocal.netNsName:
         fingerprint = 'Kernel device for %s is not in correct network namespace'
         data = intfName
         raise VerifyException( fingerprint, data )
   elif statusLocal.netNsName not in [ 'default', '' ]:
      fingerprint = 'Kernel device for %s is not in correct network namespace'
      data = intfName
      raise VerifyException( fingerprint, data )

   if not devicePresent( devName, nsName ):
      fingerprint = 'Interface %s kernel device is not present'
      data = intfName
      raise VerifyException( fingerprint, data )
   if not deviceEnabled( devName, nsName ):
      fingerprint = 'Interface %s kernel device is not up'
      data = intfName
      raise VerifyException( fingerprint, data )

   if 'vlan' not in intfName.lower() and '.' not in intfName:
      return 

   macAddr = config.addr if config.addr != '00:00:00:00:00:00' else \
                            bridgingConfig.bridgeMacAddr
   if status.addr != macAddr:
      fingerprint = 'Interface %s status mac address %s is incorrect, ' \
                     'config.addr=%s, bridgeMacAddr=%s'
      data = ( intfName, status.addr, config.addr, bridgingConfig.bridgeMacAddr )
      raise VerifyException( fingerprint, data )
   devMacAddr = deviceMacAddr( devName, nsName )
   if devMacAddr != macAddr:
      fingerprint = 'Interface %s kernel mac address %s is incorrect, ' \
                     'config.addr=%s, status.addr=%s, bridgeMacAddr=%s'
      data = ( intfName, devMacAddr, config.addr, status.addr,
               bridgingConfig.bridgeMacAddr)
      raise VerifyException( fingerprint, data )
