# /usr/bin/env python
# Copyright (c) 2006-2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""Arastra networking testing utilities.

This module contains utilities for testing networking code.
"""

import Tac, AgentDirectory, Arnet.Device, CliTestMode
import os, re
import Cell
from Arnet.NsLib import DEFAULT_NS, runMaybeInNetNs

def hwAddrForDev( deviceName, dut=None ):
   if dut:
      try:
         ifConfigOutput = dut.run( [ 'ifconfig', deviceName ],
                                   stdout=Tac.CAPTURE )
      except Tac.SystemCommandError:
         # device name not found
         return None
   else:
      try:
         ifConfigOutput = Tac.run( [ 'ifconfig', deviceName ], stdout=Tac.CAPTURE )
      except Tac.SystemCommandError:
         # device name not found
         return None
   if ifConfigOutput:
      pattern = deviceName + ': .*?ether ([a-fA-F0-9:]+)'
      match = re.search( pattern, ifConfigOutput, re.M|re.S )
      if match:
         return match.group( 1 )
   return None

def newTestIntf( sysdbRoot, name, deviceName=None ):
   """Creates the TestIntfStatus object for interface <name>. Relies on Sysdb
   reactors to create the associated TestIntfConfig object. Returns the
   TestIntfStatus object. """
   interface = sysdbRoot[ 'interface' ]
   intfStatusDir = interface[ 'status' ][ 'test' ][ 'intf' ]
   # If you get a KeyError here, make sure that Intf-testlib is installed.
   intfStatusLocalDir = Cell.root( sysdbRoot )[ 'interface' ][ 'status' ][ 'test' ] \
       [ 'local' ]
   allIntfCounterDir = interface[ 'counter' ][ 'test' ][ 'intf' ]
   # BUG629 when creating a Tacc object with the same name but different
   # constructor arguments, since there is no harm in deleting a non-existent
   # object, do so preventively till the behavior is modified
   if intfStatusDir.has_key( name ) and \
      intfStatusDir[ name ]. genId != intfStatusDir.genIdMarker:
      del intfStatusDir[ name ]
      if name in intfStatusLocalDir:
         del intfStatusLocalDir[ name ]
   intfConfig = None
   intfStatus = intfStatusDir.newMember( name, intfConfig,
                                         intfStatusDir.genIdMarker )
   intfStatusDir.genIdMarker += 1
   intfStatusLocal = intfStatusLocalDir.newMember( name )
   if deviceName:
      intfStatus.deviceName = deviceName
      hwAddr = hwAddrForDev( deviceName )
      if hwAddr:
         ethAddr = Tac.Value( "Arnet::EthAddr" )
         ethAddr.stringValue = hwAddr
         intfStatus.eui64 = ethAddr.eui64()
   intfCounterDir = allIntfCounterDir.newMember( name )
   intfCounter = intfCounterDir.newMember( 'current' )
   return intfStatus, intfStatusLocal, intfCounter, intfConfig

def moveIntfNetNs( name, intfStatus, intfStatusLocal, netNs ):
   currNs = intfStatusLocal.netNsName
   devName = intfStatus.deviceName
   if devName:
      cmd = [ "ip", "link", "set", devName, "netns", netNs ]
      runMaybeInNetNs( currNs, cmd )
      runMaybeInNetNs( netNs, [ "ifconfig", devName, "up" ] )
   intfStatusLocal.netNsName = netNs

class TestIntf( object ):
   """Creates and caches an Interface::TestIntfStatus Entity, but does not
   create a corresponding kernel tap interface.  The TestIntfStatus entity
   is automatically deleted when this TestIntf object is destructed."""
   def __init__( self, sysdbRoot, name, deviceName=None ):
      self.system_ = sysdbRoot.parent.name
      self.intfStatus_, self.intfStatusLocal_, \
          self.intfCounter_, self.intfConfig_ = \
          newTestIntf( sysdbRoot, name, deviceName )
   def intfStatus( self ):
      return self.intfStatus_
   def intfStatusLocal( self ):
      return self.intfStatusLocal_
   def intfConfig( self ):
      return self.intfConfig_
   def intfCounter( self ):
      return self.intfCounter_
   def __del__( self ):
      # Delete the interface if Sysdb is still running
      if AgentDirectory.agent( self.system_, 'Sysdb' ):
         name = self.intfStatus_.intfId
         del self.intfStatus_.parent.intfStatus[ name ]
         del self.intfStatusLocal_.parent.intfStatusLocal[ name ]
         del self.intfCounter_.parent.parent[ name ]

class Tap( Arnet.Device.Tap ):
   def __init__( self, name, sysdbRoot, deviceName, netNs=DEFAULT_NS,
                 ip6Enabled=True, dadEnabled=True ):
      """Creates a kernel tap interface (Arnet.Device.Tap) and a corresponding
      Interface::TestIntfStatus.  Enables the kernel interface."""
      Arnet.Device.Tap.__init__( self, deviceName, netNs=netNs,
                                 ip6Enabled=ip6Enabled, dadEnabled=dadEnabled )
      self.intfStatus_, self.intfStatusLocal_, self.intfCounter_, \
          self.intfConfig_ = newTestIntf( sysdbRoot, name, deviceName )
      self.intfStatusLocal_.netNsName = netNs
      self.ifconfig( "up" )

def cleanIpRules():
   sp = os.popen( "ip rule list" )
   r = re.compile(".*(iif \S+|from [\d\.]+)")
   for line in sp:
      m = r.match( line )
      if m:
         args = ['ip', 'rule', 'del'] + m.group(1).split()
         try:
            Tac.run( args, stdout=Tac.CAPTURE, asRoot=True )
         except Tac.SystemCommandError:
            # There is a race where 'ip rule list' can show a route,
            # but this command can fail because the route has gone
            # away.
            pass

# This following is a base class for creating standalone interfaces
# with status/config/counter sysdb objects and optionally create 
# corresponding TAP devices.
# @see Ebra/test/PhyIntfTestLib.py for creating PhyIntf sysdb objects.

# from abc import ABCMeta, abstractmethod # - If we ever get python2.6

class BaseIntfHelper( object ):
   # __metaclass__ = ABCMeta # - python2.6
   
   def __init__( self, em, cli ):
      assert em      
      self.em_ = em
      self.taps_ = {}
      self.tapnames_ = {}
      self.intfs_ = {}
      self.primaryIpAddr_ = {}
      self.cc_ = cli
      self.mounted_ = False
      self.statusDir = None
      self.statusLocalDir = None
      self.configDir = None
      self.configReqDir = None
      self.counterDir = None
 
   # @abstractmethod # - python2.6
   def _mountStatusDir( self, _mountGroup ):
      raise Exception( "Derived class did not implement _mountStatusDir" )

   def _mountStatusLocalDir( self, _mountGroup ):
      raise Exception( "Derived class did not implement _mountStatusLocalDir" )

   # @abstractmethod # - python2.6
   def _mountConfigDir( self, _mountGroup ):
      raise Exception( "Derived class did not implement _mountConfigDir" )

   # @abstractmethod # - python2.6
   def _mountConfigReqDir( self, _mountGroup ):
      raise Exception( "Derived class did not implement _mountConfigDir" )

   # @abstractmethod # - python2.6
   def _mountCounterDir( self, _mountGroup ):
      raise Exception( "Derived class did not implement _mountCounterDir" )

   def mount( self ):
      if not self.mounted_:
         mg = self.em_.mountGroup()         
         self.statusDir = self._mountStatusDir( mg )
         self.statusLocalDir = self._mountStatusLocalDir( mg )
         self.configDir = self._mountConfigDir( mg )
         self.configReqDir = self._mountConfigReqDir( mg )
         self.counterDir = self._mountCounterDir( mg )         
         mg.close( blocking=True )
         self.mounted_ = True
      return self.em_

   def tap( self, name ):
      return self.taps_.get( name )

   def createTapDevice( self, name, withTap ):
      self.mount()
      if withTap:
         if type(withTap) is str:
            tapname = withTap
            tap1 = None
         else:
            tap1 = Arnet.Device.Tap()
            tapname = tap1.name
         self.taps_[name] =  tap1
         self.tapnames_[name] =  tapname
      else:
         tapname = None
      return tapname
   
   def testIntfIpIs( self, name, ip, delete=False, secondary=False, 
                     netNs=DEFAULT_NS ):
      cc = self.cc_
      if cc:
         cc.gotoMode( CliTestMode.configIfMode, name )
         cc.runCmd( "%sip address %s %s" % ( "no " if delete else "", ip, 
                                             "secondary" if secondary else "" ) )
      tap1 = self.tapnames_.get(name)
      if tap1:
         if not secondary:
            if not delete:
               # we are setting a new primary IP, delete the old one first
               if self.primaryIpAddr_.has_key( name ):
                  runMaybeInNetNs( netNs, ['ip', 'addr', 
                                           'del', self.primaryIpAddr_[ name ],
                                           'broadcast', '+', 'dev', tap1],
                                   stdout=Tac.DISCARD )
               self.primaryIpAddr_[ name ] = ip
            elif self.primaryIpAddr_.has_key( name ):
               del self.primaryIpAddr_[ name ]
         runMaybeInNetNs( netNs, [ 'ip', 'addr', 'del' if delete else 'add', ip,
                          'broadcast', '+', 'dev', tap1 ],
                          stdout=Tac.CAPTURE )
         runMaybeInNetNs( netNs, [ 'ifconfig', tap1, 'up' ], stdout=Tac.CAPTURE )

   def testIntfIp6Is( self, name, ip6Addr=None, ip6Enable=True, delete=False,
                      netNs=DEFAULT_NS ):
      cc = self.cc_
      if cc:
         cc.gotoMode( CliTestMode.configIfMode, name )
         if ip6Enable:
            cc.runCmd( "ipv6 enable" )
         if ip6Addr:
            cc.runCmd( "%sipv6 address %s" % \
                        ( "no " if delete else "", ip6Addr ) )
      tap1 = self.tapnames_.get(name)
      if tap1:
         # We need to enable Ipv6 to configure ipv6 addr or even to
         # just enable ipv6 on the intf
         if ip6Enable or ( ip6Addr and not delete ):
            enableIpv6 = True
         else:
            enableIpv6 = False
         # Enable Ipv6 on the Tap interface
         cmdStr = 'echo %u > /proc/sys/net/ipv6/conf/%s/disable_ipv6' % \
                        ( ( 0 if enableIpv6 else 1 ), tap1 )
         runMaybeInNetNs( netNs, [ 'sh', '-c', cmdStr ], asRoot=True )

         # Configure Ip6 addr
         if ip6Addr:
            runMaybeInNetNs( netNs, [ 'ip', '-6', 'addr', 'del' if delete else 'add',
                       str( ip6Addr ),
                       'dev', tap1 ],
                     stdout=Tac.CAPTURE )
         runMaybeInNetNs( netNs, [ 'ifconfig', tap1, 'up' ], stdout=Tac.CAPTURE )

# The TestIntfHelper is derived from BaseIntfHelper and specifies
# the creation of TestIntf sysdb objects.
class TestIntfHelper( BaseIntfHelper ):
   def __init__( self, entMan, cli=None ):
      BaseIntfHelper.__init__( self, entMan, cli )                               

   def _mountStatusDir( self, mountGroup ):
      return mountGroup.mount( 'interface/status/test/intf',
                               "Interface::TestIntfStatusDir", "w" )
   def _mountStatusLocalDir( self, mountGroup ):
      return mountGroup.mount( Cell.path( 'interface/status/test/local' ),
                               "Interface::TestIntfStatusLocalDir", "w" )
   def _mountConfigDir( self, mountGroup ):
      return mountGroup.mount( 'interface/config/test/intf',
                               "Interface::TestIntfConfigDir", "w" )
   def _mountConfigReqDir( self, mountGroup ):
      return None

   def _mountCounterDir( self, mountGroup ):
      return mountGroup.mount( 'interface/counter/test/intf',
                               "Interface::AllTestIntfCounterDir", "w" )

   def moveIntfNetNs( self, name, netNs ):
      intfStatusLocal = self.intfs_[ name ].intfStatusLocal()
      intfStatus = self.intfs_[ name ].intfStatus()
      moveIntfNetNs( name, intfStatus, intfStatusLocal, netNs )

   def testIntfIs( self, name, withTap=True, ip=None,
                   ip6Addr=None, ip6Enable=False, netNs=DEFAULT_NS ):
      ''' Create a new TestIntf with an optional TAP device and IP addr.
      Returns a tuple of TestIntfStatus/Config/Counter objects.
      Note: intfConfig is currently NoneType. '''
      tapname = self.createTapDevice( name, withTap )
      t1 = TestIntf( self.em_.root(), name, tapname )
      self.intfs_[ name ] = t1
      if netNs != DEFAULT_NS:
         self.moveIntfNetNs( name, netNs )
         if self.tap( name ):
            self.tap( name ).netNsIs( netNs )
      if ip:
         self.testIntfIpIs( name, ip, netNs=netNs )
      # Use a default MTU > IPV6 min MTU
      t1.intfStatus().mtu = 1500
      if ip6Addr or ip6Enable:
         self.testIntfIp6Is( name, ip6Addr=ip6Addr, ip6Enable=ip6Enable, 
                             netNs=netNs )
      # Use a default MTU > IPV6 min MTU
      t1.intfStatus().mtu = 1500
      t1.intfStatus().operStatus = 1
      return ( t1.intfStatus(), t1.intfConfig(), t1.intfCounter() )

   def testIntfStatusLocal( self, name ):
      assert name in self.intfs_
      t1 = self.intfs_[ name ]
      return t1.intfStatusLocal()

   def deleteIntf( self, name ):
      assert name in self.intfs_
      del self.intfs_[ name ]
